/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <asm/irq.h>
#include <asm/atomic.h>
#include <linux/kernel.h>
#include <linux/jiffies.h>
#include <linux/completion.h>
#include <linux/semaphore.h>
#include <linux/kthread.h>
#include <linux/sonos_kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/timer.h>
#include <linux/time.h>
#include <linux/rtc.h>
#include <linux/list.h>
#include <linux/workqueue.h>
#include "mdp.h"
#include <asm/uaccess.h>
#include <linux/i2c.h>
#include <linux/delay.h>
#include <linux/crc32.h>
#include <linux/of_gpio.h>
#include <linux/of.h>
#include <linux/kconfig.h>
#include <linux/slab.h>
#include <linux/firmware.h>
#include "blackbox.h"
#include "sdd.h"
#include "sge_common.h"
#include "captouch_sim.h"
#include "sdd_hal.h"
#include "sonos_lock.h"
#include "sonos_thread_priorities.h"
#include "sdd_rt_pmux.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#if defined(CONFIG_SONOS_SDD_RTC)
#include "sdd_rtc_api.h"
#endif

#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX7ULP)
int event_queue_send_event(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
    return 0;
}
#endif

#define LOCK_I2C(adapter)	i2c_lock_adapter(adapter)
#define UNLOCK_I2C(adapter)	i2c_unlock_adapter(adapter)

struct psoc_image_info {
    u8 siliconid[12];
    u8 checksum[2];
};

struct psoc_image_version {
	u8 ident;
	u16 version;
};

struct psoc_image_version psoc_image_version_map[] = {
	{.ident = SGE_IDENT_ALPINE,            .version = SGE_VERSION_ALPINE},
	{.ident = SGE_IDENT_APOLLO,            .version = SGE_VERSION_APOLLO},
	{.ident = SGE_IDENT_APOLLO_FNI_BL483,  .version = SGE_VERSION_APOLLO_FNI_BL483},
	{.ident = SGE_IDENT_BOOTLEG_4100,      .version = SGE_VERSION_BOOTLEG_4100},
	{.ident = SGE_IDENT_BOOTLEG_4200,      .version = SGE_VERSION_BOOTLEG_4200},
	{.ident = SGE_IDENT_DOMINO,            .version = SGE_VERSION_DOMINO},
	{.ident = SGE_IDENT_ELREY,             .version = SGE_VERSION_ELREY},
	{.ident = SGE_IDENT_BRAVO,             .version = SGE_VERSION_BRAVO},
	{.ident = SGE_IDENT_ENCORE,            .version = SGE_VERSION_ENCORE},
	{.ident = SGE_IDENT_FURY,              .version = SGE_VERSION_FURY},
	{.ident = SGE_IDENT_FURY_P1,           .version = SGE_VERSION_FURY_P1},
	{.ident = SGE_IDENT_HIDEOUT,           .version = SGE_VERSION_HIDEOUT},
	{.ident = SGE_IDENT_KAPITAL,           .version = SGE_VERSION_KAPITAL},
	{.ident = SGE_IDENT_KAPITAL_FNI_BL553, .version = SGE_VERSION_KAPITAL_FNI_BL553},
	{.ident = SGE_IDENT_NEPTUNE,           .version = SGE_VERSION_NEPTUNE},
	{.ident = SGE_IDENT_NEPTUNE_4247,      .version = SGE_VERSION_NEPTUNE_4247},
	{.ident = SGE_IDENT_TITAN,             .version = SGE_VERSION_TITAN},
	{.ident = SGE_IDENT_TUPELO,            .version = SGE_VERSION_TUPELO}
};

#define SDD_I2C_MAX_RETRIES  12

#define SGE_RECOVERY_TIME	(60*1000)
#define SGE_MAX_STUCK_COUNT  4
#define SGE_MONITOR_TIME	 (SGE_RECOVERY_TIME/SGE_MAX_STUCK_COUNT)
#define SDD_I2C_BLOCK_SIZE 16

#define SDD_CRC32_CALC_TIME_MS  2010

sdd_gesture_match_t sdd_gesture_match_default = {0};

sdd_data_t *Sdd_data[SDD_MAX_PSOC];
int sdd_psoc_inst = 0;

static inline void sdd_free_irq(sdd_data_t *psd)
{
	if (gpio_is_valid(psd->gpio_table[SCI_INDEX_GPIO_IRQ].gpio)) {
		free_irq(psd->irq, psd);
	}
}

static void sdd_fatal_error_report(sdd_data_t *psd, int reg, int bytes, char *op)
{
	if (psd->sge_init_done) {
		if (op == NULL) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "fatal error, disabling sigdetail");
		} else {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "fatal error %s 0x%02x bytes from register 0x%02x, disabling sigdetail",
				   op, bytes, reg);
		}
		if (gpio_is_valid(psd->gpio_table[SCI_INDEX_GPIO_IRQ].gpio)) {
			disable_irq(psd->irq);
		}
	}
	psd->disabled_by_fatal_error = 1;
}

static void sdd_fatal_error(sdd_data_t *psd)
{
	sdd_fatal_error_report(psd, 0, 0, NULL);
}

static int sdd_get_image_version(int ident)
{
	int i;

	for (i = 0; i < (sizeof(psoc_image_version_map) / sizeof(struct psoc_image_version)); i++) {
		if (psoc_image_version_map[i].ident == ident) {
			return psoc_image_version_map[i].version;
		}
	}

	return -1;
}

static void sdd_i2c_retry_delay(sdd_data_t *psd)
{
	psd->i2c_retry_count++;
	msleep_interruptible(1);
}

static int __sdd_write_sge_block(sdd_data_t *psd, u8 reg, u8 len, const u8 *buf)
{
	s32 bytes;
	int error=0;

	psd->i2c_write_count += (len * sizeof(u8));
	bytes = i2c_smbus_write_i2c_block_data(psd->i2c_client, reg, len, buf);
	if (bytes < 0) {
		psd->i2c_error_count++;
		error = -EIO;
	}
	return error;
}

static int __sdd_read_sge_block(sdd_data_t *psd, u8 reg, u8 len, u8 *buf)
{
	s32 bytes;
	int error=0;

	psd->i2c_read_count += 4;
	bytes = i2c_smbus_read_i2c_block_data(psd->i2c_client, reg, len, buf);
	if (bytes < 0) {
		psd->i2c_error_count++;
		error = -EIO;
	}
	return error;
}

int sdd_write_sge_block(sdd_data_t *psd, u8 reg, u8 len, const u8 *buf)
{
	int error;
	int retries = SDD_I2C_MAX_RETRIES;

	if (psd->disabled_by_fatal_error) {
		return 0;
	}

	while (retries-- > 0) {
		error = __sdd_write_sge_block(psd, reg, len, buf);
		if (!error) {
			return 0;
		}
		sdd_i2c_retry_delay(psd);
	}
	sdd_fatal_error_report(psd, reg, len, "write");
	return error;
}

int sdd_read_sge_block(sdd_data_t *psd, u8 reg, u8 len, u8 *buf)
{
	int error;
	int retries = SDD_I2C_MAX_RETRIES;

	if (psd->disabled_by_fatal_error) {
		memset(buf, 0, len);
		return 0;
	}

	while (retries-- > 0) {
		error = __sdd_read_sge_block(psd, reg, len, buf);
		if (!error) {
			return 0;
		}
		sdd_i2c_retry_delay(psd);
	}
	sdd_fatal_error_report(psd, reg, len, "read");
	return error;
}

static int __sdd_read_sge_word(sdd_data_t *psd, u8 reg, u16 *word)
{
	s32 ret;

	psd->i2c_read_count += sizeof(*word);
	ret = i2c_smbus_read_word_data(psd->i2c_client, reg);
	if (ret < 0) {
		psd->i2c_error_count++;
		return ret;
	}
	*word = ret;
	return 0;
}

int sdd_read_sge_word(sdd_data_t *psd, u8 reg, u16 *word)
{
	int error;
	int retries = SDD_I2C_MAX_RETRIES;

	if (psd->disabled_by_fatal_error) {
		*word = 0;
		return 0;
	}

	while (retries-- > 0) {
		error = __sdd_read_sge_word(psd, reg, word);
		if (!error) {
			return 0;
		}
		sdd_i2c_retry_delay(psd);
	}
	sdd_fatal_error_report(psd, reg, 2, "read");
	return error;
}

static int __sdd_read_sge_byte(sdd_data_t *psd, u8 reg, u8 *byte)
{
	s32 ret;

	psd->i2c_read_count++;
	ret = i2c_smbus_read_byte_data(psd->i2c_client, reg);
	if (ret < 0) {
		psd->i2c_error_count++;
		return ret;
	}
	*byte = ret;
	return 0;
}

int sdd_read_sge_byte(sdd_data_t *psd, u8 reg, u8 *byte)
{
	int error;
	int retries = SDD_I2C_MAX_RETRIES;

	if (psd->disabled_by_fatal_error) {
		*byte = 0;
		return 0;
	}

	while (retries-- > 0) {
		error = __sdd_read_sge_byte(psd, reg, byte);
		if (!error) {
			return 0;
		}
		sdd_i2c_retry_delay(psd);
	}
	sdd_fatal_error_report(psd, reg, 1, "read");
	return error;
}

static int __sdd_write_sge_byte(sdd_data_t *psd, u8 reg, u8 byte)
{
	s32 ret;

	psd->i2c_write_count++;
	ret = i2c_smbus_write_byte_data(psd->i2c_client, reg, byte);
	if (ret < 0) {
		psd->i2c_error_count++;
		return ret;
	}
	return 0;
}

int sdd_write_sge_byte(sdd_data_t *psd, u8 reg, u8 byte)
{
	int error;
	int retries = SDD_I2C_MAX_RETRIES;

	if (psd->disabled_by_fatal_error) {
		return 0;
	}

	while (retries-- > 0) {
		error = __sdd_write_sge_byte(psd, reg, byte);
		if (!error) {
			return 0;
		}
		sdd_i2c_retry_delay(psd);
	}
	sdd_fatal_error_report(psd, reg, 1, "write");
	return error;
}

static int __sdd_write_sge_word(sdd_data_t *psd, u8 reg, u16 word)
{
	s32 ret;

	psd->i2c_write_count += sizeof(word);
	ret = i2c_smbus_write_word_data(psd->i2c_client, reg, word);
	if (ret < 0) {
		psd->i2c_error_count++;
		return ret;
	}
	return 0;
}

int sdd_write_sge_word(sdd_data_t *psd, u8 reg, u16 word)
{
	int error;
	int retries = SDD_I2C_MAX_RETRIES;

	if (psd->disabled_by_fatal_error) {
		return 0;
	}

	while (retries-- > 0) {
		error = __sdd_write_sge_word(psd, reg, word);
		if (!error) {
			return 0;
		}
		sdd_i2c_retry_delay(psd);
	}
	sdd_fatal_error_report(psd, reg, 2, "write");
	return error;
}

static int __sdd_rmw_sge_byte(sdd_data_t *psd, u8 reg, u8 mask, u8 byte)
{
	u8  value;
	s32 ret;

	ret = __sdd_read_sge_byte(psd, reg, &value);
	if (ret < 0) {
		return ret;
	}
	value &= ~mask;
	value |= byte;
	return __sdd_write_sge_byte(psd, reg, value);
}

int sdd_rmw_sge_byte(sdd_data_t *psd, u8 reg, u8 mask, u8 byte)
{
	int error;
	int retries = SDD_I2C_MAX_RETRIES;

	if (psd->disabled_by_fatal_error) {
		return 0;
	}

	while (retries-- > 0) {
		error = __sdd_rmw_sge_byte(psd, reg, mask, byte);
		if (!error) {
			return 0;
		}
		sdd_i2c_retry_delay(psd);
	}
	sdd_fatal_error_report(psd, reg, 1, "read-modify-write");
	return error;
}

int sdd_wait_for_ind_ready(sdd_data_t *psd)
{
	u16 status = 0;
	int i, error, max=600;

	for (i = 0; i < max; i++) {
		error = sdd_read_sge_word(psd, SGE_REG_STATUS, &status);
		if (!error && (status & SGE_REG_STATUS_IND_READY)) {
			return 0;
		}
		mdelay(10);
	}

	bb_log(BB_MOD_SDD, BB_LVL_ERR, "Timeout waiting for indirect registers to be ready, status reg=0x%x, error=%d", status, error);
	return -1;
}

static int sdd_get_features(sdd_data_t *psd)
{
	int error;
	u16 features;

	error = sdd_read_sge_word(psd, SGE_REG_FEATURES, &features);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SGE status could not be determined, code %d", error);
		return -1;
	}
	psd->psoc_features = features;

	return 0;
}

static int sdd_verify_psoc(sdd_data_t *psd)
{
	u16 status_reg = 0;
	int error;
	unsigned int size;
	char *start;

	error = sdd_get_features(psd);
	if (error) {
		return error;
	}

	if (psd->psoc_features & SGE_REG_FEATURES_CRC32) {
		u32 calc_crc, read_crc;
		int i, max=12000;

		start = GetPsocFlashImage(psd->psoc_cfg, &size);

		calc_crc = crc32(0xffffffff, start, size) ^ 0xffffffff;

		mdelay(SDD_CRC32_CALC_TIME_MS);
		for (i = 0; i < max; i++) {
			udelay(1);
			error = sdd_read_sge_word(psd, SGE_REG_STATUS, &status_reg);
			if (!error && (status_reg & SGE_REG_STATUS_CRC32_READY)) {
				break;
			}
		}

		if (i >= max) {
			bb_log(BB_MOD_SDD, BB_LVL_WARNING, "CRC32 timeout");
			return 1;
		}

		error = sdd_ind_get_crc32(psd, &read_crc);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "CRC32 read error %d", error);
		} else {
			if (calc_crc != read_crc) {
				error = 1;
				bb_log(BB_MOD_SDD, BB_LVL_WARNING, "CRC32 mismatch: calc %08x, read %08x", calc_crc, read_crc);
			}
		}
	} else {
		LOCK_I2C(psd->i2c_client->adapter);
		error = VerifyPsocFlash(psd->psoc_cfg);
		UNLOCK_I2C(psd->i2c_client->adapter);
	}

	return error;
}

static int sdd_erase_psoc(sdd_data_t *psd)
{
	int error;
	error = ErasePsocFlash(psd->psoc_cfg);
	return error;
}

static void sdd_program_psoc(sdd_data_t *psd, int cmdprog)
{
	int error;

	if (!cmdprog && (psd->dts_config_flags & SDD_DTS_CHECK_POWER)) {
		int rc = sdd_psoc_program_check();
		if (rc != 1) {
			if (rc == 0) {
				bb_log(BB_MOD_SDD, BB_LVL_WARNING,
				       "psoc%d program skipped, USB-C power required", psd->inst);
				return;
			} else {
				bb_log(BB_MOD_SDD, BB_LVL_WARNING,
				       "psoc%d program skipped, can't determine power state, rc %d", psd->inst, rc);
				return;
			}
		}
	}
	if (psd->dts_config_flags & SDD_DTS_RPMSG) {
		sdd_disable_psoc_interrupt(psd);
	}
	if (psd->dts_config_flags & SDD_DTS_SHARE_SWD_I2C) {
		sdd_psoc_program_notify(1);
		mdelay(250);
	}

	LOCK_I2C(psd->i2c_client->adapter);
	error = ProgramPsocFlash(psd->psoc_cfg);
	if (!error) {
		psd->psoc_erased = 0;
	}
	UNLOCK_I2C(psd->i2c_client->adapter);
	printk("PSoC%d program %s\n", psd->inst, error ? "failed" : "success");

	if (psd->dts_config_flags & SDD_DTS_SHARE_SWD_I2C) {
		sdd_psoc_program_notify(0);
	}
	if (psd->dts_config_flags & SDD_DTS_RPMSG) {
		sdd_enable_psoc_interrupt(psd);
	}
}

int sdd_show(struct seq_file *m, void *v)
{
	sdd_data_t *psd = (sdd_data_t *)(m->private);
	u16 status;
	int error;

	seq_printf(m, "\nConfig\n");
	seq_printf(m, "  Touch: %s\n", psd->touch_enable ? "enabled" : "disabled");
	seq_printf(m, "  Press Hold Time: %u\n", psd->press_hold_time);
	seq_printf(m, "  Repeat Time: %u\n", psd->repeat_time);
	if (psd->touch_timer != NULL) {
		seq_printf(m, "  Touch Time: %u\n", psd->touch_timer->timeout_ms);
	}
	seq_printf(m, "  Finger Threshold: %u\n", psd->finger_threshold);
	seq_printf(m, "  Noise Threshold: %u\n", psd->noise_threshold);
	seq_printf(m, "  Finger Hysteresis: %u\n", psd->finger_hysteresis);
	seq_printf(m, "  Move Interval: %u\n", psd->move_interval);
	seq_printf(m, "  PSoC Low Power: %ssupported\n", (psd->dts_config_flags & SDD_DTS_LOW_POWER) ? "" : "NOT ");
	seq_printf(m, "  BLE: %s\n", psd->btle_enabled ? "enabled" : "disabled");
	seq_printf(m, "  BLE Carrier Wave Transmit: %s\n", psd->ble_carrier_tx_enabled ? "enabled" : "disabled");

	error = sdd_read_sge_word(psd, SGE_REG_STATUS, &status);
	if (!error) {
		seq_printf(m, "  HFCLK: %s\n", (status & SGE_REG_STATUS_ECO_UNCONF) ? "Misconfigured" : "Configured");
	}

	if (psd->dts_config_flags & SDD_DTS_ECO_TRIM_CAP) {
		u32 trim;
		int error = sdd_ind_get_eco_trim_cap(psd, &trim);
		if (error) {
			seq_printf(m, "  ECO trim cap: error reading\n");
		} else {
			seq_printf(m, "  ECO trim cap: %04x\n", trim);
		}
	}

	seq_printf(m, "\nStatus\n");
	seq_printf(m, "  Disabled by fatal i2c error: %s\n", psd->disabled_by_fatal_error ? "yes" : "no");
	seq_printf(m, "  SGE irq %u\n", atomic_read(&psd->sge_irq_pending));
	if (psd->psoc_features & SGE_REG_FEATURES_DIE_TEMP) {
		int8_t	die_temp;
		int	error;
		error = sdd_ind_get_die_temp(psd, &die_temp);
		if (!error) {
			seq_printf(m, "  PSoC temperature:  %d C\n", die_temp);
		} else {
			seq_printf(m, "  PSoC temperature:  error reading\n");
		}
	}

	seq_printf(m, "\nStats\n");
	seq_printf(m, "%9u passes\n", psd->passes);
	seq_printf(m, "%9u completion errors\n", psd->completion_errors);
	seq_printf(m, "%9u i2c reads\n",  psd->i2c_read_count);
	seq_printf(m, "%9u i2c writes\n", psd->i2c_write_count);
	seq_printf(m, "%9u i2c errors\n", psd->i2c_error_count);
	seq_printf(m, "%9u i2c retries\n", psd->i2c_retry_count);
	seq_printf(m, "%9u received interrupts\n", psd->irq_count);
	seq_printf(m, "%9u processed interrupts\n", psd->irq_processed);
	seq_printf(m, "%9u spurious interrupts\n", psd->irq_spurious);
	seq_printf(m, "%9u baseline updates\n", psd->baseline_update_count);
	seq_printf(m, "%9u SGE recoveries\n", psd->sge_recoveries);

	if (!list_empty(&(psd->timers))) {
		sdd_proc_read_timers(m, v);
	}
	return 0;
}

static int sdd_open(struct inode *inode, struct file *file)
{
	return single_open(file, sdd_show, PDE_DATA(inode));
}

ssize_t
sdd_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	char buf[40];
	int  match;

	if (!is_mdp_authorized(MDP_AUTH_FLAG_BUTTON_DEBUG)) {
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "captouch simulation command not authorized");
		return -EACCES;
	}

	if (count >= sizeof(buf)) {
		return -EIO;
	}
	if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	}
	buf[count] = 0;
	match = captouch_sim_process_cmd(buf);
	if (!match) {
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "captouch simulation command not supported, valid options:");
		captouch_sim_print_cmds();
	}
	return count;
}

static struct file_operations sdd_proc_operations = {
	.owner    = THIS_MODULE,
	.open     = sdd_open,
	.write    = sdd_write,
	.read     = seq_read,
	.llseek   = seq_lseek,
	.release  = single_release,
};

static int sge_show(struct seq_file *m, void *v)
{
	sdd_data_t *psd = sdd_get_data(0);
	int reg;
	u8 val;

	for (reg = 0; reg < SGE_REG_SET_SIZE; reg++) {
		sdd_read_sge_byte(psd, reg, &val);
		if ((reg % 64) == 0) {
			seq_printf(m, "\n\n");
		} else if ((reg % 16) == 0) {
			seq_printf(m, "\n");
		} else if ((reg % 8) == 0) {
			seq_printf(m, "  ");
		}
		seq_printf(m, "%02x ", val);
	}
	seq_printf(m, "\n\n");
	return 0;
}

static ssize_t
sge_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	sdd_data_t *psd = sdd_get_data(0);
	s32 temp;
	char buf[40];
	char *peq;
	int error;
	u8 val, reg;

	if (count >= sizeof(buf))
		return -EIO;
	if (copy_from_user(buf, buffer, count))
		return -EFAULT;
	buf[count] = 0;

	peq = strchr(buf, '=');
	if (peq != NULL) {
		*peq = '\0';
		error = kstrtoint(peq+1, 16, &temp);
		val = temp;
		if (strncmp(buf, "reg", 3) == 0) {
			error = kstrtoint(peq-2, 16, &temp);
			reg = temp;
			error = sdd_write_sge_byte(psd, reg, val);
			if (error) {
				bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: error %d", __func__, error);
			}
		}
	}
	return count;
}

static int sge_open(struct inode *inode, struct file *file)
{
	return single_open(file, sge_show, PDE_DATA(inode));
}

static struct file_operations sge_proc_operations = {
	.owner    = THIS_MODULE,
	.open     = sge_open,
	.write    = sge_write,
	.read     = seq_read,
	.llseek   = seq_lseek,
	.release  = single_release,
};

void sdd_proc_init(void)
{
#ifdef CONFIG_PROC_FS
	sdd_data_t *psd = sdd_get_data(0);
	int error;
	struct proc_dir_entry *entry;

	if (psd->inst != 0) {
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "proc interface not supported for PSoC inst %d", psd->inst);
		return;
	}

	entry = proc_create_data("driver/sdd", 0666, NULL, &sdd_proc_operations, psd);
	if (entry == NULL) {
		return;
	}
	entry = proc_create("driver/sge", 0666, NULL, &sge_proc_operations);
	if (entry == NULL) {
		return;
	}

	error = sdd_ind_init(psd);
	if (error) {
		return;
	}
	if (!(psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH)) {
		error = sdd_rec_init(psd);
		if (error) {
			return;
		}
		psd->grr_event_log = sdd_init_log(psd);
	}
#if defined(CONFIG_SONOS_SDD_RTC)
	if (psd->dts_config_flags & SDD_DTS_PSOC_RTC) {
		error = sdd_rtc_proc_init();
		if (error) {
			return;
		}
	}
#endif
	error = sddcmd_init();
	if (psd->gesture_match == NULL) {
		psd->gesture_match = &sdd_gesture_match_default;
	}
	captouch_sim_init(NULL, psd->gesture_match);
#endif

}

void sdd_proc_remove(void)
{
#ifdef CONFIG_PROC_FS
	sdd_data_t *psd = sdd_get_data(0);
#if defined(CONFIG_SONOS_SDD_RTC)
	if (psd->dts_config_flags & SDD_DTS_PSOC_RTC) {
		sdd_rtc_remove();
	}
#endif
	remove_proc_entry("driver/sdd", NULL);
	remove_proc_entry("driver/sge", NULL);

	sdd_ind_remove();
	if (!(psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH)) {
		sdd_rec_remove();
		sdd_log_remove();
	}
	sddcmd_remove();
#endif
}

int sdd_set_btle(sdd_data_t *psd, int enable)
{
	s32 ret;
	u8 val = enable ? SGE_REG_CR0_BTLE_ENABLE : 0;
	ret = sdd_rmw_sge_byte(psd, SGE_REG_CR0, SGE_REG_CR0_BTLE_ENABLE, val);
	if (!ret) {
		psd->btle_enabled = enable ? 1 : 0;
	}
	return ret;
}

static inline int sdd_capsense(sdd_data_t *psd, int enable)
{
	u8 val = enable ? SGE_REG_CR0_CAPSENSE_ENABLE : 0;
	return sdd_rmw_sge_byte(psd, SGE_REG_CR0, SGE_REG_CR0_CAPSENSE_ENABLE, val);
}

irqreturn_t sdd_isr(int irq, void *p)
{
	sdd_data_t *psd = (sdd_data_t *)p;
	int ret = IRQ_NONE;
	static int sdd_unknown_irq = 0;

	if (irq == psd->irq) {
		psd->intr_jiffies = jiffies;
		disable_irq_nosync(irq);
		atomic_inc(&psd->sge_irq_pending);
		complete(&psd->sdd_completion);
		psd->irq_count++;
		ret = IRQ_HANDLED;
	} else {
		if (sdd_unknown_irq++ < 10) {
			bb_log(BB_MOD_SDD, BB_LVL_WARNING, "%s: unknown irq %i", __func__, irq);
		} else {
			ret = IRQ_HANDLED;
		}
	}
	return ret;
}

static int sdd_ident_match(sdd_data_t *psd, u8 ident)
{
	return (ident == psd->psoc_cfg->ident);
}

static int sdd_version_match(sdd_data_t *psd, u16 version)
{
	return (version == psd->psoc_cfg->version);
}

static void sdd_config_swd_pins(sdd_data_t *psd)
{
	int ret;
	struct pinctrl *p_ctrl;
	struct pinctrl_state *s1;

	p_ctrl = devm_pinctrl_get(&psd->i2c_client->dev);
	if (IS_ERR(p_ctrl)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to get I2C client device for pin ctrl");
		return;
	}
	s1 = pinctrl_lookup_state(p_ctrl, "mode-swd");
	if (!IS_ERR(s1)) {
		ret = pinctrl_select_state(p_ctrl, s1);
		if (ret < 0) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d: Unable to select mode-swd pin ctrl state", psd->inst);
		}
	}
}

static void sdd_disable_ble_uart(sdd_data_t *psd)
{
	int ret;
	struct pinctrl *p_ctrl;
	struct pinctrl_state *s1;

	p_ctrl = devm_pinctrl_get(&psd->i2c_client->dev);
	if (IS_ERR(p_ctrl)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to get I2C client device for pin ctrl");
	} else {
		s1 = pinctrl_lookup_state(p_ctrl, "mode-uart-hiz");
		if (!IS_ERR(s1)) {
			ret = pinctrl_select_state(p_ctrl, s1);
			if (ret < 0) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to select mode-uart-hiz pin ctrl state");
			}
		}

		s1 = pinctrl_lookup_state(p_ctrl, "mode-swd");
		if (!IS_ERR(s1)) {
			ret = pinctrl_select_state(p_ctrl, s1);
			if (ret < 0) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d: Unable to select mode-swd pin ctrl state", psd->inst);
			}
		}
	}

	sdd_rt_pmux_disable_uart();

	if (psd->dts_config_flags & SDD_DTS_SHARE_SWD_UART)
	{
		gpio_request_one(psd->gpio_table[SCI_INDEX_GPIO_IO].gpio,
				psd->gpio_table[SCI_INDEX_GPIO_IO].flags,
				psd->gpio_table[SCI_INDEX_GPIO_IO].label);
		gpio_request_one(psd->gpio_table[SCI_INDEX_GPIO_SCL].gpio,
				psd->gpio_table[SCI_INDEX_GPIO_SCL].flags,
				psd->gpio_table[SCI_INDEX_GPIO_SCL].label);
	}
}

static void sdd_enable_ble_uart(sdd_data_t *psd)
{
	int ret, error;
	struct pinctrl *p_ctrl;
	struct pinctrl_state *s1;

	if (psd->dts_config_flags & SDD_DTS_SHARE_SWD_UART)
	{
		gpio_free(psd->gpio_table[SCI_INDEX_GPIO_IO].gpio);
		gpio_free(psd->gpio_table[SCI_INDEX_GPIO_SCL].gpio);
	}

	sdd_rt_pmux_enable_uart();

	p_ctrl = devm_pinctrl_get(&psd->i2c_client->dev);
	if (IS_ERR(p_ctrl)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to get I2C client device for pin ctrl");
	} else {
		s1 = pinctrl_lookup_state(p_ctrl, "mode-flow-ctrl");
		if (!IS_ERR(s1)) {
			ret = pinctrl_select_state(p_ctrl, s1);
			if (ret < 0) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to select mode-flow-ctrl pin ctrl state");
			} else {
				error = sdd_set_btle(psd, 1);
				if (error) {
					bb_log(BB_MOD_SDD, BB_LVL_ERR, "SGE Unable to set Bluetooth LE enable flag");
				}
			}
		}
	}
}

static int sdd_get_gpio(sdd_data_t *psd, int signal)
{
	return psd->gpio_table[signal].gpio;
}

static void sdd_psoc_reset(sdd_data_t *psd)
{
	struct gpio *pgpio = &psd->gpio_table[SCI_INDEX_GPIO_RST];
	int val = (pgpio->flags == GPIOF_OUT_INIT_HIGH) ? 1 : 0;
	gpio_direction_output(pgpio->gpio, val);
}

static void sdd_psoc_unreset(sdd_data_t *psd)
{
	struct gpio *pgpio = &psd->gpio_table[SCI_INDEX_GPIO_RST];
	int val = (pgpio->flags == GPIOF_OUT_INIT_HIGH) ? 0 : 1;
	gpio_direction_output(pgpio->gpio, val);
}

static int sdd_load_bin(sdd_data_t *psd)
{
	const struct firmware *fw = NULL;
	struct psoc_image_info *img_info;
	u8 silicon_id[SWD_SILICON_ID_BYTE_LENGTH], checksum[SWD_CHECKSUM_BYTE_LENGTH];
	char img_name[20];
	int ret, datalen;

	sprintf(img_name, "psoc-%02x.bin", psd->psoc_cfg->ident);

	ret = request_firmware_direct(&fw, img_name, &psd->i2c_client->dev);
	if (ret) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Failed to load PSoC image %s, error=%d", img_name, ret);
		return -1;
	}

	datalen = fw->size - sizeof(struct psoc_image_info);
	img_info = (struct psoc_image_info *)(fw->data + datalen);
	checksum[0] = img_info->checksum[1];
	checksum[1] = img_info->checksum[0];
	memcpy(silicon_id, &img_info->siliconid[2], SWD_SILICON_ID_BYTE_LENGTH);

	ret = SetPsocImage(psd->psoc_cfg, checksum, silicon_id, fw->data, datalen);
	release_firmware(fw);

	return ret;
}

static int sdd_select_image(sdd_data_t *psd, int ident)
{
	int i, error, ret, found = 0;

	psd->psoc_cfg = &psd->psoc_cfgs[0];

	if (psd->num_psoc_cfgs > 1) {
		if (ident != SGE_IDENT_INVALID) {
			for (i = 0; i < psd->num_psoc_cfgs; i++) {
				if (psd->psoc_cfgs[i].ident == ident) {
					psd->psoc_cfg = &psd->psoc_cfgs[i];
					found = 1;
					break;
				}
			}
			if (!found) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "Read PSoC ident 0x%x not found", ident);
				return -1;
			}
		} else {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "determine PSoC image");
			psd->psoc_cfg = &psd->psoc_cfgs[1];
			error = sdd_load_bin(psd);
			if (error) {
				return error;
			}
			ret = PsocDeviceMatch(psd->psoc_cfg);
			if (ret) {
				bb_log(BB_MOD_SDD, BB_LVL_INFO, "PSoC image 1 selected");
				return 0;
			} else {
				ReleasePsocImage(psd->psoc_cfg);
				psd->psoc_cfg = &psd->psoc_cfgs[0];
				bb_log(BB_MOD_SDD, BB_LVL_INFO, "PSoC image 0 selected");
			}
		}
	}

	error = sdd_load_bin(psd);
	if (error) {
		return error;
	}
	return 0;
}

#define PSOC_PARTNAME_4127_1 "CY8C4127LQI-BL473"
#define PSOC_PARTNAME_4128_1 "CY8C4128LQI-BL543"
#define PSOC_PARTNAME_4146_1 "CY8C4146LQI-S433"
#define PSOC_PARTNAME_4245_1 "CY8C4245LQI-483"
#define PSOC_PARTNAME_4247_1 "CY8C4247LQI-BL453"
#define PSOC_PARTNAME_4247_2 "CY8C4247FNI-BL483"
#define PSOC_PARTNAME_4248_1 "CY8C4248LQI-BL553"
#define PSOC_PARTNAME_4248_2 "CY8C4248FNI-BL553"

struct psoc_part_name {
	u8 ident;
	char *name;
};

struct psoc_part_name psoc_part_names[] = {
	{.ident = SGE_IDENT_ALPINE,            .name = PSOC_PARTNAME_4247_1},
	{.ident = SGE_IDENT_APOLLO,            .name = PSOC_PARTNAME_4247_1},
	{.ident = SGE_IDENT_APOLLO_FNI_BL483,  .name = PSOC_PARTNAME_4247_2},
	{.ident = SGE_IDENT_BOOTLEG_4100,      .name = PSOC_PARTNAME_4146_1},
	{.ident = SGE_IDENT_BOOTLEG_4200,      .name = PSOC_PARTNAME_4245_1},
	{.ident = SGE_IDENT_DOMINO,            .name = PSOC_PARTNAME_4247_1},
	{.ident = SGE_IDENT_ELREY,             .name = PSOC_PARTNAME_4247_1},
	{.ident = SGE_IDENT_BRAVO,             .name = PSOC_PARTNAME_4248_1},
	{.ident = SGE_IDENT_ENCORE,            .name = PSOC_PARTNAME_4245_1},
	{.ident = SGE_IDENT_FURY,              .name = PSOC_PARTNAME_4248_1},
	{.ident = SGE_IDENT_FURY_P1,           .name = PSOC_PARTNAME_4248_1},
	{.ident = SGE_IDENT_HIDEOUT,           .name = PSOC_PARTNAME_4247_1},
	{.ident = SGE_IDENT_KAPITAL,           .name = PSOC_PARTNAME_4248_1},
	{.ident = SGE_IDENT_KAPITAL_FNI_BL553, .name = PSOC_PARTNAME_4248_2},
	{.ident = SGE_IDENT_NEPTUNE,           .name = PSOC_PARTNAME_4127_1},
	{.ident = SGE_IDENT_NEPTUNE_4247,      .name = PSOC_PARTNAME_4247_1},
	{.ident = SGE_IDENT_TITAN,             .name = PSOC_PARTNAME_4128_1},
	{.ident = SGE_IDENT_TUPELO,            .name = PSOC_PARTNAME_4247_1},
};
#define PSOC_PARTNAME_TABLE_SIZE (sizeof(psoc_part_names) / sizeof(struct psoc_part_name))

static char * sdd_psoc_partname(u8 ident)
{
	int i;
	for (i = 0; i < PSOC_PARTNAME_TABLE_SIZE; i++) {
		struct psoc_part_name *pn = &psoc_part_names[i];
		if (pn->ident == ident) {
			return pn->name;
		}
	}
	return "unknown";
}

void sdd_configure_eco_trim_cap(sdd_data_t *psd)
{
	u32 trim;
	char *prop = "eco-trim-cap";
	int error = of_property_read_u32_array(psd->i2c_client->dev.of_node, prop, &trim, 1);
	if (error < 0) {
		bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "%s not found in DTB", prop);
		return;
	}
	psd->dts_config_flags |= SDD_DTS_ECO_TRIM_CAP;
	error = sdd_ind_set_eco_trim_cap(psd, trim);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "configure eco trim cap error %d", error);
	}
}

static int sdd_init_sge(sdd_data_t *psd, int psoc_recovery)
{
	int error, attempts, max_attempts=3;
	u8  val=0, ident=0;
	u16 version=0;
#if defined(CONFIG_SONOS_SDD_RTC)
	int rtc_state_saved = 0;
#endif
	u16 status_reg=0;
        int verify_flash = !psoc_recovery && !psd->psoc_flash_verify_disabled;

	sdd_disable_ble_uart(psd);

	for (attempts = 0; attempts < max_attempts; attempts++) {
		int ident_reads, max_ident_reads = 1000;
		mdelay(75);
		sdd_psoc_unreset(psd);
		mdelay(50);
		UNLOCK_I2C(psd->i2c_client->adapter);
		psd->disabled_by_fatal_error = 0;

		for (ident_reads = 0; ident_reads < max_ident_reads; ident_reads++) {
			error = __sdd_read_sge_byte(psd, SGE_REG_IDENT, &ident);
			if (!error) {
				break;
			}
			mdelay(1);
		}
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d ident could not be determined, code %d", psd->inst, error);
			ident = SGE_IDENT_INVALID;
		}

		error = sdd_select_image(psd, ident);
		if (error) {
			sdd_fatal_error(psd);
			return error;
		}

		if (sdd_ident_match(psd, ident)) {
			error = sdd_read_sge_word(psd, SGE_REG_VERSION, &version);
			if (!error && sdd_version_match(psd, version)) {
				if (verify_flash && (attempts == 0)) {
					error = sdd_verify_psoc(psd);
					mdelay(50);
					if (!error) {
						break;
					}
				} else {
                                    break;
                                }
			} else {
				if (error) {
					bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d version could not be determined, code %d", psd->inst, error);
				} else {
					bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d version mismatch, got %04x, expected %04x", psd->inst, version, psd->psoc_cfg->version);
				}
			}
		} else {
                    bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d ident mismatch, got %02x, expected %02x", psd->inst, ident, psd->psoc_cfg->ident);
		}

#if defined(CONFIG_SONOS_SDD_RTC)
		if (PRODUCT_ID_IS_DHUEZ && !rtc_state_saved) {
			sdd_protect_rtc_content(0);
			rtc_state_saved = 1;
		}
#endif
		sdd_program_psoc(psd, 0);
		ReleasePsocImage(psd->psoc_cfg);
		mdelay(50);

		if (attempts < (max_attempts - 1)) {
			LOCK_I2C(psd->i2c_client->adapter);
		}
	}

	if (attempts >= max_attempts) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d init fatal error", psd->inst);
		sdd_fatal_error(psd);
		return -1;
	}

	error = sdd_get_features(psd);
	if (error) {
		return error;
	}

	error = sdd_wait_for_ind_ready(psd);
	if (error) {
		return error;
	}

	mutex_lock(&psd->ind_lock);
	psd->suspended = 0;
	mutex_unlock(&psd->ind_lock);

#if defined(CONFIG_SONOS_SDD_RTC)
	if (PRODUCT_ID_IS_DHUEZ && rtc_state_saved) {
		sdd_protect_rtc_content(1);
		rtc_state_saved = 0;
	}
#endif

	error = sdd_ind_get_num_sensors(psd, &val);
	if (error) {
		return error;
	}
	psd->num_sensors = val;

	if (gpio_is_valid(psd->gpio_table[SCI_INDEX_GPIO_IRQ].gpio)) {
		psd->irq = gpio_to_irq(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ));
		error = request_irq(psd->irq, sdd_isr, IRQF_TRIGGER_FALLING, "sdd_isr", psd);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: request_irq (virtual %u) failed %d", __func__, psd->irq, error);
			return error;
		}
	}
	psd->sge_init_done = 1;

	error = sdd_read_sge_word(psd, SGE_REG_STATUS, &status_reg);
	if (!error && (status_reg & SGE_REG_STATUS_ECO_UNCONF)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "HFCLK is not configured for ECO");
	}

	error = sdd_configure_sge(psd);
	if (error) {
		return error;
	}
	sdd_configure_eco_trim_cap(psd);

	sdd_enable_ble_uart(psd);

	if (!(psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH)) {
		error = sdd_set_touch(psd, 1);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d could not be enabled", psd->inst);
		}
		if (!error) {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d init success, ident %02x, version %04x, %s",
			       psd->inst, ident, version, sdd_psoc_partname(ident));
		}
	} else {
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d init success with no captouch, ident %02x, version %04x, %s",
		       psd->inst, ident, version, sdd_psoc_partname(ident));
	}

#if !defined(SONOS_ARCH_ATTR_SOC_IS_IMX7ULP)
	sdd_start_timer(psd->sge_monitor_timer);
#endif
	return 0;
}

static int sdd_process_interrupt(sdd_data_t *psd)
{
	u16 grr;
	int error;
	unsigned long entry = jiffies;

	if (psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH) {
		return 0;
	}

	psd->irq_processed++;

	error = sdd_read_sge_word(psd, SGE_REG_GRR, &grr);
	if (error) {
		return error;
	}

	if (grr != psd->prev_grr) {
		struct sdd_grr_change_entry grr_change_entry;
#ifndef CONFIG_SONOS_DIAGS
		sdd_gesture_match_t *gest = psd->gesture_match;
		u16 grr_change;
		enum HWEVTQ_EventInfo info;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		enum HWEVTQ_EventInfo hwevtq_info;
#endif
#endif
		grr_change_entry.prev_grr = psd->prev_grr;
		grr_change_entry.new_grr = grr;
		grr_change_entry.prev_msec = jiffies_to_msecs(jiffies - psd->prev_jiffies);
		grr_change_entry.intr_msec = jiffies_to_msecs(entry - psd->intr_jiffies);
		grr_change_entry.i2c_msec = jiffies_to_msecs(jiffies - entry);
		sdd_log_event(psd->grr_event_log, &grr_change_entry);
		psd->prev_jiffies = jiffies;

#ifndef CONFIG_SONOS_DIAGS
		if (grr == 0) {
			sdd_stop_timer(psd->touch_timer);
		} else if (psd->prev_grr == 0) {
			sdd_start_timer(psd->touch_timer);
		}

		grr_change = grr ^ psd->prev_grr;
		if (grr_change != 0) {
			int swipe = 0;
			if (grr_change & SGE_REG_GRR_PP_SWIPE) {
				if (grr & SGE_REG_GRR_PP_SWIPE) {
					event_queue_send_event(HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_SWIPED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_send_event(HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_SWIPED);
#endif
					swipe = 1;
				} else if (grr == 0) {
					if ((psd->prev_grr & (SGE_REG_GRR_VOL1_TAP | SGE_REG_GRR_VOL2_TAP)) == 0) {
						event_queue_send_event(HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
						hwevtq_send_event(HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED);
#endif
					}
				}
			}
			if (grr_change & gest->zonea) {
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_info = (grr & gest->zonea) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
#endif
				info = (grr & gest->zonea) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
				if (!swipe || (info == HWEVTQINFO_PRESSED)) {
					event_queue_send_event(HWEVTQSOURCE_CAPZONEA, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                        hwevtq_send_event(HWEVTQSOURCE_CAPZONEA, hwevtq_info);
#endif
				}
			}
			if (grr_change & gest->zoneb) {
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                hwevtq_info = (grr & gest->zoneb) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
#endif
				info = (grr & gest->zoneb) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
				event_queue_send_event(HWEVTQSOURCE_CAPZONEB, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(HWEVTQSOURCE_CAPZONEB, hwevtq_info);
#endif
			}
			if (grr_change & gest->zonec) {
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                hwevtq_info = (grr & gest->zonec) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
#endif
				info = (grr & gest->zonec) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
				if (!swipe || (info == HWEVTQINFO_PRESSED)) {
					event_queue_send_event(HWEVTQSOURCE_CAPZONEC, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                        hwevtq_send_event(HWEVTQSOURCE_CAPZONEC, hwevtq_info);
#endif
				}
			}
			if (grr_change & gest->zonem) {
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                hwevtq_info = (grr & gest->zonem) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
#endif
				info = (grr & gest->zonem) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
				event_queue_send_event(HWEVTQSOURCE_CAPZONEM, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(HWEVTQSOURCE_CAPZONEM, hwevtq_info);
#endif
			}
			if (grr_change & gest->zonea_hold) {
				event_queue_send_event(HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_REPEATED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                hwevtq_send_event(HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_REPEATED);
#endif
			}
			if (grr_change & gest->zonec_hold) {
				event_queue_send_event(HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_REPEATED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                hwevtq_send_event(HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_REPEATED);
#endif
			}
			if (grr_change & gest->zonem_hold) {
				event_queue_send_event(HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_REPEATED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                hwevtq_send_event(HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_REPEATED);
#endif
			}
			if (grr_change & SGE_REG_GRR_CAT) {
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                hwevtq_info = (grr & SGE_REG_GRR_CAT) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
#endif
				info = (grr & SGE_REG_GRR_CAT) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
				event_queue_send_event(HWEVTQSOURCE_CAPZONECAT, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                                hwevtq_send_event(HWEVTQSOURCE_CAPZONECAT, hwevtq_info);
#endif
			}
		}
		sdd_ind_update_stats(psd, psd->prev_grr, grr);
		psd->prev_grr = grr & ~psd->all_holds_bitmap;
#endif
	}
	return error;
}

static void sdd_touch_timer_handler(void *p)
{
	sdd_data_t *psd = (sdd_data_t *)p;

	if (psd->prev_grr != 0) {
		sdd_update_sge_baselines(psd);
		psd->baseline_update_count++;
		sdd_start_timer(psd->touch_timer);
		bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "captouch sensor baselines recalibrated");
	}
}

static void sdd_log_sge_registers(sdd_data_t *psd)
{
	u8 dd[SDD_I2C_BLOCK_SIZE];
	int off = 0;
	for (off = 0; off < sizeof(struct SGE_REGISTERS); off += SDD_I2C_BLOCK_SIZE) {
		int error = sdd_read_sge_block(psd, off, SDD_I2C_BLOCK_SIZE, dd);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_WARNING, "SGE registers, offset %02x, error %d", off, error);
			continue;
		}
		if ((off % 0x40) == 0) {
			bb_log(BB_MOD_SDD, BB_LVL_WARNING, "SGE registers, offset %02x", off);
		}
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "%02x %02x %02x %02x %02x %02x %02x %02x   %02x %02x %02x %02x %02x %02x %02x %02x",
			   dd[0], dd[1], dd[2],  dd[3],  dd[4],  dd[5],  dd[6],  dd[7],
			   dd[8], dd[9], dd[10], dd[11], dd[12], dd[13], dd[14], dd[15]);
	}
}

static void sdd_sge_monitor_timer_handler(void *p)
{
	sdd_data_t *psd = (sdd_data_t *)p;
	int error;
	u16 pass;

	if (!(psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH)) {
		if (psd->check_raw_cap) {
			sdd_ind_check_rawcap(psd);
			psd->check_raw_cap = 0;
		}
	}

	error = sdd_read_sge_word(psd, SGE_REG_PASS, &pass);
	if (error || (pass == psd->sge_prev_pass)) {
		psd->sge_stuck_count++;
		if (psd->sge_stuck_count >= SGE_MAX_STUCK_COUNT) {
			sdd_free_irq(psd);
			bb_log(BB_MOD_SDD, BB_LVL_WARNING, "SGE on PSoC not responding, restarting, i2c error %d, pass %04x", error, pass);
			if (!error) {
				sdd_log_sge_registers(psd);
			}
			LOCK_I2C(psd->i2c_client->adapter);
			psd->sge_init_done = 0;
			sdd_psoc_reset(psd);
			error = sdd_init_sge(psd, 1);
			if (error) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "SGE recovery failed");
				sdd_fatal_error(psd);
				return;
			} else {
				psd->sge_recoveries++;
			}
		}
	} else {
		psd->sge_stuck_count = 0;
	}
	psd->sge_prev_pass = pass;
	sdd_start_timer(psd->sge_monitor_timer);
}

void sdd_init_timers(sdd_data_t *psd)
{
	INIT_LIST_HEAD(&psd->timers);
	psd->touch_timer = sdd_create_timer(psd, "touch", (30*1000), sdd_touch_timer_handler);
	psd->sge_monitor_timer = sdd_create_timer(psd, "SGE-monitor", SGE_MONITOR_TIME, sdd_sge_monitor_timer_handler);
}

static int sdd_thread(void* pdata)
{
	sdd_data_t *psd = pdata;
	int error;

	psd->prev_jiffies = jiffies;
	psd->prev_grr = 0;

	while (!kthread_should_stop()) {
		psd->passes++;

		if (atomic_read(&psd->sge_irq_pending) == 0) {
			error = wait_for_completion_interruptible(&psd->sdd_completion);
			if (error) {
				psd->completion_errors++;
				continue;
			}
		}

		if (!(psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH)) {
			if (atomic_read(&psd->sge_irq_pending) > 0) {
				error = sdd_process_interrupt(psd);
				atomic_set(&psd->sge_irq_pending, 0);
				enable_irq(psd->irq);
			}
		}

		sdd_process_timers(psd);
	}

	sdd_stop_timer(psd->sge_monitor_timer);
	sdd_stop_timer(psd->touch_timer);
	return 0;
}

static int sdd_init_thread(sdd_data_t *psd)
{
	int error;
	struct sched_param param = {.sched_priority = SONOS_THREAD_PRIORITY_RT_SDD};

	psd->sddtask = kthread_run(sdd_thread, psd, "sdd-psoc%d", psd->inst);
	if ( (error = IS_ERR(psd->sddtask)) ) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: error %d", __func__, error);
		return error;
	}

	sched_setscheduler(psd->sddtask, SCHED_FIFO, &param);
	return error;
}

static int sdd_exit_thread(sdd_data_t *psd)
{
	sdd_change_timer(psd->sge_monitor_timer, 100);
	return (kthread_stop(psd->sddtask));
}

static void sdd_get_gpio_pin(sdd_data_t *psd, int index, char *name, int flags, char *label)
{
	struct device_node *node = psd->i2c_client->dev.of_node;
	struct gpio *pgpio = &psd->gpio_table[index];
	pgpio->gpio = of_get_named_gpio(node, name, 0);
	if (gpio_is_valid(pgpio->gpio)) {
		pgpio->flags = flags;
		pgpio->label = label;
		psd->num_gpio++;
	} else {
		bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "gpio %s missing from dtb", name);
	}
}

void sdd_init_gpio_table(sdd_data_t *psd)
{
	int xres_flags = (psd->psoc_reset) ? GPIOF_OUT_INIT_HIGH : GPIOF_OUT_INIT_LOW;
	sdd_get_gpio_pin(psd, SCI_INDEX_GPIO_IO,  "io-gpio",   GPIOF_DIR_IN, "PSoC IO");
	sdd_get_gpio_pin(psd, SCI_INDEX_GPIO_SCL, "scl-gpio",  GPIOF_DIR_IN, "PSoC SCL");
	sdd_get_gpio_pin(psd, SCI_INDEX_GPIO_RST, "xres-gpio", xres_flags, "PSoC Reset");
	sdd_get_gpio_pin(psd, SCI_INDEX_GPIO_IRQ, "irq-gpio",  GPIOF_DIR_IN, "PSoC Interrupt");

	if (IS_CHAPLIN) {
		if (sys_mdp.mdp_revision <= MDP_REVISION_CHAPLIN_PROTO2A) {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "using chaplin P2a SWD GPIOs");
			psd->gpio_table[SCI_INDEX_GPIO_IO].gpio  = 169;
			psd->gpio_table[SCI_INDEX_GPIO_SCL].gpio = 168;
		} else if (sys_mdp.mdp_revision == MDP_REVISION_CHAPLIN_PROTO2B) {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "using chaplin P2b SWD GPIOs");
			psd->gpio_table[SCI_INDEX_GPIO_IO].gpio  = 201;
			psd->gpio_table[SCI_INDEX_GPIO_SCL].gpio = 200;
		}
	}
}

struct sdd_zone_map {
	char *valstr;
	u16   grr_zone;
	u16   grr_hold;
};
struct sdd_zone_map sdd_zone_maps[] = {
	{"vol1", SGE_REG_GRR_VOL1_TAP, SGE_REG_GRR_VOL1_HOLD},
	{"pp",   SGE_REG_GRR_PP_TAP,   SGE_REG_GRR_PP_HOLD},
	{"vol2", SGE_REG_GRR_VOL2_TAP, SGE_REG_GRR_VOL2_HOLD},
	{"mic",  SGE_REG_GRR_ZONEM,    SGE_REG_GRR_ZONEM_HOLD},
};
#define SDD_MAX_ZONES (sizeof(sdd_zone_maps) / sizeof(struct sdd_zone_map))

static void sdd_get_zone_bitmaps(struct device_node *node, char *name, u16 *zone_mask, u16 *hold_mask)
{
	int len;
	char *cfg = (char *)of_get_property(node, name, &len);
	if (cfg == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "property %s missing", name);
		*zone_mask = 0;
		*hold_mask = 0;
	} else {
		int i;
		for (i = 0; i < SDD_MAX_ZONES; i++) {
			struct sdd_zone_map *zmap = &sdd_zone_maps[i];
			if (strcmp(cfg, zmap->valstr) == 0) {
				*zone_mask = zmap->grr_zone;
				*hold_mask = zmap->grr_hold;
				bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "property %s, value %s, %04x %04x", name, cfg, *zone_mask, *hold_mask);
				return;
			}
		}
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "property %s has unknown value %s", name, cfg);
	}
}

void sdd_set_all_holds_bitmap(sdd_data_t *psd)
{
	int i;
	psd->all_holds_bitmap = 0;
	for (i = 0; i < SDD_MAX_ZONES; i++) {
		psd->all_holds_bitmap |= sdd_zone_maps[i].grr_hold;
	}
}

static sdd_gesture_match_t * sdd_init_gesture_table(sdd_data_t *psd, struct device_node *node)
{
	sdd_get_zone_bitmaps(node, "zonea", &sdd_gesture_match_default.zonea, &sdd_gesture_match_default.zonea_hold);
	sdd_get_zone_bitmaps(node, "zoneb", &sdd_gesture_match_default.zoneb, &sdd_gesture_match_default.zoneb_hold);
	sdd_get_zone_bitmaps(node, "zonec", &sdd_gesture_match_default.zonec, &sdd_gesture_match_default.zonec_hold);
	sdd_get_zone_bitmaps(node, "zonem", &sdd_gesture_match_default.zonem, &sdd_gesture_match_default.zonem_hold);
	sdd_set_all_holds_bitmap(psd);
	return &sdd_gesture_match_default;
}

#if defined(CONFIG_SONOS_SDD_RTC)
static void sdd_set_dts_psoc_rtc(sdd_data_t *psd)
{
	u32 enable=0;
	char *property_name = "psoc-rtc";
	int error = of_property_read_u32_array(psd->i2c_client->dev.of_node, property_name, &enable, 1);
	if (error < 0) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "property %s missing", property_name);
	}
	if (enable) {
		psd->dts_config_flags |= SDD_DTS_PSOC_RTC;
	} else {
		psd->dts_config_flags &= ~SDD_DTS_PSOC_RTC;
	}
}
#endif

static int sdd_init_image_data(sdd_data_t *psd, struct device_node *psoc_images_node, int preempt_disable)
{
	int error;
	int num_images = 0;
	struct device_node *image_node;

	image_node = of_get_child_by_name(psoc_images_node, "image");
	if (!image_node) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "PSoC image node not found");
		return -ENOENT;
	}

	while (image_node != NULL) {
		if (num_images >= SGE_MAX_PSOC_IMAGES) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Max number of PSoC images = %d exceeded", SGE_MAX_PSOC_IMAGES);
			return -ENOENT;
		}

		if (of_property_read_u32(image_node, "model-num", &psd->psoc_cfgs[num_images].model) != 0) {
			int ret = of_property_read_u32(image_node, "model", &psd->psoc_cfgs[num_images].model);
			bb_log(BB_MOD_SDD, BB_LVL_WARNING, " PSoC model-num missing from DTB, model%s", ret ? " missing" : " exists");
		}
		of_property_read_u32(image_node, "ident", &psd->psoc_cfgs[num_images].ident);
		of_property_read_u32(image_node, "flash-rows", &psd->psoc_cfgs[num_images].flash_rows);
		of_property_read_u32(image_node, "flash-row-size", &psd->psoc_cfgs[num_images].flash_row_size);
		of_property_read_u32(image_node, "flash-protection-size", &psd->psoc_cfgs[num_images].flash_protection_size);

		psd->psoc_cfgs[num_images].version = sdd_get_image_version(psd->psoc_cfgs[num_images].ident);
		psd->psoc_cfgs[num_images].inst = psd->inst;

		error = InitPsocConfig(&psd->psoc_cfgs[num_images]);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: InitPsocConfig error %d", __func__, error);
			return error;
		}
		psd->psoc_cfgs[num_images].swd_preempt_disable = preempt_disable;
		num_images++;

		image_node = of_get_next_available_child(psoc_images_node, image_node);
	}

	psd->num_psoc_cfgs = num_images;
	psd->psoc_cfg = &psd->psoc_cfgs[0];

	return 0;
}

static void sdd_set_psoc_cfg_gpio(sdd_data_t *psd)
{
	int i;
	for (i = 0; i < psd->num_psoc_cfgs; i++) {
		psd->psoc_cfgs[i].gpio_table = psd->gpio_table;
	}
}

int sdd_init_device(struct i2c_client *i2c_client)
{
	sdd_data_t *psd;
	int error=0, should=256, swd_preempt_disable=0;
	struct device_node *images_node;

	if (sdd_psoc_inst >= SDD_MAX_PSOC) {
		return -ENXIO;
	}
	psd = (sdd_data_t *)kmalloc(sizeof(*psd), GFP_KERNEL);
	if (psd == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "no memory for PSoC inst %d", sdd_psoc_inst);
		return -ENOMEM;
	}
	memset(psd, 0, sizeof(*psd));
	Sdd_data[sdd_psoc_inst] = psd;
	psd->inst = sdd_psoc_inst++;

	if (sizeof(struct SGE_REGISTERS) != should) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "SGE register size is %zu, it should be %d", sizeof(struct SGE_REGISTERS), should);
		sdd_fatal_error(psd);
		return -EIO;
	}

	images_node = of_get_child_by_name(i2c_client->dev.of_node, "psoc-images");
	if (images_node == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc-images node not found");
		return -ENODEV;
	}

	if (of_property_read_bool(i2c_client->dev.of_node, "swd-preempt-disable")) {
		swd_preempt_disable = 1;
	}
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "SWD preempt disable %sconfigured", swd_preempt_disable ? "" : "not ");
	sdd_init_image_data(psd, images_node, swd_preempt_disable);

	if (of_property_read_bool(i2c_client->dev.of_node, "active-high-psoc-reset")) {
		psd->psoc_reset = 1;
		bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "psoc%d reset is active HIGH", psd->inst);
	} else {
		bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "psoc%d reset is active LOW", psd->inst);
	}

	if (of_property_read_bool(i2c_client->dev.of_node, "psoc-skip-init-reset")) {
		psd->psoc_skip_init_reset = 1;
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d reset is being skipped...", psd->inst);
	}
	if (of_property_read_bool(i2c_client->dev.of_node, "psoc-flash-verify-disabled")) {
		psd->psoc_flash_verify_disabled = 1;
		bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "psoc%d flash verify disabled", psd->inst);
	}

	if (of_property_read_bool(i2c_client->dev.of_node, "psoc-rst-config")) {
		psd->rst_pin_config = 1;
	}
	LOCK_I2C(i2c_client->adapter);
	psd->i2c_client = i2c_client;
	sdd_init_gpio_table(psd);
	sdd_set_psoc_cfg_gpio(psd);
	if (psd->psoc_skip_init_reset) {
		psd->gpio_table[SCI_INDEX_GPIO_RST].flags = ((psd->gpio_table[SCI_INDEX_GPIO_RST].flags == GPIOF_OUT_INIT_HIGH) ?
										GPIOF_OUT_INIT_LOW : GPIOF_OUT_INIT_HIGH);
	}
	sdd_config_swd_pins(psd);
	error = gpio_request_array(psd->gpio_table, psd->num_gpio);
	if (psd->psoc_skip_init_reset) {
		psd->gpio_table[SCI_INDEX_GPIO_RST].flags = ((psd->gpio_table[SCI_INDEX_GPIO_RST].flags == GPIOF_OUT_INIT_HIGH) ?
										GPIOF_OUT_INIT_LOW : GPIOF_OUT_INIT_HIGH);
	}
	if (error && error != -ENOSYS) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "gpio_request_array returned error: %d", error);
		UNLOCK_I2C(i2c_client->adapter);
		return error;
	}
	bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "gpio_request_array - putting PSoC in reset");
	psd->gesture_match = sdd_init_gesture_table(psd, i2c_client->dev.of_node);
	if (of_property_read_bool(i2c_client->dev.of_node, "psoc-supports-low-power")) {
		psd->dts_config_flags |= SDD_DTS_LOW_POWER;
	}
	if (of_property_read_bool(i2c_client->dev.of_node, "psoc-no-captouch")) {
		psd->dts_config_flags |= SDD_DTS_NO_CAPTOUCH;
	}
	if (of_property_read_bool(i2c_client->dev.of_node, "uses-rpmsg")) {
		psd->dts_config_flags |= SDD_DTS_RPMSG;
	}
	if (of_property_read_bool(i2c_client->dev.of_node, "program-requires-usbc-power")) {
		psd->dts_config_flags |= SDD_DTS_CHECK_POWER;
	}
	if (of_property_read_bool(i2c_client->dev.of_node, "shared-i2c-swd-pins")) {
		psd->dts_config_flags |= SDD_DTS_SHARE_SWD_I2C;
	}

	init_completion(&psd->sdd_completion);
	mutex_init(&psd->ind_lock);

#if defined(CONFIG_SONOS_SWD_BYPASS_GPIOLIB)
	error = cypress_swd_init(&psd->i2c_client->dev, psd->inst);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "sci_swd_init() failed");
		gpio_free_array(psd->gpio_table, psd->num_gpio);
		UNLOCK_I2C(i2c_client->adapter);
		cypress_swd_exit();
		return error;
	}
#endif

	error = sdd_rt_pmux_init(psd);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "sdd_rt_pmux_init() failed");
		gpio_free_array(psd->gpio_table, psd->num_gpio);
		UNLOCK_I2C(i2c_client->adapter);
		return error;
	}
	sdd_init_timers(psd);

#if defined(CONFIG_SONOS_SDD_RTC)
	sdd_set_dts_psoc_rtc(psd);
	if (psd->dts_config_flags & SDD_DTS_PSOC_RTC) {
		sdd_rtc_init(psd);
	}
#endif

	if (psd->dts_config_flags & SDD_DTS_RPMSG) {
		error = sdd_rpmsg_init();
		if ( error ) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc rpmsg register failed %d\n", error);
			return error;
		}
		sdd_ind_ops_rpmsg(psd);
        } else {
		sdd_ind_ops_nocop(psd);
	}

	sdd_psoc_on(psd);

	error = sdd_init_sge(psd, 0);
	if (error) {
		return error;
	}
	return sdd_init_thread(psd);
}

sdd_data_t * sdd_data_from_i2c(struct i2c_client *i2c_client)
{
	int i;
	for (i = 0; i < SDD_MAX_PSOC; i++) {
		sdd_data_t *psd = sdd_get_data(i);
		if (psd->i2c_client == i2c_client) {
			return psd;
		}
	}
	return NULL;
}

void sdd_stop_device(struct i2c_client *i2c_client)
{
	sdd_data_t *psd = sdd_data_from_i2c(i2c_client);
	if (psd == NULL) {
		return;
	}
#if defined(CONFIG_RPMSG)
	if (psd->dts_config_flags & SDD_DTS_RPMSG) {
		sdd_rpmsg_exit();
	}
#endif
	sdd_free_irq(psd);
	sdd_exit_thread(psd);
	gpio_free_array(psd->gpio_table, psd->num_gpio);
	Sdd_data[psd->inst] = NULL;
	kfree(psd);
}

void sdd_stop_devices(void)
{
	int i;
	for (i = 0; i < SDD_MAX_PSOC; i++) {
		sdd_data_t *psd = sdd_get_data(i);
		if (psd != NULL) {
			sdd_stop_device(psd->i2c_client);
		}
	}
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "all devices stopped");
}

void sdd_get_capzone(enum HWEVTQ_EventInfo *capzonea, enum HWEVTQ_EventInfo *capzoneb, enum HWEVTQ_EventInfo *capzonec, enum HWEVTQ_EventInfo *capzonem)
{
	sdd_data_t *psd = sdd_get_data(0);
	u16 grr = psd->prev_grr;
	sdd_gesture_match_t *gest = psd->gesture_match;

	*capzonea = (grr & gest->zonea) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
	*capzoneb = (grr & gest->zoneb) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
	*capzonec = (grr & gest->zonec) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
	*capzonem = (grr & gest->zonem) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
}

void sdd_set_grr(enum HWEVTQ_EventSource source)
{
	sdd_data_t *psd = sdd_get_data(0);
	sdd_gesture_match_t *gest = psd->gesture_match;

	switch(source) {
	case HWEVTQSOURCE_CAPZONEA:
		psd->prev_grr = gest->zonea;
		break;
	case HWEVTQSOURCE_CAPZONEB:
		psd->prev_grr = gest->zoneb;
		break;
	case HWEVTQSOURCE_CAPZONEC:
		psd->prev_grr = gest->zonec;
		break;
	case HWEVTQSOURCE_CAPZONEM:
		psd->prev_grr = gest->zonem;
		break;
	default:
		psd->prev_grr = 0;
	}
}

int sdd_set_press_hold_time(sdd_data_t *psd, u32 msec32)
{
	int error;
	u16 msec16 = msec32;
	error = sdd_ind_set_hold_time(psd, msec16);
	if (!error) {
		psd->press_hold_time = msec16;
	}
	return error;
}

int sdd_set_repeat_time(sdd_data_t *psd, u32 msec32)
{
	int error;
	u16 msec16 = msec32;
	error = sdd_ind_set_repeat_time(psd, msec16);
	if (!error) {
		psd->repeat_time = msec16;
	}
	return error;
}

int sdd_set_touch(sdd_data_t *psd, int enable)
{
	int error = 0;
	if (!(psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH)) {
		error = sdd_capsense(psd, enable);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "SGE cap sense %sable failure",
			       enable ? "en" : "dis");
		}
		if (!error) {
			psd->touch_enable = enable;
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "SGE cap sense %sabled",
			       enable ? "en" : "dis");
		}
	}
	return error;
}

int sdd_set_min_samples_per_gesture(sdd_data_t *psd, u32 samples)
{
	return 0;
}

int sdd_set_min_samples_per_tap(sdd_data_t *psd, u32 samples)
{
	return 0;
}

int sdd_set_finger_threshold(sdd_data_t *psd, u32 threshold)
{
	int error;
	u8 thresh = threshold;
	error = sdd_ind_set_finger_threshold(psd, thresh);
	if (!error) {
		psd->finger_threshold = thresh;
	}
	return error;
}

int sdd_set_noise_threshold(sdd_data_t *psd, u32 threshold)
{
	int error;
	u8 thresh = threshold;
	error = sdd_ind_set_noise_threshold(psd, thresh);
	if (!error) {
		psd->noise_threshold = thresh;
	}
	return error;
}

int sdd_set_dts_share_swd_uart(sdd_data_t *psd, u32 enable)
{
	if (enable) {
		psd->dts_config_flags |= SDD_DTS_SHARE_SWD_UART;
	} else {
		psd->dts_config_flags &= ~SDD_DTS_SHARE_SWD_UART;
	}

	return 0;
}

int sdd_set_finger_hysteresis(sdd_data_t *psd, u32 hysteresis)
{
	int error;
	u8 hyst = hysteresis;

	error = sdd_ind_set_finger_hysteresis(psd, hyst);
	if (!error) {
		psd->finger_hysteresis = hyst;
	}
	return error;
}

int sdd_erase_sge(sdd_data_t *psd)
{
	int error;

	sdd_disable_ble_uart(psd);

	if (psd->sge_init_done) {
		sdd_free_irq(psd);
		sdd_stop_timer(psd->sge_monitor_timer);
		psd->disabled_by_fatal_error = 1;
		psd->sge_init_done = 0;
	}

	if (psd->psoc_erased) {
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d flash already erased", psd->inst);
	} else {
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "Erasing psoc%d flash", psd->inst);
		error = sdd_erase_psoc(psd);
		if (!error) {
			psd->psoc_erased = 1;
		}
	}
	return 0;
}

int sdd_program_sge(sdd_data_t *psd)
{
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "Writing image to psoc%d", psd->inst);
#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX7ULP)
        if (sdd_select_image(psd, SGE_IDENT_INVALID)) {
            bb_log(BB_MOD_SDD, BB_LVL_WARNING, "psoc%d select image failed", psd->inst);
        }
	sdd_program_psoc(psd, 1);
	sdd_psoc_unreset(psd);
        ReleasePsocImage(psd->psoc_cfg);
#else
	LOCK_I2C(psd->i2c_client->adapter);
	sdd_init_sge(psd, 0);
#endif
	return 0;
}

int sdd_set_move_interval(sdd_data_t *psd, u32 move_interval)
{
	int error;
	u8 interval = move_interval;
	error = sdd_ind_set_move_interval(psd, interval);
	if (!error) {
		psd->move_interval = interval;
	}
	return error;
}

void sdd_reset_psoc(void)
{
	int i;
	for (i = 0; i < sdd_psoc_inst; i++) {
		sdd_data_t *psd = sdd_get_data(i);
		sdd_psoc_reset(psd);
	}
	printk("reset PSoC\n");
}
EXPORT_SYMBOL(sdd_reset_psoc);

int sdd_change_repeat_time(u32 msec32)
{
	int i, error;
	for (i = 0; i < sdd_psoc_inst; i++) {
		sdd_data_t *psd = sdd_get_data(i);
		error = sdd_set_repeat_time(psd, msec32);
		if (error) {
			return error;
		}
	}
	return 0;
}

static int sdd_enter_psoc_low_power_mode(sdd_data_t *psd)
{
	int error, attempts, max_attempts = 1200;
	u8 lpstate = SGE_LOW_POWER_STATE_OFF;
	int old_suspended_value;

	if (!(psd->dts_config_flags & SDD_DTS_LOW_POWER)) {
		return 0;
	}

	sdd_stop_timer(psd->sge_monitor_timer);

	free_irq(psd->irq, psd);
	gpio_direction_output(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ), SGE_LOW_POWER_GPIO_OFF);

	mutex_lock(&psd->ind_lock);
	old_suspended_value = psd->suspended;
	psd->suspended = 1;
	mutex_unlock(&psd->ind_lock);

	error = sdd_rmw_sge_byte(psd, SGE_REG_CR0, SGE_REG_CR0_LOW_POWER, SGE_REG_CR0_LOW_POWER);
	if (error) {
		int error2;
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d enter low power RMW error %d", psd->inst, error);
		gpio_direction_input(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ));
		error2 = request_irq(psd->irq, sdd_isr, IRQF_TRIGGER_FALLING, "sdd_isr", psd);
		if (error2) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d request_irq error %d after PSoC i2c failure", psd->inst, error2);
		}
		mutex_lock(&psd->ind_lock);
		psd->suspended = old_suspended_value;
		mutex_unlock(&psd->ind_lock);
		return error;
	}

	for (attempts = 0; attempts < max_attempts; attempts++) {
		mdelay(1);
		error = __sdd_read_sge_byte(psd, SGE_REG_LOW_POWER_STATE, &lpstate);
		if (!error && (lpstate == SGE_LOW_POWER_STATE_ENTER)) {
			break;
		}
	}

	if (attempts >= max_attempts) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d enter low power state timeout, state %02x attempts %d", psd->inst, lpstate, attempts);
	} else {
		gpio_set_value(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ), SGE_LOW_POWER_GPIO_ON);
	}
	return 0;
}

static int sdd_suspend_psoc(sdd_data_t *psd)
{
	int error;

	if (!(psd->dts_config_flags & SDD_DTS_LOW_POWER)) {
		return 0;
	}

	sdd_disable_ble_uart(psd);
	gpio_direction_output(sdd_get_gpio(psd, SCI_INDEX_GPIO_SCL), 0);

	error = sdd_enter_psoc_low_power_mode(psd);

	if (psd->rst_pin_config == 1) {
		pr_info(" GPIO_RST output low\n");
		sdd_psoc_unreset(psd);
	} else {
		gpio_direction_input(sdd_get_gpio(psd, SCI_INDEX_GPIO_RST));
	}
	return error;
}

int sdd_suspend(void)
{
	int i, error;
	for (i = 0; i < sdd_psoc_inst; i++) {
		sdd_data_t *psd = sdd_get_data(i);
		error = sdd_suspend_psoc(psd);
		if (error) {
			return error;
		}
	}
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "SDD/PSoC suspend success");
	return 0;
}

static void sdd_log_low_power_stats(sdd_data_t *psd, char *s, int polls)
{
	int error;
	u16 sleeps=0, pulses=0;

	if (!(psd->dts_config_flags & SDD_DTS_LOW_POWER)) {
		return;
	}

	error = sdd_ind_get_deep_sleeps(psd, &sleeps);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d error %d reading sleeps", psd->inst, error);
	}
	error = sdd_ind_get_pmu_pulses(psd, &pulses);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d error %d reading wake pulses", psd->inst, error);
	}
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "psoc%d %s, polls %u, (deep or normal) sleeps %u, wake pulses %d", psd->inst, s, polls, sleeps, pulses);
}

static int sdd_resume_psoc(sdd_data_t *psd)
{
	int error, attempts, max_attempts = 1200;
	u8 lpstate = SGE_LOW_POWER_STATE_ON;

	if (!(psd->dts_config_flags & SDD_DTS_LOW_POWER)) {
		return 0;
	}

	sdd_psoc_unreset(psd);

	gpio_set_value(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ), SGE_LOW_POWER_GPIO_OFF);

	for (attempts = 0; attempts < max_attempts; attempts++) {
		mdelay(1);
		error = __sdd_read_sge_byte(psd, SGE_REG_LOW_POWER_STATE, &lpstate);
		if (!error && (lpstate == SGE_LOW_POWER_STATE_RESUME)) {
			break;
		}
	}
	if (attempts < max_attempts) {
		error = sdd_rmw_sge_byte(psd, SGE_REG_CR0, SGE_REG_CR0_LOW_POWER, 0);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d error %d resuming PSoC", psd->inst, error);
			goto restart_psoc;
		}

		gpio_direction_input(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ));
		error = request_irq(psd->irq, sdd_isr, IRQF_TRIGGER_FALLING, "sdd_isr", psd);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: psoc%d request_irq failed %d",  __func__, psd->inst, error);
			return 0;
		}
		mutex_lock(&psd->ind_lock);
		psd->suspended = 0;
		mutex_unlock(&psd->ind_lock);
		sdd_log_low_power_stats(psd, "resume success", attempts);

		gpio_direction_input(sdd_get_gpio(psd, SCI_INDEX_GPIO_SCL));
		sdd_enable_ble_uart(psd);
		sdd_start_timer(psd->sge_monitor_timer);
		mdelay(24);
		sdd_log_low_power_stats(psd, "SDD/PSoC resume success", attempts);
		return 0;
	}

	bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d resume fail, error %d, state %02x, attempts %d", psd->inst, error, lpstate, attempts);

restart_psoc:
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "resetting psoc%d", psd->inst);
	gpio_direction_input(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ));
	LOCK_I2C(psd->i2c_client->adapter);
	psd->sge_init_done = 0;
	sdd_psoc_reset(psd);
	error = sdd_init_sge(psd, 1);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unable to reset/reinit psoc%d, error %d", psd->inst, error);
	}
	return 0;
}

int sdd_resume(void)
{
	int i, error;
	for (i = 0; i < sdd_psoc_inst; i++) {
		sdd_data_t *psd = sdd_get_data(i);
		error = sdd_resume_psoc(psd);
		if (error) {
			return error;
		}
	}
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "SDD/PSoC resume success");
	return 0;
}

static int sdd_psoc_off(sdd_data_t *psd)
{
	int error;
#if defined(CONFIG_SONOS_SDD_RTC)
        time64_t alarm;
#endif

	if (!(psd->dts_config_flags & SDD_DTS_LOW_POWER)) {
		return 0;
	}

#if defined(CONFIG_SONOS_SDD_RTC)
        error = sdd_rtc_get_alarm(&alarm);
        if (error) {
            printk("%s: alarm get error\n", __func__);
        } else {
            printk("%s: alarm 0x%016llx\n", __func__, alarm);
        }
#endif

	error = sdd_set_btle(psd, 0);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "psoc%d: unable to disable BLE, error %d", psd->inst, error);
		return error;
	}

	mdelay(12);
	error = sdd_enter_psoc_low_power_mode(psd);
	if (psd->rst_pin_config == 1) {
		pr_info(" GPIO_RST output low\n");
		sdd_psoc_unreset(psd);
	} else {
		gpio_direction_input(sdd_get_gpio(psd, SCI_INDEX_GPIO_RST));
	}
	return error;
}

int sdd_system_off(void)
{
	int i, error;
	for (i = 0; i < sdd_psoc_inst; i++) {
		sdd_data_t *psd = sdd_get_data(i);
		error = sdd_psoc_off(psd);
		if (error) {
			return error;
		}
	}
	pr_info("SDD/PSoC system off: enter low power success\n");
	return 0;
}

int sdd_psoc_on(sdd_data_t *psd)
{
	int error, attempts, max_attempts = 2000;
	u8 lpstate = SGE_LOW_POWER_STATE_ON;

	if (!(psd->dts_config_flags & SDD_DTS_LOW_POWER)) {
		return 0;
	}

	gpio_direction_output(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ), SGE_LOW_POWER_GPIO_OFF);
	UNLOCK_I2C(psd->i2c_client->adapter);

	for (attempts = 0; attempts < max_attempts; attempts++) {
		mdelay(1);
		error = __sdd_read_sge_byte(psd, SGE_REG_LOW_POWER_STATE, &lpstate);
		if (!error && ((lpstate == SGE_LOW_POWER_STATE_RESUME) || (lpstate == SGE_LOW_POWER_STATE_OFF))) {
			break;
		}
	}

	if (attempts < max_attempts) {
		if (lpstate == SGE_LOW_POWER_STATE_RESUME) {
			error = sdd_rmw_sge_byte(psd, SGE_REG_CR0, SGE_REG_CR0_LOW_POWER, 0);
			if (error) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "error %d leaving low power mode", error);
				sdd_psoc_reset(psd);
				return 0;
			}
			sdd_log_low_power_stats(psd, "PSoC exit low power off-mode", attempts);
		} else {
			bb_log(BB_MOD_SDD, BB_LVL_INFO, "PSoC was not in low power off-mode");
		}

		gpio_direction_input(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ));
		LOCK_I2C(psd->i2c_client->adapter);
		return 0;
	}

	bb_log(BB_MOD_SDD, BB_LVL_ERR, "PSoC system on fail, error %d, state %02x, attempts %d", error, lpstate, attempts);
	if (!error) {
		sdd_log_sge_registers(psd);
	}
	gpio_direction_input(sdd_get_gpio(psd, SCI_INDEX_GPIO_IRQ));
	sdd_psoc_reset(psd);
	return 0;
}
