/*
 * Copyright (c) 2016-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/version.h>
#include <generated/autoconf.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <linux/cdev.h>
#include <linux/sonos_kernel.h>
#include <linux/reboot.h>
#include "blackbox.h"
#include "sdd_hal.h"
#include "sdd.h"
#if defined(CONFIG_SONOS_SDD_RTC)
#include "sdd_rtc_api.h"
#endif
#include "sonos_device.h"
#include "sdd_rt_pmux.h"

static int devno = -1;

static int device_init_success = 0;
int device_readers = 0;

struct file_operations sdd_fops;

static int sdd_device_open(struct inode *inode, struct file *file)
{
	if (file->f_mode & FMODE_READ) {
		++device_readers;
	}
	return (device_init_success ? 0 : -ENODEV);
};

static int sdd_device_release(struct inode *inode, struct file *file)
{
	if (file->f_mode & FMODE_READ) {
		--device_readers;
	}
	return 0;
}

static ssize_t sdd_device_read(struct file *filp, char *data, size_t len, loff_t *ppos)
{
	int ret;
	struct SDD_READ sdd_read_data;

	if (len < sizeof(sdd_read_data)) {
		return -EINVAL;
	}

	sdd_get_capzone(&sdd_read_data.capzonea, &sdd_read_data.capzoneb, &sdd_read_data.capzonec, &sdd_read_data.capzonem);
	if (copy_to_user(data, (char *)&sdd_read_data, sizeof(sdd_read_data))) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: copy_to_user failed", __func__);
		ret = -EFAULT;
	} else {
		ret = sizeof(sdd_read_data);
	}
	return ret;
}

static long sdd_device_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = SDD_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case 1:
	{
		int repeat_time, error;
		if (copy_from_user(&repeat_time, (uint32_t *)arg, sizeof(uint32_t))) {
			return -EACCES;
		}
		error = sdd_change_repeat_time(repeat_time);
		if (error) {
			return -EFAULT;
		}
		break;
	}
	case 2:
	{
		int repeat_time = sdd_report_repeat_time();
		if (copy_to_user((uint32_t *)arg, &repeat_time, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case 3:
	{
		struct sdd_info sdd_info;
		int error = 0, i;
		uint32_t instance;
		sdd_data_t *psd;

		if (copy_from_user(&sdd_info, (struct sdd_info *)arg, sizeof(struct sdd_info))) {
			return -EACCES;
		}
		instance = sdd_info.instance;

		psd = sdd_get_data(instance);

		memset(&sdd_info, 0, sizeof(sdd_info));

		error = sdd_read_sge_word(psd, SGE_REG_VERSION, &sdd_info.version);
		if (error) {
			return -EACCES;
		}

		sdd_info.num_sensors = psd->num_sensors;

		for (i = 0; i < sdd_info.num_sensors; i++) {
			error = sdd_ind_get_sensor32(psd, SGE_REG_IND_IDX_PARASITIC_CAP, i, &sdd_info.parasitic_cap[i]);
			if (error) {
				return -EACCES;
			}

			error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_RAW_CAP, i, &sdd_info.raw_cap[i]);
			if (error) {
				return -EACCES;
			}

			error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_MOD_IDAC, i, &sdd_info.mod_idac[i]);
			if (error) {
				return -EACCES;
			}

			error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_COMP_IDAC, i, &sdd_info.comp_idac[i]);
			if (error) {
				return -EACCES;
			}

			error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_SENSE_CLK_DIV, i, &sdd_info.sense_clk_div[i]);
			if (error) {
				return -EACCES;
			}

			error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_MOD_CLK_DIV, i, &sdd_info.mod_clk_div[i]);
			if (error) {
				return -EACCES;
			}

			error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_SIGNAL, i, &sdd_info.signal[i]);
			if (error) {
				return -EACCES;
			}

			error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_BASELINE, i, &sdd_info.baseline[i]);
			if (error) {
				return -EACCES;
			}

			error = sdd_ind_get_sensor16(psd, SGE_REG_IND_IDX_MAX_SIG, i, &sdd_info.max_sig[i]);
			if (error) {
				return -EACCES;
			}
		}

		if (copy_to_user((struct sdd_info *)arg, &sdd_info, sizeof(sdd_info))) {
			return -EACCES;
		}
		break;
	}
	case 4:
	{
		int error;
		uint32_t instance;
		sdd_data_t *psd;

		if (copy_from_user(&instance, (uint32_t *)arg, sizeof(uint32_t))) {
			return -EACCES;
		}

		psd = sdd_get_data(instance);

		error = sdd_erase_sge(psd);
		if (error) {
			return -EACCES;
		}
		break;
	}
	case 5:
	{
		int error;
		uint32_t instance;
		sdd_data_t *psd;

		if (copy_from_user(&instance, (uint32_t *)arg, sizeof(uint32_t))) {
			return -EACCES;
		}

		psd = sdd_get_data(instance);

		error = sdd_program_sge(psd);
		if (error) {
			return -EACCES;
		}
		break;
	}
	case 6:
	{
		u16 grr;
		int error;
		uint32_t instance;
		sdd_data_t *psd;

		if (copy_from_user(&instance, (uint32_t *)arg, sizeof(uint32_t))) {
			return -EACCES;
		}

		psd = sdd_get_data(instance);

		error = sdd_read_sge_word(psd, SGE_REG_GRR, &grr);
		if (error) {
			return -EACCES;
		}

		grr |= SGE_REG_GRR_PP_TAP;
		error = sdd_write_sge_word(psd, SGE_REG_GRR, grr);
		if (error) {
			return -EACCES;
		}
		break;
	}
	case 7:
	{
		int error;
		uint32_t instance;
		uint32_t cmod_val;
		sdd_data_t *psd;

		if (copy_from_user(&instance, (uint32_t *)arg, sizeof(uint32_t))) {
			return -EACCES;
		}

		psd = sdd_get_data(instance);

		error = sdd_ind_get_cmod_cap(psd, &cmod_val);
		if (error) {
			return -EACCES;
		}

		if (copy_to_user((uint32_t *)arg, &cmod_val, sizeof(uint32_t))) {
			return -EACCES;
		}

		break;
	}
	case 8:
	{
#if defined(CONFIG_SONOS_SDD_RTC)
		int error;
		int64_t instance;
		int64_t rtc_time;
		sdd_data_t *psd;

		if (copy_from_user(&instance, (int64_t *)arg, sizeof(time64_t))) {
			return -EACCES;
		}

		psd = sdd_get_data((uint32_t)instance);

		error = sdd_rtc_get_time(&rtc_time);
		if (error) {
			return -EACCES;
		}

		if (copy_to_user((int64_t *)arg, &rtc_time, sizeof(int64_t))) {
			return -EACCES;
		}
		break;
#else
		return -EINVAL;
#endif
	}
	case 9:
	{
		int error;
		sdd_data_t *psd;
		struct ble_settings ble_settings;

		if (copy_from_user(&ble_settings, (struct ble_settings *)arg, sizeof(struct ble_settings))) {
			return -EACCES;
		}

		psd = sdd_get_data(ble_settings.instance);

		if (psd->ble_carrier_tx_enabled) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR,
				"Can't enable/disable the BLE stack while carrier wave transmit is enabled!");
			return -EACCES;
		}

		error = sdd_set_btle(psd, (ble_settings.enable ? 1 : 0));
		if (error) {
			return -EACCES;
		}
		break;
	}
	case 10:
	{
		int error;
		sdd_data_t *psd;
		struct ble_carrier_settings ble_carrier_settings;

		if (copy_from_user(&ble_carrier_settings, (struct ble_carrier_settings *)arg, sizeof(struct ble_carrier_settings))) {
			return -EACCES;
		}

		psd = sdd_get_data(ble_carrier_settings.instance);

		error = sdd_ind_set_ble_carrier_chan(psd, ble_carrier_settings.channel);
		if (error) {
			return -EACCES;
		}

		error = sdd_ind_transmit_ble_carrier(psd, (ble_carrier_settings.transmit ? 1 : 0));
		if (error) {
			return -EACCES;
		}
		break;
	}
	case 11:
	{
		int error;
		sdd_data_t *psd;
		struct capsense_settings _settings;

		if (copy_from_user(&_settings, (struct _settings *)arg,
		                   sizeof(struct capsense_settings))) {
			return -EACCES;
		}

		psd = sdd_get_data(_settings.instance);

		error = sdd_set_touch(psd, (_settings.enable ? 1 : 0));
		if (error) {
			return -EACCES;
		}
		break;
	}
	case 12:
	{
		sdd_data_t *psd;
		struct capsense_settings _settings;

		if (copy_from_user(&_settings, (struct _settings *)arg, sizeof(_settings))) {
			return -EACCES;
		}

		psd = sdd_get_data(_settings.instance);
		if (!psd) {
			return -EACCES;
		}

		_settings.enable = (psd->touch_enable ? 1 : 0);

		if (copy_to_user((struct _settings *) arg, &_settings, sizeof(_settings))) {
			return -EACCES;
		}
		break;
	}
	default:
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Unrecognized IOCTL command %u\n", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}

struct file_operations sdd_fops =
{
	.unlocked_ioctl = sdd_device_ioctl,
	.open		= sdd_device_open,
	.release	= sdd_device_release,
	.read		= sdd_device_read,
};

static int
sdd_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	return sdd_init_device(i2c_client);
}

static int
sdd_i2c_remove(struct i2c_client *i2c_client)
{
	sdd_stop_device(i2c_client);
	return 0;
}

#ifdef CONFIG_PM
static int sdd_i2c_suspend(struct device *dev)
{
	return sdd_suspend();
}

static int sdd_i2c_resume(struct device *dev)
{
	return sdd_resume();
}
#endif

static const struct i2c_device_id sdd_i2c_id[] = {
	{ "psoc", 0 },
	{ "psoc", 1 },
	{ }
};
MODULE_DEVICE_TABLE(sdd_i2c, sdd_i2c_id);

static struct of_device_id sdd_ids[] = {
	{ .compatible = "Sonos,sge-psoc4" },
	{  }
};

#ifdef CONFIG_PM
#ifdef SET_LATE_SYSTEM_SLEEP_PM_OPS
static const struct dev_pm_ops sdd_i2c_pm_ops = {
	SET_LATE_SYSTEM_SLEEP_PM_OPS(sdd_i2c_suspend, sdd_i2c_resume)
};
#else
static const struct dev_pm_ops sdd_i2c_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(sdd_i2c_suspend, sdd_i2c_resume)
};
#endif
#endif

static struct i2c_driver sdd_i2c_driver = {
	.driver = {
		.name		= "sdd",
		.owner		= THIS_MODULE,
#ifdef CONFIG_PM
		.pm		= &sdd_i2c_pm_ops,
#endif
		.of_match_table = sdd_ids,
	},
	.id_table	= sdd_i2c_id,
	.probe		= sdd_i2c_probe,
	.remove		= sdd_i2c_remove,
};

struct cdev sdd_chr_dev;

void (*sdd_init_funcs[])(void) = {
	NULL
};

void sdd_run_init_funcs(void)
{
	int idx;
	for(idx = 0; sdd_init_funcs[idx]; idx++) {
		sdd_init_funcs[idx]();
	}
}

void (*sdd_exit_funcs[])(void) =  {
	NULL
};

void sdd_run_exit_funcs(void)
{
	int idx;
	for(idx = 0; sdd_exit_funcs[idx]; idx++) {
		sdd_exit_funcs[idx]();
	}
}

static int sdd_reboot_notifier(struct notifier_block *nb, unsigned long action, void *unused)
{
	if (action == SYS_POWER_OFF) {
		int error = 0;
		error = sdd_system_off();
		if (error) {
			return NOTIFY_BAD;
		}
	}
	return NOTIFY_DONE;
}

static struct notifier_block sdd_reboot_nb = {
	.notifier_call = sdd_reboot_notifier,
	.priority = 0x7fffffff,
};

static int __init sdd_init(void)
{
	int error;
	struct device *class_dev = NULL;

	devno = MKDEV(SDD_MAJOR_NUMBER, 0);

	sdd_run_init_funcs();

	error = i2c_add_driver(&sdd_i2c_driver);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "i2c_add_driver failed");
		sdd_run_exit_funcs();
		return error;
	}
#ifdef CONFIG_DEVTMPFS
	error = alloc_chrdev_region(&devno, 0, 1, SDD_DEVICE_NAME);
#else
	error = register_chrdev_region(devno, 1, SDD_DEVICE_NAME);
#endif
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Error %d registering device %s", error, SDD_DEVICE_NAME);
		i2c_del_driver(&sdd_i2c_driver);
		sdd_run_exit_funcs();
		return error;
	}
	cdev_init(&sdd_chr_dev, &sdd_fops);
	sdd_chr_dev.owner = THIS_MODULE;

	error = cdev_add(&sdd_chr_dev, devno, 1);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Couldn't add character device (%d).", error);
		sdd_stop_devices();
		unregister_chrdev_region(devno, 1);
		sdd_run_exit_funcs();
		return error;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, SDD_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Error creating sdd class.\n");
		cdev_del(&sdd_chr_dev);
		unregister_chrdev_region(devno, 1);
		error = PTR_ERR(class_dev);
		sdd_run_exit_funcs();
		return error;
	}

	if (sdd_get_data(0) == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_INFO, "No SGE entry in device tree.\n");
		return 0;
	}

	sdd_proc_init();
#ifdef CONFIG_DEBUG_FS
	sdd_debugfs_init();
#endif

#if defined(CONFIG_SONOS_SDD_RTC)
#if defined(SONOS_ARCH_ATTR_SOC_IS_A113)
	if (PRODUCT_ID_IS_DHUEZ) {
		sdd_rtc_alarm_recovery();
	}
#endif
#endif

	error = register_reboot_notifier(&sdd_reboot_nb);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: register reboot notifier error %d", __func__, error);
	}

	device_init_success = 1;
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "sdd module registered for %d, version %d", MAJOR(devno), SDD_VERSION);
	return 0;
}
module_init(sdd_init);

static void __exit sdd_exit(void)
{
	if (sdd_get_data(0) != NULL) {
		unregister_reboot_notifier(&sdd_reboot_nb);
#ifdef CONFIG_DEBUG_FS
		sdd_debugfs_remove();
#endif
		sdd_proc_remove();
	}
	sonos_device_destroy(devno);
	cdev_del(&sdd_chr_dev);
	unregister_chrdev_region(devno, 1);
	i2c_del_driver(&sdd_i2c_driver);
	sdd_run_exit_funcs();
#if defined(CONFIG_SONOS_SWD_BYPASS_GPIOLIB)
	cypress_swd_exit();
#endif
	sdd_rt_pmux_exit();
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "SDD exit");
}
module_exit(sdd_exit);

MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("Sonos, Inc.");
MODULE_DESCRIPTION ("SDD Module");
