/*
 * Copyright (c) 2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#ifdef CONFIG_DEBUG_FS
#include <linux/debugfs.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include "sdd.h"
#include "sge_common.h"
#include "blackbox.h"

#define MAX_PARSABLE_NUMBER_SIZE 24

static struct dentry *dfs_tree;

#define SEQ_FOPS(fops, getter, setter) \
	static int fops ## _open(struct inode *inodep, struct file *filp) \
	{ \
		return single_open(filp, getter, inodep->i_private); \
	} \
	static const struct file_operations fops = { \
		.open    = fops ## _open, \
		.read	 = seq_read, \
		.llseek	 = seq_lseek, \
		.write   = setter, \
		.release = single_release \
	}; \

static int sdd_ind_get(struct seq_file *m, void *p);
static ssize_t sdd_ind_set(struct file *file, const char __user *buf,
			 size_t len, loff_t *ppos);
SEQ_FOPS(fops_sdd_ind, sdd_ind_get, sdd_ind_set);

static int sdd_ind_stats(struct seq_file *m, void *p);
SEQ_FOPS(fops_sdd_stats, sdd_ind_stats, NULL);

static int sdd_max_signal_get(struct seq_file *m, void *p);
static ssize_t sdd_max_signal_set(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos);
SEQ_FOPS(fops_sdd_max_signal, sdd_max_signal_get, sdd_max_signal_set);

static int sge_show(struct seq_file *m, void *v);
static ssize_t sge_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
SEQ_FOPS(fops_sdd_sge, sge_show, sge_write);

static int sdd_xy_show(struct seq_file *m, void *v);
SEQ_FOPS(fops_sdd_xy, sdd_xy_show, NULL);

static int eco_trim_cap_show(struct seq_file *m, void *v);
static ssize_t eco_trim_cap_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
SEQ_FOPS(fops_eco_trim_cap, eco_trim_cap_show, eco_trim_cap_write);

static int sdd_swipe_distance_show(struct seq_file *m, void *v);
static ssize_t sdd_swipe_distance_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
SEQ_FOPS(fops_sdd_swipe_distance, sdd_swipe_distance_show, sdd_swipe_distance_write);

static int sdd_press_hold_time_show(struct seq_file *m, void *v);
static ssize_t sdd_press_hold_time_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
SEQ_FOPS(fops_sdd_press_hold_time, sdd_press_hold_time_show, sdd_press_hold_time_write);

static int sdd_double_tap_time_show(struct seq_file *m, void *v);
static ssize_t sdd_double_tap_time_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
SEQ_FOPS(fops_sdd_double_tap_time, sdd_double_tap_time_show, sdd_double_tap_time_write);

int sdd_log_show(struct seq_file *m, void *p);
SEQ_FOPS(fops_sdd_log, sdd_log_show, NULL);

int sdd_touchpad_log_show(struct seq_file *m, void *p);
SEQ_FOPS(fops_sdd_touchpad_log, sdd_touchpad_log_show, NULL);

ssize_t sdd_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
static struct file_operations fops_sdd_capsim = {
	.owner    = THIS_MODULE,
	.write    = sdd_write,
};

int sdd_show(struct seq_file *m, void *v);
static ssize_t sdd_data_set(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos);
SEQ_FOPS(fops_sdd_data, sdd_show, sdd_data_set);

typedef struct {
	const char* filename;
	umode_t access;
	int idx;
} sdd_ind_file;

static const sdd_ind_file sdd_ind_files[] = {
	{ "baseline", S_IRUGO, SGE_REG_IND_IDX_BASELINE },
	{ "baseline-inits", S_IRUGO, SGE_REG_IND_IDX_BASELINE_INITS },
	{ "ble-carrier-channel", S_IRUGO, SGE_REG_IND_IDX_BLE_CARRIER_CHAN },
	{ "cmod-cap", S_IRUGO, SGE_REG_IND_IDX_CMOD_CAP },
	{ "compensation-idac", S_IRUGO, SGE_REG_IND_IDX_COMP_IDAC },
	{ "crc32", S_IRUGO, SGE_REG_IND_IDX_CRC32 },
	{ "deep-sleeps", S_IRUGO, SGE_REG_IND_IDX_DEEP_SLEEPS },
	{ "die-temp", S_IRUGO, SGE_REG_IND_IDX_DIE_TEMP },
	{ "finger-hysteresis", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_FINGER_HYSTERESIS },
	{ "finger-threshold", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_FINGER_THRESHOLD },
	{ "hold-time", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_HOLD_TIME },
	{ "modulation-idac", S_IRUGO, SGE_REG_IND_IDX_MOD_IDAC },
	{ "move-interval", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_MOVE_INTERVAL },
	{ "no-diffs", S_IRUGO, SGE_REG_IND_IDX_NO_DIFFS },
	{ "noise-threshold", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_NOISE_THRESHOLD },
	{ "num-sensors", S_IRUGO, SGE_REG_IND_IDX_NUM_SENSORS },
	{ "parasitic-cap", S_IRUGO, SGE_REG_IND_IDX_PARASITIC_CAP },
	{ "pmu-pulses", S_IRUGO, SGE_REG_IND_IDX_PMU_PULSES },
	{ "raw-cap", S_IRUGO, SGE_REG_IND_IDX_RAW_CAP },
	{ "repeat-time", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_REPEAT_TIME },
	{ "signal", S_IRUGO, SGE_REG_IND_IDX_SIGNAL },
	{ "tpintr-delta", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_TP_INTR_DELTA },
	{ "prox-finger-hysteresis", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_PROX_HYSTERESIS},
	{ "prox-finger-threshold", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_PROX_FINGER_THRESH },
	{ "prox-noise-threshold", S_IRUGO | S_IWUGO, SGE_REG_IND_IDX_PROX_NOISE_THRESH },
	{ NULL }
};

typedef struct {
	const sdd_ind_file* sdd_reg;
	int sdd_idx;
} sdd_ind_register;

static sdd_ind_register sdd_ind_reg[SDD_MAX_PSOC][SGE_REG_SET_SIZE];

void sdd_debugfs_init(void)
{
	const sdd_ind_file* ind_idx;
	sdd_ind_register* reg;
	struct dentry* result;
	struct dentry* parent;
	sdd_data_t *psd;
	char path[32];
	int i;

	dfs_tree = debugfs_create_dir("sdd", NULL);
	if (dfs_tree == NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs for SDD");
		return ;
	}

	result = debugfs_create_file("stats", S_IRUGO, dfs_tree, NULL, &fops_sdd_stats);
	if (result == NULL)  {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs stats node for SDD");
		return ;
	}

	psd = sdd_get_data(0);
	if (!(psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH)) {
		result = debugfs_create_file("grr", S_IRUGO, dfs_tree, psd, &fops_sdd_log);
		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs log node for SDD");
			return ;
		}

		result = debugfs_create_file("cap-touch-sim", S_IWUGO, dfs_tree, NULL, &fops_sdd_capsim);
		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs cap touch simulation node for SDD");
			return ;
		}

		result = debugfs_create_file("touchpad-log", S_IWUGO, dfs_tree, NULL, &fops_sdd_touchpad_log);
		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs touchpad log");
			return ;
		}
	}

	for (i = 0; i < sdd_psoc_inst; i++) {
		psd = sdd_get_data(i);

		scnprintf(path, sizeof(path), "%i", i);
		parent = debugfs_create_dir(path, dfs_tree);

		if (parent == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs device node for SDD");
			continue ;
		}

		result = debugfs_create_file("data", S_IRUGO, parent, psd, &fops_sdd_data);

		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs data node for SDD");
			continue ;
		}

		result = debugfs_create_file("sge", S_IRUGO | S_IWUGO, parent, psd, &fops_sdd_sge);

		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs sge node for SDD");
			continue ;
		}

		if (!(psd->dts_config_flags & SDD_DTS_NO_CAPTOUCH)) {
			result = debugfs_create_file("max-signal", S_IRUGO | S_IWUGO, parent, psd, &fops_sdd_max_signal);

			if (result == NULL)  {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs max-signal node for SDD");
				continue ;
			}
		}

		if (psd->dts_config_flags & SDD_DTS_ECO_TRIM_CAP) {
			result = debugfs_create_file("eco_trim_cap", S_IRUGO | S_IWUGO, parent, psd, &fops_eco_trim_cap);
			if (result == NULL)  {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs eco_trim_cap node for SDD");
				continue ;
			}
		}

		result = debugfs_create_file("xy", S_IRUGO | S_IWUGO, parent, psd, &fops_sdd_xy);
		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs xy node for SDD");
			continue ;
		}

		result = debugfs_create_file("swipe-distance", S_IRUGO | S_IWUGO, parent, psd, &fops_sdd_swipe_distance);
		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs swipe-distance node for SDD");
			continue;
		}

		result = debugfs_create_file("press-hold-time", S_IRUGO | S_IWUGO, parent, psd, &fops_sdd_press_hold_time);
		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs hold-time node for SDD");
			continue;
		}

		result = debugfs_create_file("double-tap-time", S_IRUGO | S_IWUGO, parent, psd, &fops_sdd_double_tap_time);
		if (result == NULL)  {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs double-tap-time node for SDD");
			continue;
		}

		for (ind_idx = &sdd_ind_files[0]; ind_idx->filename != NULL; ind_idx++) {
			reg = &sdd_ind_reg[i][ind_idx->idx];
			reg->sdd_reg = ind_idx;
			reg->sdd_idx = i;

			result = debugfs_create_file(ind_idx->filename, ind_idx->access, parent, reg, &fops_sdd_ind);

			if (result == NULL)  {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "Could not create debugfs node %s for SDD", ind_idx->filename);
				continue ;
			}
		}
	}
}

void sdd_debugfs_remove(void)
{
	debugfs_remove_recursive(dfs_tree);
}

static int sdd_ind_get(struct seq_file *m, void *p)
{
	sdd_ind_register *priv = m->private;
	sdd_data_t *psd;
	int error = 0;
	u32 val = 0;
	u8 option;
	int i;

	psd = sdd_get_data(priv->sdd_idx);
	option = priv->sdd_reg->idx;

	switch (option) {
	case SGE_REG_IND_IDX_FINGER_HYSTERESIS:
		error = sdd_ind_get_finger_hysteresis(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_FINGER_THRESHOLD:
		error = sdd_ind_get_finger_threshold(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_REPEAT_TIME:
		error = sdd_ind_get_repeat_time(psd, (u16 *)&val);
		break;
	case SGE_REG_IND_IDX_HOLD_TIME:
		error = sdd_ind_get_hold_time(psd, (u16 *)&val);
		break;
	case SGE_REG_IND_IDX_NOISE_THRESHOLD:
		error = sdd_ind_get_noise_threshold(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_MOVE_INTERVAL:
		error = sdd_ind_get_move_interval(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_DIE_TEMP:
		error = sdd_ind_get_die_temp(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_CRC32:
		error = sdd_ind_get_crc32(psd, &val);
		break;
	case SGE_REG_IND_IDX_NUM_SENSORS:
		error = sdd_ind_get_num_sensors(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_BASELINE_INITS:
		error = sdd_ind_get_baseline_inits(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_NO_DIFFS:
		error = sdd_ind_get_no_diffs(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_PMU_PULSES:
		error = sdd_ind_get_pmu_pulses(psd, (u16 *)&val);
		break;
	case SGE_REG_IND_IDX_DEEP_SLEEPS:
		error = sdd_ind_get_deep_sleeps(psd, (u16 *)&val);
		break;
	case SGE_REG_IND_IDX_CMOD_CAP:
		error = sdd_ind_get_cmod_cap(psd, &val);
		break;
	case SGE_REG_IND_IDX_BLE_CARRIER_CHAN:
		error = sdd_ind_get_ble_carrier_chan(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_TP_INTR_DELTA:
		error = sdd_ind_get_tpintr_delta(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_PROX_HYSTERESIS:
		error = sdd_ind_get_prox_finger_hysteresis(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_PROX_FINGER_THRESH:
		error = sdd_ind_get_prox_finger_threshold(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_PROX_NOISE_THRESH:
		error = sdd_ind_get_prox_noise_threshold(psd, (u8 *)&val);
		break;
	case SGE_REG_IND_IDX_PARASITIC_CAP:
		for (i = 0; i < psd->num_sensors; i++) {
			error = sdd_ind_get_sensor32(psd, option, i, &val);
			if (error) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "error reading index 0x%x: %d\n", option, error);
				return 0;
			} else {
				seq_printf(m, "%s%04x", (i == 0) ? "" : ",", val);
			}
		}
		seq_printf(m, "\n");
		return 0;
	case  SGE_REG_IND_IDX_RAW_CAP:
	case  SGE_REG_IND_IDX_MOD_IDAC:
	case  SGE_REG_IND_IDX_COMP_IDAC:
	case  SGE_REG_IND_IDX_SIGNAL:
	case  SGE_REG_IND_IDX_BASELINE:
		for (i = 0; i < psd->num_sensors; i++) {
			error = sdd_ind_get_sensor16(psd, option, i, (u16 *)&val);
			if (error) {
				bb_log(BB_MOD_SDD, BB_LVL_ERR, "error reading index 0x%x: %d\n", option, error);
				return 0;
			} else {
				seq_printf(m, "%s%04x", i ? "," : "", val);
			}
		}
		seq_printf(m, "\n");
		return 0;
	default:
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "unknown option %d", option);
		return 0;
	}

	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "error reading index 0x%x: %d", option, error);
		return 0;
	}

	seq_printf(m, "%x\n", val);

	return 0;
}

static ssize_t sdd_ind_set(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos)
{
	sdd_ind_register *priv = file_inode(filep)->i_private;
	sdd_data_t *psd;
	u8 option;
	int error = 0;
	int val = 0;
	char buff[MAX_PARSABLE_NUMBER_SIZE];

	if (*ppos) {
		return -EINVAL;
	}

	psd = sdd_get_data(priv->sdd_idx);
	option = priv->sdd_reg->idx;

	if (simple_write_to_buffer(buff, sizeof(buff), ppos, ubuf, len) != len) {
		return -EIO;
	}
	if (len < sizeof(buff)) {
		buff[len] = 0;
	}
	if (kstrtoint(buff, 16, &val)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "Cannot parse %s", buff);
		return -EINVAL;
	}

	switch (option) {
	case SGE_REG_IND_IDX_FINGER_HYSTERESIS:
		error = sdd_ind_set_finger_hysteresis(psd, (u8)val);
		break;
	case SGE_REG_IND_IDX_FINGER_THRESHOLD:
		error = sdd_ind_set_finger_threshold(psd, (u8)val);
		break;
	case SGE_REG_IND_IDX_REPEAT_TIME:
		error = sdd_ind_set_repeat_time(psd, (u16)val);
		break;
	case SGE_REG_IND_IDX_HOLD_TIME:
		error = sdd_ind_set_hold_time(psd, (u16)val);
		break;
	case SGE_REG_IND_IDX_NOISE_THRESHOLD:
		error = sdd_ind_set_noise_threshold(psd, (u8)val);
		break;
	case SGE_REG_IND_IDX_MOVE_INTERVAL:
		error = sdd_ind_set_move_interval(psd, (u8)val);
		break;
	case SGE_REG_IND_IDX_TP_INTR_DELTA:
		error = sdd_ind_set_tpintr_delta(psd, (u8)val);
		break;
	case SGE_REG_IND_IDX_PROX_HYSTERESIS:
		error = sdd_ind_set_prox_finger_hysteresis(psd, (u8)val);
		break;
	case SGE_REG_IND_IDX_PROX_FINGER_THRESH:
		error = sdd_ind_set_prox_finger_threshold(psd, (u8)val);
		break;
	case SGE_REG_IND_IDX_PROX_NOISE_THRESH:
		error = sdd_ind_set_prox_noise_threshold(psd, (u8)val);
		break;
	default:
		bb_log(BB_MOD_SDD, BB_LVL_WARNING,  "write for 0x%x not supported", option);
		return 0;
	}

	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR,  "Could not set 0x%x: %i", option, error);
		return error;
	}

	return len;
}

static int sdd_max_signal_get(struct seq_file *m, void *p)
{
	const u8 option = SGE_REG_IND_IDX_MAX_SIG;
	sdd_data_t *psd = m->private;
	int error = 0;
	u16 val;
	int i;

	for (i = 0; i < psd->num_sensors; i++) {
		error = sdd_ind_get_sensor16(psd, option, i, &val);
		if (error) {
			bb_log(BB_MOD_SDD, BB_LVL_ERR, "error reading index 0x%x: %d\n", option, error);
			return 0;
		} else {
			seq_printf(m, "%s%04x", i ? "," : "", val);
		}
	}

	seq_printf(m, "\n");
	return 0;
}

static ssize_t sdd_max_signal_set(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos)
{
	sdd_data_t *psd = file_inode(filep)->i_private;
	char buff[12] = {0};
	int trim;

	if (simple_write_to_buffer(buff, sizeof(buff), ppos, ubuf, len) != len) {
		return -EIO;
	}

	for (trim = len - 1; trim >=0 && buff[trim] < 0x20; trim--) buff[trim] = 0;

	if (strcmp(buff, "clear") == 0) {
		sdd_ind_clear_max_signal(psd);
	}

	return len;
}

static int eco_trim_cap_show(struct seq_file *m, void *p)
{
	sdd_data_t *psd = m->private;
	int error = 0;
	u32 trim;

	error = sdd_ind_get_eco_trim_cap(psd, &trim);
	if (error) {
		seq_printf(m, "  ECO trim cap: error reading\n");
		return 0;
	} else {
		seq_printf(m, "  ECO trim cap: %04x\n", trim);
	}

	return 0;
}

static ssize_t eco_trim_cap_write(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos)
{
	sdd_data_t *psd = file_inode(filep)->i_private;
	int error = 0;
	char buff[12] = {0};
	u32 trim;

	if (simple_write_to_buffer(buff, sizeof(buff), ppos, ubuf, len) != len) {
		return -EIO;
	}

	if (kstrtouint(buff, 16, &trim) !=0 ) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "sdd debugfs: Failed to parse the value.");
		return -EFAULT;
	}

	error = sdd_ind_set_eco_trim_cap(psd, trim);
	if (error) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "configure eco trim cap error %d", error);
	}


	return len;
}

static ssize_t sdd_data_set(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos)
{
	sdd_data_t *psd = file_inode(filep)->i_private;
	char buff[12] = {0};
	int trim;

	if (simple_write_to_buffer(buff, sizeof(buff), ppos, ubuf, len) != len) {
		return -EIO;
	}

	for (trim = len - 1; trim >=0 && buff[trim] < 0x20; trim--) buff[trim] = 0;

	if (strcmp(buff, "erase") == 0) {
		sdd_erase_sge(psd);
	} if (strcmp(buff, "program") == 0) {
		sdd_program_sge(psd);
	}

	return len;
}

static int sdd_ind_stats(struct seq_file *m, void *p)
{
	sdd_data_t *psd;
	int i;

	for (i = 0; i < sdd_psoc_inst; i++) {
		psd = sdd_get_data(i);
		seq_printf(m, "\nStats %i\n", i);
		seq_printf(m, "%09u ZoneA presses\n", psd->zonea_press_count);
		seq_printf(m, "%09u ZoneA repeats\n", psd->zonea_repeat_count);
		seq_printf(m, "%09u ZoneA releases\n", psd->zonea_release_count);
		seq_printf(m, "%09u ZoneB presses\n", psd->zoneb_press_count);
		seq_printf(m, "%09u ZoneB repeats\n", psd->zoneb_repeat_count);
		seq_printf(m, "%09u ZoneB releases\n", psd->zoneb_release_count);
		seq_printf(m, "%09u ZoneC presses\n", psd->zonec_press_count);
		seq_printf(m, "%09u ZoneC repeats\n", psd->zonec_repeat_count);
		seq_printf(m, "%09u ZoneC releases\n", psd->zonec_release_count);
		if (psd->gesture_match->zonem != 0) {
			seq_printf(m, "%09u ZoneM presses\n", psd->zonem_press_count);
			seq_printf(m, "%09u ZoneM repeats\n", psd->zonem_repeat_count);
			seq_printf(m, "%09u ZoneM releases\n", psd->zonem_release_count);
		}
		seq_printf(m, "%09u Swipes\n", psd->swipe_count);
	}
	seq_printf(m, "\n");

	return 0;
}

static int sge_show(struct seq_file *m, void *v)
{
	sdd_data_t *psd = (sdd_data_t *)(m->private);
	int reg;
	u8 val;

	for (reg = 0; reg < SGE_REG_SET_SIZE; reg++) {
		sdd_read_sge_byte(psd, reg, &val);
		if ((reg % 64) == 0) {
			seq_printf(m, "\n\n");
		} else if ((reg % 16) == 0) {
			seq_printf(m, "\n");
		} else if ((reg % 8) == 0) {
			seq_printf(m, "  ");
		}
		seq_printf(m, "%02x ", val);
	}
	seq_printf(m, "\n\n");
	return 0;
}

static ssize_t
sge_write(struct file *filep, const char __user * buffer, size_t count, loff_t *data)
{
	sdd_data_t *psd = file_inode(filep)->i_private;
	s32 temp;
	char buf[40];
	char *peq;
	int error;
	u8 val, reg;

	if (count >= sizeof(buf))
		return -EIO;
	if (copy_from_user(buf, buffer, count))
		return -EFAULT;
	buf[count] = 0;

	peq = strchr(buf, '=');
	if (peq != NULL) {
		*peq = '\0';
		error = kstrtoint(peq+1, 16, &temp);
		val = temp;
		if (strncmp(buf, "reg", 3) == 0) {
			error = kstrtoint(peq-2, 16, &temp);
			reg = temp;
			error = sdd_write_sge_byte(psd, reg, val);
			if (error) {
				bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: error %d", __func__, error);
			}
		}
	}
	return count;
}

static int sdd_xy_show(struct seq_file *m, void *v)
{
	sdd_data_t *psd = (sdd_data_t *)(m->private);
	int reg = SGE_REG_XY;
	u32 val;

        sdd_read_sge_block(psd, reg, sizeof(val), (u8 *)&val);
        seq_printf(m, "%08x\n", val);
	return 0;
}

static int sdd_swipe_distance_show(struct seq_file *m, void *p)
{
	sdd_data_t *psd = m->private;
	int error = 0;
	u32 dist;

	error = sdd_read_swipe_distance(psd, &dist);
	if (!error) {
		seq_printf(m, "%d\n", dist);
	}
	return 0;
}

static ssize_t sdd_swipe_distance_write(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos)
{
	sdd_data_t *psd = file_inode(filep)->i_private;
	int error = 0;
	char buff[12] = {0};
	u32 dist;

	if (simple_write_to_buffer(buff, sizeof(buff), ppos, ubuf, len) != len) {
		return -EIO;
	}
	if (kstrtouint(buff, 10, &dist) !=0 ) {
	    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: Failed to parse the value", __func__);
		return -EFAULT;
	}
	error = sdd_write_swipe_distance(psd, dist);
	if (error) {
	    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: error %d", __func__, error);
	}
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: dist %d", __func__, dist);
	return len;
}

static int sdd_press_hold_time_show(struct seq_file *m, void *p)
{
	sdd_data_t *psd = m->private;
	int error = 0;
	u32 msec;

	error = sdd_read_press_hold(psd, &msec);
	if (!error) {
		seq_printf(m, "%d\n", msec);
	}
	return 0;
}

static ssize_t sdd_press_hold_time_write(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos)
{
	sdd_data_t *psd = file_inode(filep)->i_private;
	int error = 0;
	char buff[12] = {0};
	u32 msec;

	if (simple_write_to_buffer(buff, sizeof(buff), ppos, ubuf, len) != len) {
		return -EIO;
	}
	if (kstrtouint(buff, 10, &msec) !=0 ) {
	    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: Failed to parse the value", __func__);
		return -EFAULT;
	}
	error = sdd_write_press_hold(psd, msec);
	if (error) {
	    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: error %d", __func__, error);
	}
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: msec %d", __func__, msec);
	return len;
}

static int sdd_double_tap_time_show(struct seq_file *m, void *p)
{
	sdd_data_t *psd = m->private;
	int error = 0;
	u32 msec;

	error = sdd_read_double_tap(psd, &msec);
	if (!error) {
		seq_printf(m, "%d\n", msec);
	}
	return 0;
}

static ssize_t sdd_double_tap_time_write(struct file *filep, const char __user *ubuf,
			 size_t len, loff_t *ppos)
{
	sdd_data_t *psd = file_inode(filep)->i_private;
	int error = 0;
	char buff[12] = {0};
	u32 msec;

	if (simple_write_to_buffer(buff, sizeof(buff), ppos, ubuf, len) != len) {
		return -EIO;
	}
	if (kstrtouint(buff, 10, &msec) !=0 ) {
	    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: Failed to parse the value", __func__);
		return -EFAULT;
	}
	error = sdd_write_double_tap(psd, msec);
	if (error) {
	    bb_log(BB_MOD_SDD, BB_LVL_ERR, "%s: error %d", __func__, error);
	}
	bb_log(BB_MOD_SDD, BB_LVL_INFO, "%s: msec %d", __func__, msec);
	return len;
}

#endif
