/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include "blackbox.h"
#include "sdd.h"

typedef struct {
	char *option;
	int   value;
} sddcmd_str_t;

typedef struct {
	unsigned int min;
	unsigned int max;
	unsigned int base;
} sddcmd_num_t;

#define SDDCMD_NUM   0
#define SDDCMD_STR   1

typedef struct {
	char *name;
	int   type;
	void  *values;
	void (*action)(sdd_data_t *, u32);
	int  hide;
} sddcmd_t;

sddcmd_num_t sddcmd_press_hold_time = {.min = 0, .max = 5000, .base = 10};
void sddcmd_set_press_hold_time(sdd_data_t *psd, u32 p)
{
	sdd_set_press_hold_time(psd, p);
}

sddcmd_num_t sddcmd_repeat_time = {.min = 0, .max = 5000, .base = 10};
void sddcmd_set_repeat_time(sdd_data_t *psd, u32 p)
{
	sdd_set_repeat_time(psd, p);
}

sddcmd_str_t sddcmd_generic_enable[] = {
	{"disable", 0},
	{"enable",  1},
	{NULL, 0}
};
void sddcmd_set_touch(sdd_data_t *psd, u32 p)
{
	int enable = (int)p;
	sdd_set_touch(psd, enable);
}

sddcmd_num_t sddcmd_samples = {.min = 1, .max = 50, .base = 10};
void sddcmd_set_min_samples_per_gesture(sdd_data_t *psd, u32 p)
{
	sdd_set_min_samples_per_gesture(psd, p);
}
void sddcmd_set_min_samples_per_tap(sdd_data_t *psd, u32 p)
{
	sdd_set_min_samples_per_tap(psd, p);
}

sddcmd_num_t sddcmd_threshold = {.min = 1, .max = 254, .base = 10};
void sddcmd_set_finger_threshold(sdd_data_t *psd, u32 p)
{
	sdd_set_finger_threshold(psd, p);
}
void sddcmd_set_noise_threshold(sdd_data_t *psd, u32 p)
{
	sdd_set_noise_threshold(psd, p);
}

sddcmd_num_t sddcmd_hysteresis = {.min = 0, .max = 254, .base = 10};
void sddcmd_set_finger_hysteresis(sdd_data_t *psd, u32 p)
{
	sdd_set_finger_hysteresis(psd, p);
}

sddcmd_num_t sddcmd_move_interval = {.min = 1, .max = 255, .base = 10};
void sddcmd_set_move_interval(sdd_data_t *psd, u32 p)
{
	sdd_set_move_interval(psd, p);
}

sddcmd_num_t sddcmd_touch_timer= {.min = 0, .max = 10*60*1000, .base = 10};
void sddcmd_set_touch_timer(sdd_data_t *psd, u32 p)
{
	u32 msecs = p;
	sdd_change_timer(psd->touch_timer, msecs);
}

sddcmd_str_t sddcmd_max_signal[] = {
	{"clear", 0},
	{NULL, 0}
};
void sddcmd_max_signal_func(sdd_data_t *psd, u32 p)
{
	sdd_ind_clear_max_signal(psd);
}

sddcmd_str_t sddcmd_psoc_flash[] = {
	{"erase",    SDD_FLASH_CMD_ERASE},
	{"program",  SDD_FLASH_CMD_PROGRAM},
	{NULL, 0}
};
void sddcmd_psoc_flash_func(sdd_data_t *psd, u32 p)
{
	int cmd = (int)p;

	switch(cmd) {
	case SDD_FLASH_CMD_ERASE:
		sdd_erase_sge(psd);
		break;
	case SDD_FLASH_CMD_PROGRAM:
		sdd_program_sge(psd);
		break;
	}
}

int sddcmd_active_psoc = 0;
sddcmd_num_t sddcmd_psoc_num = {.min = 0, .max = 1, .base = 10};
void sddcmd_set_psoc_num(sdd_data_t *psd, u32 p)
{
	sddcmd_active_psoc = p;
}

static sddcmd_t sddcmds[] = {
	{"touch",             SDDCMD_STR, &sddcmd_generic_enable, sddcmd_set_touch},
	{"press-hold-time",   SDDCMD_NUM, &sddcmd_press_hold_time, sddcmd_set_press_hold_time},
	{"repeat-time",       SDDCMD_NUM, &sddcmd_press_hold_time, sddcmd_set_repeat_time},
	{"finger-threshold",  SDDCMD_NUM, &sddcmd_threshold, sddcmd_set_finger_threshold},
	{"noise-threshold",   SDDCMD_NUM, &sddcmd_threshold, sddcmd_set_noise_threshold},
	{"finger-hysteresis", SDDCMD_NUM, &sddcmd_hysteresis, sddcmd_set_finger_hysteresis},
	{"move-interval",     SDDCMD_NUM, &sddcmd_move_interval, sddcmd_set_move_interval},
	{"touch-timer",       SDDCMD_NUM, &sddcmd_touch_timer, sddcmd_set_touch_timer},
	{"max-signal",        SDDCMD_STR, &sddcmd_max_signal, sddcmd_max_signal_func},
	{"psoc-flash",        SDDCMD_STR, &sddcmd_psoc_flash, sddcmd_psoc_flash_func},
	{"psoc",              SDDCMD_NUM, &sddcmd_psoc_num, sddcmd_set_psoc_num},
	{.name = NULL}
};

int sddcmd_show(struct seq_file *m, void *v)
{
	sddcmd_t *cmd;
	sddcmd_num_t *cmdnum;
	sddcmd_str_t *cmdstr;

	seq_printf(m, "\n");
	cmd = sddcmds;
	while (cmd->name != NULL) {
		if (cmd->hide) {
			cmd++;
			continue;
		}
		if (cmd->type == SDDCMD_NUM) {
			cmdnum = cmd->values;
			if (cmdnum->base == 16) {
				seq_printf(m, "%s: min 0x%X, max 0x%X\n\n",
					   cmd->name, cmdnum->min, cmdnum->max);
			} else {
				seq_printf(m, "%s: min %u, max %u\n\n",
					   cmd->name, cmdnum->min, cmdnum->max);
			}
		} else if (cmd->type == SDDCMD_STR) {
			cmdstr = cmd->values;
			seq_printf(m, "%s\n", cmd->name);
			while (cmdstr->option != NULL) {
				seq_printf(m, "  %s\n", cmdstr->option);
				cmdstr++;
			}
			seq_printf(m, "\n");
		}
		cmd++;
	}
	seq_printf(m, "active psoc %d\n", sddcmd_active_psoc);
	seq_printf(m, "\n");
	return 0;
}

int sddcmd_run(sdd_data_t *psd, char *s)
{
	sddcmd_t *cmd;
	sddcmd_num_t *cmdnum;
	sddcmd_str_t *cmdstr;
	char *param, *keyword, *end;
	int done;
	int error;
	u32 ul;

	keyword = s;
	param = strchr(s, '=');
	if (param == NULL) {
		return 0;
	}
	*param++ = '\0';
	end = param;
	while (*end++ >= 0x20);
	*(end-1) = '\0';

	done = 0;
	cmd = sddcmds;
	while (!done && (cmd->name != NULL)) {
		if (strcmp(keyword, cmd->name) == 0) {
			if (cmd->type == SDDCMD_NUM) {
				cmdnum = cmd->values;
				error = kstrtouint(param, cmdnum->base, &ul);
				if ((ul >= cmdnum->min) && (ul <= cmdnum->max)) {
					cmd->action(psd, ul);
					done = 1;
				} else {
					bb_log(BB_MOD_SDD, BB_LVL_WARNING, "%s: min %u, max %u, value %u\n",
					       cmd->name, cmdnum->min, cmdnum->max, ul);
				}
			} else if (cmd->type == SDDCMD_STR) {
				cmdstr = cmd->values;
				while (cmdstr->option != NULL) {
					if (strcmp(param, cmdstr->option) == 0) {
						cmd->action(psd, cmdstr->value);
						done = 1;
						break;
					}
					cmdstr++;
				}
			}
			break;
		}
		cmd++;
	}
	return done;
}

static ssize_t sddcmd_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	sdd_data_t *psd = sdd_get_data(sddcmd_active_psoc);
	char buf[100];

	if (count >= sizeof(buf))
		return -EIO;
	if (copy_from_user(buf, buffer, count))
		return -EFAULT;
	buf[count] = 0;
	if (!sddcmd_run(psd, buf)) {
		bb_log(BB_MOD_SDD, BB_LVL_ERR, "sddcmd failed");
	}
	return count;
}

static int sddcmd_open(struct inode *inode, struct file *file)
{
	return single_open(file, sddcmd_show, PDE_DATA(inode));
}

static struct file_operations sddcmd_proc_operations = {
	.owner    = THIS_MODULE,
	.open     = sddcmd_open,
	.write    = sddcmd_write,
	.read     = seq_read,
	.llseek   = seq_lseek,
	.release  = single_release,
};

int sddcmd_init(void)
{
	struct proc_dir_entry *entry;
	entry = proc_create("driver/cmd", 0666, NULL, &sddcmd_proc_operations);
	if( !entry ) {
		return( -EIO );
	}
	return 0;
}

void sddcmd_remove(void)
{
	remove_proc_entry("driver/cmd", NULL);
}
