/*
 * Copyright (c) 2014-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#ifndef __SIG_DETAIL_H__
#define __SIG_DETAIL_H__

#include <linux/timer.h>
#include <linux/list.h>
#include <asm/uaccess.h>
#include <linux/interrupt.h>
#include <linux/completion.h>
#include <linux/i2c.h>
#include <linux/mutex.h>
#include <linux/gpio.h>
#include "adp8863.h"
#include "sge_common.h"
#include "event_queue_api.h"
#include "cypress_swd_pub.h"

#define SGE_PRESS_HOLD_TIME             340
#define SGE_REPEAT_TIME                 160
#define SGE_MIN_SAMPLES_PER_GESTURE       1
#define SGE_MIN_SAMPLES_PER_TAP           1
#define SGE_MOVE_INTERVAL                51

#define SDD_FLASH_CMD_ERASE     1
#define SDD_FLASH_CMD_PROGRAM   2

#define SDD_TIMER_STATE_STOP    0
#define SDD_TIMER_STATE_RUN     1
#define SDD_TIMER_STATE_EXPIRED 2

#define SDD_DTS_SHARE_SWD_UART  0x01
#define SDD_DTS_PSOC_RTC        0x02
#define SDD_DTS_LOW_POWER       0x04
#define SDD_DTS_ECO_TRIM_CAP    0x08
#define SDD_DTS_NO_CAPTOUCH	0x10
#define SDD_DTS_RPMSG           0x20
#define SDD_DTS_CHECK_POWER     0x40
#define SDD_DTS_SHARE_SWD_I2C   0x80

#define SDD_MAX_GPIO            4
#define SGE_MAX_PSOC_IMAGES     2

typedef struct _sdd_gesture_match {
	char   *name;
	u16     zonea;
	u16     zoneb;
	u16     zonec;
	u16     zonem;
	u16     zonea_hold;
	u16     zoneb_hold;
	u16     zonec_hold;
	u16     zonem_hold;
} sdd_gesture_match_t;

#define SDD_TIMER_NAME_LEN	32
typedef struct _sdd_timer {
	struct list_head    list;
	char                name[SDD_TIMER_NAME_LEN];
	int                 state;
	struct timer_list   timer;
	unsigned int        timeout_ms;
	void (*handler)(void *);
	unsigned int        timeout_count;
	unsigned int        processed_count;
	unsigned int        stop_count;
	unsigned int        restart_count;
	spinlock_t          lock;
	struct _sdd_data_s *sdd_data;
} sdd_timer_t;

struct sdd_grr_change_entry {
	u16 prev_grr;
	u16 new_grr;
	u32 prev_msec;
	u32 intr_msec;
	u32 i2c_msec;
};

struct sdd_log {
	u32 head;
	u32 tail;
	struct sdd_grr_change_entry entries[1];
};

typedef struct _sdd_data_s {
	int                 inst;
	int                 disabled_by_fatal_error;
	int                 sge_init_done;
	int                 psoc_erased;
	int                 suspended;
	int                 rst_pin_config;
	struct i2c_client   *i2c_client;
	sdd_gesture_match_t *gesture_match;
	struct sge_config_param *sge_config_table;
	sdd_timer_t         *touch_timer;
	sdd_timer_t         *sge_monitor_timer;
	unsigned int        sge_prev_pass;
	unsigned int        sge_stuck_count;
	unsigned int        sge_recoveries;
	struct sdd_log      *grr_event_log;
        struct sdd_indirect_ops *indirect_ops;
	unsigned int        i2c_read_count;
	unsigned int        i2c_write_count;
	unsigned int        i2c_error_count;
	unsigned int        i2c_retry_count;
	unsigned int        irq_count;
	unsigned int        irq_processed;
	unsigned int        irq_spurious;
	unsigned int        touch_count;
	unsigned int        baseline_update_count;
	unsigned int        zonea_press_count;
	unsigned int        zoneb_press_count;
	unsigned int        zonec_press_count;
	unsigned int        zonem_press_count;
	unsigned int        zonea_release_count;
	unsigned int        zoneb_release_count;
	unsigned int        zonec_release_count;
	unsigned int        zonem_release_count;
	unsigned int        zonea_repeat_count;
	unsigned int        zoneb_repeat_count;
	unsigned int        zonec_repeat_count;
	unsigned int        zonem_repeat_count;
	unsigned int        swipe_count;
	unsigned int        completion_errors;
	struct completion   sdd_completion;
	struct list_head    timers;
	struct gpio         gpio_table[SDD_MAX_GPIO];
	int                 num_gpio;
	int                 psoc_reset;
	int                 psoc_skip_init_reset;
	int                 psoc_flash_verify_disabled;
	struct task_struct *sddtask;
	atomic_t            sge_irq_pending;
	unsigned int        passes;
	u32                 irq;
	unsigned long       prev_jiffies;
	unsigned long       intr_jiffies;
	u16                 prev_grr;
	u16                 press_hold_time;
	u16                 repeat_time;
	u16                 all_holds_bitmap;
	int                 touch_enable;
	u8                  min_samples_per_gesture;
	u8                  min_samples_per_tap;
	u8                  finger_threshold;
	u8                  noise_threshold;
	u8                  finger_hysteresis;
	u8                  move_interval;
	u8                  baseline_interval;
	u32                 dts_config_flags;
	u16                 psoc_features;
	struct mutex        ind_lock;
	struct psoc_cfg     psoc_cfgs[SGE_MAX_PSOC_IMAGES];
	struct psoc_cfg     *psoc_cfg;
	int                 num_psoc_cfgs;
	int                 num_sensors;
	int                 check_raw_cap;
	int                 btle_enabled;
	int                 ble_carrier_tx_enabled;
} sdd_data_t;

struct sdd_indirect_ops {
	int (*read)(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len);
	int (*write)(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len);
	int (*read_special)(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len);
};

struct sge_config_param {
	int (*func)(sdd_data_t *psd, u32 value);
	u32 value;
	char *property_name;
};

#define SDD_MAX_PSOC 2
extern sdd_data_t *Sdd_data[SDD_MAX_PSOC];
extern int sdd_psoc_inst;
static inline sdd_data_t * sdd_get_data(unsigned int inst)
{
	if (inst >= SDD_MAX_PSOC) {
		return NULL;
	}
	return Sdd_data[inst];
}

extern struct mutex sdd_swd_lock;

extern sdd_data_t * sdd_data_from_i2c(struct i2c_client *i2c_client);

extern int  sdd_get_buttons(void);
extern void sdd_set_gesture_match(int use_default);
extern int  sdd_set_press_hold_time(sdd_data_t *, u32 msec);
extern int  sdd_set_repeat_time(sdd_data_t *, u32 msec);
extern int  sdd_set_min_samples_per_gesture(sdd_data_t *, u32 samples);
extern int  sdd_set_min_samples_per_tap(sdd_data_t *, u32 samples);
extern int  sdd_set_finger_threshold(sdd_data_t *, u32 threshold);
extern int  sdd_set_noise_threshold(sdd_data_t *, u32 threshold);
extern int  sdd_set_finger_hysteresis(sdd_data_t *psd, u32 hysteresis);
extern int  sdd_set_move_interval(sdd_data_t *, u32 move_interval);

extern int sdd_ind_read(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len);
extern int __sdd_ind_read(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len);
extern int sdd_ind_write(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len);
extern int __sdd_ind_write(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len);
extern int sdd_ind_read_special(sdd_data_t *psd, u8 index, u8 flags, u32 param, u8 *data, u32 len);
extern int sdd_ind_get_finger_hysteresis(sdd_data_t *psd, u8 *val);
extern int sdd_ind_set_finger_hysteresis(sdd_data_t *psd, u8 val);
extern int sdd_ind_set_finger_threshold(sdd_data_t *psd, u8 val);
extern int sdd_ind_get_finger_threshold(sdd_data_t *psd, u8 *val);
extern int sdd_ind_get_repeat_time(sdd_data_t *psd, u16 *val);
extern int sdd_ind_set_repeat_time(sdd_data_t *psd, u16 val);
extern int sdd_ind_get_hold_time(sdd_data_t *psd, u16 *val);
extern int sdd_ind_set_hold_time(sdd_data_t *psd, u16 val);
extern int sdd_ind_get_noise_threshold(sdd_data_t *psd, u8 *val);
extern int sdd_ind_set_noise_threshold(sdd_data_t *psd, u8 val);
extern int sdd_ind_get_move_interval(sdd_data_t *psd, u8 *val);
extern int sdd_ind_set_move_interval(sdd_data_t *psd, u8 val);
extern int sdd_ind_get_die_temp(sdd_data_t *psd, u8 *val);
extern int sdd_ind_get_crc32(sdd_data_t *psd, u32 *val);
extern int sdd_ind_get_sensor16(sdd_data_t *psd, u8 index, u8 sensor, u16 *val);
extern int sdd_ind_get_sensor32(sdd_data_t *psd, u8 index, u8 sensor, u32 *val);
extern int sdd_ind_get_num_sensors(sdd_data_t *psd, u8 *val);
extern int sdd_ind_get_baseline_inits(sdd_data_t *psd, u8 *val);
extern int sdd_ind_get_pass(sdd_data_t *psd, u16 *val);
extern int sdd_ind_get_no_diffs(sdd_data_t *psd, u8 *val);
extern int sdd_ind_get_pmu_pulses(sdd_data_t *psd, u16 *val);
extern int sdd_ind_get_deep_sleeps(sdd_data_t *psd, u16 *val);
extern int sdd_ind_get_cmod_cap(sdd_data_t *psd, u32 *val);
extern int sdd_ind_set_ble_carrier_chan(sdd_data_t *psd, u8 val);
extern int sdd_ind_get_ble_carrier_chan(sdd_data_t *psd, u8 *val);
extern int sdd_ind_transmit_ble_carrier(sdd_data_t *psd, u8 enable);
extern int sdd_ind_get_tpintr_delta(sdd_data_t *psd, u8 *delta);
extern int sdd_ind_set_tpintr_delta(sdd_data_t *psd, u8 delta);
extern int sdd_ind_set_prox_finger_hysteresis(sdd_data_t *psd, u8 val);
extern int sdd_ind_get_prox_finger_hysteresis(sdd_data_t *psd, u8 *val);
extern int sdd_ind_set_prox_finger_threshold(sdd_data_t *psd, u8 val);
extern int sdd_ind_get_prox_finger_threshold(sdd_data_t *psd, u8 *val);
extern int sdd_ind_set_prox_noise_threshold(sdd_data_t *psd, u8 val);
extern int sdd_ind_get_prox_noise_threshold(sdd_data_t *psd, u8 *val);

extern int sdd_configure_sge(sdd_data_t *psd);
extern int sdd_set_dts_share_swd_uart(sdd_data_t *psd, u32 enable);

static inline int sdd_set_prox_noise_threshold(sdd_data_t *psd, u32 threshold)
{
    return sdd_ind_set_prox_noise_threshold(psd, threshold);
}
static inline int sdd_set_prox_finger_threshold(sdd_data_t *psd, u32 threshold)
{
    return sdd_ind_set_prox_finger_threshold(psd, threshold);
}
static inline int sdd_set_prox_finger_hysteresis(sdd_data_t *psd, u32 hyst)
{
    return sdd_ind_set_prox_finger_hysteresis(psd, hyst);
}


extern int  sdd_set_touch(sdd_data_t *, int enable);
extern int  sdd_erase_sge(sdd_data_t *psd);
extern int  sdd_program_sge(sdd_data_t *psd);
extern void sdd_get_capzone(enum HWEVTQ_EventInfo *capzonea, enum HWEVTQ_EventInfo *capzoneb, enum HWEVTQ_EventInfo *capzonec, enum HWEVTQ_EventInfo *capzonem);

static inline int sdd_disable_touch(sdd_data_t *psd, int *prev)
{
	*prev = psd->touch_enable;
	return sdd_set_touch(psd, 0);
}

static inline int sdd_enable_touch(sdd_data_t *psd, int *prev)
{
	*prev = psd->touch_enable;
	return sdd_set_touch(psd, 1);
}

extern int sdd_read_sge_byte(sdd_data_t *psd, u8 reg, u8 *byte);
extern int sdd_write_sge_byte(sdd_data_t *psd, u8 reg, u8 byte);
extern int sdd_read_sge_word(sdd_data_t *psd, u8 reg, u16 *word);
extern int sdd_write_sge_word(sdd_data_t *psd, u8 reg, u16 word);
extern int sdd_read_sge_block(sdd_data_t *psd, u8 reg, u8 len, u8 *buf);
extern int sdd_rmw_sge_byte(sdd_data_t *psd, u8 reg, u8 mask, u8 byte);

static inline int sdd_update_sge_baselines(sdd_data_t *psd)
{
	return sdd_rmw_sge_byte(psd, SGE_REG_CR0, SGE_REG_CR0_UPDATE_BASELINE, SGE_REG_CR0_UPDATE_BASELINE);
}

extern int  sddcmd_init(void);
extern void sddcmd_remove(void);

extern int  sdd_ind_init(sdd_data_t *);
extern void sdd_ind_ops_nocop(sdd_data_t *psd);
extern void sdd_ind_remove(void);
extern void sdd_ind_update_stats(sdd_data_t *psd, u16 old, u16 new);
extern void sdd_ind_clear_max_signal(sdd_data_t *psd);
extern void sdd_ind_check_rawcap(sdd_data_t *psd);
extern int  sdd_ind_set_eco_trim_cap(sdd_data_t *psd, u32 trim);
extern int  sdd_ind_get_eco_trim_cap(sdd_data_t *psd, u32 *trim);
extern int  sdd_change_repeat_time(u32 msec32);
static inline u32 sdd_report_repeat_time(void)
{
	sdd_data_t *psd = sdd_get_data(0);
	return psd->repeat_time;
}

extern void sdd_start_timer(sdd_timer_t *sdt);
extern void sdd_stop_timer(sdd_timer_t *sdt);
extern void sdd_change_timer(sdd_timer_t *sdt, unsigned int msecs);
extern void sdd_inc_timer(sdd_timer_t *sdt, unsigned int msecs);
extern void sdd_dec_timer(sdd_timer_t *sdt, unsigned int msecs);
extern void sdd_timer(unsigned long x);
extern void sdd_process_timers(sdd_data_t *psd);
extern int  sdd_proc_read_timers(struct seq_file *m, void *v);
extern sdd_timer_t * sdd_create_timer(sdd_data_t *psd, char *name, unsigned int msecs, void (*action)(void *));

extern void sdd_log_event(struct sdd_log *sl, struct sdd_grr_change_entry *sgce);
extern struct sdd_log * sdd_init_log(sdd_data_t *);
extern void sdd_log_remove(void);

extern int  sdd_init_device(struct i2c_client *);
extern void sdd_stop_device(struct i2c_client *);
extern void sdd_stop_devices(void);
extern void sdd_proc_init(void);
extern void sdd_proc_remove(void);
extern int sdd_set_btle(sdd_data_t *psd, int enable);
#ifdef CONFIG_DEBUG_FS
extern void sdd_debugfs_init(void);
extern void sdd_debugfs_remove(void);
#endif
extern void sdd_set_grr(enum HWEVTQ_EventSource source);
extern int  sdd_get_thread_pid(void);
extern int sdd_suspend(void);
extern int sdd_resume(void);
extern int sdd_system_off(void);
extern int sdd_psoc_on(sdd_data_t *psd);

#ifdef SGE_REV1_IMAGE
extern int sdd_write_sge_block(sdd_data_t *psd, u8 reg, u8 len, const u8 *buf);
extern int  sdd_rec_init(sdd_data_t *);
extern void sdd_rec_remove(void);
#endif
#if defined(CONFIG_SONOS_SDD_RTC)
extern int sdd_rtc_init(sdd_data_t *);
extern int sdd_rtc_proc_init(void);
extern void sdd_rtc_remove(void);
#endif

#ifdef CONFIG_RPMSG
extern int sdd_psoc_program_notify(bool enter);
extern int sdd_psoc_program_check(void);
extern int sdd_rpmsg_init(void);
extern void sdd_rpmsg_exit(void);
extern void sdd_ind_ops_rpmsg(sdd_data_t *psd);
extern int  sdd_read_touchpad_log(u8 *data, u32 len);
extern int  sdd_read_swipe_distance(sdd_data_t *psd, u32 *dist);
extern int  sdd_write_swipe_distance(sdd_data_t *psd, u32 dist);
extern int  sdd_read_press_hold(sdd_data_t *psd, u32 *msec);
extern int  sdd_write_press_hold(sdd_data_t *psd, u32 msec);
extern int  sdd_read_double_tap(sdd_data_t *psd, u32 *msec);
extern int  sdd_write_double_tap(sdd_data_t *psd, u32 msec);
extern int  sdd_disable_psoc_interrupt(sdd_data_t *psd);
extern int  sdd_enable_psoc_interrupt(sdd_data_t *psd);
#else
static inline int sdd_psoc_program_notify(bool enter) {return 0;}
static inline int sdd_psoc_program_check(void) {return 0;}
static inline int sdd_rpmsg_init(void) {return 0;}
static inline void sdd_rpmsg_exit(void){}
static inline void sdd_ind_ops_rpmsg(sdd_data_t *psd){}
static inline int  sdd_read_touchpad_log(u8 *data, u32 len){return 1;};
static inline int  sdd_read_swipe_distance(sdd_data_t *psd, u32 *dist){return -EINVAL;}
static inline int  sdd_write_swipe_distance(sdd_data_t *psd, u32 dist){return -EINVAL;}
static inline int  sdd_read_press_hold(sdd_data_t *psd, u32 *msec){return -EINVAL;}
static inline int  sdd_write_press_hold(sdd_data_t *psd, u32 msec){return -EINVAL;}
static inline int  sdd_read_double_tap(sdd_data_t *psd, u32 *msec){return -EINVAL;}
static inline int  sdd_write_double_tap(sdd_data_t *psd, u32 msec){return -EINVAL;}
static inline int  sdd_disable_psoc_interrupt(sdd_data_t *psd){return 0;}
static inline int  sdd_enable_psoc_interrupt(sdd_data_t *psd){return 0;}
#endif

#endif
