/*
 * Copyright (c) 2015-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/stddef.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/workqueue.h>
#include <linux/jiffies.h>
#include "blackbox.h"
#include "sdd.h"
#include "captouch_sim.h"

static struct delayed_work	captouch_work;
int captouch_msec_remaining;
unsigned int captouch_msec_interval;
struct captouch_match_entry *captouch_active_entry = NULL;
struct captouch_match_entry  *captouch_match_table;

struct captouch_sequence_entry captouch_ac_swipe_sequence[] = {
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED, 124},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_SWIPED,   24},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED, 124},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED,  0},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT}
};

struct captouch_sequence_entry captouch_ca_swipe_sequence[] = {
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED, 172},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_SWIPED,   68},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED, 260},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED,  0},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_ab_bad_sequence[] = {
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED, 124},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_SWIPED,   32},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED,  0},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_cb_bad_sequence[] = {
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED, 172},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_SWIPED,   68},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED,  0},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_acac_sequence[] = {
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED, 156},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED,  44},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED,776},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED,544},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_acca_sequence[] = {
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED, 138},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED,  96},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED, 94},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED, 94},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_caca_sequence[] = {
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED,  444},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED,  332},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED, 772},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED, 114},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT,  0}
};

struct captouch_sequence_entry captouch_caac_sequence[] = {
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED, 112},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED,  88},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED, 34},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED, 12},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_cat_sequence[] = {
	{HWEVTQSOURCE_CAPZONEC,   HWEVTQINFO_PRESSED, 142},
	{HWEVTQSOURCE_CAPZONEB,   HWEVTQINFO_SWIPED,   38},
	{HWEVTQSOURCE_CAPZONECAT, HWEVTQINFO_PRESSED, 444},
	{HWEVTQSOURCE_CAPZONEB,   HWEVTQINFO_RELEASED, 12},
	{HWEVTQSOURCE_CAPZONECAT, HWEVTQINFO_RELEASED,  0},
	{HWEVTQSOURCE_NO_SOURCE,  HWEVTQINFO_NO_EVENT,  0}
};

struct captouch_sequence_entry captouch_baab_sequence[] = {
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_PRESSED,  99},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED,  86},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED, 33},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED, 12},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_cbcb_sequence[] = {
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED, 377},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_PRESSED, 136},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED, 33},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED, 12},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_match_entry captouch_default_table[] = {
	{.keyword = "zonea", .source = HWEVTQSOURCE_CAPZONEA},
	{.keyword = "zoneb", .source = HWEVTQSOURCE_CAPZONEB},
	{.keyword = "zonec", .source = HWEVTQSOURCE_CAPZONEC},
	{.keyword = "swipe-good-ac", .sequence = captouch_ac_swipe_sequence},
	{.keyword = "swipe-good-ca", .sequence = captouch_ca_swipe_sequence},
	{.keyword = "swipe-bad-ab", .sequence = captouch_ab_bad_sequence},
	{.keyword = "swipe-bad-cb", .sequence = captouch_cb_bad_sequence},
	{.keyword = "multi-acac", .sequence = captouch_acac_sequence},
	{.keyword = "multi-acca", .sequence = captouch_acca_sequence},
	{.keyword = "multi-caca", .sequence = captouch_caca_sequence},
	{.keyword = "multi-caac", .sequence = captouch_caac_sequence},
	{.keyword = "multi-baab", .sequence = captouch_baab_sequence},
	{.keyword = "multi-cbcb", .sequence = captouch_cbcb_sequence},
	{.keyword = "cat", .sequence = captouch_cat_sequence},
	{NULL, NULL}
};

struct captouch_sequence_entry captouch_mabbam_sequence[] = {
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_PRESSED, 312},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED, 249},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_PRESSED, 387},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED, 81},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED, 85},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_RELEASED, 19},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_cmacma_sequence[] = {
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED, 291},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_PRESSED, 444},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED, 184},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED, 91},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_RELEASED, 67},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED, 75},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_bcmmbc_sequence[] = {
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_PRESSED, 287},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED, 112},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_PRESSED, 222},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_RELEASED, 44},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED, 82},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED, 41},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_cmcm_sequence[] = {
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_PRESSED, 111},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_PRESSED, 124},
	{HWEVTQSOURCE_CAPZONEC, HWEVTQINFO_RELEASED, 94},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_RELEASED, 33},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_mbbm_sequence[] = {
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_PRESSED, 311},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_PRESSED, 112},
	{HWEVTQSOURCE_CAPZONEB, HWEVTQINFO_RELEASED, 79},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_RELEASED, 82},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_sequence_entry captouch_amma_sequence[] = {
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_PRESSED, 212},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_PRESSED, 187},
	{HWEVTQSOURCE_CAPZONEM, HWEVTQINFO_RELEASED, 93},
	{HWEVTQSOURCE_CAPZONEA, HWEVTQINFO_RELEASED, 12},
	{HWEVTQSOURCE_NO_SOURCE, HWEVTQINFO_NO_EVENT, 0}
};

struct captouch_match_entry captouch_default_zonem_table[] = {
	{.keyword = "zonea", .source = HWEVTQSOURCE_CAPZONEA},
	{.keyword = "zoneb", .source = HWEVTQSOURCE_CAPZONEB},
	{.keyword = "zonec", .source = HWEVTQSOURCE_CAPZONEC},
	{.keyword = "zonem", .source = HWEVTQSOURCE_CAPZONEM},
	{.keyword = "swipe-good-ac", .sequence = captouch_ac_swipe_sequence},
	{.keyword = "swipe-good-ca", .sequence = captouch_ca_swipe_sequence},
	{.keyword = "swipe-bad-ab", .sequence = captouch_ab_bad_sequence},
	{.keyword = "swipe-bad-cb", .sequence = captouch_cb_bad_sequence},
	{.keyword = "multi-acac", .sequence = captouch_acac_sequence},
	{.keyword = "multi-acca", .sequence = captouch_acca_sequence},
	{.keyword = "multi-caca", .sequence = captouch_caca_sequence},
	{.keyword = "multi-caac", .sequence = captouch_caac_sequence},
	{.keyword = "multi-baab", .sequence = captouch_baab_sequence},
	{.keyword = "multi-cbcb", .sequence = captouch_cbcb_sequence},
	{.keyword = "cat", .sequence = captouch_cat_sequence},
	{.keyword = "multi-mabbam", .sequence = captouch_mabbam_sequence},
	{.keyword = "multi-cmacma", .sequence = captouch_cmacma_sequence},
	{.keyword = "multi-bcmmbc", .sequence = captouch_bcmmbc_sequence},
	{.keyword = "multi-cmcm", .sequence = captouch_cmcm_sequence},
	{.keyword = "multi-mbbm", .sequence = captouch_mbbm_sequence},
	{.keyword = "multi-amma", .sequence = captouch_amma_sequence},
	{NULL, NULL}
};

struct captouch_match_entry captouch_empty_table[] = {
	{NULL, NULL}
};

void captouch_sim(struct work_struct *d)
{
	enum HWEVTQ_EventInfo info;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	enum HWEVTQ_EventInfo hwevtq_info;
#endif
	captouch_msec_remaining -= captouch_msec_interval;
	info = (captouch_msec_remaining <= 0) ? HWEVTQINFO_RELEASED : HWEVTQINFO_REPEATED;
	event_queue_send_event(captouch_active_entry->source, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	hwevtq_info = (captouch_msec_remaining <= 0) ? HWEVTQINFO_RELEASED : HWEVTQINFO_REPEATED;
	hwevtq_send_event(captouch_active_entry->source, hwevtq_info);
#endif
	if (captouch_msec_remaining > 0) {
		captouch_msec_interval = (captouch_msec_remaining > SGE_REPEAT_TIME) ? SGE_REPEAT_TIME : captouch_msec_remaining;
		schedule_delayed_work(&captouch_work, msecs_to_jiffies(captouch_msec_interval));
	} else {
		captouch_active_entry = NULL;
		sdd_set_grr(HWEVTQSOURCE_NO_SOURCE);
	}
}

void captouch_sim_init(struct captouch_match_entry *match_table, sdd_gesture_match_t *supported)
{
	if (match_table == NULL) {
		if (supported->zonem) {
			bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "using default zonem captouch sim table");
			captouch_match_table = captouch_default_zonem_table;
		} else {
			bb_log(BB_MOD_SDD, BB_LVL_DEBUG, "using default captouch sim table");
			captouch_match_table = captouch_default_table;
		}
	} else {
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "optional captouch sim table not implemented, aborted");
		captouch_match_table = captouch_empty_table;
	}
	INIT_DELAYED_WORK(&captouch_work, captouch_sim);
}

static inline int captouch_sim_cmd_match(char *cmd, struct captouch_match_entry *cme)
{
	return (strncmp(cmd, cme->keyword, strlen(cme->keyword)) == 0);
}

static unsigned long captouch_sim_get_duration(char *param)
{
	u32 uval = 0;
	u32 max_duration = 1000 * 10;
	u32 duration = 0;
	int error;

	if (*param == '=') {
		if ((error = kstrtouint(param+1, 10, &uval))) {
			bb_log(BB_MOD_SDD, BB_LVL_WARNING, "unable to parse string to uint using kstrtouint, default to 0ms");
			return duration;
		}

		if (uval < max_duration) {
			duration = uval;
		} else {
			bb_log(BB_MOD_SDD, BB_LVL_WARNING, "duration of %ums exceeds max of %ums, default to 0ms",
				   uval, max_duration);
		}
	}
	return duration;
}

int captouch_sim_process_cmd(char *cmd)
{
	struct captouch_match_entry *cme;

	if (captouch_active_entry != NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "simulated button event already in progress");
		return 0;
	}

	cme = captouch_match_table;
	while (cme->keyword != NULL) {
		if (captouch_sim_cmd_match(cmd, cme)) {
			struct captouch_sequence_entry *cse = cme->sequence;
			if (cse == NULL) {
				u32 duration = captouch_sim_get_duration(cmd+strlen(cme->keyword));
				captouch_msec_interval = (duration > SGE_PRESS_HOLD_TIME) ? SGE_PRESS_HOLD_TIME : duration;
				captouch_msec_remaining = duration;
				captouch_active_entry = cme;
				sdd_set_grr(cme->source);
				event_queue_send_event(cme->source, HWEVTQINFO_PRESSED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(cme->source, HWEVTQINFO_PRESSED);
#endif
				schedule_delayed_work(&captouch_work, msecs_to_jiffies(captouch_msec_interval));
			} else {
				while (cse->source != HWEVTQSOURCE_NO_SOURCE) {
					event_queue_send_event(cse->source, cse->info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_send_event(cse->source, cse->info);
#endif
					msleep(cse->post_event_delay);
					cse++;
				}
			}
			return 1;
		}
		cme++;
	}
	return 0;
}

void captouch_sim_print_cmds(void)
{
	struct captouch_match_entry *cme = captouch_match_table;
	while (cme->keyword != NULL) {
		bb_log(BB_MOD_SDD, BB_LVL_WARNING, "  %s", cme->keyword);
		cme++;
	}
}
