/*
 * Copyright (c) 2018-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Sonos PWM LED control module
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sysfs.h>
#include <linux/of.h>
#include <linux/slab.h>
#include <linux/pwm.h>

#include "blackbox.h"
#include "sonos_device.h"

#define PWM_MM_INFO(msg, args...)		bb_log(BB_MOD_LEDCTL, BB_LVL_INFO, "PWM_MM: "msg, ##args)
#define PWM_MM_ERR(msg, args...)		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "PWM_MM: "msg, ##args)
#define PWM_MM_BRIGHTNESS_INIT_VAL		1000

static int pwm_mm_led_set_brightness(int brightness);

static struct pwm_mm_led_device {
	char *name;
	struct pinctrl *pin;
	struct pwm_device *pwm;
	bool init;
} *led_dev;

static int brightness;
static int devno = -1;

static ssize_t brightness_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	if (led_dev && led_dev->init) {
		return scnprintf(buf, PAGE_SIZE, "%d\n", brightness);
	}

	PWM_MM_ERR("led_dev device not initialized.");
	return -EFAULT;
}

static ssize_t brightness_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int val;
	int ret;

	if (led_dev && led_dev->init) {
		if (kstrtoint(buf, 0, &val) != 0) {
			PWM_MM_ERR("Failed to parse the entry.");
			return -EFAULT;
		}

		if ((ret = pwm_mm_led_set_brightness(val))) {
			return ret;
		}
		return count;
	}

	PWM_MM_ERR("led_dev device not initialized.");
	return -EFAULT;

}
static DEVICE_ATTR(brightness, (S_IWUSR|S_IRUSR|S_IWGRP|S_IRGRP), brightness_show, brightness_store);


int pwm_mm_led_set_brightness(int val)
{
	struct pwm_state pstate;

	if (val < 1 || val > 1000) {
		PWM_MM_ERR("Brightness value out of bounds. Expect: (1..1000)");
		return -EFAULT;
	}

	pwm_get_state(led_dev->pwm, &pstate);
	pstate.duty_cycle = (val * pstate.period )  / 1000;
	brightness = val;
	pwm_apply_state(led_dev->pwm, &pstate);

	return 0;
}

int __init pwm_mm_led_init(void)
{
	struct device *dev;
	struct device_node *of_node;
	struct pwm_state pstate;
	int ret = 0;

	ret = alloc_chrdev_region(&devno, 0, 1, "pwm_mm_led");
	if (ret) {
		PWM_MM_ERR("Failed to allocate character device");
		return ret;
	}

	dev = sonos_device_create(NULL, devno, NULL, "pwm_mm_led");
	if (IS_ERR(dev)) {
		PWM_MM_ERR("Failed to create Sonos class device");
		unregister_chrdev_region(devno, 1);
		return PTR_ERR(dev);
	}

	of_node = of_find_node_by_name(NULL, "sonos_pwm_led");
	dev->of_node = of_node;

	if (of_node) {
		led_dev = kzalloc(sizeof(struct pwm_mm_led_device), GFP_KERNEL);
		if (!led_dev) {
			PWM_MM_ERR("Failed to allocate memory");
			return -ENOMEM;
		}

		led_dev->pin = devm_pinctrl_get_select(dev, "default");
		if (IS_ERR(led_dev->pin)) {
			PWM_MM_ERR("unable to select led_pwm_pins pinctrl.");
			ret = -ENODEV;
			goto end;
		}

		led_dev->pwm = devm_of_pwm_get(dev, of_node, NULL);
		if (IS_ERR(led_dev->pwm)) {
			PWM_MM_ERR("unable to request PWM0");
			ret = -ENODATA;
			goto end;
		}

		pwm_init_state(led_dev->pwm, &pstate);
		pstate.enabled = 1;
		pstate.duty_cycle = (PWM_MM_BRIGHTNESS_INIT_VAL * pstate.period)  / 1000;
		brightness = PWM_MM_BRIGHTNESS_INIT_VAL;
		pwm_apply_state(led_dev->pwm, &pstate);

		PWM_MM_INFO("PWM Mic Mute LED Brightness module registered.");
		led_dev->init = 1;
	} else {
		PWM_MM_ERR("Failed to parse config from DT");
		goto end;
	}

	ret = device_create_file(dev, &dev_attr_brightness);
	if (ret) {
		PWM_MM_ERR("Failed to create the brightness control file");
		goto end;
	}

	return 0;
end:
	sonos_device_destroy(devno);
	unregister_chrdev_region(devno, 1);
	kfree(led_dev);
	return -ENODEV;
}

void __exit pwm_mm_led_exit(void)
{
	sonos_device_destroy(devno);
	unregister_chrdev_region(devno, 1);
	if (led_dev) {
		kfree(led_dev);
		PWM_MM_INFO("PWM Mic Mute LED Brightness module removed.");
	}
}

module_init(pwm_mm_led_init);
module_exit(pwm_mm_led_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for Sonos PWM Mic-Mute LED");
MODULE_LICENSE("GPL");
