/*
 * Copyright (c) 2016-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#include <linux/version.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/kthread.h>
#include <linux/module.h>
#include <linux/reboot.h>
#include <linux/sonos_kernel.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
#include <linux/slab.h>
#else
#include <linux/kernel.h>
#endif
#include <linux/workqueue.h>
#include <linux/of_gpio.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/slab.h>

#include "blackbox.h"
#include "ampctl.h"

enum psmon_trigger_type {
	PSMON_ADC_MODE,
	PSMON_GPIO_MODE,
};

struct psmon_io_params {
	enum psmon_trigger_type type;
	int trigger_gpio;
	unsigned long flag;
	int adc_port;
	int adc_scale;
	int shutdown_voltage;
	int shutdown_gpio_5v;
	int shutdown_gpio_amp_hiz;
	int shutdown_gpio_24v;
	int shutdown_gpio_2g;
	int shutdown_gpio_5g;
	int shutdown_phy;
	int shutdown_mipi;
	int shutdown_earc;
	int shutdown_led;
	int shutdown_psoc;
};

struct task_struct *vm_task;

typedef struct {
	struct work_struct psmon_work;
	struct psmon_io_params *ptr_psmon_io;
	int    lp_count;
} psmon_work_t;

psmon_work_t *pd_work;

void psmon_do_halt(struct psmon_io_params *psmon_io)
{

	if (gpio_is_valid(psmon_io->shutdown_gpio_5v)){
		gpio_set_value(psmon_io->shutdown_gpio_5v, 0);
	}
	if (gpio_is_valid(psmon_io->shutdown_gpio_amp_hiz)){
		gpio_set_value(psmon_io->shutdown_gpio_amp_hiz, 0);
	}
	if (gpio_is_valid(psmon_io->shutdown_gpio_24v)){
		gpio_set_value(psmon_io->shutdown_gpio_24v, 0);
	}
	if (gpio_is_valid(psmon_io->shutdown_gpio_2g)){
		gpio_set_value(psmon_io->shutdown_gpio_2g, 0);
	}
	if (gpio_is_valid(psmon_io->shutdown_gpio_5g)){
		gpio_set_value(psmon_io->shutdown_gpio_5g, 0);
	}
	if (gpio_is_valid(psmon_io->shutdown_phy)){
		gpio_direction_output(psmon_io->shutdown_phy, 0);
	}
	if (gpio_is_valid(psmon_io->shutdown_mipi)){
		gpio_direction_output(psmon_io->shutdown_mipi, 1);
	}
	if (gpio_is_valid(psmon_io->shutdown_earc)){
		gpio_set_value(psmon_io->shutdown_earc, 0);
	}
	if (gpio_is_valid(psmon_io->shutdown_led)){
		gpio_set_value(psmon_io->shutdown_led, 0);
	}
	if (gpio_is_valid(psmon_io->shutdown_psoc)){
		gpio_set_value(psmon_io->shutdown_psoc, 0);
	}

	ampctl_shutdown();

#ifdef CONFIG_MTD
	while(!nand_shutdown_access(1));
#else
	printk(KERN_ERR "psmon: PS_OFF\n");
#endif
	kernel_restart("psmon: PS_OFF");
}

static int psmon_volt_thread(void *data)
{
	struct psmon_io_params *psmon_io = (struct psmon_io_params *)data;
	int ret = 0;
	int mvolts;
	int scale = psmon_io->adc_scale;

	bb_log(BB_MOD_PSMON, BB_LVL_DEBUG, "Brownout detection thread, shutdown limit %d mV.", psmon_io->shutdown_voltage);

	while (!kthread_should_stop()) {
		(void)read_adc_voltage(psmon_io->adc_port, &mvolts);

		if((scale * mvolts / 1000) < psmon_io->shutdown_voltage) {
			psmon_do_halt(psmon_io);
			msleep(500);
			kernel_restart("brownout detected");
		}
		msleep(50);
	}
	return ret;
}

irqreturn_t psmon_isr(int irq, void *data)
{
	if (pd_work->lp_count == 0) {
		pd_work->lp_count ++;
		schedule_work_on(0, ((struct work_struct *)pd_work ));
	}
	return IRQ_HANDLED;
}

static void psmon_wq_function( struct work_struct *work_arg)
{
	psmon_work_t *lp_work = (psmon_work_t *)work_arg;

	psmon_do_halt(lp_work->ptr_psmon_io);
	msleep(500);
	kernel_restart("brownout detected");

	lp_work->lp_count = 0;
	return;
}

int psmon_init_io( struct device_node *np, struct psmon_io_params *psmon_io)
{
	int ret = 0;

	if (!np) {
		bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Could not find node psmon");
		ret = -ENODEV;
	}

	psmon_io->trigger_gpio = of_get_named_gpio(np, "psmon-gpio", 0);
	if (gpio_is_valid(psmon_io->trigger_gpio)) {
		psmon_io->type = PSMON_GPIO_MODE;

		of_property_read_u32(np, "trigger-edge", (u32 *)&psmon_io->flag);
		if (!psmon_io->flag) {
			bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Psmon gpio interrupt triggering edge not found!");
			ret = -EINVAL;
		}
	} else {
		psmon_io->type = PSMON_ADC_MODE;
		of_property_read_u32(np, "shutdown-voltage", &psmon_io->shutdown_voltage);
		if (!psmon_io->shutdown_voltage){
			bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Psmon adc monitor shutdown voltage not found!");
			ret = -EINVAL;
		}
		if (of_property_read_u32(np, "adc-port", &psmon_io->adc_port)) {
			bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Psmon adc monitor port not found!");
			ret = -EINVAL;
		}
		of_property_read_u32(np, "adc-scale", &psmon_io->adc_scale);
		if (!psmon_io->adc_scale){
			bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Psmon adc monitor scale not found!");
			ret = -EINVAL;
		}
		if (ret >= 0) {
			bb_log(BB_MOD_PSMON, BB_LVL_INFO, "psmon in ADC_MODE on adc port %d, shutdown_voltage = %d", \
				psmon_io->adc_port, psmon_io->shutdown_voltage);
		}
	}

	psmon_io->shutdown_gpio_5v = of_get_named_gpio(np, "5v-shutdown-gpio", 0);
	psmon_io->shutdown_gpio_amp_hiz = of_get_named_gpio(np, "amp-hiz-shutdown-gpio", 0);
	psmon_io->shutdown_gpio_24v = of_get_named_gpio(np, "24v-shutdown-gpio", 0);
	psmon_io->shutdown_gpio_2g = of_get_named_gpio(np, "2g-shutdown-gpio", 0);
	psmon_io->shutdown_gpio_5g = of_get_named_gpio(np, "5g-shutdown-gpio", 0);
	psmon_io->shutdown_phy = of_get_named_gpio(np, "shutdown-phy", 0);
	psmon_io->shutdown_mipi = of_get_named_gpio(np, "shutdown-mipi", 0);
	psmon_io->shutdown_earc = of_get_named_gpio(np, "shutdown-earc", 0);
	psmon_io->shutdown_led = of_get_named_gpio(np, "shutdown-led", 0);
	psmon_io->shutdown_psoc = of_get_named_gpio(np, "shutdown-psoc", 0);

	return ret;
}

static int psmon_init(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct psmon_io_params *psmon_io;
	int ret = 0;

	pd_work = NULL;

	psmon_io = devm_kzalloc(&pdev->dev, sizeof(*psmon_io), GFP_KERNEL);
	if (!psmon_io)
		return -ENOMEM;

	if (psmon_init_io(np, psmon_io)) {
		bb_log(BB_MOD_PSMON, BB_LVL_INFO, "Power supply monitoring not supported on this unit");
		ret = -EINVAL;
		goto out_err;
	}

	switch (psmon_io->type) {
		case PSMON_ADC_MODE:
			vm_task = kthread_run(&psmon_volt_thread, psmon_io, "Voltage Monitor");
			if (IS_ERR(vm_task)) {
				bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Failed to start voltage monitor thread (%ld).", PTR_ERR(vm_task));
				ret = PTR_ERR(vm_task);
				goto out_err;
			}
			break;
		case PSMON_GPIO_MODE:
			pd_work = (psmon_work_t *)kmalloc(sizeof(psmon_work_t), GFP_KERNEL);
			if (pd_work == NULL) {
				bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Failed to kmalloc the psmon workqueue");
				ret = -EFAULT;
				goto out_err;
			}
			INIT_WORK( (struct work_struct *)pd_work, psmon_wq_function );
			pd_work->ptr_psmon_io = psmon_io;
			pd_work->lp_count = 0;
			if (gpio_request_one(psmon_io->trigger_gpio, GPIOF_IN, "PS monitor pin")) {
				bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Failed to register voltage monitor GPIO.");
				ret = -EFAULT;
				goto out_err;
			}
			if (request_irq(gpio_to_irq(psmon_io->trigger_gpio), psmon_isr, (psmon_io->flag), "PS monitor", NULL)) {
				bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Failed to request voltage monitor interrupt.");
				gpio_free(psmon_io->trigger_gpio);
				ret = -EFAULT;
				goto out_err;
			}
			bb_log(BB_MOD_PSMON, BB_LVL_DEBUG, "Got GPIO %d and IRQ %d.", psmon_io->trigger_gpio, gpio_to_irq(psmon_io->trigger_gpio));
			break;
		default:
			bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Unknow psmon monitor type!");
			ret = -EFAULT;
			goto out_err;
	}

	if (gpio_is_valid(psmon_io->shutdown_gpio_5v)){
		if (gpio_request_one(psmon_io->shutdown_gpio_5v, GPIOF_OUT_INIT_HIGH, "PS shutdown pin")) {
			bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Failed to request GPIO shutdown pin.");
			ret = -EFAULT;
		}
	}

	platform_set_drvdata(pdev, psmon_io);
	bb_log(BB_MOD_PSMON, BB_LVL_INFO, "Started power supply brownout monitor.");

	return ret;

out_err:
	if (pd_work)
		kfree(pd_work);
	if (psmon_io)
		devm_kfree(&pdev->dev, psmon_io);
	return ret;
}

static int psmon_exit(struct platform_device *pdev)
{
	struct psmon_io_params *psmon_io = platform_get_drvdata(pdev);

	if (!psmon_io)
		return 0;

	switch (psmon_io->type) {
		case PSMON_ADC_MODE:
			if (vm_task != NULL) {
				kthread_stop(vm_task);
			}
			break;
		case PSMON_GPIO_MODE:
			kfree(pd_work);
			free_irq(gpio_to_irq(psmon_io->trigger_gpio), NULL);
			gpio_free(psmon_io->trigger_gpio);
			break;
		default:
			bb_log(BB_MOD_PSMON, BB_LVL_ERR, "Unknow psmon monitor type!");
			break;
	}

	if (gpio_is_valid(psmon_io->shutdown_gpio_5v))
		gpio_free(psmon_io->shutdown_gpio_5v);

	devm_kfree(&pdev->dev, psmon_io);

	return 0;
}

static const struct of_device_id psmon_match[] = {
        { .compatible = "sonos,psmon" },
        {},
};
MODULE_DEVICE_TABLE(of, psmon_match);

static struct platform_driver sonos_psmon_driver = {
        .probe          = psmon_init,
        .remove         = psmon_exit,
        .driver         = {
                .name           = KBUILD_MODNAME,
                .of_match_table = psmon_match,
        },
};

module_platform_driver(sonos_psmon_driver);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Power supply brownout monitor");
MODULE_LICENSE("GPL");
