/*
 * Copyright (c) 2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/timekeeping.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/cdev.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/clk.h>
#include <linux/debugfs.h>
#include <linux/notifier.h>
#include <linux/reboot.h>

#define DRIVER_NAME "MT8518s watchdog reason Driver"

static void __iomem *MTK_WDT_NONRST_REG;
static struct dentry* mtk_wdt_reason_fs;
static struct dentry* mtk_wdt_prev_reason_fs;
static u64 previous_wdt_reason;

static int shutdown_notify_sys(struct notifier_block* this, unsigned long code, void* unused);

static struct notifier_block shutdown_notifier = {
	.notifier_call = shutdown_notify_sys,
};


static int nonrst_write_op(void* data, u64 reg)
{
	u32 value = readl(MTK_WDT_NONRST_REG) & ~0xFFFF;
	value |= (u32)(reg & 0xFFFF);
	writel(value, MTK_WDT_NONRST_REG);
	return 0;
}

static int nonrst_read_op(void* data, u64* reg)
{
	*reg = readl(MTK_WDT_NONRST_REG) & 0xFFFF;
	return 0;
}

static int shutdown_notify_sys(struct notifier_block* this, unsigned long code, void* unused)
{
	nonrst_write_op(NULL, 0x8000 | code);
	return NOTIFY_DONE;
}

DEFINE_SIMPLE_ATTRIBUTE(nonrst2_fops, nonrst_read_op, nonrst_write_op, "%llu\n");

static int __init wdt_reason_driver_init(void)
{
	register_reboot_notifier(&shutdown_notifier);

	MTK_WDT_NONRST_REG = ioremap(0x10007020, 4);

	mtk_wdt_prev_reason_fs = debugfs_create_u64("mtk_wdt_prev_reason", 0444, NULL, &previous_wdt_reason);

	if (!mtk_wdt_prev_reason_fs) {
		printk(KERN_ALERT "mtk_wdt_reason: cannot create previous reason debugfs file");
		return -1;
	}

	mtk_wdt_reason_fs = debugfs_create_file("mtk_wdt_reason", 0666, NULL, NULL, &nonrst2_fops);

	if (!mtk_wdt_reason_fs) {
		printk(KERN_ALERT "mtk_wdt_reason: cannot create reason debugfs file");
		return -1;
	}

	nonrst_read_op(NULL, &previous_wdt_reason);
	printk(KERN_NOTICE "mtk_wdt_reason: previous reset reason code %llu\n", previous_wdt_reason);
	nonrst_write_op(NULL, 0);

	return 0;
}


static void __exit wdt_reason_driver_exit(void)
{
	debugfs_remove(mtk_wdt_prev_reason_fs);
	debugfs_remove(mtk_wdt_reason_fs);
	iounmap(MTK_WDT_NONRST_REG);
	unregister_reboot_notifier(&shutdown_notifier);
}

module_init(wdt_reason_driver_init);
module_exit(wdt_reason_driver_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("MTK8518 watchdog reboot reason driver");
MODULE_LICENSE("GPL");
