/*
 * Copyright (c) 2017-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/poll.h>
#include <linux/firmware.h>
#include <linux/platform_device.h>
#include <linux/cdev.h>
#include <linux/clk.h>
#include <linux/err.h>
#include <linux/debugfs.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/kthread.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/circ_buf.h>
#include <asm/io.h>
#include "blackbox.h"
#include "mtk_m4.h"
#include "mtk_m4_common.h"

#define CM4_FIRMWARE_IMAGE        "cm4.bin"

#define CM4_REG_RESET_CTL         0x00000000
#define CM4_REG_CLOCK_CTL         0x00000008
#define CM4_REG_CONFIGURE         0x0000002c
#define CM4_REG_EVT_CNTRL         0x00000010
#define CM4_REG_EVT_CLEAR         0x00000014
#define CM4_REG_CM4_SEMA4         0x00000020

#define CM4_MSK_CPU_RESET         BIT(0)
#define CM4_MSK_CPU_CK_EN         BIT(0)
#define CM4_MSK_MPU_DISAB         BIT(0)
#define CM4_MSK_CM2AP_CLR         BIT(1)
#define CM4_MSK_SOFT_IRQ0         BIT(2)
#define CM4_MSK_SEMA4_BIT         BIT(0)

#define CM4_CODE_SRAM_SIZE        0x00030000
#define CM4_DATA_SRAM_SIZE        0x00020000
#define CM4_SW_WDG_TIMEOUT        15000

#define CM4_MSG_PRINT(LEVEL, FORMAT, ARG...) bb_log(BB_MOD_M4, LEVEL, FORMAT, ## ARG)
#define CM4_MSG_INFO(FORMAT, ARG...) CM4_MSG_PRINT(BB_LVL_INFO, FORMAT, ## ARG)
#define CM4_MSG_WARN(FORMAT, ARG...) CM4_MSG_PRINT(BB_LVL_WARNING, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define CM4_MSG_ERROR(FORMAT, ARG...) CM4_MSG_PRINT(BB_LVL_ERR, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define CM4_MSG_DEBUG(FORMAT, ARG...) CM4_MSG_PRINT(BB_LVL_DEBUG, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)

#define cmsys_read(addr) readl(((volatile void *)(((u32)(addr))+g_cmsys_base_address)))
#define cmsys_write(addr, val) writel((val), ((volatile void *)(((u32)(addr))+g_cmsys_base_address)))
#define cmsys_msk_write(addr, msk, val) cmsys_write((addr), ((cmsys_read(addr) & (~(msk))) | ((val) & (msk))))


struct cm4_private_data {
	struct clk *cmsys_clk;
	void __iomem *cmsys_base_address;
	void __iomem *cmsys_code_sram;
	void __iomem *cmsys_data_sram;
	int cmsys_irq;
	struct timer_list sw_wdog_timer;
	struct task_struct *kthread_task;
	struct semaphore ipc_recv_sem;
};

static int devno = -1;

static void (*ipc_callback_funcs[IPC_PKT_MAX])(uint8_t *payload) = {NULL};
static sys_status_t sys_stat = {0};
ipc_config_t ipc_config;
static int sw_wdog_exp_count = 0;
bool quit_flag=0;

static const struct debugfs_reg32 cm4_register_list[] = {
	{ "[Reg 00] Reset Control", 0x00 },
	{ "[Reg 01] Remap Control", 0x04 },
	{ "[Reg 02] Clock Gate Control", 0x08 },
	{ "[Reg 03] Debug Control", 0x0C },
	{ "[Reg 04] Event Control", 0x10 },
	{ "[Reg 05] Event Clear", 0x14 },
	{ "[Reg 06] IRQ Polarity Control", 0x18 },
	{ "[Reg 07] Interrupt Status Clear", 0x1C },
	{ "[Reg 08] Semaphore Register", 0x20 },
	{ "[Reg 09] Share Register", 0x24 },
	{ "[Reg 10] Systick Calibration", 0x28 },
	{ "[Reg 11] Configure Register", 0x2C },
	{ "[Reg 12] GPR 0", 0x30 },
	{ "[Reg 13] GPR 1", 0x34 },
	{ "[Reg 14] WDT Control", 0x38 },
	{ "[Reg 15] Timer Control", 0x3C },
	{ "[Reg 16] Timer Set Value", 0x40 },
	{ "[Reg 17] Timer Count Value", 0x44 },
	{ "[Reg 18] Share Reg 1", 0x48 },
	{ "[Reg 19] Share Reg 2", 0x4C },
	{ "[Reg 20] Mbist Done", 0x50 },
	{ "[Reg 21] Mbist Fail 0", 0x54 },
	{ "[Reg 22] Mbist Fail 1", 0x58 },
	{ "[Reg 23] HBUS Debug Mon 0", 0x5C },
	{ "[Reg 24] HBUS Debug Mon 1", 0x60 },
	{ "[Reg 25] Debug Reg 0", 0x64 },
	{ "[Reg 26] Debug Reg 1", 0x68 },
};


static int __init mtk_cm4_driver_init(void);
static void __exit mtk_cm4_driver_exit(void);
static int cm4_debug_init(struct platform_device *pdev);
static int cm4_open(struct inode *inodep, struct file *filp);
static int cm4_release(struct inode *inodep, struct file *filp);
static int cm4_probe(struct platform_device *pdev);
static int cm4_remove(struct platform_device *pdev);
static ssize_t cm4_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset);
static ssize_t cm4_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset);
static loff_t cm4_llseek(struct file *filp, loff_t offset, int seek_type);
static unsigned int cm4_poll(struct file *filp, struct poll_table_struct *table);
static int cm4_get_firmware(struct platform_device *pdev);
static void cm4_power_on(int on);
static int cm4_ipc_recv_thread(void *data);
static void cm4_wdog_callback(unsigned long data);
static void cm4_log_print(uint8_t *buf);

DECLARE_WAIT_QUEUE_HEAD(read_queue);
struct cdev cm4_chr_dev;
static void __iomem *g_cmsys_base_address = NULL;
static struct class *cm4_dev_class = NULL;
static struct dentry *cm4_debugfs_root_dir;
static struct debugfs_blob_wrapper cm4_code_sram;
static struct debugfs_blob_wrapper cm4_data_sram;
static struct debugfs_regset32 cm4_reg_set;

void platform_resource_lock(void)
{
	unsigned int tmp=0;

	for (tmp = 0; tmp < 1000; tmp++ ) {
		cmsys_msk_write(CM4_REG_CM4_SEMA4, CM4_MSK_SEMA4_BIT, CM4_MSK_SEMA4_BIT);
		if ((cmsys_read(CM4_REG_CM4_SEMA4)) & CM4_MSK_SEMA4_BIT) {
			return;
		}
	}

	CM4_MSG_ERROR("Failed to obtain resource lock");
	return;
}

void platform_resource_unlock(void)
{
	cmsys_msk_write(CM4_REG_CM4_SEMA4, CM4_MSK_SEMA4_BIT, CM4_MSK_SEMA4_BIT);

	if ((cmsys_read(CM4_REG_CM4_SEMA4)) & CM4_MSK_SEMA4_BIT) {
		CM4_MSG_ERROR("Failed to unlock resource. We do not own the lock!");
	}
	return;
}

void platform_cpu_to_cpu_interrupt(void)
{
	cmsys_msk_write(CM4_REG_EVT_CNTRL, CM4_MSK_SOFT_IRQ0, ~(CM4_MSK_SOFT_IRQ0));
}

inline void clear_cpu_to_cpu_interrupt(void)
{
	cmsys_msk_write(CM4_REG_EVT_CLEAR, CM4_MSK_CM2AP_CLR, CM4_MSK_CM2AP_CLR);
}

irqreturn_t cpu_to_cpu_isr(int irq, void *data)
{
	struct cm4_private_data *cm4_priv;

	clear_cpu_to_cpu_interrupt();
	cm4_priv = platform_get_drvdata(data);

	up(&cm4_priv->ipc_recv_sem);

	return IRQ_HANDLED;
}


static void cm4_wdog_callback(unsigned long data)
{
	sw_wdog_exp_count++;
	if (sw_wdog_exp_count > 3)  {
		CM4_MSG_ERROR("He's dead, Jim!");
	}
}

void cm4_register_ipc_rcv_callback(void (*ipc_callback)(uint8_t *payload), IPC_PKT_TYPE pkt_type)
{
	ipc_callback_funcs[pkt_type] = ipc_callback;
}
EXPORT_SYMBOL(cm4_register_ipc_rcv_callback);

void cm4_unregister_ipc_rcv_callback(IPC_PKT_TYPE pkt_type)
{
	ipc_callback_funcs[pkt_type] = NULL;
}
EXPORT_SYMBOL(cm4_unregister_ipc_rcv_callback);


int cm4_ipc_send_packet(IPC_PKT_TYPE pkt_type, uint8_t *payload, uint32_t len)
{
	switch (pkt_type) {
		case IPC_PKT_CT1:
		case IPC_PKT_CT2:
			break;
		default: CM4_MSG_ERROR("Attempting to send disallowed packet type!");
			return -EPERM;
	}
	return ipc_send(pkt_type, payload, len, &ipc_config);
}
EXPORT_SYMBOL(cm4_ipc_send_packet);

static int cm4_ipc_recv_thread(void *data)
{
	int ret = 0;
	uint8_t payload[IPC_MAX_PAYLOAD_LEN];
	IPC_PKT_TYPE pkt_type;
	struct cm4_private_data *cm4_priv = platform_get_drvdata(data);

	CM4_MSG_INFO("CM4 IPC Receive thread started");

	while (!kthread_should_stop()) {
		if (!down_interruptible(&cm4_priv->ipc_recv_sem)) {
			ret = ipc_recv(&pkt_type, payload, &ipc_config);
			if (ret >= 0 ) {
				switch (pkt_type) {
					case IPC_PKT_WDG:
						mod_timer(&cm4_priv->sw_wdog_timer, jiffies + msecs_to_jiffies(CM4_SW_WDG_TIMEOUT));
						sw_wdog_exp_count = 0;
						break;
					case IPC_PKT_LOG:
					case IPC_PKT_SYS:
					case IPC_PKT_CT1:
					case IPC_PKT_CT2:
						if (ipc_callback_funcs[pkt_type] != NULL) {
							ipc_callback_funcs[pkt_type](payload);
						} else {
							CM4_MSG_ERROR("No Callback function registered for this packet type(%d)!", pkt_type);
						}
						break;
					default:
						CM4_MSG_ERROR("IPC with an unknown packet type received!");
						break;
				}
			} else if (!quit_flag) {
				CM4_MSG_ERROR("IPC receive function returned error: %d!", ret);
			}
		}
	}

	return ret;
}

static void cm4_power_on(int on)
{
	if (on) {
		cmsys_msk_write(CM4_REG_CONFIGURE, CM4_MSK_MPU_DISAB, CM4_MSK_MPU_DISAB);
		cmsys_msk_write(CM4_REG_CLOCK_CTL, CM4_MSK_CPU_CK_EN, CM4_MSK_CPU_CK_EN);
		cmsys_msk_write(CM4_REG_RESET_CTL, CM4_MSK_CPU_RESET, CM4_MSK_CPU_RESET);
		CM4_MSG_INFO("CM4 power and clock turned ON");
	} else {
		cmsys_msk_write(CM4_REG_RESET_CTL, CM4_MSK_CPU_RESET, ~CM4_MSK_CPU_RESET);
		cmsys_msk_write(CM4_REG_CLOCK_CTL, CM4_MSK_CPU_CK_EN, ~CM4_MSK_CPU_CK_EN);
		CM4_MSG_INFO("CM4 power and clock turned OFF");
	}
}


static int cm4_get_firmware(struct platform_device *pdev)
{
	int ret;
	struct cm4_private_data *cm4_priv = NULL;
	static const struct firmware *cm4_firmware;

	cm4_priv = platform_get_drvdata(pdev);
	if (IS_ERR(cm4_priv)) {
		CM4_MSG_ERROR("Failed to get driver data!");
		ret = PTR_ERR(cm4_priv);
	}

	CM4_MSG_INFO("Obtaining CM4 firmware image");
	ret = request_firmware(&cm4_firmware, CM4_FIRMWARE_IMAGE, &pdev->dev);
	if (ret) {
		CM4_MSG_ERROR("Requesting firmware failed (%d)!", ret);
		return ret;
	}

	if (cm4_firmware->size > CM4_CODE_SRAM_SIZE) {
		CM4_MSG_ERROR("The binary does not fit in the internal M4 SRAM!");
		return -EINVAL;
	}

	__iowrite32_copy((void *)cm4_priv->cmsys_code_sram, cm4_firmware->data, (cm4_firmware->size / 4));

	release_firmware(cm4_firmware);
	CM4_MSG_INFO("CM4 firmware image successfully obtained and loaded");
	cm4_power_on(1);

	return 0;
}

static void cm4_log_print(uint8_t *buf)
{
	bb_log(BB_MOD_M4, BB_LVL_INFO, "[CM4]-%s", (char *)buf);
}

static void cm4_system_stat(uint8_t *buf)
{
	memcpy(&sys_stat, buf, sizeof(sys_stat));
}

static ssize_t cm4_run_control(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	long run;

	if (kstrtol(buf, 0, &run) != 0) {
		CM4_MSG_ERROR("Failed to parse the entry");
		return -EFAULT;
	}

	if ((run != 0) && (run != 1)) {
		CM4_MSG_ERROR("Invalid Entry. Input 1/0 for run control");
		return -EINVAL;
	}

	cm4_power_on((int)run);
	return count;
}
static DEVICE_ATTR(run, (S_IWUSR|S_IWGRP), NULL, cm4_run_control);

static int cm4_stats_show(struct seq_file *m, void *priv)
{
	int i;
	char status[64];
	uint32_t cpu_usage[64];
	sys_status_t stat;

	memcpy(&stat, &sys_stat, sizeof(stat));
	stat.total_run_time = DIV_ROUND_CLOSEST(stat.total_run_time , 100);

	seq_printf(m, "----------------------------------\n");
	seq_printf(m, "     CM4 Runtime Statistics \n");
	seq_printf(m, "..................................\n");
	seq_printf(m, " Free Heap   : %08d\n", stat.free_heap);
	seq_printf(m, " IPC pkts(RX): %08d\n", stat.recv_pkt_cnt);
	seq_printf(m, " IPC pkts(TX): %08d\n\n", stat.send_pkt_cnt);
	seq_printf(m, "..................................\n");
	seq_printf(m, " TASK       CPU%%    P    S    STK\n");
	seq_printf(m, "..................................\n");
	stat.task_name[0][4] = ' ';
	stat.task_name[0][5] = ' ';
	stat.task_name[0][6] = ' ';
	stat.task_name[0][7] = '\0';
	for (i = 0; i < stat.num_tasks; i++) {
		switch(stat.task_status[i]) {
			case 0:
				status[i] = 'A';
				break;
			case 1:
				status[i] = 'R';
				break;
			case 2:
				status[i] = 'B';
				break;
			case 3:
				status[i] = 'S';
				break;
			case 4:
				status[i] = 'D';
				break;
			default:
				status[i] = '-';
				break;
		}
		cpu_usage[i] = DIV_ROUND_CLOSEST(stat.task_runtime[i], stat.total_run_time);
		seq_printf(m, " %s    %02d %%    %01d    %c    %03d\n", stat.task_name[i], ((cpu_usage[i] <= 99 ) ? cpu_usage[i] : 99), stat.task_curr_prio[i], status[i], stat.task_stack_hi_mark[i]);
	}
	seq_printf(m, "----------------------------------\n");

	return 0;
}

static int cm4_stats_open(struct inode *inode, struct file *file)
{
	return single_open(file, cm4_stats_show, inode->i_private);
}

static const struct file_operations cm4_stats_fops = {
	.open = cm4_stats_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};


static int cm4_debug_open(struct inode *inodep, struct file *filp)
{
	filp->private_data = inodep->i_private;
	return 0;
}


static ssize_t cm4_debug_write(struct file *filep, const char __user *user_buffer,
		size_t count, loff_t *position)
{
	unsigned char buff;
	struct cm4_private_data *cm4_priv = NULL;

	cm4_priv = platform_get_drvdata(filep->private_data);

	if (copy_from_user(&buff, user_buffer, sizeof(buff))) {
		CM4_MSG_ERROR("Failed to copy data from user");
		return -ENODEV;
	}

	buff -= '0';
	switch(buff)
	{
		case 0:
		case 1:
		case 2:
		case 3:
			break;
		default:
			bb_log(BB_MOD_M4, BB_LVL_ERR, "Invalid debug mode specified! (Echo 0, 1, 2 or 3)"
				"\n0. Disable debug modes\n1. Enable CM4 debug prints\n2. Enable CM4 run time statistics\n3. Enable both");
			return count;
	}

	if (ipc_send(IPC_PKT_DBG, &buff, sizeof(buff), &ipc_config)) {
		CM4_MSG_ERROR("IPC send failed!");
	}
	return count;
}

static const struct file_operations cm4_debug_fops = {
		.open = cm4_debug_open,
		.write = cm4_debug_write,
};

static int cm4_debug_init(struct platform_device *pdev)
{
	int ret = 0;
	struct dentry *cm4_reg;
	struct dentry *cm4_code;
	struct dentry *cm4_data;
	struct dentry *cm4_ipc_rx;
	struct dentry *cm4_ipc_tx;
	struct dentry *cm4_debug;
	struct dentry *cm4_stats;
	struct cm4_private_data *cm4_priv = NULL;

	cm4_priv = platform_get_drvdata(pdev);

	cm4_reg_set.regs = cm4_register_list;
	cm4_reg_set.nregs = ARRAY_SIZE(cm4_register_list);
	cm4_reg_set.base = cm4_priv->cmsys_base_address;

	cm4_code_sram.data = cm4_priv->cmsys_code_sram;
	cm4_code_sram.size = CM4_CODE_SRAM_SIZE;
	cm4_data_sram.data = cm4_priv->cmsys_data_sram;
	cm4_data_sram.size = CM4_DATA_SRAM_SIZE;

	cm4_debugfs_root_dir = debugfs_create_dir(CM4_DEVICE_NAME, NULL);
	if (IS_ERR(cm4_debugfs_root_dir)) {
		CM4_MSG_ERROR("Failed to create debugfs root directory");
		ret = PTR_ERR(cm4_debugfs_root_dir);
	}

	cm4_reg = debugfs_create_regset32("registers", S_IRUGO, cm4_debugfs_root_dir, &cm4_reg_set);
	if (IS_ERR(cm4_reg)) {
		CM4_MSG_ERROR("Failed to create debugfs 'registers' file");
		ret = PTR_ERR(cm4_reg);
	}

	cm4_code = debugfs_create_blob("code_sram", S_IRUGO, cm4_debugfs_root_dir, &cm4_code_sram);
	if (IS_ERR(cm4_code)) {
		CM4_MSG_ERROR("Failed to create debugfs 'code_sram' file");
		ret = PTR_ERR(cm4_code);
	}

	cm4_data = debugfs_create_blob("data_sram", S_IRUGO, cm4_debugfs_root_dir, &cm4_data_sram);
	if (IS_ERR(cm4_data)) {
		CM4_MSG_ERROR("Failed to create debugfs 'data_sram' file");
		ret = PTR_ERR(cm4_data);
	}

	cm4_ipc_rx = debugfs_create_u32("rx_pkt_cnt", S_IRUGO, cm4_debugfs_root_dir, &ipc_config.rcv_pkt_cnt);
	if (IS_ERR(cm4_ipc_rx)) {
		CM4_MSG_ERROR("Failed to create debugfs 'rx_pkt_cnt' file");
		ret = PTR_ERR(cm4_ipc_rx);
	}

	cm4_ipc_tx = debugfs_create_u32("tx_pkt_cnt", S_IRUGO, cm4_debugfs_root_dir, &ipc_config.snd_pkt_cnt);
	if (IS_ERR(cm4_ipc_tx)) {
		CM4_MSG_ERROR("Failed to create debugfs 'tx_pkt_cnt' file");
		ret = PTR_ERR(cm4_ipc_tx);
	}

	cm4_debug = debugfs_create_file("debug", S_IRUGO, cm4_debugfs_root_dir, pdev, &cm4_debug_fops);
	if (IS_ERR(cm4_debug)) {
		CM4_MSG_ERROR("Failed to create debugfs 'debug' file");
		ret = PTR_ERR(cm4_debug);
	}

	cm4_stats = debugfs_create_file("stats", S_IRUGO, cm4_debugfs_root_dir, pdev, &cm4_stats_fops);
	if (IS_ERR(cm4_stats)) {
		CM4_MSG_ERROR("Failed to create debugfs 'stats' file");
		ret = PTR_ERR(cm4_stats);
	}

	if (ret) {
		CM4_MSG_ERROR("Debugfs init failed!");
		debugfs_remove_recursive(cm4_debugfs_root_dir);
	}

	return ret;
}


static int cm4_open(struct inode *inodep, struct file *filp)
{
	CM4_MSG_DEBUG("Device special file opened");
	return 0;
}


static ssize_t cm4_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset)
{
	CM4_MSG_DEBUG("Read from device special file by userspace process");
	return size;
}


static long cm4_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = CM4_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	default:
		CM4_MSG_ERROR("Unrecognized IOCTL command %u.", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}


static unsigned int cm4_poll(struct file *filp, struct poll_table_struct *table)
{
	unsigned int mask = 0;
	mask |= (POLLIN | POLLRDNORM);
	poll_wait(filp, &read_queue, table);
	return mask;
}


static ssize_t cm4_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset)
{
	CM4_MSG_INFO("Device special file was written to by a userspace application");
	return size;
}


static int cm4_release(struct inode *inodep, struct file *filp)
{
	CM4_MSG_DEBUG("Device special file closed");
	return 0;
}


static loff_t cm4_llseek(struct file *filp, loff_t offset, int seek_type)
{
	CM4_MSG_ERROR("Seek not supported in this driver!");
	return -EINVAL;
}


static const struct file_operations cm4_fops =
{
	.open		= cm4_open,
	.read		= cm4_read,
	.poll		= cm4_poll,
	.write		= cm4_write,
	.llseek		= cm4_llseek,
	.release	= cm4_release,
	.unlocked_ioctl	= cm4_ioctl,
};


static int cm4_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct resource *res;
	struct cm4_private_data *cm4_priv;

	cm4_priv = devm_kzalloc(&pdev->dev, sizeof(*cm4_priv), GFP_KERNEL);
	if (!cm4_priv) {
		CM4_MSG_ERROR("Allocating memory for the private structure failed");
		return -ENOMEM;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	cm4_priv->cmsys_base_address = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(cm4_priv->cmsys_base_address)) {
		return PTR_ERR(cm4_priv->cmsys_base_address);
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	cm4_priv->cmsys_code_sram = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(cm4_priv->cmsys_code_sram)) {
		return PTR_ERR(cm4_priv->cmsys_code_sram);
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 2);
	cm4_priv->cmsys_data_sram = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(cm4_priv->cmsys_data_sram)) {
		return PTR_ERR(cm4_priv->cmsys_data_sram);
	}

	cm4_priv->cmsys_irq = platform_get_irq(pdev, 0);
	if (cm4_priv->cmsys_irq < 0) {
		CM4_MSG_ERROR("No IRQ resource found in the DT");
		return -ENODEV;
	}

	platform_set_drvdata(pdev, cm4_priv);
	g_cmsys_base_address = cm4_priv->cmsys_base_address;

	ret = cm4_debug_init(pdev);
	if (ret) {
		CM4_MSG_ERROR("Setting up debug file system failed");
		return ret;
	}

	cm4_priv->cmsys_clk=devm_clk_get(&pdev->dev, "cmsys_sel");
	if (IS_ERR(cm4_priv->cmsys_clk)) {
		CM4_MSG_ERROR("Failure getting CMSYS sel clk");
		return PTR_ERR(cm4_priv->cmsys_clk);
	}

	ret = clk_prepare_enable(cm4_priv->cmsys_clk);
	if (ret < 0) {
		CM4_MSG_ERROR("Failed to enable m4 clk (%d)", ret);
		return -EFAULT;
	}

	clear_cpu_to_cpu_interrupt();

	if (devm_request_irq(&pdev->dev, cm4_priv->cmsys_irq, cpu_to_cpu_isr, 0, CM4_DEVICE_NAME, pdev)) {
		CM4_MSG_ERROR("Failed to request IRQ");
	}

	sema_init(&cm4_priv->ipc_recv_sem, 0);

	memset(cm4_priv->cmsys_data_sram, 0 , CM4_DATA_SRAM_SIZE);
	ret = ipc_init(cm4_priv->cmsys_data_sram, CM4_DATA_SRAM_SIZE, IPC_NODE_AP, &ipc_config);
	if (ret) {
		CM4_MSG_ERROR("Initializing IPC failed");
		clk_disable_unprepare(cm4_priv->cmsys_clk);
		return -EFAULT;
	}

	cm4_register_ipc_rcv_callback(cm4_log_print, IPC_PKT_LOG);
	cm4_register_ipc_rcv_callback(cm4_system_stat, IPC_PKT_SYS);
	quit_flag=0;

	ret = cm4_get_firmware(pdev);
	if (ret < 0) {
		CM4_MSG_ERROR("Getting firmware image failed (%d)", ret);
		clk_disable_unprepare(cm4_priv->cmsys_clk);
		return -EFAULT;
	}

	setup_timer(&cm4_priv->sw_wdog_timer, cm4_wdog_callback, 0);
	mod_timer(&cm4_priv->sw_wdog_timer, jiffies + msecs_to_jiffies(CM4_SW_WDG_TIMEOUT));
	sw_wdog_exp_count = 0;

	cm4_priv->kthread_task = kthread_run(&cm4_ipc_recv_thread, pdev, "CM4-IPC-RX");
	if (IS_ERR(cm4_priv->kthread_task)) {
		CM4_MSG_ERROR("Starting kthread failed");
		del_timer(&cm4_priv->sw_wdog_timer);
		clk_disable_unprepare(cm4_priv->cmsys_clk);
		return -EFAULT;
	}

	CM4_MSG_DEBUG("Sonos mtk CM4 driver probed!");
	return ret;
}


static int cm4_remove(struct platform_device *pdev)
{
	struct cm4_private_data *cm4_priv = NULL;

	cm4_priv = platform_get_drvdata(pdev);
	if (IS_ERR(cm4_priv)) {
		CM4_MSG_ERROR("Failed to get driver data!");
		return PTR_ERR(cm4_priv);
	}

	quit_flag = 1;
	up(&cm4_priv->ipc_recv_sem);
	kthread_stop(cm4_priv->kthread_task);
	del_timer(&cm4_priv->sw_wdog_timer);
	ipc_exit(&ipc_config);
	cm4_power_on(0);
	clk_disable_unprepare(cm4_priv->cmsys_clk);
	debugfs_remove_recursive(cm4_debugfs_root_dir);

	CM4_MSG_DEBUG("Sonos mtk CM4 driver removed!");

	return 0;
}

static const struct of_device_id cm4_dt_ids[] = {
	{ .compatible = "sonos,mtk-m4", },
	{  }
};
MODULE_DEVICE_TABLE(of, cm4_dt_ids);

static struct platform_driver cm4_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "CM4 Driver",
		.of_match_table = cm4_dt_ids,
		},
	.probe = cm4_probe,
	.remove = cm4_remove,
};


static int __init mtk_cm4_driver_init(void)
{
	int ret;
	struct device *cm4_device = NULL;

	devno = MKDEV(CM4_MAJOR_NUMBER, 0);

#ifdef CONFIG_DEVTMPFS
	ret = alloc_chrdev_region(&devno, 0, 1, CM4_DEVICE_NAME);
#else
	ret = register_chrdev_region(devno, 1, CM4_DEVICE_NAME);
#endif
	if (ret) {
		CM4_MSG_ERROR("Failed to register CM4 character node (%d)", ret);
		return ret;
	}

	cdev_init(&cm4_chr_dev, &cm4_fops);
	cm4_chr_dev.owner = THIS_MODULE;

	ret = cdev_add(&cm4_chr_dev, devno, 1);
	if (ret) {
		CM4_MSG_ERROR("Failed to add CM4 chrdev %d:0 (%d)", MAJOR(devno), ret);
		unregister_chrdev_region(devno, 1);
		return ret;
	}

	ret = platform_driver_register(&cm4_driver);
	if (ret) {
		CM4_MSG_ERROR("Failed to register CM4 driver");
		cdev_del(&cm4_chr_dev);
		unregister_chrdev_region(devno, 1);
		return ret;
	} else {
		CM4_MSG_INFO("Registered Sonos mtk M4 driver with major number: %d", MAJOR(devno));
	}

	cm4_dev_class = class_create(THIS_MODULE, CM4_CLASS_NAME);
	if (IS_ERR(cm4_dev_class)) {
		ret = PTR_ERR(cm4_dev_class);
		CM4_MSG_ERROR("Failed creating the device class");
		goto err_out;
	}

	cm4_device = device_create(cm4_dev_class, NULL, devno, NULL, CM4_DEVICE_NAME);
	if (IS_ERR(cm4_device)) {
		ret = PTR_ERR(cm4_device);
		CM4_MSG_ERROR("Failed creating the device in class");
		goto err_out;
	}

	ret = device_create_file(cm4_device, &dev_attr_run);
	if (ret) {
		CM4_MSG_ERROR("Failed to create the run control file");
		goto err_out;
	}

	CM4_MSG_DEBUG("CM4 driver initialization succeeded");
	return 0;

err_out:
	platform_driver_unregister(&cm4_driver);
	cdev_del(&cm4_chr_dev);
	unregister_chrdev_region(devno, 1);
	return ret;
}


static void __exit mtk_cm4_driver_exit(void)
{
	device_destroy(cm4_dev_class, devno);
	class_destroy(cm4_dev_class);
	platform_driver_unregister(&cm4_driver);
	cdev_del(&cm4_chr_dev);
	unregister_chrdev_region(devno, 1);
	CM4_MSG_INFO("Unloaded Sonos mtk M4 driver");
}


module_init(mtk_cm4_driver_init);
module_exit(mtk_cm4_driver_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Sonos Mediatek CM4 driver");
MODULE_LICENSE("GPL");
MODULE_FIRMWARE(CM4_FIRMWARE_IMAGE);
