/*
 * Copyright (c) 2018~2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/timekeeping.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/cdev.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/clk.h>

#include "blackbox.h"
#include "sonos_device.h"
#include "mtk_irrx.h"
#include "mtk_ir_regs.h"

#define DRIVER_NAME "IR Driver"
#define IR_PROC_FILE            "driver/"IR_DEVICE_NAME
#define IR_PROC_PERM            (S_IRUSR | S_IRGRP | S_IROTH | S_IWUSR)

#define IR_MSG_PRINT(LEVEL, FORMAT, ARG...) bb_log(BB_MOD_IR, LEVEL, FORMAT, ## ARG)
#define IR_MSG_INFO(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_INFO, FORMAT, ## ARG)
#define IR_MSG_WARN(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_WARNING, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define IR_MSG_ERROR(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_ERR, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define IR_MSG_DEBUG(FORMAT, ARG...) bb_log_dbg(BB_MOD_IR, FORMAT, ## ARG)

struct ir_private_data {
	bool ir_ready;
	bool record_stat;
	uint32_t user_count;
	uint32_t ir_len;
	uint32_t ir_data[IR_SYMBOL_MAX_NUM_EDGES];
};

static int devno = -1;

static int __init ir_driver_init(void);
static void __exit ir_driver_exit(void);
static int ir_proc_init(void);
static int ir_proc_show(struct seq_file *m, void *v);
static int ir_proc_open(struct inode *inode, struct  file *file);
static ssize_t ir_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
static int ir_open(struct inode *inodep, struct file *filp);
static int ir_release(struct inode *inodep, struct file *filp);
static int ir_probe(struct platform_device *pdev);
static int ir_remove(struct platform_device *pdev);
static ssize_t ir_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset);
static ssize_t ir_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset);
static loff_t ir_llseek(struct file *filp, loff_t offset, int seek_type);
static unsigned int ir_poll(struct file *filp, struct poll_table_struct *table);

DEFINE_RWLOCK(ir_rwlock);
DECLARE_WAIT_QUEUE_HEAD(read_queue);

struct cdev ir_chr_dev;
static struct ir_private_data ir_priv = {
	.user_count = 0,
	.record_stat = 0,
	.ir_len = 0,
	.ir_ready = 0,
};

static void __iomem *irrx_base_addr;

static void IR_WRITE32(u32 offset, u32 value)
{
	__raw_writel(value, ((void *)(irrx_base_addr + offset)));
}

static u32 IR_READ32(u32 offset)
{
	return __raw_readl((void *)(irrx_base_addr + offset));
}

static void IR_WRITE_MASK(u32 u4Addr, u32 u4Mask, u32 u4Offset, u32 u4Val)
{
	IR_WRITE32(u4Addr, (IR_READ32(u4Addr) & ~u4Mask) | ((u4Val << u4Offset) & u4Mask));
	dsb(sy);
}

static void ir_enable_hwirq(void)
{
	IR_WRITE_MASK(IRRX_IRINT_CLR, IRRX_INTCLR_MASK, IRRX_INTCLR_OFFSET, 0x1);
	IR_WRITE_MASK(IRRX_IRCLR, IRRX_IRCLR_MASK, IRRX_IRCLR_OFFSET, 0x1);
	IR_WRITE_MASK(IRRX_IRINT_EN, IRRX_INTEN_MASK, IRRX_INTEN_OFFSET, 0x1);
}

static void ir_disable_hwirq(void)
{
	IR_WRITE_MASK(IRRX_IRINT_CLR, IRRX_INTCLR_MASK,IRRX_INTCLR_OFFSET, 0x1);
	IR_WRITE_MASK(IRRX_IRINT_EN, IRRX_INTEN_MASK, IRRX_INTEN_OFFSET, 0x0);
}

static irqreturn_t ir_core_irq(int irq, void *dev_id)
{
	u32* output = &ir_priv.ir_data[0];
	u32 ir_pulse_word;
	int i;
	int idx;

	ir_disable_hwirq();

	write_lock(&ir_rwlock);
	ir_priv.ir_len = (IR_READ32(IRRX_EXPBCNT) >> IRRX_IRCHK_CNT_OFFSET) & IRRX_IRCHK_CNT;

	for (i = 0; i < ir_priv.ir_len; i += 4) {
		ir_pulse_word = IR_READ32(IRRX_CHKDATA0 + i);
		for (idx = 0; idx < 32; idx += 8) {
			*(output++) = (ir_pulse_word >> idx) & 0xFF;
		}
	}

	ir_priv.ir_ready = 1;
	if(ir_priv.user_count > 0) {
		wake_up(&read_queue);
	}
	write_unlock(&ir_rwlock);

	ir_enable_hwirq();
	return IRQ_HANDLED;
}

static int ir_proc_show(struct seq_file *m, void *v)
{
	int i = 0;
	bool ir_ready;
	bool record_stat;
	uint32_t temp_len;
	uint32_t temp_buff[IR_SYMBOL_MAX_NUM_EDGES];
	struct ir_private_data *priv_data = (struct ir_private_data *)m->private;

	read_lock(&ir_rwlock);
	ir_ready = priv_data->ir_ready;
	record_stat = priv_data->record_stat;
	temp_len = priv_data->ir_len;
	for(i = 0; i < temp_len; i++) {
		temp_buff[i] = priv_data->ir_data[i];
	}
	read_unlock(&ir_rwlock);

	if(record_stat == IR_RECORD_ON) {
		if(ir_ready) {
			seq_printf(m, "replay=");
			for(i=0; i<temp_len; i++) {
				seq_printf(m, "%08X,", temp_buff[i]);
			}
		} else {
			seq_printf(m, "NO IR Data!\n");
		}
	} else {
		if(ir_ready) {
			seq_printf(m, "IR Data: ");
			for(i=0; i<temp_len; i++) {
				seq_printf(m, "%08X,", temp_buff[i]);
			}
			seq_printf(m, "\n");
		} else {
			seq_printf(m, "NO IR Data!\n");
		}
	}

	write_lock(&ir_rwlock);
	priv_data->ir_ready = 0;
	write_unlock(&ir_rwlock);
	return 0;
}


static int ir_proc_open(struct inode *inode, struct  file *file)
{
	return single_open(file, ir_proc_show, PDE_DATA(inode));
}


static ssize_t ir_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	struct ir_private_data *priv_data = PDE_DATA(file_inode(file));
	char *temp = NULL;
	char *buf = NULL;
	char *tok = NULL;
	int i = 0;
	int ret;
	unsigned long value;

	buf = (char *)kmalloc(sizeof(char)*2048, GFP_KERNEL);
	if (buf == NULL) {
		IR_MSG_ERROR("Failed to allocate memory for buffer");
		ret = -ENOMEM;
		goto err_out;
	}

	if (count > 2047) {
		IR_MSG_ERROR("The string is too long");
		ret = -EIO;
		goto err_out;
	} else if (copy_from_user(buf, buffer, count)) {
		ret = -EFAULT;
		goto err_out;
	}

	buf[count] = '\0';
	temp = strchr(buf, '=');
	if(temp == NULL) {
		IR_MSG_ERROR("Invalid format for proc command");
		IR_MSG_ERROR("Use <command>=<value(s)>");
		ret = -EINVAL;
		goto err_out;
	}
	temp++;

	if (strncmp(buf, "record", 6) == 0) {
		if (strncmp(temp, "on", 2) == 0) {
			write_lock(&ir_rwlock);
			priv_data->record_stat = IR_RECORD_ON;
			write_unlock(&ir_rwlock);

			IR_MSG_INFO("IR record mode ON. The IR data received will be raw edge timestamps");
		} else if (strncmp(temp, "off", 3) == 0) {
			write_lock(&ir_rwlock);
			priv_data->record_stat = IR_RECORD_OFF;
			write_unlock(&ir_rwlock);

			IR_MSG_INFO("IR record mode OFF. The IR data received will be normal");
		} else {
			IR_MSG_ERROR("Failed to parse the command: %s", temp);
			ret = -EINVAL;
			goto err_out;
		}
	} else if (strncmp(buf, "replay", 6) == 0) {
		write_lock(&ir_rwlock);
		ir_priv.ir_len = 0;

		for (i = 0; i < IR_SYMBOL_MAX_NUM_EDGES; i++) {
			tok = strsep(&temp, ",");
			if (temp == NULL){
				break;
			}
			if (kstrtoul(tok, 16, &value)) {
				IR_MSG_ERROR("Failed to parse the value: %s", tok);
				ret = -EINVAL;
				goto err_out;
			} else {
				ir_priv.ir_data[i] = (uint32_t)value;
				ir_priv.ir_len++;
			}
		}

		ir_priv.ir_ready = 1;
		if(ir_priv.user_count > 0) {
			wake_up(&read_queue);
		}

		write_unlock(&ir_rwlock);

		IR_MSG_INFO("IR in replay mode. Will replay the edges provided");
	} else {
		IR_MSG_ERROR("Unknown command: %s", buf);
		ret = -EINVAL;
		goto err_out;
	}

	ret = count;
err_out:
	kfree(buf);
	return ret;
}


static const struct file_operations ir_proc_fops = {
	.owner = THIS_MODULE,
	.open = ir_proc_open,
	.write = ir_proc_write,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int ir_proc_init(void)
{
	struct proc_dir_entry *ir_proc_entry;
	ir_proc_entry = proc_create_data(IR_PROC_FILE, IR_PROC_PERM, NULL, &ir_proc_fops, &ir_priv);
	if (!ir_proc_entry) {
		return -EIO;
	}
	return 0;
}


static int ir_open(struct inode *inodep, struct file *filp)
{
	write_lock(&ir_rwlock);
	ir_priv.user_count++;
	write_unlock(&ir_rwlock);
	return 0;
}


static ssize_t ir_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset)
{
	int i;
	int ret;
	char kbuf[IR_SYMBOL_MAX_NUM_EDGES];
	uint32_t temp_len;

	read_lock(&ir_rwlock);
	temp_len = ir_priv.ir_len;
	read_unlock(&ir_rwlock);

	if (size < IR_SYMBOL_MAX_NUM_EDGES) {
		IR_MSG_ERROR("Invalid size! Read must be at least %d bytes (1 max-length symbol).", IR_SYMBOL_MAX_NUM_EDGES);
		return -EINVAL;
	}

	read_lock(&ir_rwlock);
	for (i = 0; i < temp_len; i++) {
		kbuf[i] = (char)ir_priv.ir_data[i];
	}
	read_unlock(&ir_rwlock);

	ret = copy_to_user(buffer, kbuf, temp_len);
	if (ret) {
		IR_MSG_ERROR("Failed to copy buffer from kernel to userspace");
		return ret;
	}

	IR_MSG_DEBUG("IR symbol read by userspace process");

	write_lock(&ir_rwlock);
	ir_priv.ir_ready = 0;
	write_unlock(&ir_rwlock);
	return temp_len;
}

static long ir_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = IR_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			IR_MSG_ERROR("copy_to_user() failed");
			return -EACCES;
		}
		break;
	}
	default:
		IR_MSG_ERROR("Unrecognized IOCTL command %u.", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}


static unsigned int ir_poll(struct file *filp, struct poll_table_struct *table)
{
	unsigned int mask = 0;
	read_lock(&ir_rwlock);
	if (ir_priv.ir_ready) {
		IR_MSG_DEBUG("IR Data ready to be read!");
		mask |= (POLLIN | POLLRDNORM);
	}
	read_unlock(&ir_rwlock);
	poll_wait(filp, &read_queue, table);
	return mask;
}
static ssize_t ir_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset)
{
	IR_MSG_ERROR("Writing to the device file not implemented yet!");
	return -EINVAL;
}


static int ir_release(struct inode *inodep, struct file *filp)
{
	write_lock(&ir_rwlock);
	ir_priv.user_count--;
	write_unlock(&ir_rwlock);
	return 0;
}


static loff_t ir_llseek(struct file *filp, loff_t offset, int seek_type)
{
	IR_MSG_ERROR("Seek not supported in this driver!");
	return -EINVAL;
}

static const struct file_operations ir_fops =
{
	.open		= ir_open,
	.read		= ir_read,
	.poll		= ir_poll,
	.write		= ir_write,
	.llseek		= ir_llseek,
	.release	= ir_release,
	.unlocked_ioctl	= ir_ioctl,
};

static int ir_config_hw(struct platform_device *pdev)
{
	const char *clkname = "irrx_clock";
	struct clk *irrx_clk;
	struct pinctrl *irrx_pinctrl1;
	struct pinctrl_state *irrx_pins_default;
	u32 irrx_irq;
	u32 irrx_irq_type;
	int ret;

	irrx_base_addr = of_iomap(pdev->dev.of_node, 0);
	if (!irrx_base_addr) {
		IR_MSG_ERROR(" Cannot get irrx register base address!");
		return -ENODEV;
	}
	irrx_clk = devm_clk_get(&pdev->dev, clkname);
	if (IS_ERR(irrx_clk)) {
		IR_MSG_ERROR(" Cannot get irrx clock!");
		ret = PTR_ERR(irrx_clk);
		IR_MSG_ERROR(" ret=%d!", ret);
		iounmap(irrx_base_addr);
		return ret;
	}

	ret = clk_prepare_enable(irrx_clk);
	if (ret) {
		IR_MSG_ERROR(" Enable clk failed!");
		iounmap(irrx_base_addr);
		return -ENODEV;
	}

	irrx_pinctrl1 = devm_pinctrl_get(&pdev->dev);
	if (IS_ERR(irrx_pinctrl1)) {
		ret = PTR_ERR(irrx_pinctrl1);
		IR_MSG_ERROR("Cannot find pinctrl %d!", ret);
		iounmap(irrx_base_addr);
		clk_disable_unprepare(irrx_clk);
		return ret;
	}

	irrx_pins_default = pinctrl_lookup_state(irrx_pinctrl1, "default");
	if (IS_ERR(irrx_pins_default)) {
		ret = PTR_ERR(irrx_pins_default);
		IR_MSG_INFO("Cannot find pinctrl default %d!", ret);
		iounmap(irrx_base_addr);
		clk_disable_unprepare(irrx_clk);
		return ret;
	}
	pinctrl_select_state(irrx_pinctrl1, irrx_pins_default);

	ir_disable_hwirq();
	IR_WRITE32(IRRX_CONFIG_HIGH_REG, MTK_IRRX_CONFIG);
	IR_WRITE32(IRRX_CONFIG_LOW_REG, MTK_IRRX_SAPERIOD);
	IR_WRITE32(IRRX_THRESHOLD_REG, MTK_IRRX_THRESHOLD);

	irrx_irq = irq_of_parse_and_map(pdev->dev.of_node, 0);
	irrx_irq_type = irq_get_trigger_type(irrx_irq);
	IR_MSG_INFO("irq: %d, type: %d!", irrx_irq, irrx_irq_type);

	ret = request_threaded_irq(irrx_irq, NULL, ir_core_irq, irrx_irq_type | IRQF_ONESHOT,
			DRIVER_NAME, NULL);
	if (ret) {
		IR_MSG_INFO("fail to request irq(%d), ir_dev ret = %d !!!",
			   irrx_irq, ret);
		return ret;
	}
	ir_enable_hwirq();

	return 0;
}

static int ir_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct device *class_dev = NULL;

	ret = ir_config_hw(pdev);
	if (ret) {
		IR_MSG_ERROR("Failed to configure IRRX hardware!");
		return ret;
	}

	ret = ir_proc_init();
	if (ret) {
		IR_MSG_ERROR("Setting up proc file system failed");
		return ret;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, IR_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		IR_MSG_ERROR("Invalid pointer to the Sonos device struct!");
		ret = PTR_ERR(class_dev);
		remove_proc_entry(IR_PROC_FILE, NULL);
		return ret;
	}

	return ret;
}


static int ir_remove(struct platform_device *pdev)
{
	remove_proc_entry(IR_PROC_FILE, NULL);
	sonos_device_destroy(devno);

	return 0;
}

static const struct of_device_id ir_dt_ids[] = {
	{.compatible = "sonos,mt8518-ir-rcvr",},
	{  }
};
MODULE_DEVICE_TABLE(of, ir_dt_ids);

static struct platform_driver ir_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = DRIVER_NAME,
		.of_match_table = ir_dt_ids,
		},
	.probe = ir_probe,
	.remove = ir_remove,
};


static int __init ir_driver_init(void)
{
	int ret;

	devno = MKDEV(IR_MAJOR_NUMBER, 0);

#ifdef CONFIG_DEVTMPFS
	ret = alloc_chrdev_region(&devno, 0, 1, IR_DEVICE_NAME);
#else
	ret = register_chrdev_region(devno, 1, IR_DEVICE_NAME);
#endif
	if (ret) {
		IR_MSG_ERROR("Failed to register IR character node (%d)", ret);
		return ret;
	}

	cdev_init(&ir_chr_dev, &ir_fops);
	ir_chr_dev.owner = THIS_MODULE;

	ret = cdev_add(&ir_chr_dev, devno, 1);
	if (ret) {
		IR_MSG_ERROR("Failed to add IR chrdev %d:0 (%d)", MAJOR(devno), ret);
		goto cdev_err;
	}

	ret = platform_driver_register(&ir_driver);
	if (ret) {
		IR_MSG_ERROR("Failed to register IR Receiver driver");
		goto plat_err;
	} else {
		IR_MSG_INFO("Registered IR Receiver driver with major number: %d", MAJOR(devno));
	}

	return 0;

plat_err:
	cdev_del(&ir_chr_dev);
cdev_err:
	unregister_chrdev_region(devno, 1);

	return ret;
}


static void __exit ir_driver_exit(void)
{
	platform_driver_unregister(&ir_driver);
	cdev_del(&ir_chr_dev);
	unregister_chrdev_region(devno, 1);
	IR_MSG_INFO("Unloaded IR Receiver driver");
}

module_init(ir_driver_init);
module_exit(ir_driver_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("MTK8521 IR Receiver driver");
MODULE_LICENSE("GPL");
