/*
 * Copyright (c) 2016-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the TI PCM1864 Mic Aggregator/ADC
 */

#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>

#include "blackbox.h"
#include "micctl.h"
#include "pcm1864.h"
#include "mdp.h"
#include "sonos_lock.h"

#include "micctl_core.h"

struct adc_priv_data {
	struct i2c_client *i2c_client;
	u32 dev_id;
	struct micctl_device dev;
};

static int pcm1864_proc_init(struct adc_priv_data *);
static void pcm1864_proc_remove(struct adc_priv_data *);

#define PCM_WRITE(R, V)	i2c_smbus_write_byte_data(pcm1864->i2c_client, R, V)
#define PCM_READ(R)	i2c_smbus_read_byte_data(pcm1864->i2c_client, R)

static int pcm1864_init_regs(struct adc_priv_data *pcm1864)
{
	int ret = 0;
	ret = PCM_WRITE(PCM1864_ADC1_L, PCM1864_CONF_ADC1_L);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setup ADC 1 L returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_ADC1_R, PCM1864_CONF_ADC1_R);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setup ADC 1 R returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_ADC2_L, PCM1864_CONF_ADC2_L);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setup ADC 2 L returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_ADC2_R, PCM1864_CONF_ADC2_R);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setup ADC 2 R returned %d.", ret);
	}

	ret = PCM_WRITE(PCM1864_PGA_CONF, PCM1864_CONF_PGA);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setting PGA configuration returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_PGA_CH1L, PCM1864_CONF_PGA_VAL);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setting PGA gain returned %d.", ret);
	}

	ret = PCM_WRITE(PCM1864_TDM_FMT, PCM1864_CONF_TDM_FMT);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setting TDM format returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_TDM_MODE, PCM1864_CONF_TDM_MODE);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setting TDM mode returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_TDM_OFFSET, PCM1864_CONF_TDM_OFFSET(pcm1864->dev_id));
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setting TDM offset returned %d.", ret);
	}

	ret = PCM_WRITE(PCM1864_DSP_CTL, PCM1864_CONF_DSP_CTL);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setting DSP config returned %d.", ret);
	}

	ret = PCM_WRITE(PCM1864_DIGMIC, 0x00);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Disabling digital mics returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_IRQ_POL, PCM1864_CONF_IRQ_POL);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setup IRQ line returned %d.", ret);
	}

	return ret;
}

static int pcm1864_init_pdm(struct adc_priv_data *pcm1864)
{
	int ret = 0;

	ret = PCM_WRITE(PCM1864_GPIO0_GPIO1, PCM1864_CONF_GPIO0_GPIO1);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setup GPIO0 and GPIO1 returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_GPIO2_GPIO3, PCM1864_CONF_GPIO2_GPIO3);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setup GPIO2 and GPIO3 returned %d.", ret);
	}
	ret = PCM_WRITE(PCM1864_DIGMIC, PCM1864_CONF_DIGMIC);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Setup digital mics returned %d.", ret);
	}

	return 0;
}

static int pcm1864_is_muted(struct micctl_device *dev)
{
	u32 val = 0;
	struct adc_priv_data *pcm1864 = container_of(dev, struct adc_priv_data, dev);
	val = PCM_READ(PCM1864_PWR_STATE);
	return (val == 0);
}

static int pcm1864_set_mute(struct micctl_device *dev, int on)
{
	int ret = 0;
	u32 val = 0;
	struct adc_priv_data *pcm1864 = container_of(dev, struct adc_priv_data, dev);
	val = PCM_READ(PCM1864_PWR_CTL);
	if (on) {
		val |= (PCM1864_PWR_DIGI | PCM1864_PWR_SLEEP | PCM1864_PWR_ADC);
	} else {
		val &= ~(PCM1864_PWR_DIGI | PCM1864_PWR_SLEEP | PCM1864_PWR_ADC);
	}
	ret = PCM_WRITE(PCM1864_PWR_CTL, val);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "%sute operation returned %d. PCM1864 state: %x.", (on ? "M" : "Unm"), ret, PCM_READ(PCM1864_PWR_STATE));
	}

	return ret;
}


static int pcm1864_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	struct device_node *node = client->dev.of_node;
	struct adc_priv_data *pcm1864 = devm_kzalloc(&(client->dev), sizeof(struct adc_priv_data), GFP_KERNEL);
	if (!pcm1864) {
		bb_log_dev(&(client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Memory allocation failed");
		return -ENOMEM;
	}

	dev_set_drvdata(&(client->dev), pcm1864);

	pcm1864->i2c_client = client;
	if (of_property_read_u32(node, "dev-id", &(pcm1864->dev_id))) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Could not get device ID.");
		return -ENODEV;
	}

	ret = pcm1864_init_regs(pcm1864);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Failed to setup ADC registers.");
		return ret;
	}
#ifdef PCM1864_DEFAULT_PDM
	ret = pcm1864_init_pdm(pcm1864);
	if (ret) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_ERR, "Failed to set up PDM registers.");
		return ret;
	}
#endif

	ret = pcm1864_proc_init(pcm1864);

	pcm1864->dev.of_node = node;
	pcm1864->dev.mute = pcm1864_set_mute;
	pcm1864->dev.is_muted = pcm1864_is_muted;
	micctl_register_device(&pcm1864->dev);

	bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_INFO, "ADC %u registered.", pcm1864->dev_id);
	return ret;
}

static int pcm1864_remove(struct i2c_client *client)
{
	struct adc_priv_data *pcm1864 = dev_get_drvdata(&(client->dev));
	pcm1864_proc_remove(pcm1864);
	pcm1864_set_mute(&(pcm1864->dev), 1);
	bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_INFO, "ADC %u removed.", pcm1864->dev_id);
	return 0;
}

const char * pcm1864_pwr_state_str(struct adc_priv_data *pcm1864)
{
	switch (PCM_READ(PCM1864_PWR_STATE)) {
	case 0x00:
		return "Power down";
	case 0x01:
		return "Wait for stable clock";
	case 0x02:
		return "Release reset";
	case 0x03:
		return "Standby";
	case 0x04:
		return "Fade-in";
	case 0x05:
		return "Fade-out";
	case 0x09:
		return "Sleep";
	case 0x0f:
		return "Running";
	default:
		return "Unknown";
	}
}

const char * pcm1864_fs_state_str(struct adc_priv_data *pcm1864)
{
	switch (PCM_READ(PCM1864_FS_STATE)) {
	case 0x00:
		return "Low range or stopped FS";
	case 0x01:
		return "8kHz";
	case 0x02:
		return "16kHz";
	case 0x03:
		return "32-48kHz";
	case 0x04:
		return "88.2-96kHz";
	case 0x05:
		return "176.4-192kHz";
	case 0x06:
		return "High range";
	case 0x07:
		return "Invalid";
	default:
		return "Unknown";
	}
}

const char * pcm1864_bclk_state_str(struct adc_priv_data *pcm1864)
{
	u32 state = (PCM_READ(PCM1864_BCLK_STATE) >> 4);
	switch (state) {
	case 0x00:
		return "Low range or stopped BCLK";
	case 0x01:
		return "32fs";
	case 0x02:
		return "48fs";
	case 0x03:
		return "64fs";
	case 0x04:
		return "256fs";
	case 0x06:
		return "High range";
	case 0x07:
		return "Invalid ratio or stopped FS";
	default:
		return "Unknown";
	}
}

const char * pcm1864_sclk_state_str(struct adc_priv_data *pcm1864)
{
	u32 state = (PCM_READ(PCM1864_BCLK_STATE) >> 4);
	switch (state) {
	case 0x00:
		return "Low range or stopped SCLK";
	case 0x01:
		return "128fs";
	case 0x02:
		return "256fs";
	case 0x03:
		return "384fs";
	case 0x04:
		return "512fs";
	case 0x05:
		return "768fs";
	case 0x06:
		return "High range";
	case 0x07:
		return "Invalid ratio or stopped FS";
	default:
		return "Unknown";
	}
}

#define PCM1864_PAGE_0_REGS	120

static int pcm1864_show(struct seq_file *m, void *v)
{
	u32 gain_reg;
	int regs[PCM1864_PAGE_0_REGS];
	int i;
	struct adc_priv_data *pcm1864 = (struct adc_priv_data *)m->private;

	for(i = 0; i < PCM1864_PAGE_0_REGS; i++) {
		regs[i] = PCM_READ(i);
	}
	seq_printf(m, "TI PCM1864 #%u\n\n", pcm1864->dev_id);
	seq_printf(m, "Power state: %s\n", pcm1864_pwr_state_str(pcm1864));
	seq_printf(m, "Frame sync state: %s\n", pcm1864_fs_state_str(pcm1864));
	seq_printf(m, "BCLK state: %s\n", pcm1864_bclk_state_str(pcm1864));
	seq_printf(m, "SCLK state: %s\n", pcm1864_sclk_state_str(pcm1864));
	seq_printf(m, "Microphone type: %s\n", PCM_READ(PCM1864_DIGMIC) ? "PDM" : "Analog");
	gain_reg = PCM_READ(PCM1864_PGA_CH1L);
	seq_printf(m, "Microphone gain: %d%sdB\n",
		((gain_reg >> 1) | ((gain_reg & 0x80) ? 0xffffff80 : 0)),
		((gain_reg & 1) ? ".5" : ""));
	seq_printf(m, "\nPage 0 registers:\n");
	for(i = 0; i < PCM1864_PAGE_0_REGS; i++) {
		if (!(i % 4))
			seq_printf(m, "%#04x:  ", i);
		seq_printf(m, "%02x\t", regs[i]);
		if ((i % 4) == 3)
			seq_printf(m, "\n");
	}

	return 0;
}

static ssize_t pcm1864_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct adc_priv_data *pcm1864 = PDE_DATA(file_inode(filp));
	if (!is_mdp_authorized(MDP_AUTH_FLAG_MIC_DBG_ENABLE)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_WARNING, "Writing pcm1864 debug interface is not authorized");
		return -EACCES;
	}

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "init", 4) == 0) {
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_INFO, "Register init from procfile returned %d.", pcm1864_init_regs(pcm1864));
	} else if (strncmp(buf, "mute", 4) == 0) {
		pcm1864_set_mute(&(pcm1864->dev), 1);
	} else if (strncmp(buf, "unmute", 6) == 0) {
		pcm1864_set_mute(&(pcm1864->dev), 0);
	} else if (strncmp(buf, "type", 4) == 0) {
		char *type = buf + 5;
		if (strncmp(type, "pdm", 3) == 0) {
			pcm1864_init_pdm(pcm1864);
		} else {
			pcm1864_init_regs(pcm1864);
		}
	} else if (strncmp(buf, "reg", 3) == 0) {
		unsigned long reg;
		unsigned long val;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 0, &reg)) {
			bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		PCM_WRITE(reg, val);
		bb_log_dev(&(pcm1864->i2c_client->dev), BB_MOD_MICCTL, BB_LVL_INFO, "Wrote register %lu, read back %u.", reg, PCM_READ(reg));
	}

	return count;
}

const static struct seq_operations pcm1864_op = {
	.show		= pcm1864_show
};

static int pcm1864_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, pcm1864_show, PDE_DATA(inode));
}

static struct file_operations pcm1864_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= pcm1864_proc_open,
	.write		= pcm1864_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define PCM1864_PROCFS_FILE "driver/pcm1864"

static int pcm1864_proc_init(struct adc_priv_data *pcm1864)
{
	struct proc_dir_entry *entry;
	char file_name[50];

	snprintf(file_name, sizeof(file_name), PCM1864_PROCFS_FILE"-%u", pcm1864->dev_id);

	entry = proc_create_data(file_name, 0666, NULL, &pcm1864_proc_ops, pcm1864);
	if (!entry)
		return -EIO;

	return 0;
}

static void pcm1864_proc_remove(struct adc_priv_data *pcm1864)
{
	char file_name[50];

	snprintf(file_name, sizeof(file_name), PCM1864_PROCFS_FILE"-%u", pcm1864->dev_id);
	remove_proc_entry(file_name, NULL);
}

static const struct i2c_device_id pcm1864_id[] = {
	{ "pcm0", 0},
	{ "pcm1", 0},
	{ }
};
MODULE_DEVICE_TABLE(pcm1864_i2c, pcm1864_id);

static struct of_device_id pcm1864_ids[] = {
	{ .compatible = "ti,pcm1864"},
	{ }
};

static struct i2c_driver pcm1864_i2c_driver = {
	.driver = {
		.name		= "pcm1864",
		.owner		= THIS_MODULE,
		.of_match_table	= pcm1864_ids,
	},
	.id_table	= pcm1864_id,
	.probe		= pcm1864_probe,
	.remove		= pcm1864_remove
};

module_i2c_driver(pcm1864_i2c_driver);
MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Sonos pcm1864 micctl");
MODULE_LICENSE("GPL");