/*
 * Copyright (c) 2016-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for generic Sonos microphone control.
 */
#include <linux/version.h>
#include <linux/cdev.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>

#include "blackbox.h"
#include "sonos_device.h"
#include "micctl.h"
#include "mdp.h"
#include "sonos_lock.h"

#include "micctl_core.h"

static int devno = -1;

static LIST_HEAD(device_list);
struct micctl_device_node {
	struct device_node *of_node;
	struct list_head list;
};

struct micctl_data {
	uint32_t ref_count : 31;
	uint32_t mute_state : 1;
	struct cdev chr_dev;
} global_data;

int micctl_is_muted(void)
{
	int is_muted = 1;
	int mics = 0;

	struct micctl_device *dev;
	struct micctl_device_node *node;

	if (list_empty(&device_list)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "No mic devices registered");
		return -ENODEV;
	}

	list_for_each_entry(node, &device_list, list) {
		if (micctl_get_device(node->of_node, &dev)) {
			mics++;
			if (dev->is_muted) {
				is_muted &= dev->is_muted(dev);
			}
		}
	}

	if (!mics) {
		bb_log(BB_MOD_MICCTL, BB_LVL_WARNING,
		       "Tried check mute on the mic hw with incorrect devices registered.");
		return -ENODEV;
	}

	return is_muted;
}

int micctl_set_mute(int on)
{
	int ret = 0;
	int mics = 0;

	struct micctl_device *dev;
	struct micctl_device_node *node;

	if (list_empty(&device_list)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "No mic devices registered");
		return -ENODEV;
	}

	list_for_each_entry(node, &device_list, list) {
		if (micctl_get_device(node->of_node, &dev)) {
			if (dev->mute) {
				bb_log(BB_MOD_MICCTL, BB_LVL_INFO, "Mute %s",
				       (on ? "on" : "off"));
				ret |= dev->mute(dev, on);
				mics++;
			}
		}
	}

	if (!mics)
		bb_log(BB_MOD_MICCTL, BB_LVL_WARNING,
		       "Tried to set mute before any callbacks were registered.");

	if (ret) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR, "Unable to set mute (%d).",
		       ret);
	} else {
		global_data.mute_state = on;
	}

	return ret;
}

static int micctl_show(struct seq_file *m, void *v)
{
	int mutes = 0;
	struct micctl_device *dev;

	if (!list_empty(&device_list)) {
		list_for_each_entry(dev, &device_list, list) {
			mutes++;
		}
	}

	seq_printf(m, "Sonos Microphone Control\n\n");
	seq_printf(m, "Reference count: %d\n", global_data.ref_count);
	seq_printf(m, "Mute state: %s\n",
		   (global_data.mute_state ? "on" : "off"));
	seq_printf(m, "Mute callbacks registered: %d\n", mutes);

	return 0;
}

static ssize_t micctl_proc_write(struct file *filp, const char __user *buffer,
				 size_t count, loff_t *data)
{
	char buf[200];

	if (!is_mdp_authorized(MDP_AUTH_FLAG_MIC_DBG_ENABLE)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_WARNING,
		       "micctl debug commands are not authorized.");
		return -EACCES;
	}

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "mute", 4) == 0) {
		micctl_set_mute(1);
	} else if (strncmp(buf, "unmute", 6) == 0) {
		micctl_set_mute(0);
	}

	return count;
}

const static struct seq_operations micctl_op = { .show = micctl_show };

static int micctl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, micctl_show, PDE_DATA(inode));
}

static struct file_operations micctl_proc_ops = { .owner = THIS_MODULE,
						  .open = micctl_proc_open,
						  .write = micctl_proc_write,
						  .read = seq_read,
						  .llseek = seq_lseek,
						  .release = single_release };

#define MICCTL_PROCFS_FILE "driver/micctl"

static int micctl_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create(MICCTL_PROCFS_FILE, 0666, NULL, &micctl_proc_ops);
	if (!entry) {
		return -EIO;
	}

	return 0;
}

static void micctl_proc_remove(void)
{
	remove_proc_entry(MICCTL_PROCFS_FILE, NULL);
}

static long micctl_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	uint8_t on = 0;
	long ret = 0;
	if (_IOC_DIR(cmd) &
	    _IOC_WRITE) {
		if (copy_from_user(&on, (uint8_t *)arg, sizeof(uint8_t))) {
			return -EACCES;
		}
	}
	switch (_IOC_NR(cmd)) {
	case 0: {
		uint32_t ver = MICCTL_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case 1: {
		ret = micctl_set_mute((on ? 1 : 0));
		if (ret) {
			bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
			       "MICCTL_SET_MUTE IOCTL failed with %ld.", ret);
		}
		break;
	}
	default:
		bb_log(BB_MOD_MICCTL, BB_LVL_WARNING, "Unrecognized IOCTL %d",
		       _IOC_NR(cmd));
		return -EINVAL;
	}
	return ret;
}

static int micctl_open(struct inode *inodep, struct file *filp)
{
	global_data.ref_count++;
	return 0;
}

static int micctl_release(struct inode *inodep, struct file *filp)
{
	global_data.ref_count--;
	return 0;
}

const struct file_operations micctl_fops = {
	.open = micctl_open,
	.unlocked_ioctl = micctl_ioctl,
	.release = micctl_release,
};

int micctl_link_devices(struct device *dev, struct device_node *np)
{
	struct device_node *micctl_node;
	int micctl_dev_idx = 0;
	int mics = 0;
	int ret = 0;

	while ((micctl_node = of_parse_phandle(np, "micctl-device",
					       micctl_dev_idx++))) {
		if (!of_device_is_available(micctl_node)) {
			dev_err(dev, "micctl-device not available");
			of_node_put(micctl_node);
			continue;
		} else {
			struct micctl_device_node *node =
				devm_kzalloc(dev,
					     sizeof(struct micctl_device_node),
					     GFP_KERNEL);

			node->of_node = micctl_node;
			list_add(&node->list, &device_list);
			mics++;
		}
	}

	if (!micctl_dev_idx) {
		dev_err(dev, "micctl-device not specified");
		ret = -ENODEV;
	}

	if (!mics) {
		dev_err(dev, "micctl-devices not found");
		ret = -EPROBE_DEFER;
	}

	return ret;
}

void micctl_register_device_cb(void)
{
	int is_muted = 0;

	if (list_empty(&device_list)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "No mic devices registered from callback!");
		return;
	}

	micctl_set_mute(1);

	is_muted = micctl_is_muted();
	if (is_muted != -ENODEV)
		global_data.mute_state = is_muted;
}

int micctl_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct device *class_dev = NULL;

	ret = micctl_link_devices(&pdev->dev, pdev->dev.of_node);
	if (ret)
		return ret;

	devno = MKDEV(MICCTL_MAJOR_NUMBER, 0);
#ifdef CONFIG_DEVTMPFS
	ret = alloc_chrdev_region(&devno, 0, 1, MICCTL_DEVICE_NAME);
#else
	ret = register_chrdev_region(devno, 1, MICCTL_DEVICE_NAME);
#endif
	if (ret) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "Couldn't register character device region (%d).", ret);
		return ret;
	}
	cdev_init(&global_data.chr_dev, &micctl_fops);
	global_data.chr_dev.owner = THIS_MODULE;

	ret = cdev_add(&global_data.chr_dev, devno, 1);
	if (ret) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "Couldn't add character device (%d).", ret);
		unregister_chrdev_region(devno, 1);
		return ret;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, MICCTL_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "Error creating micctl class.\n");
		cdev_del(&global_data.chr_dev);
		unregister_chrdev_region(devno, 1);
		ret = PTR_ERR(class_dev);
		return ret;
	}

	ret = micctl_proc_init();
	if (ret) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "Failed to init mic procfs (%d).", ret);
	}

	return ret;
}

void micctl_device_remove(void)
{
	struct micctl_device_node *node;

	if (list_empty(&device_list)) {
		return;
	}

	list_for_each_entry(node, &device_list, list) {
		of_node_put(node->of_node);
	}
}

int micctl_remove(struct platform_device *pdev)
{
	micctl_device_remove();
	micctl_proc_remove();
	sonos_device_destroy(devno);
	cdev_del(&global_data.chr_dev);
	unregister_chrdev_region(devno, 1);

	return 0;
}

static const struct of_device_id of_match[] = {
	{ .compatible = "sonos,micctl", },
	{},
};
MODULE_DEVICE_TABLE(of, of_match);

static struct platform_driver micctl_platform_driver = {
	.probe = micctl_probe,
	.remove = micctl_remove,
	.driver = {
		.name = "micctl",
		.of_match_table = of_match,
	},
};

module_platform_driver(micctl_platform_driver);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Microphone control driver");
MODULE_LICENSE("GPL");
