/*
 * Copyright (c) 2016-2017, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the enabling/disabling MICs through the iCE40 FPGA device over i2c
 */

#include <linux/device.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/uaccess.h>
#include "blackbox.h"
#include "micctl.h"
#include "ice40_fpga.h"
#include "micctl_core.h"

#define ICE40_REG_MIC_CONTROL 0xC0
#define ICE40_MSK_MIC_MUTE BIT(7)
#define ICE40_MSK_MIC_PDM_EN BIT(6)

int ice40_mics_mute(struct micctl_device *dev, int on)
{
	return ice40_fpga_write_reg(ICE40_REG_MIC_CONTROL,
				    (on ? 0 : ICE40_MSK_MIC_MUTE));
}

int ice40_mics_probe(struct platform_device *pdev)
{
	int ret;

	struct micctl_device *mic_dev = devm_kzalloc(
		&pdev->dev, sizeof(struct micctl_device), GFP_KERNEL);


	ice40_fpga_release_reset();
	ret = ice40_fpga_write_reg(ICE40_REG_MIC_CONTROL, 0);
	if (ret < 0)
		ret = -EFAULT;
	else
		ret = 0;

	platform_set_drvdata(pdev, mic_dev);

	mic_dev->of_node = pdev->dev.of_node;
	mic_dev->mute = ice40_mics_mute;
	mic_dev->is_muted = NULL;
	micctl_register_device(mic_dev);

	return ret;
}

int ice40_mics_remove(struct platform_device *pdev)
{
	struct micctl_device *mic_dev = platform_get_drvdata(pdev);

	if (ice40_mics_mute(mic_dev, 1)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "Failed to trun OFF mics while removing the driver!");
	} else {
		bb_log(BB_MOD_MICCTL, BB_LVL_INFO,
		       "iCE40 micctl driver removed");
	}

	micctl_unregister_device(mic_dev);

	return 0;
}

static const struct of_device_id of_match[] = {
	{ .compatible = "sonos,ice40-mics", },
	{},
};
MODULE_DEVICE_TABLE(of, of_match);

static struct platform_driver ice40_mics_platform_driver = {
	.probe = ice40_mics_probe,
	.remove = ice40_mics_remove,
	.driver = {
		.name = "ice40-mics",
		.of_match_table = of_match,
	},
};

module_platform_driver(ice40_mics_platform_driver);
MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Sonos ice40 micctl");
MODULE_LICENSE("GPL");
