/*
 * Copyright (c) 2016-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#include <linux/gpio.h>
#include "blackbox.h"
#include "micctl.h"
#include <linux/of_gpio.h>

#include "micctl_core.h"

struct gpio gpio_mic_pin;
struct gpio gpio_mic_clk;

int gpio_mics_mute(struct micctl_device *dev, int on)
{
	if (gpio_is_valid(gpio_mic_clk.gpio)) {
		gpio_set_value(gpio_mic_clk.gpio, !((on ^ 1) & 0x01));
	}
	gpio_set_value(gpio_mic_pin.gpio, (on ^ 1) & 0x01);
	return 0;
}

int gpio_mics_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;

	struct micctl_device *mic_dev = devm_kzalloc(
		&pdev->dev, sizeof(struct micctl_device), GFP_KERNEL);

	mic_dev->of_node = np;
	mic_dev->mute = gpio_mics_mute;
	mic_dev->is_muted = NULL;
	micctl_register_device(mic_dev);

	platform_set_drvdata(pdev, mic_dev);

	if (of_property_read_bool(np, "no-mics-product")) {
		return -ENODEV;
	}


	gpio_mic_pin.gpio = of_get_named_gpio(np, "micctl-gpio", 0);
	gpio_mic_pin.label = "mic_ADC_reset";

	if (gpio_request_one(gpio_mic_pin.gpio, gpio_mic_pin.flags,
			     gpio_mic_pin.label)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR,
		       "Failed to request mic ADC reset.");
		return -1;
	}

	gpio_mic_clk.gpio = of_get_named_gpio(np, "micctl-clk-en", 0);
	if (gpio_is_valid(gpio_mic_clk.gpio)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_DEBUG, "micctl-clk-en in the dtb - (%d)", gpio_mic_clk.gpio);
		gpio_mic_clk.label = "mic_clock_enable";
		if (gpio_request_one(gpio_mic_clk.gpio, gpio_mic_clk.flags, gpio_mic_clk.label)) {
			bb_log(BB_MOD_MICCTL, BB_LVL_ERR, "Failed to request mic clock enable.");
			return -1;
		}
	} else {
		bb_log(BB_MOD_MICCTL, BB_LVL_DEBUG, "no micctl-clk-en in the dtb");
	}


	bb_log(BB_MOD_MICCTL, BB_LVL_INFO,
	       "gpio_mics micctl driver initialized and callbacks registered");
	return 0;
}

int gpio_mics_remove(struct platform_device *pdev)
{
	struct micctl_device *mic_dev = platform_get_drvdata(pdev);

	micctl_unregister_device(mic_dev);
	gpio_free(gpio_mic_clk.gpio);
	gpio_free(gpio_mic_pin.gpio);

	return 0;
}

static const struct of_device_id of_match[] = {
	{ .compatible = "sonos,gpio-mics", },
	{},
};
MODULE_DEVICE_TABLE(of, of_match);

static struct platform_driver gpio_mics_platform_driver = {
	.probe = gpio_mics_probe,
	.remove = gpio_mics_remove,
	.driver = {
		.name = "gpio-mics",
		.of_match_table = of_match,
	},
};

module_platform_driver(gpio_mics_platform_driver);
MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Sonos gpio micctl");
MODULE_LICENSE("GPL");
