/*
 * Copyright (c) 2015-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/module.h>
#include <linux/suspend.h>
#include <linux/kthread.h>
#include <linux/seq_file.h>
#include <linux/circ_buf.h>
#include <linux/proc_fs.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <asm/uaccess.h>
#include <linux/slab.h>
#include <linux/cdev.h>
#include <linux/err.h>
#include <asm/io.h>
#ifndef SONOS_ARCH_ATTR_IMX6_KERNEL_UPGRADE
#include <linux/mcc_config_linux.h>
#include <linux/mcc_common.h>
#include <linux/mcc_imx6sx.h>
#include <linux/mcc_linux.h>
#include <linux/imx_sema4.h>
#include <linux/mcc_api.h>

#include "blackbox.h"
#include "ir_rcvr.h"
#include "m4.h"

#define M4_MAJOR_NUMBER         IR_MAJOR_NUMBER
#define M4_DEVICE_NAME          IR_DEVICE_NAME
#define M4_DEVICE_NODE          IR_DEVICE_NODE
#define M4_PROC_FILE            "driver/"IR_DEVICE_NAME
#define M4_PROC_PERM            (S_IRUSR | S_IRGRP | S_IROTH | S_IWUSR)

#define M4_MSG_PRINT(LEVEL, FORMAT, ARG...) bb_log(BB_MOD_M4, LEVEL, FORMAT, ## ARG)
#define M4_MSG_INFO(FORMAT, ARG...) M4_MSG_PRINT(BB_LVL_INFO, FORMAT, ## ARG)
#define M4_MSG_WARN(FORMAT, ARG...) M4_MSG_PRINT(BB_LVL_WARNING, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define M4_MSG_ERROR(FORMAT, ARG...) M4_MSG_PRINT(BB_LVL_ERR, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define M4_MSG_DEBUG(FORMAT, ARG...) bb_log_dbg(BB_MOD_M4, FORMAT, ## ARG)

MCC_ENDPOINT mcc_endpoint_a9 = {0, MCC_NODE_A9, MCC_A9_PORT};
MCC_ENDPOINT mcc_endpoint_m4 = {1, MCC_NODE_M4, MCC_M4_PORT};


struct m4_private_data {
	int rsp_count;
	bool ir_ready;
	bool record_stat;
	uint32_t user_count;
	uint32_t timeout;
	uint32_t runt_duration;
	uint32_t ir_len;
	uint32_t ir_data[MCC_MSG_DATA_SIZE];
};

static int devno = -1;

static int __init m4_driver_init(void);
static void __exit m4_driver_exit(void);
static int m4_receive_thread(void *data);
static void m4_watchdog_timer_callback(unsigned long data);
static int m4_proc_init(void);
static int m4_proc_show(struct seq_file *m, void *v);
static int m4_proc_open(struct inode *inode, struct  file *file);
static ssize_t m4_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
static int m4_open(struct inode *inodep, struct file *filp);
static int m4_release(struct inode *inodep, struct file *filp);
static int m4_probe(struct platform_device *pdev);
static int m4_remove(struct platform_device *pdev);
static ssize_t m4_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset);
static ssize_t m4_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset);
static loff_t m4_llseek(struct file *filp, loff_t offset, int seek_type);
static unsigned int m4_poll(struct file *filp, struct poll_table_struct *table);

DEFINE_RWLOCK(m4_rwlock);
DECLARE_WAIT_QUEUE_HEAD(read_queue);
struct timer_list wd_timer;
struct task_struct *m4_task;
struct cdev m4_chr_dev;
static struct m4_private_data m4_priv = {
	.user_count = 0,
	.record_stat = 0,
	.ir_len = 0,
	.rsp_count = 0,
	.timeout = 11000,
	.ir_ready = 0,
	.runt_duration = 120
};


static void m4_watchdog_timer_callback(unsigned long data)
{
	M4_MSG_ERROR("He's dead, Jim!");
}


static int m4_receive_thread(void *data)
{
	int i;
	int ret = 0;
	unsigned retries = 0;
	struct mcc_message msg;
	MCC_MEM_SIZE num_of_received_bytes;

	M4_MSG_INFO("Receive thread started");

	while (!kthread_should_stop()) {
		ret = mcc_recv(&mcc_endpoint_m4, &mcc_endpoint_a9, &msg, sizeof(struct mcc_message), &num_of_received_bytes, MCC_INFINITE_TIMEOUT);
		if (MCC_SUCCESS != ret) {
			if (retries < 5 || !(retries % 100)) {
				M4_MSG_ERROR("A9 failed to receive message (%d), attempt %u", ret, retries);
			}
			retries++;
			msleep(500);
		}
		else {
			retries = 0;
			switch(msg.msg_type) {
				case WD_HEART_BEAT:
					M4_MSG_DEBUG("Watchdog Heartbeat");
					write_lock(&m4_rwlock);
					m4_priv.rsp_count += 1;
					if(INT_MAX == m4_priv.rsp_count) {
						m4_priv.rsp_count = 0;
					}
					write_unlock(&m4_rwlock);
					break;

				case IR_SYMBOL_DATA:
					M4_MSG_DEBUG("IR Received!");
					write_lock(&m4_rwlock);
					m4_priv.ir_len = msg.msg_len;
					for(i=0; i< msg.msg_len; i++) {
						m4_priv.ir_data[i] = msg.msg_data[i];
					}
					m4_priv.ir_ready = 1;
					if(m4_priv.user_count > 0) {
						wake_up(&read_queue);
					}
					write_unlock(&m4_rwlock);
					break;

				default:
					M4_MSG_ERROR("Invalid Message (%d) received from M4!", msg.msg_type);
					break;
			}
		}
		mod_timer(&wd_timer, jiffies + msecs_to_jiffies(15000));
	}
	return ret;
}



static int m4_proc_show(struct seq_file *m, void *v)
{
	int i = 0;
	int rsp_count;
	bool ir_ready;
	bool record_stat;
	uint32_t temp_len;
	uint32_t timeout;
	uint32_t runt_duration;
	uint32_t temp_buff[MCC_MSG_DATA_SIZE];
	struct m4_private_data *priv_data = (struct m4_private_data *)m->private;

	read_lock(&m4_rwlock);
	ir_ready = priv_data->ir_ready;
	record_stat = priv_data->record_stat;
	temp_len = priv_data->ir_len;
	timeout = priv_data->timeout;
	rsp_count = priv_data->rsp_count;
	runt_duration = priv_data->runt_duration;
	for(i=0; i<temp_len; i++) {
		temp_buff[i] = priv_data->ir_data[i];
	}
	read_unlock(&m4_rwlock);

	if(record_stat == IR_RECORD_ON) {
		if(ir_ready) {
			seq_printf(m, "replay=");
			for(i=0; i<temp_len; i++) {
				seq_printf(m, "%08X,", temp_buff[i]);
			}
		} else {
			seq_printf(m, "NO IR Data!\n");
		}
	} else {
		seq_printf(m, "M4 Watchdog Counter: %u\n", rsp_count);
		seq_printf(m, "IR Timeout: %uus\n", timeout);
		seq_printf(m, "IR Runt filter duration: %uus\n", runt_duration);
		if(ir_ready) {
			seq_printf(m, "IR Data: ");
			for(i=0; i<temp_len; i++) {
				seq_printf(m, "%08X,", temp_buff[i]);
			}
			seq_printf(m, "\n");
		} else {
			seq_printf(m, "NO IR Data!\n");
		}
	}

	write_lock(&m4_rwlock);
	priv_data->ir_ready = 0;
	write_unlock(&m4_rwlock);
	return 0;
}


static int m4_proc_open(struct inode *inode, struct  file *file)
{
	return single_open(file, m4_proc_show, PDE_DATA(inode));
}


static ssize_t m4_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	struct m4_private_data *priv_data = PDE_DATA(file_inode(file));
	char *temp = NULL;
	char *buf = NULL;
	char *tok = NULL;
	int i = 0;
	int ret;
	unsigned long value;
	struct mcc_message msg;

	buf = (char *)kmalloc(sizeof(char)*2048, GFP_KERNEL);
	if (buf == NULL) {
		M4_MSG_ERROR("Failed to allocate memory for buffer");
		ret = -ENOMEM;
		goto err_out;
	}

	if (count > 2047) {
		M4_MSG_ERROR("The string is too long");
		ret = -EIO;
		goto err_out;
	} else if (copy_from_user(buf, buffer, count)) {
		ret = -EFAULT;
		goto err_out;
	}

	buf[count] = '\0';
	temp = strchr(buf, '=');
	if(temp == NULL) {
		M4_MSG_ERROR("Invalid format for proc command");
		M4_MSG_ERROR("Use <command>=<value(s)>");
		ret = -EINVAL;
		goto err_out;
	}
	temp++;

	if (strncmp(buf, "timeout", 7) == 0) {
		if (temp && kstrtoul(temp, 0, &value)) {
			M4_MSG_ERROR("Failed to parse the value: %s", temp);
			ret = -EINVAL;
			goto err_out;
		} else {
			write_lock(&m4_rwlock);
			priv_data->timeout = (uint32_t)value;
			write_unlock(&m4_rwlock);
			msg.msg_type = IR_TIMEOUT;
			msg.msg_len = 1;
			msg.msg_data[0] = (uint32_t)value;
			ret = mcc_send(&mcc_endpoint_a9, &mcc_endpoint_m4, &msg, sizeof(struct mcc_message), MCC_INFINITE_TIMEOUT);
			if (ret) {
				M4_MSG_ERROR("Sending message to M4 failed");
				ret = -EFAULT;
				goto err_out;
			}
			M4_MSG_INFO("Setting timeout to %u us", priv_data->timeout);
		}
	} else if (strncmp(buf, "runt", 4) == 0) {
		if (temp && kstrtoul(temp, 0, &value)) {
			M4_MSG_ERROR("Failed to parse the value: %s", temp);
			ret = -EINVAL;
			goto err_out;
		} else {
			write_lock(&m4_rwlock);
			priv_data->runt_duration = (uint32_t)value;
			write_unlock(&m4_rwlock);
			msg.msg_type = IR_RUNT_DURATION;
			msg.msg_len = 1;
			msg.msg_data[0] = (uint32_t)value;
			ret = mcc_send(&mcc_endpoint_a9, &mcc_endpoint_m4, &msg, sizeof(struct mcc_message), MCC_INFINITE_TIMEOUT);
			if (ret) {
				M4_MSG_ERROR("Sending message to M4 failed");
				ret  = -EFAULT;
				goto err_out;
			}
			M4_MSG_INFO("Setting runt filter duration to %u us", priv_data->runt_duration);
		}
	} else if (strncmp(buf, "record", 6) == 0) {
		if (strncmp(temp, "on", 2) == 0) {
			write_lock(&m4_rwlock);
			priv_data->record_stat = IR_RECORD_ON;
			write_unlock(&m4_rwlock);
			msg.msg_data[0] = IR_RECORD_ON;
			M4_MSG_INFO("M4 record mode ON. The IR data received will be raw edge timestamps");
		} else if (strncmp(temp, "off", 3) == 0) {
			write_lock(&m4_rwlock);
			priv_data->record_stat = IR_RECORD_OFF;
			write_unlock(&m4_rwlock);
			msg.msg_data[0] = IR_RECORD_OFF;
			M4_MSG_INFO("M4 record mode OFF. The IR data received will be normal");
		} else {
			M4_MSG_ERROR("Failed to parse the command: %s", temp);
			ret = -EINVAL;
			goto err_out;
		}
		msg.msg_type = IR_RECORD_EDGES;
		msg.msg_len = 1;
		ret = mcc_send(&mcc_endpoint_a9, &mcc_endpoint_m4, &msg, sizeof(struct mcc_message), MCC_INFINITE_TIMEOUT);
		if (ret) {
			M4_MSG_ERROR("Sending message to M4 failed");
			ret = -EFAULT;
			goto err_out;
		}
	} else if (strncmp(buf, "replay", 6) == 0) {
		msg.msg_len = 0;
		for (i=0; i < MCC_MSG_DATA_SIZE; i++) {
			tok = strsep(&temp, ",");
			if (temp == NULL){
				break;
			}
			if (kstrtoul(tok, 16, &value)) {
				M4_MSG_ERROR("Failed to parse the value: %s", tok);
				ret = -EINVAL;
				goto err_out;
			} else {
				msg.msg_data[i] = (uint32_t)value;
				msg.msg_len++;
			}
		}

		M4_MSG_INFO("Message length = %d",msg.msg_len);
		msg.msg_type = IR_REPLAY_EDGES;
		ret = mcc_send(&mcc_endpoint_a9, &mcc_endpoint_m4, &msg, sizeof(struct mcc_message), MCC_INFINITE_TIMEOUT);
		if (ret) {
			M4_MSG_ERROR("Sending message to M4 failed");
			ret = -EFAULT;
			goto err_out;
		}
		M4_MSG_INFO("M4 in replay mode. M4 will replay the edges provided");
	}else {
		M4_MSG_ERROR("Unknown command: %s", buf);
		ret = -EINVAL;
		goto err_out;
	}

	ret = count;
err_out:
	kfree(buf);
	return ret;
}


static const struct file_operations m4_proc_fops = {
	.owner = THIS_MODULE,
	.open = m4_proc_open,
	.write = m4_proc_write,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int m4_proc_init(void)
{
	struct proc_dir_entry *m4_proc_entry;
	m4_proc_entry = proc_create_data(M4_PROC_FILE, M4_PROC_PERM, NULL, &m4_proc_fops, &m4_priv);
	if (!m4_proc_entry) {
		return -EIO;
	}
	return 0;
}


static int m4_open(struct inode *inodep, struct file *filp)
{
	write_lock(&m4_rwlock);
	m4_priv.user_count++;
	write_unlock(&m4_rwlock);
	return 0;
}


static ssize_t m4_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset)
{
	int i;
	int ret;
	char kbuf[MCC_MSG_DATA_SIZE];
	uint32_t temp_len;

	read_lock(&m4_rwlock);
	temp_len = m4_priv.ir_len;
	read_unlock(&m4_rwlock);

	if (size < MCC_MSG_DATA_SIZE) {
		M4_MSG_ERROR("Invalid size! Read must be at least %d bytes (1 max-length symbol).", MCC_MSG_DATA_SIZE);
		return -EINVAL;
	}

	read_lock(&m4_rwlock);
	for (i=0; i < temp_len; i++) {
		kbuf[i] = (char)m4_priv.ir_data[i];
	}
	read_unlock(&m4_rwlock);

	ret = copy_to_user(buffer, kbuf, temp_len);
	if (ret) {
		M4_MSG_ERROR("Failed to copy buffer from kernel to userspace");
		return ret;
	}

	M4_MSG_DEBUG("IR symbol read by userspace process");

	write_lock(&m4_rwlock);
	m4_priv.ir_ready = 0;
	write_unlock(&m4_rwlock);
	return temp_len;
}

static long m4_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = IR_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	default:
		M4_MSG_ERROR("Unrecognized IOCTL command %u.", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}


static unsigned int m4_poll(struct file *filp, struct poll_table_struct *table)
{
	unsigned int mask = 0;
	read_lock(&m4_rwlock);
	if (m4_priv.ir_ready) {
		M4_MSG_DEBUG("IR Data ready to be read!");
		mask |= (POLLIN | POLLRDNORM);
	}
	read_unlock(&m4_rwlock);
	poll_wait(filp, &read_queue, table);
	return mask;
}
static ssize_t m4_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset)
{
	M4_MSG_ERROR("Writing to the device file not implemented yet!");
	return -EINVAL;
}


static int m4_release(struct inode *inodep, struct file *filp)
{
	write_lock(&m4_rwlock);
	m4_priv.user_count--;
	write_unlock(&m4_rwlock);
	return 0;
}


static loff_t m4_llseek(struct file *filp, loff_t offset, int seek_type)
{
	M4_MSG_ERROR("Seek not supported in this driver!");
	return -EINVAL;
}

static const struct file_operations m4_fops =
{
	.open		= m4_open,
	.read		= m4_read,
	.poll		= m4_poll,
	.write		= m4_write,
	.llseek		= m4_llseek,
	.release	= m4_release,
	.unlocked_ioctl	= m4_ioctl,
};



static int m4_probe(struct platform_device *pdev)
{
	int i = 0;
	int ret = 0;
	MCC_INFO_STRUCT mcc_info;
	struct mcc_message msg;

	ret = mcc_initialize(MCC_NODE_A9);
	if (ret) {
		M4_MSG_ERROR("Failed to initialize MCC");
	} else {
		if (mcc_generate_cpu_to_cpu_interrupt()) {
			M4_MSG_ERROR("Failed to send interrupt to start M4.");
		}
	}

	ret = mcc_get_info(MCC_NODE_A9, &mcc_info);
	if (ret) {
		M4_MSG_ERROR("Failed to get MCC info");
		return -EINVAL;
	}
	M4_MSG_INFO("A9 MCC is initialized, MCC version is %s", mcc_info.version_string);

	if (strcmp(mcc_info.version_string, MCC_VERSION_STRING) != 0) {
		M4_MSG_ERROR("MCC version mis-match between cores");
		return -EINVAL;
	}

	ret = mcc_create_endpoint(&mcc_endpoint_a9, MCC_A9_PORT);
	if (ret) {
		M4_MSG_ERROR("Failed to create A9 mcc End Point");
		return -EFAULT;
	}

	msg.msg_type = A9_DRVR_READY;
	msg.msg_len = 0;
	ret = mcc_send(&mcc_endpoint_a9, &mcc_endpoint_m4, &msg, sizeof(struct mcc_message), MCC_INFINITE_TIMEOUT);
	if (ret) {
		for (i=0; i<5; i++) {
			msleep(2000);
			M4_MSG_INFO("Sending message to M4 failed, Retry(%d)", i+1);
			ret = mcc_send(&mcc_endpoint_a9, &mcc_endpoint_m4, &msg, sizeof(struct mcc_message), MCC_INFINITE_TIMEOUT);
			if (!ret) {
				break;
			}
		}
		if (ret) {
			M4_MSG_ERROR("Sending message to M4 failed after retrying 5 times");
			mcc_destroy_endpoint(&mcc_endpoint_a9);
			return -EFAULT;
		}
	}

	ret = m4_proc_init();
	if (ret) {
		M4_MSG_ERROR("Setting up proc file system failed");
		mcc_destroy_endpoint(&mcc_endpoint_a9);
		return ret;
	}

	setup_timer(&wd_timer, m4_watchdog_timer_callback, 0);
	mod_timer(&wd_timer, jiffies + msecs_to_jiffies(15000));
	m4_task = kthread_run(&m4_receive_thread, NULL, "M4-RX");
	if (m4_task < 0) {
		M4_MSG_ERROR("Starting kthread failed");
		del_timer(&wd_timer);
		mcc_destroy_endpoint(&mcc_endpoint_a9);
		return -EFAULT;
	}

	return ret;
}


static int m4_remove(struct platform_device *pdev)
{
	int ret;
	struct mcc_message msg;

	kthread_stop(m4_task);

	del_timer(&wd_timer);

	msg.msg_type = A9_DRVR_AWAY;
	msg.msg_len = 0;
	ret = mcc_send(&mcc_endpoint_a9, &mcc_endpoint_m4, &msg, sizeof(struct mcc_message), MCC_INFINITE_TIMEOUT);
	if (ret) {
		M4_MSG_ERROR("Sending message to M4 failed");
	}

	remove_proc_entry(M4_PROC_FILE, NULL);
	ret = mcc_destroy_endpoint(&mcc_endpoint_a9);
	if (ret) {
		M4_MSG_ERROR("Failed to destroy A9 MCC Endpoint");
		return ret;
	} else {
		M4_MSG_INFO("Destroyed A9 MCC EndPoint");
	}
	return 0;
}

static const struct of_device_id m4_dt_ids[] = {
	{ .compatible = "sonos,imx6sx-m4-ir", },
	{  }
};
MODULE_DEVICE_TABLE(of, m4_dt_ids);

static struct platform_driver m4_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "M4 Driver",
		.of_match_table = m4_dt_ids,
		},
	.probe = m4_probe,
	.remove = m4_remove,
};


static int __init m4_driver_init(void)
{
	int ret;

	devno = MKDEV(M4_MAJOR_NUMBER, 0);

#ifdef CONFIG_DEVTMPFS
	ret = alloc_chrdev_region(&devno, 0, 1, "m4");
#else
	ret = register_chrdev_region(devno, 1, "m4");
#endif
	if (ret) {
		M4_MSG_ERROR("Failed to register M4 character node (%d)", ret);
		return ret;
	}

	cdev_init(&m4_chr_dev, &m4_fops);
	m4_chr_dev.owner = THIS_MODULE;

	ret = cdev_add(&m4_chr_dev, devno, 1);
	if (ret) {
		M4_MSG_ERROR("Failed to add M4 chrdev %d:0 (%d)", MAJOR(devno), ret);
		goto cdev_err;
	}

	ret = platform_driver_register(&m4_driver);
	if (ret) {
		M4_MSG_ERROR("Failed to register M4 driver");
		goto plat_err;
	} else {
		M4_MSG_INFO("Registered M4 driver with major number: %d", MAJOR(devno));
	}

	return 0;

plat_err:
	cdev_del(&m4_chr_dev);
cdev_err:
	unregister_chrdev_region(devno, 1);

	return ret;
}


static void __exit m4_driver_exit(void)
{
	platform_driver_unregister(&m4_driver);
	cdev_del(&m4_chr_dev);
	unregister_chrdev_region(devno, 1);
	M4_MSG_INFO("Unloaded M4 driver");
}


module_init(m4_driver_init);
module_exit(m4_driver_exit);
#endif

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("M4 driver");
MODULE_LICENSE("GPL");
