/*
 * Copyright (c) 2016-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#include <asm/byteorder.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include "blackbox.h"
#include "ledctl.h"

#define LP5562_NUM_ENGINES	3
#define LP5562_BYTES_PER_ENGINE	32

struct lp5562_state {
	struct i2c_client *client;
	struct mutex lock;
	struct led_step last_step;
	u8 engines[LP5562_NUM_ENGINES][LP5562_BYTES_PER_ENGINE];
	u8 exec_mode;
	u8 op_mode;
	u8 led_map;
	u8 first_update;
};

static int lp5562_proc_init(struct lp5562_state *state);
static void lp5562_proc_remove(void);
struct lp5562_state *lp5562_global_state = NULL;

#define LP_PRINT(lvl, msg, args...)	bb_log_dev(&(state->client->dev), BB_MOD_LEDCTL, lvl, msg, ##args)
#define LP_DEBUG(msg, args...)		bb_log_dbg_dev(&(state->client->dev), BB_MOD_LEDCTL, msg, ##args)
#define LP_INFO(msg, args...)		LP_PRINT(BB_LVL_INFO, msg, ##args)
#define LP_WARNING(msg, args...)	LP_PRINT(BB_LVL_WARNING, msg, ##args)
#define LP_ERR(msg, args...)		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "LP5562: "msg, ##args)

#define ENG1_MODE(reg)	((reg >> 4) & 3)
#define ENG2_MODE(reg)	((reg >> 2) & 3)
#define ENG3_MODE(reg)	(reg & 3)

#define LP5562_REG_ENABLE		0x00
#define LP5562_EXEC_ENG1_M		0x30
#define LP5562_EXEC_ENG2_M		0x0C
#define LP5562_EXEC_ENG3_M		0x03
#define LP5562_EXEC_M			0x3F
#define LP5562_MASTER_ENABLE		0x40
#define LP5562_LOGARITHMIC_PWM		0x80
#define LP5562_EXEC_RUN			0x2A
#define LP5562_ENABLE_DEFAULT	\
	(LP5562_MASTER_ENABLE | LP5562_LOGARITHMIC_PWM)
#define LP5562_ENABLE_RUN_PROGRAM	0x3f

#define LP5562_REG_OP_MODE		0x01
#define LP5562_MODE_ENG1_M		0x30
#define LP5562_MODE_ENG2_M		0x0C
#define LP5562_MODE_ENG3_M		0x03
#define LP5562_LOAD_ENG1		0x10
#define LP5562_LOAD_ENG2		0x04
#define LP5562_LOAD_ENG3		0x01
#define LP5562_RUN_ENG1			0x20
#define LP5562_RUN_ENG2			0x08
#define LP5562_RUN_ENG3			0x02
#define LP5562_ENG1_IS_LOADING(mode)	\
	((mode & LP5562_MODE_ENG1_M) == LP5562_LOAD_ENG1)
#define LP5562_ENG2_IS_LOADING(mode)	\
	((mode & LP5562_MODE_ENG2_M) == LP5562_LOAD_ENG2)
#define LP5562_ENG3_IS_LOADING(mode)	\
	((mode & LP5562_MODE_ENG3_M) == LP5562_LOAD_ENG3)
#define LP5562_IS_LOADING(mode)		\
	(LP5562_ENG1_IS_LOADING(mode) || LP5562_ENG2_IS_LOADING(mode) || LP5562_ENG3_IS_LOADING(mode))

#define LP5562_REG_R_PWM		0x04
#define LP5562_REG_G_PWM		0x03
#define LP5562_REG_B_PWM		0x02
#define LP5562_REG_W_PWM		0x0E

#define LP5562_REG_R_CURRENT		0x07
#define LP5562_REG_G_CURRENT		0x06
#define LP5562_REG_B_CURRENT		0x05
#define LP5562_REG_W_CURRENT		0x0F

#define LP5562_REG_CONFIG		0x08
#define LP5562_PWM_HF			0x40
#define LP5562_PWRSAVE_EN		0x20
#define LP5562_CLK_INT			0x01
#define LP5562_CONFIG_DEFAULT		(LP5562_PWM_HF | LP5562_PWRSAVE_EN | LP5562_CLK_INT)

#define LP5562_REG_ENG1_PC		0x09
#define LP5562_REG_ENG2_PC		0x0A
#define LP5562_REG_ENG3_PC		0x0B

#define LP5562_REG_STATUS		0x0C
#define LP5562_STATUS_INTR(e)		(1 << (3 - (e)))

#define LP5562_REG_RESET		0x0D
#define LP5562_RESET			0xFF

#define LP5562_REG_PROG_MEM_ENG1	0x10
#define LP5562_REG_PROG_MEM_ENG2	0x30
#define LP5562_REG_PROG_MEM_ENG3	0x50

#define LP5562_REG_ENG_SEL		0x70
#define LP5562_ENG_SEL_PWM		0
#define LP5562_ENG_FOR_RGB_M		0x3F
#define LP5562_ENG_SEL_RGB		0x1B
#define LP5562_ENG_FOR_W_M		0xC0
#define LP5562_ENG_FOR_R_M		0x30
#define LP5562_ENG_FOR_G_M		0x0C
#define LP5562_ENG_FOR_B_M		0x03
#define LP5562_ENG1_FOR_W		0x40
#define LP5562_ENG2_FOR_W		0x80
#define LP5562_ENG3_FOR_W		0xC0

#define RAMP(scale, step, sign, incr)	htons( \
	((scale & 0x01) << 14) | \
	((step & 0x3f) << 8) | \
	((sign & 0x01) << 7) | \
	(incr & 0x7f))
#define WAIT(time)		RAMP(1, time, 0, 0)
#define PWM(val)		htons((1 << 14) | (val & 0xff))
#define BRANCH(loop, pc)	htons((5 << 13) | \
	((loop & 0x3f) << 7) | \
	(pc & 0x3f))
#define END(intr, reset)	htons((3 << 14) | \
	((intr & 0x01) << 12) | \
	((reset & 0x01) << 11))
#define TRIG(wait, send)	htons((7 << 13) | \
	((wait & 0x07) << 7) | \
	((send & 0x07) << 1))
#define TRIG_ENG1	0x01
#define TRIG_ENG2	0x02
#define TRIG_ENG3	0x04

static inline void lp5562_wait_opmode_done(void)
{
	usleep_range(200, 300);
}

static inline void lp5562_wait_enable_done(void)
{
	usleep_range(500, 600);
}

int lp5562_read(struct lp5562_state *state, u8 addr, u8 *val)
{
	int ret = 0;
	static u32 err_count = 0;
	ret = i2c_smbus_read_byte_data(state->client, addr);
	if (ret < 0 && !(err_count % 5)) {
		LP_ERR("I2C read error %d (attempt %u)", ret, err_count++);
	} else {
		*val = ret;
		ret = 0;
	}
	return ret;
}

int lp5562_read_block(struct lp5562_state *state, u8 addr, u8 *val, u8 len)
{
	u8 *start = val;
	int ret = 0;
	static u32 err_count = 0;

	for(; val < start + len; val++,addr++) {
		ret = i2c_smbus_read_byte_data(state->client, addr);
		if (ret < 0 && !(err_count % 5)) {
			LP_ERR("I2C read error %d (attempt %u)", ret, err_count++);
			break;
		} else {
			*val = ret;
			ret = 0;
		}
	}
	return ret;
}

int lp5562_write(struct lp5562_state *state, u8 addr, u8 val)
{
	int ret = 0;
	static u32 err_count = 0;
	ret = i2c_smbus_write_byte_data(state->client, addr, val);
	if (ret < 0 && !(err_count % 5)) {
		LP_ERR("I2C write error %d (attempt %u)", ret, err_count++);
	}
	return ret;
}

#if 0
int lp5562_write_block(struct lp5562_state *state, u8 addr, u8 *val, u8 len)
{
	int ret = 0;
	static u32 err_count = 0;
	ret = i2c_smbus_write_block_data(state->client, addr, len, val);
	if (ret < 0 && !(err_count % 5)) {
		LP_ERR("I2C write error %d (attempt %u)", ret, err_count++);
	}
	return ret;
}
#else
int lp5562_write_block(struct lp5562_state *state, u8 addr, u8 *val, u8 len)
{
	u8 *start = val;
	int ret = 0;
	static u32 err_count = 0;

	for(; val < start + len; val++,addr++) {
		ret = i2c_smbus_write_byte_data(state->client, addr, *val);
		if (ret < 0 && !(err_count % 5)) {
			LP_ERR("I2C read error %d (attempt %u)", ret, err_count++);
			break;
		}
	}
	return ret;
}
#endif

static inline struct lp5562_state *lp5562_get_state(void)
{
	return lp5562_global_state;
}

#define IS_RUNNING(e, o)\
	case e: \
		return ((o & LP5562_MODE_ENG##e##_M) == LP5562_RUN_ENG##e) || LP5562_ENG##e##_IS_LOADING(o);

static int lp5562_engine_running(u8 engine, u8 op)
{
	switch (engine) {
	IS_RUNNING(1, op)
	IS_RUNNING(2, op)
	IS_RUNNING(3, op)
	}
	return 0;
}

#define ENG_MASK(e)		(LP5562_MODE_ENG##e##_M)
#define ENG_LOAD(e, o) \
	case e: \
		return ((o & ~ENG_MASK(e)) | LP5562_LOAD_ENG##e)
#define ENG_RUN(e, o) \
	case e: \
		return ((o & ~ENG_MASK(e)) | LP5562_RUN_ENG##e)
#define ENG_DIS(e, o) \
	case e: \
		return (o & ~ENG_MASK(e))

static u8 lp5562_set_load(u8 engine, u8 op)
{
	switch (engine) {
	ENG_LOAD(1, op);
	ENG_LOAD(2, op);
	ENG_LOAD(3, op);
	}
	return op;
}

static u8 lp5562_set_run(u8 engine, u8 op)
{
	switch (engine) {
	ENG_RUN(1, op);
	ENG_RUN(2, op);
	ENG_RUN(3, op);
	}
	return op;
}

static u8 lp5562_set_disable(u8 engine, u8 op)
{
	switch (engine) {
	ENG_DIS(1, op);
	ENG_DIS(2, op);
	ENG_DIS(3, op);
	}
	return op;
}

static int lp5562_select_eng(struct lp5562_state *state, u8 *eng_num, u8 op_mode)
{
	if (!(*eng_num)) {
		(*eng_num)++;
	}
	if (!lp5562_engine_running(*eng_num, op_mode)) {
		LP_DEBUG("Selected engine is not running!");
		return 0;
	} else {
		int i = 1;
		for (; i <= LP5562_NUM_ENGINES; i++) {
			if (!lp5562_engine_running(i, op_mode)) {
				LP_DEBUG("Engine %u not running", i);
				*eng_num = i;
				return 0;
			} else {
				LP_DEBUG("Engine %u is running", i);
			}
		}
	}
	return 1;
}

static void lp5562_make_fade(struct lp5562_state *state, u16 *new_eng, u8 *eng_idx, u8 color, u16 fade, u8 sign)
{
	u8 scale, steps, incr;
	if (fade > 3591) {
		scale = 1;
		steps = fade / (color * 8);
	} else {
		scale = 0;
		steps = fade / color;
	}
	incr = color / 2;
	if (!steps) {
		steps++;
	}
	LP_DEBUG("RAMP(%u, %u, %u, %u)", scale, steps, sign, incr);
	new_eng[*eng_idx] = RAMP(scale, steps, sign, incr);
	(*eng_idx)++;
	LP_DEBUG("BRANCH(1, %u)", *eng_idx - 1);
	{
		u16 inter = ( (5 << 13) | ((1 & 0x3f) << 7) |  ((*eng_idx - 1) & 0x3f) );
		new_eng[*eng_idx] = htons(inter);
	}

	(*eng_idx)++;
}

static void lp5562_make_end(struct lp5562_state *state, u16 *new_eng, u8 *eng_idx, u8 interrupt)
{
	LP_DEBUG("END(%u, 0)", interrupt);
	new_eng[*eng_idx] = END(interrupt, 0);
	(*eng_idx)++;
	LP_DEBUG("eng_idx = %u", *eng_idx);
}

static int lp5562_fade_out(struct lp5562_state *state, u16 fade, u16 eng_num, u8 *new_op, u8 *new_exec)
{
	if (state->last_step.fade == fade) {
		u8 pc = 0, i = 0;
		for (; i < LP5562_BYTES_PER_ENGINE; i += 2) {
			if (state->engines[eng_num - 1][i] == END(0, 0)) {
				pc = (i / 2) + 1;
				LP_DEBUG("Found fade-out at %u", pc);
				break;
			}
		}
		if (pc) {
			lp5562_write(state, LP5562_REG_ENG1_PC + (eng_num - 1), pc);
			*new_op = lp5562_set_run(eng_num, *new_op);
			*new_exec = lp5562_set_run(eng_num, *new_exec);
			LP_DEBUG("Wrote new PC and starting engine");
		} else {
			LP_DEBUG("Could not find fade-out");
			return 1;
		}
	} else {
		LP_DEBUG("Different fade-out timing, new pattern required.");
		return 1;
	}
	return 0;
}

static void lp5562_garbage_collect(struct lp5562_state *state, u8 *new_op, u8 *new_exec)
{
	int i = 2;
	u8 intr = 0;
	if (state->first_update) {
		LP_DEBUG("First update, disable boot flash pattern.");
		*new_exec = (*new_exec & ~LP5562_MODE_ENG1_M);
		*new_op = (*new_op & ~LP5562_MODE_ENG1_M);
		state->first_update = 0;
	}
	lp5562_read(state, LP5562_REG_STATUS, &intr);
	LP_DEBUG("intr: %x", intr);
	for (; i >= 0; i--) {
		if (intr & LP5562_STATUS_INTR(3 - i)) {
			LP_DEBUG("Engine %d at 0", 3 - i);
			*new_op = (*new_op & ~(3 << (i * 2)));
			continue;
		}
		if (((state->op_mode >> (2 * i)) & 3) && !((state->exec_mode >> (2 * i)) & 3)) {
			u8 map = state->led_map;
			u8 used = 0;
			for (; map; map >>= 2) {
				if ((map & 3) == (3 - i)) {
					used = 1;
					break;
				}
			}
			if (!used) {
				LP_DEBUG("Disabling engine %d", 3 - i);
				*new_op = (*new_op & ~(3 << (i * 2)));
			}
		}
	}
}

static void lp5562_reset_state(struct lp5562_state *state)
{
	state->exec_mode = 0;
	state->op_mode = 0;
	state->led_map = 0;
	memset(&(state->engines), 0, sizeof(state->engines));
	memset(&(state->last_step), 0, sizeof(state->last_step));
}

int lp5562_update(struct led_step *step)
{
	struct lp5562_state *state = lp5562_get_state();
	struct led_step diff = *step;
	u16 new_eng[LP5562_NUM_ENGINES][LP5562_BYTES_PER_ENGINE / 2] = {{ 0 }};
	u8 eng_idx[LP5562_NUM_ENGINES] = { 0 };
	u8 new_op = 0;
	u8 new_exec = 0;
	u8 new_map = 0;
	if (!state) {
		LP_ERR("State uninitialized. Device probe has failed or not taken place.");
		return -1;
	}
	mutex_lock(&(state->lock));
	lp5562_read(state, LP5562_REG_OP_MODE, &new_op);
	lp5562_read(state, LP5562_REG_ENABLE, &new_exec);
	if (!(new_exec & LP5562_MASTER_ENABLE)) {
		LP_DEBUG("Master enable not set! Resetting internal state.");
		new_exec |= LP5562_MASTER_ENABLE;
		lp5562_reset_state(state);
	}
	lp5562_read(state, LP5562_REG_ENG_SEL, &(state->led_map));
	state->op_mode = new_op;
	state->exec_mode = new_exec;
	lp5562_garbage_collect(state, &new_op, &new_exec);
	if (ledctl_same_color(step, &(state->last_step))) {
		LP_DEBUG("New step is the same color as current step.");
		goto up_out;
	}
	if ((ledctl_is_white(step) && ledctl_is_white(&(state->last_step))) ||
		(!ledctl_is_white(step) && !ledctl_is_white(&(state->last_step)))) {
		diff = ledctl_diff(step, &(state->last_step));
		diff.fade = step->fade;
		diff.hold_time = step->hold_time;
		LP_DEBUG("diff %02x%02x%02x", diff.r, diff.g, diff.b);
	}
	if (ledctl_is_white(&diff) && ledctl_is_white(step)) {
		u8 eng_num = state->led_map >> 6;
		LP_DEBUG("White engine currently %u", eng_num);
		if (lp5562_select_eng(state, &eng_num, new_op)) {
			int i = 0;
			for (; i <= 6; i += 2) {
				if (eng_num == ((state->led_map >> i) & 0x3)) {
					state->led_map &= ~(0x3 << i);
				}
			}
			eng_idx[eng_num - 1] = 0;
		}
		LP_DEBUG("Picked engine %u", eng_num);
		if (step->fade) {
			u8 sign;
			if (ledctl_is_white(&(state->last_step))) {
				LP_DEBUG("PWM(%u)", state->last_step.r);
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(state->last_step.r);
			} else {
				LP_DEBUG("PWM(0)");
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(0);
			}
			sign = (ledctl_is_white(&(state->last_step)) && state->last_step.r > step->r);
			lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), diff.r, step->fade, sign);
		} else {
			LP_DEBUG("PWM(%u)", step->r);
			new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(step->r);
		}
		lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), (step->r ? 0 : 1));
		if (step->fade) {
			LP_DEBUG("PWM(%u)", step->r);
			new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(step->r);
			lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), step->r, step->fade, 1);
			lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), 1);
		}
		new_op = lp5562_set_load(eng_num, new_op);
		new_map = (new_map & ~LP5562_ENG_FOR_W_M) | (eng_num << 6);
	} else if (ledctl_is_white(step)) {
		new_map = (new_map & ~LP5562_ENG_FOR_W_M) | (state->led_map & LP5562_ENG_FOR_W_M);
	} else if (ledctl_is_white(&(state->last_step))) {
		u8 eng_num = (state->led_map & LP5562_ENG_FOR_W_M) >> 6;
		LP_DEBUG("White engine was %u", eng_num);
		if (step->fade) {
			if (lp5562_fade_out(state, step->fade, eng_num, &new_op, &new_exec)) {
				lp5562_select_eng(state, &eng_num, new_op);
				LP_DEBUG("PWM(%u)", state->last_step.r);
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(state->last_step.r);
				lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]),
					state->last_step.r, step->fade, 1);
				lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), 1);
				new_op = lp5562_set_load(eng_num, new_op);
			}
			new_map = (new_map & ~LP5562_ENG_FOR_W_M) | (eng_num << 6);
		} else {
			new_op = lp5562_set_disable(eng_num, new_op);
			new_exec = lp5562_set_disable(eng_num, new_exec);
		}
	}
	if (step->r && diff.r && !ledctl_is_white(step)) {
		u8 eng_num = (state->led_map & LP5562_ENG_FOR_R_M) >> 4;
		LP_DEBUG("Red engine currently %u", eng_num);
		if (lp5562_select_eng(state, &eng_num, new_op)) {
			if (new_map >> 6) {
				eng_num = new_map >> 6;
				new_map &= (~LP5562_ENG_FOR_W_M);
			}
			eng_idx[eng_num - 1] = 0;
		}
		LP_DEBUG("Picked engine %u", eng_num);
		if (step->fade) {
			u8 sign;
			if (state->last_step.r && !ledctl_is_white(&(state->last_step))) {
				LP_DEBUG("PWM(%u)", state->last_step.r);
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(state->last_step.r);
			} else {
				LP_DEBUG("PWM(0)");
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(0);
			}
			sign = (!ledctl_is_white(&(state->last_step)) && state->last_step.r > step->r);
			lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), diff.r, step->fade, sign);
		} else {
			LP_DEBUG("PWM(%u)", step->r);
			new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(step->r);
		}
		lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), (step->r ? 0 : 1));
		if (step->fade) {
			LP_DEBUG("PWM(%u)", step->r);
			new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(step->r);
			lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), step->r, step->fade, 1);
			lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), 1);
		}
		new_op = lp5562_set_load(eng_num, new_op);
		new_map = (new_map & ~LP5562_ENG_FOR_R_M) | (eng_num << 4);
	} else if (step->r && !ledctl_is_white(step)) {
		new_map = (new_map & ~LP5562_ENG_FOR_R_M) | (state->led_map & LP5562_ENG_FOR_R_M);
	} else if (state->last_step.r && !ledctl_is_white(&(state->last_step))) {
		u8 eng_num = (state->led_map & LP5562_ENG_FOR_R_M) >> 4;
		LP_DEBUG("Red engine was %u", eng_num);
		if (step->fade) {
			if (lp5562_fade_out(state, step->fade, eng_num, &new_op, &new_exec)) {
				lp5562_select_eng(state, &eng_num, new_op);
				LP_DEBUG("PWM(%u)", state->last_step.r);
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(state->last_step.r);
				lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]),
					state->last_step.r, step->fade, 1);
				lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), 1);
				new_op = lp5562_set_load(eng_num, new_op);
			}
			new_map = (new_map & ~LP5562_ENG_FOR_R_M) | (eng_num << 4);
		} else {
			new_op = lp5562_set_disable(eng_num, new_op);
			new_exec = lp5562_set_disable(eng_num, new_exec);
		}
	}
	if (step->g && diff.g && !ledctl_is_white(step)) {
		u8 eng_num = (state->led_map & LP5562_ENG_FOR_G_M) >> 2;
		LP_DEBUG("Green engine currently %u", eng_num);
		if (lp5562_select_eng(state, &eng_num, new_op)) {
			if (new_map >> 6) {
				eng_num = new_map >> 6;
				new_map &= (~LP5562_ENG_FOR_W_M);
			}
			eng_idx[eng_num - 1] = 0;
		}
		LP_DEBUG("Picked engine %u", eng_num);
		if (step->fade) {
			u8 sign;
			if (state->last_step.g && !ledctl_is_white(&(state->last_step))) {
				LP_DEBUG("PWM(%u)", state->last_step.g);
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(state->last_step.g);
			} else {
				LP_DEBUG("PWM(0)");
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(0);
			}
			sign = (!ledctl_is_white(&(state->last_step)) && state->last_step.g > step->g);
			lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), diff.g, step->fade, sign);
		} else {
			LP_DEBUG("PWM(%u)", step->g);
			new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(step->g);
		}
		lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), (step->g ? 0 : 1));
		if (step->fade) {
			LP_DEBUG("PWM(%u)", step->g);
			new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(step->g);
			lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), step->g, step->fade, 1);
			lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), 1);
		}
		new_op = lp5562_set_load(eng_num, new_op);
		new_map = (new_map & ~LP5562_ENG_FOR_G_M) | (eng_num << 2);
	} else if (step->g && !ledctl_is_white(step)) {
		new_map = (new_map & ~LP5562_ENG_FOR_G_M) | (state->led_map & LP5562_ENG_FOR_G_M);
	} else if (state->last_step.g && !ledctl_is_white(&(state->last_step))) {
		u8 eng_num = (state->led_map & LP5562_ENG_FOR_G_M) >> 2;
		LP_DEBUG("Green engine was %u", eng_num);
		if (step->fade) {
			if (lp5562_fade_out(state, step->fade, eng_num, &new_op, &new_exec)) {
				lp5562_select_eng(state, &eng_num, new_op);
				LP_DEBUG("PWM(%u)", state->last_step.g);
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(state->last_step.g);
				lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]),
					state->last_step.g, step->fade, 1);
				lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), 1);
				new_op = lp5562_set_load(eng_num, new_op);
			}
			new_map = (new_map & ~LP5562_ENG_FOR_G_M) | (eng_num << 2);
		} else {
			new_op = lp5562_set_disable(eng_num, new_op);
			new_exec = lp5562_set_disable(eng_num, new_exec);
		}
	}
	if (step->b && diff.b && !ledctl_is_white(step)) {
		u8 eng_num = (state->led_map & LP5562_ENG_FOR_B_M);
		LP_DEBUG("Blue engine currently %u", eng_num);
		if (lp5562_select_eng(state, &eng_num, new_op)) {
			if (new_map >> 6) {
				eng_num = new_map >> 6;
				new_map &= (~LP5562_ENG_FOR_W_M);
			}
			eng_idx[eng_num - 1] = 0;
		}
		LP_DEBUG("Picked engine %u", eng_num);
		if (step->fade) {
			u8 sign;
			if (state->last_step.b && !ledctl_is_white(&(state->last_step))) {
				LP_DEBUG("PWM(%u)", state->last_step.b);
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(state->last_step.b);
			} else {
				LP_DEBUG("PWM(0)");
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(0);
			}
			sign = (!ledctl_is_white(&(state->last_step)) && state->last_step.b > step->b);
			lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), diff.b, step->fade, sign);
		} else {
			LP_DEBUG("PWM(%u)", step->b);
			new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(step->b);
		}
		lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), (step->b ? 0 : 1));
		if (step->fade) {
			LP_DEBUG("PWM(%u)", step->b);
			new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(step->b);
			lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), step->b, step->fade, 1);
			lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), 1);
		}
		new_op = lp5562_set_load(eng_num, new_op);
		new_map = (new_map & ~LP5562_ENG_FOR_B_M) | eng_num;
	} else if (step->b && !ledctl_is_white(step)) {
		new_map = (new_map & ~LP5562_ENG_FOR_B_M) | (state->led_map & LP5562_ENG_FOR_B_M);
	} else if (state->last_step.b && !ledctl_is_white(&(state->last_step))) {
		u8 eng_num = (state->led_map & LP5562_ENG_FOR_B_M);
		LP_DEBUG("Blue engine was %u", eng_num);
		if (step->fade) {
			if (lp5562_fade_out(state, step->fade, eng_num, &new_op, &new_exec)) {
				lp5562_select_eng(state, &eng_num, new_op);
				LP_DEBUG("PWM(%u)", state->last_step.b);
				new_eng[eng_num - 1][eng_idx[eng_num - 1]++] = PWM(state->last_step.b);
				lp5562_make_fade(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]),
					state->last_step.b, step->fade, 1);
				lp5562_make_end(state, new_eng[eng_num - 1], &(eng_idx[eng_num - 1]), 1);
				new_op = lp5562_set_load(eng_num, new_op);
			}
			new_map = (new_map & ~LP5562_ENG_FOR_B_M) | eng_num;
		} else {
			new_op = lp5562_set_disable(eng_num, new_op);
			new_exec = lp5562_set_disable(eng_num, new_exec);
		}
	}
	state->exec_mode = new_exec;
	state->op_mode = new_op;
	state->led_map = new_map;
	if (LP5562_ENG1_IS_LOADING(state->op_mode)) {
		new_exec = (new_exec & ~LP5562_EXEC_ENG1_M);
		new_op = (new_op & ~LP5562_MODE_ENG1_M);
	}
	if (LP5562_ENG2_IS_LOADING(state->op_mode)) {
		new_exec = (new_exec & ~LP5562_EXEC_ENG2_M);
		new_op = (new_op & ~LP5562_MODE_ENG2_M);
	}
	if (LP5562_ENG3_IS_LOADING(state->op_mode)) {
		new_exec = (new_exec & ~LP5562_EXEC_ENG3_M);
		new_op = (new_op & ~LP5562_MODE_ENG3_M);
	}
	if (LP5562_IS_LOADING(state->op_mode)) {
		lp5562_write(state, LP5562_REG_ENABLE, new_exec);
		lp5562_wait_enable_done();
		lp5562_write(state, LP5562_REG_OP_MODE, new_op);
		lp5562_wait_opmode_done();
		lp5562_write(state, LP5562_REG_OP_MODE, state->op_mode);
		lp5562_wait_opmode_done();
	}
	if (LP5562_ENG1_IS_LOADING(state->op_mode)) {
		LP_DEBUG("Load engine 1 (state %p).", state->engines[0]);
		memcpy(state->engines[0], new_eng[0], LP5562_BYTES_PER_ENGINE);
		lp5562_write_block(state, LP5562_REG_PROG_MEM_ENG1, state->engines[0], LP5562_BYTES_PER_ENGINE);
		state->exec_mode = ((state->exec_mode & ~LP5562_EXEC_ENG1_M) | LP5562_RUN_ENG1);
		state->op_mode = ((state->op_mode & ~LP5562_MODE_ENG1_M) | LP5562_RUN_ENG1);
	}
	if (LP5562_ENG2_IS_LOADING(state->op_mode)) {
		LP_DEBUG("Load engine 2 (state %p).", state->engines[1]);
		memcpy(state->engines[1], new_eng[1], LP5562_BYTES_PER_ENGINE);
		lp5562_write_block(state, LP5562_REG_PROG_MEM_ENG2, state->engines[1], LP5562_BYTES_PER_ENGINE);
		state->exec_mode = ((state->exec_mode & ~LP5562_EXEC_ENG2_M) | LP5562_RUN_ENG2);
		state->op_mode = ((state->op_mode & ~LP5562_MODE_ENG2_M) | LP5562_RUN_ENG2);
	}
	if (LP5562_ENG3_IS_LOADING(state->op_mode)) {
		LP_DEBUG("Load engine 3 (state %p).", state->engines[2]);
		memcpy(state->engines[2], new_eng[2], LP5562_BYTES_PER_ENGINE);
		lp5562_write_block(state, LP5562_REG_PROG_MEM_ENG3, state->engines[2], LP5562_BYTES_PER_ENGINE);
		state->exec_mode = ((state->exec_mode & ~LP5562_EXEC_ENG3_M) | LP5562_RUN_ENG3);
		state->op_mode = ((state->op_mode & ~LP5562_MODE_ENG3_M) | LP5562_RUN_ENG3);
	}
	LP_DEBUG("Write exec: %x op: %x map: %x", state->exec_mode, state->op_mode, state->led_map);
	lp5562_write(state, LP5562_REG_OP_MODE, state->op_mode);
	lp5562_write(state, LP5562_REG_ENABLE, state->exec_mode);
	lp5562_wait_enable_done();
	lp5562_write(state, LP5562_REG_ENG_SEL, state->led_map);
	state->last_step = *step;
up_out:
	mutex_unlock(&(state->lock));
	LP_DEBUG("**************************");
	return step->hold_time + step->fade;
}

static int lp5562_init_regs(struct lp5562_state *state)
{
	int ret = 0;
	ret = lp5562_write(state, LP5562_REG_ENABLE, LP5562_ENABLE_DEFAULT);
	if (ret) {
		LP_ERR("Failed to enable chip, abort (code %d).", ret);
		return ret;
	}
	lp5562_wait_enable_done();

	ret = lp5562_write(state, LP5562_REG_CONFIG, LP5562_CONFIG_DEFAULT);
	if (ret) {
		LP_ERR("Failed to configure chip (code %d).", ret);
	}

	ret = lp5562_write(state, LP5562_REG_W_CURRENT, 0xc8);
	if (ret) {
		LP_ERR("Failed to set white LED current (code %d).", ret);
	}

	return ret;
}

static void lp5562_init_state(struct lp5562_state *state)
{
	int ret = 0;
	ret = lp5562_read(state, LP5562_REG_ENABLE, &(state->exec_mode));
	if (ret) {
		LP_ERR("Failed to read exec mode (%d).", ret);
	} else {
		LP_DEBUG("Exec mode: %x", state->exec_mode);
	}
	ret = lp5562_read(state, LP5562_REG_OP_MODE, &(state->op_mode));
	if (ret) {
		LP_ERR("Failed to read op mode (%d).", ret);
	} else {
		LP_DEBUG("Op mode: %x", state->op_mode);
	}
	ret = lp5562_read(state, LP5562_REG_ENG_SEL, &(state->led_map));
	if (ret) {
		LP_ERR("Failed to read LED map (%d).", ret);
	} else {
		LP_DEBUG("LED map: %x", state->led_map);
	}

	if (state->exec_mode & LP5562_ENABLE_RUN_PROGRAM) {
		LP_DEBUG("Program running, stop so we can read the SRAM");
		state->first_update = 1;
		lp5562_write(state, LP5562_REG_ENABLE, LP5562_ENABLE_DEFAULT);
		lp5562_write(state, LP5562_REG_OP_MODE, 0x00);
		lp5562_wait_enable_done();
		lp5562_write(state, LP5562_REG_OP_MODE, LP5562_LOAD_ENG1 | LP5562_LOAD_ENG2 | LP5562_LOAD_ENG3);
		lp5562_wait_opmode_done();
	}

	ret = lp5562_read_block(state, LP5562_REG_PROG_MEM_ENG1, state->engines[0], LP5562_BYTES_PER_ENGINE);
	if (ret) {
		LP_ERR("Failed to read Engine 1 program (%d).", ret);
	}
	ret = lp5562_read_block(state, LP5562_REG_PROG_MEM_ENG2, state->engines[1], LP5562_BYTES_PER_ENGINE);
	if (ret) {
		LP_ERR("Failed to read Engine 2 program (%d).", ret);
	}
	ret = lp5562_read_block(state, LP5562_REG_PROG_MEM_ENG3, state->engines[2], LP5562_BYTES_PER_ENGINE);
	if (ret) {
		LP_ERR("Failed to read Engine 3 program (%d).", ret);
	}

	if (state->exec_mode & LP5562_ENABLE_RUN_PROGRAM) {
		LP_DEBUG("Restoring program state");
		lp5562_write(state, LP5562_REG_OP_MODE, state->op_mode);
		lp5562_write(state, LP5562_REG_ENABLE, state->exec_mode);
		lp5562_wait_enable_done();
	}
}

static struct ledctl_hw_ops ops = {
	.update		= lp5562_update,
	.scale		= NULL,
};

static int lp5562_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	struct lp5562_state *state = devm_kzalloc(&(client->dev), sizeof(struct lp5562_state), GFP_KERNEL);
	if (!state) {
		LP_ERR("Memory allocation failed");
		return -ENOMEM;
	}
	dev_set_drvdata(&(client->dev), state);
	mutex_init(&(state->lock));
	lp5562_global_state = state;

	mutex_lock(&(state->lock));

	state->client = client;
	ret = lp5562_read(state, LP5562_REG_ENABLE, &(state->exec_mode));
	if (ret) {
		LP_ERR("Error %d reading exec mode register, abort.", ret);
		mutex_unlock(&(state->lock));
		lp5562_global_state = (struct lp5562_state *)NULL;
		return -ENODEV;
	} else if ((state->exec_mode & LP5562_MASTER_ENABLE) == 0) {
		ret = lp5562_init_regs(state);
		if (ret) {
			LP_ERR("Error %d initializing chip.", ret);
		}
	} else {
		LP_DEBUG("Chip initialized by u-boot, skipping reg init");
	}

	lp5562_init_state(state);

	mutex_unlock(&(state->lock));

	ret = lp5562_proc_init(state);
	if (ret) {
		LP_ERR("Procfs init failed");
	}

	ledctl_hw_register(&ops);

	LP_INFO("LED controller registered.");
	return ret;
}

static int lp5562_remove(struct i2c_client *client)
{
	struct lp5562_state *state = dev_get_drvdata(&(client->dev));

	ledctl_hw_unregister();
	lp5562_proc_remove();
	lp5562_write(state, LP5562_REG_RESET, LP5562_RESET);
	LP_INFO("LED controller removed.");
	return 0;
}

const char * lp5562_exec_mode_str(u8 val)
{
	switch (val) {
	case 0:
		return "Hold";
	case 1:
	case 3:
		return "Step";
	case 2:
		return "Run";
	default:
		return "Unknown";
	}
}

const char * lp5562_op_mode_str(u8 val)
{
	switch (val) {
	case 0:
		return "Disabled";
	case 1:
		return "Load";
	case 2:
		return "Exec";
	case 3:
		return "Direct";
	default:
		return "Unknown";
	}
}

#define LP5562_NUM_REGS	0x71
#define LED_MAP_W(x)		((x >> 6) & 3)
#define LED_MAP_R(x)		((x >> 4) & 3)
#define LED_MAP_G(x)		((x >> 2) & 3)
#define LED_MAP_B(x)		(x & 3)
#define LED_MAP_STR(led, eng)	((LED_MAP_##led(regs[LP5562_REG_ENG_SEL]) == eng) ? "*" : " ")

static int lp5562_show(struct seq_file *m, void *v)
{
	u8 regs[LP5562_NUM_REGS];
	int i;
	struct lp5562_state *state = (struct lp5562_state *)m->private;

	lp5562_read_block(state, 0x00, regs, LP5562_NUM_REGS);
	seq_printf(m, "TI LP5562\n\n");
	seq_printf(m, "Engine configuration:\n");
	seq_printf(m, "\t1: Exec %s, Op %s\n",
		lp5562_exec_mode_str(ENG1_MODE(regs[LP5562_REG_ENABLE])),
		lp5562_op_mode_str(ENG1_MODE(regs[LP5562_REG_OP_MODE])));
	seq_printf(m, "\t2: Exec %s, Op %s\n",
		lp5562_exec_mode_str(ENG2_MODE(regs[LP5562_REG_ENABLE])),
		lp5562_op_mode_str(ENG2_MODE(regs[LP5562_REG_OP_MODE])));
	seq_printf(m, "\t3: Exec %s, Op %s\n",
		lp5562_exec_mode_str(ENG3_MODE(regs[LP5562_REG_ENABLE])),
		lp5562_op_mode_str(ENG3_MODE(regs[LP5562_REG_OP_MODE])));

	seq_printf(m, "LED <-> Engine Map:\n");
	seq_printf(m, "///// 1 | 2 | 3 | D |\n");
	seq_printf(m, "| R | %s | %s | %s | %s |\n",
		LED_MAP_STR(R, 1),
		LED_MAP_STR(R, 2),
		LED_MAP_STR(R, 3),
		LED_MAP_STR(R, 0));
	seq_printf(m, "| G | %s | %s | %s | %s |\n",
		LED_MAP_STR(G, 1),
		LED_MAP_STR(G, 2),
		LED_MAP_STR(G, 3),
		LED_MAP_STR(G, 0));
	seq_printf(m, "| B | %s | %s | %s | %s |\n",
		LED_MAP_STR(B, 1),
		LED_MAP_STR(B, 2),
		LED_MAP_STR(B, 3),
		LED_MAP_STR(B, 0));
	seq_printf(m, "| W | %s | %s | %s | %s |\n",
		LED_MAP_STR(W, 1),
		LED_MAP_STR(W, 2),
		LED_MAP_STR(W, 3),
		LED_MAP_STR(W, 0));

	mutex_lock(&(state->lock));
	seq_printf(m, "\nCurrent engine programs:\n");
	for (i = 0; i < LP5562_NUM_ENGINES; i++) {
		u8 idx = 0;
		u16 instr = 0;
		seq_printf(m, "%d:", i + 1);
		do {
			instr = (state->engines[i][idx] << 8) | state->engines[i][idx + 1];
			seq_printf(m, "\t%02x: %04x", (idx >> 1), instr);
			if ((instr & 0xe000) == 0xe000) {
				seq_printf(m, "\ttrig\t%u,%u\n", ((instr >> 7) & 3), ((instr >> 1) & 3));
			} else if ((instr & 0xe000) == 0xc000) {
				seq_printf(m, "\tend\t%u,%u\n", ((instr >> 12) & 1), ((instr >> 11) & 1));
			} else if ((instr & 0xe000) == 0xa000) {
				seq_printf(m, "\tbr\t%u,%u\n", ((instr >> 7) & 0x3f), (instr & 0x3f));
			} else if (instr == 0) {
				seq_printf(m, "\tgts\n");
			} else if ((instr & 0xff00) == 0x4000) {
				seq_printf(m, "\tpwm\t%u\n", (instr & 0xff));
			} else if ((instr & 0x007f) == 0) {
				seq_printf(m, "\twait\t%u,%u\n", ((instr >> 14) & 1), ((instr >> 8) & 0x3f));
			} else {
				seq_printf(m, "\tramp\t%u,%u,%u,%u\n",
					((instr >> 14) & 1), ((instr >> 8) & 0x3f),
					((instr >> 7) & 1), (instr & 0x7f));
			}
			idx += 2;
		} while (idx < LP5562_BYTES_PER_ENGINE && instr != 0x0000);
		seq_printf(m, "\n");
	}
	mutex_unlock(&(state->lock));

	seq_printf(m, "Register dump:");
	for (i = 0; i < LP5562_NUM_REGS; i++) {
		if (!(i % 4)) {
			seq_printf(m, "\n%#04x:  ", i);
		}
		seq_printf(m, "%02x\t", regs[i]);
	}
	seq_printf(m, "\n");

	return 0;
}

static ssize_t lp5562_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct lp5562_state *state = PDE_DATA(file_inode(filp));
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "reg", 3) == 0) {
		unsigned long reg;
		unsigned long val;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 16, &reg)) {
			LP_WARNING("Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 16, &val)) {
			LP_WARNING("Cannot parse value %s.", val_str);
			return count;
		}
		lp5562_write(state, reg, val);
		lp5562_read(state, reg, (u8 *)&val);
		LP_INFO("Wrote register %lu, read back %lx.", reg, val);
	}

	return count;
}

const static struct seq_operations lp5562_op = {
	.show		= lp5562_show
};

static int lp5562_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, lp5562_show, PDE_DATA(inode));
}

static struct file_operations lp5562_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= lp5562_proc_open,
	.write		= lp5562_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define LP5562_PROCFS_FILE "driver/lp5562"

static int lp5562_proc_init(struct lp5562_state *state)
{
	struct proc_dir_entry *entry;
	entry = proc_create_data(LP5562_PROCFS_FILE, 0666, NULL, &lp5562_proc_ops, state);
	if (!entry) {
		return -EIO;
	}

	return 0;
}

static void lp5562_proc_remove(void)
{
	remove_proc_entry(LP5562_PROCFS_FILE, NULL);
}

static const struct i2c_device_id lp5562_id[] = {
	{ "led0", 0},
	{ }
};
MODULE_DEVICE_TABLE(lp5562_i2c, lp5562_id);

static struct of_device_id lp5562_ids[] = {
	{ .compatible = "ti,lp5562"},
	{ }
};

static struct i2c_driver lp5562_i2c_driver = {
	.driver = {
		.name		= "lp5562",
		.owner		= THIS_MODULE,
		.of_match_table	= lp5562_ids,
	},
	.id_table	= lp5562_id,
	.probe		= lp5562_probe,
	.remove		= lp5562_remove
};

module_i2c_driver(lp5562_i2c_driver);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for HW using lp5562 to control the LED");
MODULE_LICENSE("GPL");
