/*
 * Copyright (c) 2017, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for MT8521p Timer peripheral
 */

#include <linux/clk.h>
#include <linux/device.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/of_platform.h>
#include <linux/proc_fs.h>

#include "lla.h"
#include "blackbox.h"

#define DRV_NAME		"mtk_gpt"
#define PROCFS_REGS_FILE	"driver/"DRV_NAME

#define GPT_READ(R)		readl(&(gpt->regs->R))
#define GPT_WRITE(V, R)		writel(V, &(gpt->regs->R))

#define GPT_REG_CON		0x00
#define GPT_REG_CLK		0x04
#define GPT_REG_CNT		0x08
#define GPT_REG_CMP		0x0c

struct mtk_gpt_regs {
	u32 con;
	u32 clk;
	u32 cnt;
	u32 cmp;
};

struct mtk_gpt {
	struct mtk_gpt_regs __iomem	*regs;
	struct lla_hw_clk		lla_clk_src;
	struct resource			*res;
	struct device			*dev;
	struct clk			*clk;
};

static int procfs_regs_show(struct seq_file *m, void *v)
{
	struct mtk_gpt *gpt = m->private;

#define _GPT_PROCFS_REG_SHOW(reg) seq_printf(m, "[%03x] GPT_%-7s = %08x\n", (&(gpt->regs->reg) - (u32*)gpt->regs) * sizeof(u32), #reg, GPT_READ(reg))

	seq_printf(m, "%08x-%08x\n", gpt->res->start, gpt->res->end);
	_GPT_PROCFS_REG_SHOW(con);
	_GPT_PROCFS_REG_SHOW(clk);
	_GPT_PROCFS_REG_SHOW(cnt);
	_GPT_PROCFS_REG_SHOW(cmp);

	return 0;
}

static int procfs_regs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_regs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_regs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_regs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static void procfs_remove(void)
{
	remove_proc_entry(PROCFS_REGS_FILE, NULL);
}

static int procfs_init(struct mtk_gpt *gpt)
{
	struct proc_dir_entry *file;

	file = proc_create_data(PROCFS_REGS_FILE, (S_IRUSR | S_IRGRP | S_IROTH), NULL, &procfs_regs_ops, gpt);
	if (!file) {
		bb_log_dev(gpt->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_REGS_FILE);
		return -1;
	}

	return 0;
}

static u32 mtk_gpt_lla_clock_read(void *context)
{
	struct mtk_gpt *gpt = context;
	return GPT_READ(cnt);
}

static struct lla_hw_clk_ops lla_clk_ops = {
	.read = mtk_gpt_lla_clock_read,
};

static int mtk_gpt_probe(struct platform_device *pdev)
{
	struct mtk_gpt *gpt;
	gpt = devm_kzalloc(&pdev->dev, sizeof(*gpt), GFP_KERNEL);
	if (!gpt) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Memory allocation failed!");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, gpt);
	gpt->dev = &(pdev->dev);

	gpt->res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (IS_ERR(gpt->res)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Could not get resources.");
		return PTR_ERR(gpt->res);
	}

	gpt->regs = devm_ioremap_resource(&pdev->dev, gpt->res);
	if (IS_ERR(gpt->regs)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Could not map registers.");
		return PTR_ERR(gpt->regs);
	}

	gpt->clk = devm_clk_get(gpt->dev, "system-clk");
	if (IS_ERR(gpt->clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Could not get clock.");
		return PTR_ERR(gpt->clk);
	}
	clk_prepare_enable(gpt->clk);

	GPT_WRITE(0x31, con);

	if (lla_init_clk(&(gpt->lla_clk_src), DRV_NAME, (gpt->res->start + GPT_REG_CNT), clk_get_rate(gpt->clk), 1, &lla_clk_ops, gpt)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Could not set up LLA clock.");
		return -EFAULT;
	}

	if (lla_register_clk(&(gpt->lla_clk_src))) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Could not register LLA clock.");
		return -ENODEV;
	}

	procfs_init(gpt);

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "Registered GPT at %08x for the LLA. Rate: %luHz", gpt->res->start, clk_get_rate(gpt->clk));

	return 0;
}

static int mtk_gpt_remove(struct platform_device *pdev)
{
	struct mtk_gpt *gpt = platform_get_drvdata(pdev);

	procfs_remove();

	GPT_WRITE(0, con);

	clk_disable_unprepare(gpt->clk);

	return 0;
}

static const struct of_device_id mtk_gpt_ids[] = {
	{ .compatible = "sonos,mtk-timer", },
	{ }
};

static struct platform_driver mtk_gpt_driver = {
	.probe = mtk_gpt_probe,
	.remove = mtk_gpt_remove,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = mtk_gpt_ids,
	},
};

int mtk_gpt_init(void)
{
	return platform_driver_register(&mtk_gpt_driver);
}

void mtk_gpt_exit(void)
{
	platform_driver_unregister(&mtk_gpt_driver);
}
