/*
 * Copyright (c) 2016-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/of.h>
#include <linux/bitrev.h>
#include <sound/pcm.h>
#include <asm/barrier.h>

#include "mt8521p-afe.h"
#include "mt8521p-aud-gpio.h"
#include "mt8521p-afe-clk.h"
#include "mt8521p-private.h"

#include "lla.h"
#include "blackbox.h"

#include "mt8521p_audio.h"
#include "mt8521p_spdif.h"

#define DRV_NAME	"spdif"

struct audio_priv *global_data = NULL;

static void spdif_callback(enum afe_spdifrx_callback_event event);
static int spdif_rx_lla_get_csb(struct lla_csb *csb, void *context);

int iec1_chst_flag = 1;
int iec2_chst_flag = 1;

static inline int spdif_code_to_fs(int code)
{
	switch (code) {
	case SPDIFIN_OUT_RANGE:
		return 0;
	case SPDIFIN_32K:
		return 32000;
	case SPDIFIN_44K:
		return 44100;
	case SPDIFIN_48K:
		return 48000;
	case SPDIFIN_64K:
		return 64000;
	case SPDIFIN_88K:
		return 88200;
	case SPDIFIN_96K:
		return 96000;
	case SPDIFIN_128K:
		return 128000;
	case SPDIFIN_176K:
		return 176400;
	case SPDIFIN_192K:
		return 192000;
	default:
		return 0;
	}
}

static inline char *spdif_port_to_str(enum afe_spdifrx_port port)
{
	switch (port) {
	case SPDIFRX_PORT_NONE:
		return "none";
	case SPDIFRX_PORT_OPT:
		return "OPTICAL";
	case SPDIFRX_PORT_ARC:
		return "HDMI ARC";
	default:
		return "<invalid>";
	};
}

static int spdif_procfs_show(struct seq_file *m, void *v)
{
	struct audio_priv *spdif_data = m->private;
	const volatile struct afe_dir_info *spdif_state = afe_spdifrx_state();
	u32 now_ptr, idx;
	struct lla_csb csb;
	if (spdif_data->rx_params->enable)  {
		afe_memif_pointer(AFE_MEM_ULMCH, &now_ptr);
		seq_printf(m, "SPDIF RX rate: %d\n", spdif_code_to_fs(spdif_state->rate));

		seq_printf(m, "SPDIF RX CSB: ");
		if (spdif_rx_lla_get_csb(&csb, spdif_data) == 0) {
			seq_printf(m, "%24ph\n", csb.data);
		}
		else {
			seq_printf(m, "n/a\n");
		}

		seq_printf(m, "SPDIF RX port: %s\n", spdif_port_to_str(spdif_data->port));
		seq_printf(m, "RX IRQ Count: %u\n", spdif_data->rx_irq_cnt);
		seq_printf(m, "Current RX DMA location: %08x\n", now_ptr);
		for (idx = 0; idx < COMPLETION_HIST_SIZE; idx++) {
			if (idx == COMPLETION_HIST_SIZE - 1) {
				seq_printf(m, "%u+: %u\n", idx, spdif_data->rx_completion_hist[idx]);
			} else {
				seq_printf(m, "%u: %u%c", idx, spdif_data->rx_completion_hist[idx],
					(idx && !((idx + 1) % 5) ? '\n' : '\t'));
			}
		}
	}
	if (spdif_data->tx_params->enable)  {
		now_ptr = afe_spdif_cur(MT_STREAM_IEC1);
		seq_printf(m, "Current TX DMA location: %08x\n", now_ptr);
		seq_printf(m, "TX IRQ Count: %u\n", spdif_data->tx_irq_cnt);
		seq_printf(m, "Discontinuous buffer commits: %u\n", spdif_data->completion_discont_cnt);
		seq_printf(m, "Buffer completion histogram:\n");
		seq_printf(m, "Zero-buffer completions: %u\n", spdif_data->zero_comp_cnt);
		seq_printf(m, "TX Buffer completion histogram:\n");
		for (idx = 0; idx < COMPLETION_HIST_SIZE; idx++) {
			if (idx == COMPLETION_HIST_SIZE - 1) {
				seq_printf(m, "%u+: %u\n", idx, spdif_data->tx_completion_hist[idx]);
			} else {
				seq_printf(m, "%u: %u%c", idx, spdif_data->tx_completion_hist[idx],
				(idx && !((idx + 1) % 5) ? '\n' : '\t'));
			}
		}
		seq_printf(m, "Buffer slices:\n");
		for (idx = 0; idx < spdif_data->tx_params->buffer_cnt; idx++) {
			seq_printf(m, "\t%3u (%p): %c", idx,
				spdif_data->slices.spdif[idx].addr,
				(spdif_data->slices.spdif[idx].state ?
					(spdif_data->slices.spdif[idx].state == BUF_SLICE_ERASED ? 'z' : 'c') : '.'));
			if (idx & 1) {
				seq_printf(m, "\n");
			}
		}
	}
	return 0;
}

static int spdif_procfs_open(struct inode *inode, struct file *file)
{
	return single_open(file, spdif_procfs_show, PDE_DATA(inode));
}

static ssize_t spdif_procfs_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	struct audio_priv *spdif_data = PDE_DATA(file_inode(file));
	char input[32];
	char *newline;
	ssize_t result = count;
	char *parse;
	char *cmd;
	char *param;

	if (count >= sizeof(input)) {
		return -EIO;
	} else if (copy_from_user(&input, buffer, count)) {
		return -EFAULT;
	} else {
		input[count] = '\0';

		newline = strchr(input, '\n');
		if (newline != NULL) {
			*newline = '\0';
		}

		parse = input;
		cmd = strsep(&parse, "=");
		param = parse;

		if (strcmp(cmd, "port") == 0) {
			if (strcmp(param, "opt") == 0) {
				spdif_data->port = SPDIFRX_PORT_OPT;
			} else if (strcmp(param, "arc") == 0) {
				spdif_data->port = SPDIFRX_PORT_ARC;
			} else {
				bb_log_dev(spdif_data->dev, BB_MOD_LLA, BB_LVL_ERR, "invalid parameter '%s'", param);
				return -EINVAL;
			}

			bb_log_dev(spdif_data->dev, BB_MOD_LLA, BB_LVL_WARNING, "spdif port switching to %s", spdif_port_to_str(spdif_data->port));
			afe_spdifrx_stop();
			afe_spdifrx_start(spdif_data->port, spdif_callback);

			spdif_callback(SPDIFRX_CALLBACK_EVENT_UNLOCK);
		} else {
			bb_log_dev(spdif_data->dev, BB_MOD_LLA, BB_LVL_ERR, "invalid command '%s'", cmd);
			return -EINVAL;
		}
	}

	return result;
}

static const struct file_operations procfs_spdif_ops = {
	.owner = THIS_MODULE,
	.open = spdif_procfs_open,
	.read = seq_read,
	.write = spdif_procfs_write,
	.llseek = seq_lseek,
	.release = single_release,
};

static int spdif_snoop_show(struct seq_file *m, void *v)
{
	struct audio_priv *spdif_data = m->private;
	seq_write(m, spdif_data->rx_shared->buffers.k, spdif_data->rx_shared->buffers_total);
	return 0;
}

static int spdif_snoop_open(struct inode *inode, struct file *file)
{
	return single_open(file, spdif_snoop_show, PDE_DATA(inode));
}

static const struct file_operations snoop_spdif_ops = {
	.owner = THIS_MODULE,
	.open = spdif_snoop_open,
	.read = seq_read,
	.write = NULL,
	.llseek = seq_lseek,
	.release = single_release,
};

static int spdif_procfs_init(struct audio_priv *spdif_data)
{
	struct proc_dir_entry *file;
	file = proc_create_data("spdif", S_IRUGO, spdif_data->parent, &procfs_spdif_ops, spdif_data);
	if (!file) {
		bb_log_dev(spdif_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to create SPDIF procfs file.");
		return -1;
	}
	file = proc_create_data("spdif_snoop", S_IRUGO, spdif_data->parent, &snoop_spdif_ops, spdif_data);
	if (!file) {
		bb_log_dev(spdif_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to create SPDIF snoop file.");
		return -1;
	}
	return 0;
}

static void spdif_procfs_exit(struct audio_priv *spdif_data)
{
	remove_proc_entry("spdif", spdif_data->parent);
	remove_proc_entry("spdif_snoop", spdif_data->parent);
}

static unsigned int spdif_rx_lla_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	unsigned int mask = 0;
	struct audio_priv *spdif_data = context;


	poll_wait(filp, &(spdif_data->rx_poll_event), table);

	if ((spdif_data->rx_shared->poll_num < spdif_data->rx_shared->buffers_complete) &&
	    (spdif_data->rx_shared->flags & LLA_SHARED_FLAG_RUNNING)) {
		mask |= POLLIN | POLLRDNORM;
	}

	return mask;
}

static int spdif_rx_lla_get_csb(struct lla_csb *csb, void *context)
{
	struct audio_priv *spdif_data = context;
	int ret = -EAGAIN;

	spin_lock_irq(&spdif_data->spdif_rx_csb_lock);
	if (spdif_data->spdif_rx_csb_valid) {
		memcpy(csb->data, spdif_data->spdif_rx_csb_data, sizeof(spdif_data->spdif_rx_csb_data));
		ret = 0;
	}
	spin_unlock_irq(&spdif_data->spdif_rx_csb_lock);

	csb->len = sizeof(spdif_data->spdif_rx_csb_data);
	return ret;
}

static struct lla_ops lla_rx_ops = {
	.poll		= spdif_rx_lla_poll,
	.get_csb	= spdif_rx_lla_get_csb,
};

static int spdif_rx_lla_init(struct audio_priv *spdif_data)
{
	if (lla_init_dev(&(spdif_data->rx_lla_dev), spdif_data->dev, DRV_NAME"_rx", LLA_DEV_TYPE_SPDIF_RX, &lla_rx_ops, spdif_data)) {
		return -1;
	}

	spdif_data->rx_shared = lla_allocate_shared_buffers(
			&(spdif_data->rx_lla_dev),
			spdif_data->rx_params->channel_len,
			spdif_data->rx_params->channels_per_frame,
			spdif_data->rx_params->frames_per_buffer,
			spdif_data->rx_params->buffer_cnt,
			1, NULL);
	if (spdif_data->rx_shared == NULL) {
		return -1;
	}

	spdif_data->rx_shared->flags |= LLA_SHARED_FLAG_RX;
	spdif_data->rx_shared->channel_mask = spdif_data->rx_params->channel_mask;

	lla_update_sample_rate(&(spdif_data->rx_lla_dev), 0);
	return 0;
}

static const struct afe_multilinein_config mli_config = {
	.dsd_mode	= 0,
	.ch_num		= AFE_MULTILINEIN_2CH,
	.endian		= AFE_MULTILINEIN_LITTILE_ENDIAN,
	.fmt		= FMT_64CYCLE_32BIT_I2S,
	.mss		= AFE_MULTILINE_FROM_RX,
	.intr_period	= AFE_MULTILINEIN_INTR_PERIOD_64
};

static int spdif_update_rx_stats(struct audio_priv *spdif_data, u32 time)
{
	u32 now_ptr = 0;
	u32 buf_idx, num_completed = 0;
	afe_memif_pointer(AFE_MEM_ULMCH, &now_ptr);
	now_ptr -= spdif_data->rx_shared->buffers.p;
	buf_idx = now_ptr / spdif_data->rx_shared->buffer_len;
	if (spdif_data->last_rx_buf > buf_idx) {
		num_completed = (spdif_data->rx_shared->buffer_cnt - spdif_data->last_rx_buf) + buf_idx;
	} else if (spdif_data->last_rx_buf < buf_idx) {
		num_completed = (buf_idx - spdif_data->last_rx_buf);
	}
	if (num_completed < COMPLETION_HIST_SIZE) {
		spdif_data->rx_completion_hist[num_completed]++;
	} else {
		spdif_data->rx_completion_hist[COMPLETION_HIST_SIZE - 1]++;
	}
	if (num_completed) {
		spdif_data->rx_shared->complete_time = time;
		dmb();
		spdif_data->rx_shared->buffers_complete += num_completed;
		dmb();
		spdif_data->rx_shared->complete_time2 = time;
	}
	spdif_data->last_rx_buf = buf_idx;
	spdif_data->last_rx_ptr = now_ptr;
	return num_completed;
}

static void spdif_store_csb(const volatile struct afe_dir_info *info, int valid)
{
	int byte;
	volatile u8 *src = (volatile u8*)info->c_bit;

	spin_lock(&global_data->spdif_rx_csb_lock);

	if (valid) {
		for (byte = 0; byte < sizeof(global_data->spdif_rx_csb_data); byte++) {
			global_data->spdif_rx_csb_data[byte] = bitrev8(src[byte]);
		}
	}

	global_data->spdif_rx_csb_valid = valid;

	spin_unlock(&global_data->spdif_rx_csb_lock);
}

static void spdif_callback(enum afe_spdifrx_callback_event event)
{
	const volatile struct afe_dir_info *spdif_state = afe_spdifrx_state();

	switch (event) {
		case SPDIFRX_CALLBACK_EVENT_LOCK:
			{
				int rate = spdif_code_to_fs(spdif_state->rate);
				u32 last_buf = global_data->last_rx_buf;
				u32 completed = global_data->rx_shared->buffers_complete;
				global_data->mem_config.fs = fs_enum(rate);
				afe_memif_configurate(AFE_MEM_ULMCH, &(global_data->mem_config));
				afe_memif_enable(AFE_MEM_ULMCH, 1);
				afe_multilinein_configurate(&mli_config);
				afe_multilinein_enable(1);
				audio_irq_enable(IRQ_AFE_MULTIIN, 1);
				spdif_update_rx_stats(global_data, lla_clock_read());
				bb_log_dev(global_data->dev, BB_MOD_LLA, BB_LVL_INFO, "SPDIF RX lock! Rate: %d, buffer %u->%u, skipped %llu", rate, last_buf, global_data->last_rx_buf, (global_data->rx_shared->buffers_complete - completed));
				global_data->rx_shared->flags |= LLA_SHARED_FLAG_RUNNING;
				lla_update_sample_rate(&(global_data->rx_lla_dev), rate);
				spdif_store_csb(spdif_state, 0);
			}
			break;

		case SPDIFRX_CALLBACK_EVENT_UNLOCK:
			bb_log_dev(global_data->dev, BB_MOD_LLA, BB_LVL_INFO, "SPDIF RX unlock!");
			afe_memif_enable(AFE_MEM_ULMCH, 0);
			afe_multilinein_enable(0);
			audio_irq_enable(IRQ_AFE_MULTIIN, 0);
			global_data->rx_shared->flags &= ~LLA_SHARED_FLAG_RUNNING;
			global_data->rx_shared->flags |= LLA_SHARED_FLAG_RESTARTED;
			lla_update_sample_rate(&(global_data->rx_lla_dev), 0);
			spdif_store_csb(spdif_state, 0);
			break;

		case SPDIFRX_CALLBACK_EVENT_C_BITS:
			spdif_store_csb(spdif_state, 1);
			break;

		case SPDIFRX_CALLBACK_EVENT_U_BITS:
			break;

		default:
			bb_log_dev(global_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Unknown SPDIF RX callback event %d",  event);
			break;
	}
}

void spdif_tx_do_zero(struct audio_priv *spdif_data, u32 start, u32 count)
{
	u32 erase_idx = start;
	u32 idx;
	for (idx = 0; idx < count; idx++, erase_idx++) {
		if (erase_idx >= spdif_data->tx_params->buffer_cnt) {
			erase_idx = 0;
		}
		if (spdif_data->slices.spdif[erase_idx].state) {
			continue;
		}
		memset(spdif_data->slices.spdif[erase_idx].addr, 0x00, spdif_data->tx_shared->buffer_len);
		spdif_data->slices.spdif[erase_idx].state = BUF_SLICE_ERASED;
	}
	spdif_data->last_erase = erase_idx;
}

static unsigned int spdif_tx_lla_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	unsigned int mask = 0;
	struct audio_priv *spdif_data = context;

	poll_wait(filp, &(spdif_data->tx_poll_event), table);

	if (spdif_data->tx_shared->poll_num < (spdif_data->tx_shared->buffers_complete + spdif_data->tx_shared->tx_latency)) {
		mask |= POLLOUT | POLLWRNORM;
	}

	return mask;
}

int spdif_tx_lla_commit(unsigned long buffer_idx, void *context)
{
	struct audio_priv *spdif_data = context;
	static u32 last_idx = 0;
	if (buffer_idx != 0 && last_idx != buffer_idx - 1) {
		spdif_data->completion_discont_cnt++;
	}
	last_idx = buffer_idx;

	if (buffer_idx >= spdif_data->tx_shared->buffer_cnt) {
		bb_log_dev(spdif_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Buffer index %lu out of bounds.", buffer_idx);
		return -EFAULT;
	}

	spdif_data->slices.spdif[buffer_idx].state = BUF_SLICE_COMMIT;

	return 0;
}

static struct lla_ops lla_tx_ops = {
	.poll	= spdif_tx_lla_poll,
	.commit	= spdif_tx_lla_commit,
};

static int spdif_tx_lla_init(struct audio_priv *spdif_data)
{
	if (lla_init_dev(&(spdif_data->tx_lla_dev), spdif_data->dev, DRV_NAME"_tx", LLA_DEV_TYPE_SPDIF_TX, &lla_tx_ops, spdif_data)) {
		return -1;
	}

	spdif_data->tx_shared = lla_allocate_shared_buffers(
			&(spdif_data->tx_lla_dev),
			spdif_data->tx_params->channel_len,
			spdif_data->tx_params->channels_per_frame,
			spdif_data->tx_params->frames_per_buffer,
			spdif_data->tx_params->buffer_cnt,
			1, NULL);
	if (spdif_data->tx_shared == NULL) {
		return -1;
	}

	spdif_data->tx_shared->flags |= LLA_SHARED_FLAG_TX | LLA_SHARED_FLAG_NEED_COMMIT_TX;
	spdif_data->tx_shared->tx_latency = (spdif_data->tx_params->buffer_cnt)/ 2;
	spdif_data->tx_shared->channel_mask = spdif_data->tx_params->channel_mask;

	lla_update_sample_rate(&(spdif_data->tx_lla_dev), spdif_data->tx_params->frames_per_second);
	return 0;
}

static int spdif_tx_zero_init(struct audio_priv *spdif_data)
{
	int i;

	spdif_data->last_erase = 0;
	for (i = 0; i < spdif_data->tx_params->buffer_cnt; i++) {
		spdif_data->slices.spdif[i].addr = spdif_data->tx_shared->buffers.k + (i * spdif_data->tx_shared->buffer_len);
	}

	tasklet_init(&(spdif_data->erase_task), audio_tx_zero_work, (unsigned long)spdif_data);
	return 0;
}

static int spdif_tx_start(struct audio_priv *spdif_data)
{
	struct snd_pcm_runtime runtime = {
		.frame_bits	= (spdif_data->tx_params->channels_per_frame) * (spdif_data->tx_params->channel_len) * 8,
		.period_size	= 1024,
		.rate		= spdif_data->tx_params->frames_per_second,
		.sample_bits	= spdif_data->tx_params->channel_len * 8,
		.dma_addr	= spdif_data->tx_shared->buffers.p,
		.dma_bytes	= spdif_data->tx_shared->buffers_total
	};

	afe_iec_power_on(MT_STREAM_IEC1, 1);
	afe_iec_clock_on(MT_STREAM_IEC1, 5644800);
	afe_iec_configurate(MT_STREAM_IEC1, 0, &runtime);
	afe_iec_enable(MT_STREAM_IEC1);
	audio_irq_enable(IRQ_AFE_SPDIF, 1);

	lla_register_dev(&(spdif_data->tx_lla_dev));
	spdif_data->tx_shared->flags |= LLA_SHARED_FLAG_RUNNING;

	return 0;
}

static u32 spdif_isr(struct audio_priv *spdif_data, u32 status)
{
	u32 cleared = 0;
	if (spdif_data->rx_params->enable) {
		if (status & AFE_MULTIIN_IRQ) {
			u32 completion_time = lla_clock_read();
			spdif_data->rx_irq_cnt++;
			if (spdif_update_rx_stats(spdif_data, completion_time)) {
				wake_up_interruptible(&(spdif_data->rx_poll_event));
			}
			cleared |= AFE_MULTIIN_IRQ;
		}
		if (status & AFE_SPDIFIN_IRQ) {
			afe_spdifrx_isr();
			cleared |= AFE_SPDIFIN_IRQ;
		}
	}
	if (spdif_data->tx_params->enable) {
		if (status & AFE_SPDIF_IRQ) {
			u32 nsadr;
			if (afe_iec_burst_info_is_ready(MT_STREAM_IEC1)) {
				nsadr = afe_iec_nsadr(MT_STREAM_IEC1);
				nsadr += afe_iec_burst_len(MT_STREAM_IEC1);
				if (nsadr >= afe_spdif_end(MT_STREAM_IEC1))
					nsadr = afe_spdif_base(MT_STREAM_IEC1);
				afe_iec_set_nsadr(MT_STREAM_IEC1, nsadr);
				afe_iec_burst_info_clear_ready(MT_STREAM_IEC1);
			}
			cleared |= AFE_SPDIF_IRQ;
		}
		if (status & AFE_IRQ1_IRQ) {
			u32 now_ptr, i, buf_idx, num_completed = 0;
			now_ptr = afe_spdif_cur(MT_STREAM_IEC1);
			if (!now_ptr) {
				spdif_data->tx_completion_hist[0]++;
				goto no_cur;
			}
			now_ptr -= spdif_data->tx_shared->buffers.p;
			buf_idx = now_ptr / spdif_data->tx_shared->buffer_len;
			if (spdif_data->last_tx_buf > buf_idx) {
				for (i = spdif_data->last_tx_buf; i < spdif_data->tx_shared->buffer_cnt; i++) {
					spdif_data->slices.spdif[i].state = BUF_SLICE_DONE;
					num_completed++;
				}
				spdif_data->last_tx_buf = 0;
				spdif_data->last_tx_ptr = 0;
			}
			for (i = spdif_data->last_tx_buf; i < buf_idx; i++) {
				spdif_data->slices.spdif[i].state = BUF_SLICE_DONE;
				num_completed++;
			}
			if (num_completed < COMPLETION_HIST_SIZE) {
				spdif_data->tx_completion_hist[num_completed]++;
			} else {
				spdif_data->tx_completion_hist[COMPLETION_HIST_SIZE - 1]++;
			}
			if (num_completed == 0 && ((now_ptr - spdif_data->last_tx_ptr) >= MTK_DMA_PTR_SLOP(spdif_data->tx_shared->buffer_len))) {
				num_completed = 1;
				buf_idx++;
			}
			if (num_completed) {
				u32 completion_time = lla_clock_read();
				spdif_data->last_tx_buf = buf_idx;
				spdif_data->tx_shared->complete_time = completion_time;
				dmb();
				spdif_data->tx_shared->buffers_complete += num_completed;
				dmb();
				spdif_data->tx_shared->complete_time2 = completion_time;
			} else {
				spdif_data->zero_comp_cnt++;
			}
			spdif_data->last_tx_ptr = now_ptr;
			if (!(spdif_data->tx_irq_cnt % 16)) {
				tasklet_schedule(&(spdif_data->erase_task));
			}
no_cur:
			wake_up_interruptible(&(spdif_data->tx_poll_event));
			spdif_data->tx_irq_cnt++;
			cleared |= AFE_IRQ1_IRQ;
		}
	}

	return cleared;
}

int spdif_init(struct device *dev, struct proc_dir_entry *parent)
{
	struct audio_priv *spdif_data = NULL;
	spdif_data = devm_kzalloc(dev, sizeof(struct audio_priv), GFP_KERNEL);
	if (!spdif_data) {
		bb_log_dev(dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to allocate memory for SPDIF peripheral.");
		return -ENOMEM;
	}
	global_data = spdif_data;
	spdif_data->dev = dev;
	spdif_data->parent = parent;
	spdif_data->tx_params = audio_get_params(LLA_DEV_TYPE_SPDIF_TX);
	spdif_data->rx_params = audio_get_params(LLA_DEV_TYPE_SPDIF_RX);

	if (spdif_data->rx_params->enable)  {
		init_waitqueue_head(&(spdif_data->rx_poll_event));
		if (spdif_rx_lla_init(spdif_data)) {
			return -1;
		}
		spdif_data->mem_config.hd_audio = 1;
		spdif_data->mem_config.channel = STEREO;
		spdif_data->mem_config.first_bit = LSB_FIRST;
		spdif_data->mem_config.buffer = (struct afe_buffer) {
			.base = spdif_data->rx_shared->buffers.p,
			.size = spdif_data->rx_shared->buffers_total
		};
	}
        if (spdif_data->tx_params->enable)  {
		init_waitqueue_head(&(spdif_data->tx_poll_event));
		if (spdif_tx_lla_init(spdif_data)) {
			return -1;
		}
	}

	spin_lock_init(&spdif_data->spdif_rx_csb_lock);

	spdif_data->irq_callback = spdif_isr;
	spdif_data->do_zero = spdif_tx_do_zero;
	spdif_data->exit = spdif_exit;
	audio_register_dev(spdif_data);

	if (spdif_data->rx_params->enable)  {
		spdif_data->port = SPDIFRX_PORT_ARC;
		afe_spdifrx_start(spdif_data->port, spdif_callback);
		lla_register_dev(&(spdif_data->rx_lla_dev));
	}
	if (spdif_data->tx_params->enable)  {
		spdif_tx_zero_init(spdif_data);
		spdif_tx_start(spdif_data);
	}
	spdif_procfs_init(spdif_data);

	bb_log_dev(spdif_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Registered SPDIF");
	return 0;
}

void spdif_exit(struct audio_priv *spdif_data)
{
	spdif_procfs_exit(spdif_data);
        if (spdif_data->rx_params->enable)  {
		lla_free_shared_buffers(&(spdif_data->rx_lla_dev));
		lla_unregister_dev(&(spdif_data->rx_lla_dev));
		afe_memif_enable(AFE_MEM_ULMCH, 0);
		afe_spdifrx_stop();
		afe_multilinein_enable(0);
	}
        if (spdif_data->tx_params->enable)  {
		tasklet_kill(&(spdif_data->erase_task));
		lla_free_shared_buffers(&(spdif_data->tx_lla_dev));
		lla_unregister_dev(&(spdif_data->tx_lla_dev));
		audio_irq_enable(IRQ_AFE_SPDIF, 0);
		afe_iec_disable(MT_STREAM_IEC1);
		afe_iec_power_on(MT_STREAM_IEC1, 0);
	}
	audio_unregister_dev(spdif_data);
	bb_log_dev(spdif_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Removed SPDIF");
	return;
}
