/*
 * Copyright (c) 2016-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for MT8521p I2S peripheral
 */

#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/of_platform.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/wait.h>
#include <linux/workqueue.h>
#include <asm/barrier.h>

#include "mt8521p-afe.h"
#include "mt8521p-aud-gpio.h"
#include "mt8521p-afe-clk.h"
#include "lla.h"
#include "blackbox.h"

#include "mt8521p_audio.h"
#include "mt8521p_i2s.h"

#define DRV_NAME		"mtk_i2s"

#define SNOOP_RX

static int i2s_procfs_show(struct seq_file *m, void *v)
{
	struct audio_priv *i2s_data = m->private;
	u32 now_ptr, idx;
	if (i2s_data->rx_params->enable) {
		if (i2s_data->rx_params->ring_cnt > 1) {
			for (idx = 0; idx < i2s_data->rx_params->ring_cnt; idx++) {
				afe_memif_pointer(AFE_MEM_UL1 + idx, &now_ptr);
				seq_printf(m, "Current RX%u DMA location: %08x\n", idx + 1, now_ptr);
			}
		} else {
			afe_memif_pointer(AFE_MEM_UL2, &now_ptr);
			seq_printf(m, "Current RX2 DMA location: %08x\n", now_ptr);
		}
		seq_printf(m, "Last RX buffer: %u\n", i2s_data->last_rx_buf);
		seq_printf(m, "RX IRQ count: %u\n", i2s_data->rx_irq_cnt);
		seq_printf(m, "RX Buffer completion histogram:\n");
		for (idx = 0; idx < COMPLETION_HIST_SIZE; idx++) {
			if (idx == COMPLETION_HIST_SIZE - 1) {
				seq_printf(m, "%u+: %u\n", idx, i2s_data->rx_completion_hist[idx]);
			} else {
				seq_printf(m, "%u: %u%c", idx, i2s_data->rx_completion_hist[idx],
					(idx && !((idx + 1) % 5) ? '\n' : '\t'));
			}
		}
	}
	if (i2s_data->tx_params->enable) {
		afe_memif_pointer(AFE_MEM_DLMCH, &now_ptr);
		seq_printf(m, "Current TX DMA location: %08x\n", now_ptr);
		seq_printf(m, "Last erased buffer: %u\n", i2s_data->last_erase);
		seq_printf(m, "TX IRQ count: %u\n", i2s_data->tx_irq_cnt);
		seq_printf(m, "Discontinuous buffer commits: %u\n", i2s_data->completion_discont_cnt);
		seq_printf(m, "Zero-buffer completions: %u\n", i2s_data->zero_comp_cnt);
		seq_printf(m, "TX Buffer completion histogram:\n");
		for (idx = 0; idx < COMPLETION_HIST_SIZE; idx++) {
			if (idx == COMPLETION_HIST_SIZE - 1) {
				seq_printf(m, "%u+: %u\n", idx, i2s_data->tx_completion_hist[idx]);
			} else {
				seq_printf(m, "%u: %u%c", idx, i2s_data->tx_completion_hist[idx],
					(idx && !((idx + 1) % 5) ? '\n' : '\t'));
			}
		}
		seq_printf(m, "Buffer slices:\n");
		for (idx = 0; idx < i2s_data->tx_params->buffer_cnt; idx++) {
			seq_printf(m, "\t%3u (%p): %c", idx,
				i2s_data->slices.i2s[idx].addr,
				(i2s_data->slices.i2s[idx].state ?
					(i2s_data->slices.i2s[idx].state == BUF_SLICE_ERASED ? 'z' : 'c') : '.'));
			if (idx & 1) {
				seq_printf(m, "\n");
			}
		}
	}
	return 0;
}

static int i2s_procfs_open(struct inode *inode, struct file *file)
{
	return single_open(file, i2s_procfs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_i2s_ops = {
	.owner = THIS_MODULE,
	.open = i2s_procfs_open,
	.read = seq_read,
	.write = NULL,
	.llseek = seq_lseek,
	.release = single_release,
};

#ifdef SNOOP_RX
static int i2s_procfs_snoop_show(struct seq_file *m, void *v)
{
	struct audio_priv *i2s_data = m->private;
	seq_write(m, i2s_data->rx_shared->buffers.k, i2s_data->rx_shared->buffers_total);
	return 0;
}

static int i2s_procfs_snoop_open(struct inode *inode, struct file *file)
{
	return single_open(file, i2s_procfs_snoop_show, PDE_DATA(inode));
}

static const struct file_operations procfs_i2s_snoop_ops = {
	.owner = THIS_MODULE,
	.open = i2s_procfs_snoop_open,
	.read = seq_read,
	.write = NULL,
	.llseek = seq_lseek,
	.release = single_release,
};
#endif

static int i2s_procfs_init(struct audio_priv *i2s_data)
{
	struct proc_dir_entry *file;
#ifdef SNOOP_RX
	file = proc_create_data("snoop_i2s", S_IRUGO, i2s_data->parent, &procfs_i2s_snoop_ops, i2s_data);
	if (!file) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to create procfs file.");
		return -1;
	}
#endif
	file = proc_create_data("i2s", S_IRUGO, i2s_data->parent, &procfs_i2s_ops, i2s_data);
	if (!file) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to create procfs file.");
		return -1;
	}
	return 0;
}

static void i2s_procfs_exit(struct audio_priv *i2s_data)
{
	remove_proc_entry("i2s", i2s_data->parent);
#ifdef SNOOP_RX
	remove_proc_entry("snoop_i2s", i2s_data->parent);
#endif
}

void i2s_tx_do_zero(struct audio_priv *i2s_data, u32 start, u32 count)
{
	u32 erase_idx = start;
	u32 idx;
	for (idx = 0; idx < count; idx++, erase_idx++) {
		if (erase_idx >= i2s_data->tx_shared->buffer_cnt) {
			erase_idx = 0;
		}
		if (i2s_data->slices.i2s[erase_idx].state) {
			continue;
		}
		memset(i2s_data->slices.i2s[erase_idx].addr, 0x00, i2s_data->tx_shared->buffer_len);
		i2s_data->slices.i2s[erase_idx].state = BUF_SLICE_ERASED;
	}
	i2s_data->last_erase = erase_idx;
}

unsigned int i2s_tx_lla_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	struct audio_priv *i2s_data = (struct audio_priv *)context;
	u32 mask = 0;

	poll_wait(filp, &(i2s_data->tx_poll_event), table);

	if (i2s_data->tx_shared->poll_num < (i2s_data->tx_shared->buffers_complete + i2s_data->tx_shared->tx_latency)) {
		mask |= POLLOUT | POLLWRNORM;
	}

	return mask;
}

int i2s_tx_lla_commit(unsigned long buffer_idx, void *context)
{
	struct audio_priv *i2s_data = (struct audio_priv *)context;
	static u32 last_idx = 0;
	if (buffer_idx != 0 && last_idx != buffer_idx - 1) {
		i2s_data->completion_discont_cnt++;
	}
	last_idx = buffer_idx;

	if (buffer_idx >= i2s_data->tx_shared->buffer_cnt) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Buffer index %lu out of bounds.", buffer_idx);
		return -EFAULT;
	}

	i2s_data->slices.i2s[buffer_idx].state = BUF_SLICE_COMMIT;

	return 0;
}

static struct lla_ops lla_tx_ops = {
	.poll	= i2s_tx_lla_poll,
	.commit	= i2s_tx_lla_commit,
};

static int i2s_tx_lla_init(struct audio_priv *i2s_data)
{
	if (lla_init_dev(&(i2s_data->tx_lla_dev), i2s_data->dev, DRV_NAME"_tx", LLA_DEV_TYPE_DAC, &lla_tx_ops, i2s_data)) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "LLA TX init failed");
		return -1;
	}

	i2s_data->tx_shared = lla_allocate_shared_buffers(&(i2s_data->tx_lla_dev),
				i2s_data->tx_params->channel_len,
				i2s_data->tx_params->channels_per_frame,
				i2s_data->tx_params->frames_per_buffer,
				i2s_data->tx_params->buffer_cnt,
				i2s_data->tx_params->ring_cnt, NULL);

	if (!i2s_data->tx_shared) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "LLA shared buffer alloc failed");
		return -1;
	}

	bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Set up buffers: p %08lx k %p total bytes %u",
			i2s_data->tx_shared->buffers.p,
			i2s_data->tx_shared->buffers.k,
			i2s_data->tx_shared->buffers_total);

	i2s_data->tx_shared->flags |= LLA_SHARED_FLAG_TX | LLA_SHARED_FLAG_NEED_COMMIT_TX;
	i2s_data->tx_shared->tx_latency = i2s_data->tx_params->buffer_cnt / 2;
	i2s_data->tx_shared->channel_mask = i2s_data->tx_params->channel_mask;

	lla_update_sample_rate(&(i2s_data->tx_lla_dev), i2s_data->tx_params->frames_per_second);
	return 0;
}

static int i2s_tx_zero_init(struct audio_priv *i2s_data)
{
	int i;
	for (i = 0; i < i2s_data->tx_shared->buffer_cnt; i++) {
		i2s_data->slices.i2s[i].addr = i2s_data->tx_shared->buffers.k + (i * i2s_data->tx_shared->buffer_len);
	}

	tasklet_init(&(i2s_data->erase_task), audio_tx_zero_work, (unsigned long)i2s_data);
	return 0;
}

static int i2s_tx_start(struct audio_priv *i2s_data)
{
	struct afe_memif_config mem_config = {
		.fs		= fs_enum(i2s_data->tx_params->frames_per_second),
		.hd_audio	= 1,
		.dsd_width	= DSD_WIDTH_32BIT,
		.first_bit	= MSB_FIRST,
		.channel	= STEREO,
		.buffer		= (struct afe_buffer) {
			.base	= i2s_data->tx_shared->buffers.p,
			.size	= i2s_data->tx_shared->buffers_total
		}
	};
	struct afe_i2s_out_config i2s_out_config = {
		.fpga_test_loop	= 0,
		.data_from_sine	= 0,
		.use_asrc	= 0,
		.dsd_mode	= 0,
		.couple_mode	= 1,
		.one_heart_mode	= 1,
		.fs_invert	= i2s_data->tx_params->invert_fs,
		.slave		= 0,
		.fmt		= FMT_64CYCLE_32BIT_I2S,
		.mclk		= 256 * i2s_data->tx_params->frames_per_second,
		.mode		= fs_enum(i2s_data->tx_params->frames_per_second)
	};
	uint32_t tx_channels_per_frame = i2s_data->tx_params->channels_per_frame;

	switch(tx_channels_per_frame) {
	case 4:
		mem_config.dlmch_ch_num	= DLMCH_4CH;
		break;
	case 8:
		mem_config.dlmch_ch_num	= DLMCH_8CH;
		break;
	default:
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "invalid number of channel per frame: %d", tx_channels_per_frame);
		return -1;
	}


	itrcon_connect(I12, O15, 1);
	itrcon_connect(I13, O16, 1);
	itrcon_connect(I14, O17, 1);
	itrcon_connect(I15, O18, 1);
	itrcon_connect(I16, O19, 1);
	itrcon_connect(I17, O20, 1);
	itrcon_connect(I18, O21, 1);
	itrcon_connect(I19, O22, 1);
	afe_memif_configurate(AFE_MEM_DLMCH, &mem_config);
	afe_i2s_out_configurate(AFE_I2S_OUT_1, &i2s_out_config);
	afe_asmo_timing_set(AFE_I2S_OUT_1, i2s_out_config.mode);
	afe_i2s_out_configurate(AFE_I2S_OUT_2, &i2s_out_config);
	afe_asmo_timing_set(AFE_I2S_OUT_2, i2s_out_config.mode);
	afe_i2s_out_configurate(AFE_I2S_OUT_3, &i2s_out_config);
	afe_asmo_timing_set(AFE_I2S_OUT_3, i2s_out_config.mode);
	afe_i2s_out_configurate(AFE_I2S_OUT_4, &i2s_out_config);
	afe_asmo_timing_set(AFE_I2S_OUT_4, i2s_out_config.mode);
	if (i2s_data->tx_params->uses_mclk) {
		mt_i2s_power_on_mclk(AFE_I2S_OUT_1, 1);
	}
	afe_i2s_out_enable(AFE_I2S_OUT_4, 1);
	afe_i2s_out_enable(AFE_I2S_OUT_3, 1);
	afe_i2s_out_enable(AFE_I2S_OUT_2, 1);
	afe_i2s_out_enable(AFE_I2S_OUT_1, 1);
	afe_memif_enable(AFE_MEM_DLMCH, 1);
	lla_register_dev(&(i2s_data->tx_lla_dev));
	i2s_data->tx_shared->flags |= LLA_SHARED_FLAG_RUNNING;

	return 0;
}

unsigned int i2s_rx_lla_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	struct audio_priv *i2s_data = (struct audio_priv *)context;
	u32 mask = 0;

	poll_wait(filp, &(i2s_data->rx_poll_event), table);

	if (i2s_data->rx_shared->poll_num < i2s_data->rx_shared->buffers_complete) {
		mask |= POLLIN | POLLRDNORM;
	}

	return mask;
}

static struct lla_ops lla_rx_ops = {
	.poll	= i2s_rx_lla_poll,
};

static int i2s_rx_lla_init(struct audio_priv *i2s_data)
{
	if (lla_init_dev(&(i2s_data->rx_lla_dev), i2s_data->dev, DRV_NAME"_rx", i2s_data->rx_params->lla_type, &lla_rx_ops, i2s_data)) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "LLA TX init failed");
		return -1;
	}

	i2s_data->rx_shared = lla_allocate_shared_buffers(&(i2s_data->rx_lla_dev),
				i2s_data->rx_params->channel_len,
				i2s_data->rx_params->channels_per_frame,
				i2s_data->rx_params->frames_per_buffer,
				i2s_data->rx_params->buffer_cnt,
				i2s_data->rx_params->ring_cnt, NULL);
	if (!i2s_data->rx_shared) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "LLA shared buffer alloc failed");
		return -1;
	}

	bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_DEBUG, "Set up buffers: p %08lx k %p total bytes %u",
			i2s_data->rx_shared->buffers.p,
			i2s_data->rx_shared->buffers.k,
			i2s_data->rx_shared->buffers_total);

	dma_sync_single_for_device(i2s_data->dev, (dma_addr_t)i2s_data->rx_shared->buffers.p, i2s_data->rx_shared->buffers_total, DMA_TO_DEVICE);

	i2s_data->rx_shared->flags |= LLA_SHARED_FLAG_RX;
	i2s_data->rx_shared->channel_mask = i2s_data->rx_params->channel_mask;

	lla_update_sample_rate(&(i2s_data->rx_lla_dev), i2s_data->rx_params->virtual_fps);
	return 0;
}

static void i2s_rx_scan_magic_word(struct work_struct *w)
{
	struct delayed_work *delayed = container_of(w, struct delayed_work, work);
	struct audio_priv *i2s_data = container_of(delayed, struct audio_priv, rx_sync_search_work);
	u32 first_word[2] = { 0 };
	u32 i, j, found=1;
	static u8 flagged_warning = 0;
	if (i2s_data->rx_params->ring_cnt > 2) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "%s: magic word search only supports 2 rings right now. ring_cnt = %u", __func__, i2s_data->rx_params->ring_cnt);
		return;
	}
	for (i = 0; i < i2s_data->rx_params->ring_cnt; i++) {
		u32 cmp = (i ? 0x08000000 : 0x0c000000);
		for (j = 0; j < i2s_data->rx_shared->channels_per_frame; j++) {
			u32 word = *(u32 *)(i2s_data->rx_shared->buffers.k + (j * sizeof(u32)) +
			                   (i * i2s_data->rx_shared->buffer_len * i2s_data->rx_shared->buffer_cnt));
			if (word == cmp) {
				first_word[i] = (j + 1) % i2s_data->rx_shared->channels_per_frame;
				bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Magic word found for ring %u! offset: %u", i, first_word[i]);
				found = 1;
				break;
			} else {
				bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_DEBUG, "Magic word search: %08x != %08x", word, cmp);
				found = 0;
			}
		}
	}
	if (first_word[0] != first_word[1]) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "I2S RX rings are not in alignment! %u != %u",
			first_word[0], first_word[1]);
	} else {
		i2s_data->rx_shared->buffer_start_offset = first_word[0];
	}
	if (!found) {
		if (!flagged_warning) {
			bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_WARNING, "Couldn't find sync words. Retrying in 5 seconds.");
			flagged_warning = 1;
		}
		schedule_delayed_work(&(i2s_data->rx_sync_search_work), msecs_to_jiffies(5000));
	}
}

static int i2s_rx_setup(struct audio_priv *i2s_data, u8 input)
{
	struct afe_memif_config mem_config = {
		.fs		= fs_enum(i2s_data->rx_params->frames_per_second),
		.hd_audio	= 1,
		.dsd_width	= DSD_WIDTH_32BIT,
		.first_bit	= MSB_FIRST,
		.channel	= STEREO
	};
	struct afe_i2s_in_config i2s_in_config = {
		.fmt		= FMT_64CYCLE_32BIT_I2S,
		.mclk		= 256 * i2s_data->rx_params->frames_per_second,
		.mode		= fs_enum(i2s_data->rx_params->frames_per_second),
		.slave		= 0,
		.one_heart_mode	= i2s_data->rx_params->sync_tx_rx,
		.couple_mode	= 1
	};

	if (input > i2s_data->rx_params->ring_cnt) {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Requested invalid input line %u.", input);
		return -EIO;
	}

	if (i2s_data->rx_params->ring_cnt > 1) {
		mem_config.buffer = (struct afe_buffer) {
			.base = i2s_data->rx_shared->buffers.p + (input * (i2s_data->rx_shared->buffers_total / i2s_data->rx_shared->ring_cnt)),
			.size = (i2s_data->rx_shared->buffers_total / i2s_data->rx_shared->ring_cnt)
		};
	} else {
		mem_config.buffer = (struct afe_buffer) {
			.base = i2s_data->rx_shared->buffers.p,
			.size = i2s_data->rx_shared->buffers_total
		};
	}

	bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Input %u, buffer starts at %x size %u", input, mem_config.buffer.base, mem_config.buffer.size);

	if (input == 0) {
		itrcon_connect(I00, O00, 1);
		itrcon_connect(I01, O01, 1);
		afe_memif_configurate(AFE_MEM_UL1, &mem_config);
		afe_i2s_in_configurate(AFE_I2S_IN_1, &i2s_in_config);
		afe_asmi_timing_set(AFE_I2S_IN_1, i2s_in_config.mode);
		afe_memif_enable(AFE_MEM_UL1, 1);
	} else if (input == 1) {
		itrcon_connect(I02, O02, 1);
		itrcon_connect(I03, O03, 1);
		afe_memif_configurate(AFE_MEM_UL2, &mem_config);
		afe_i2s_in_configurate(AFE_I2S_IN_2, &i2s_in_config);
		afe_asmi_timing_set(AFE_I2S_IN_2, i2s_in_config.mode);
		if (i2s_data->rx_params->uses_mclk) {
			mt_i2s_power_on_mclk(AFE_I2S_IN_2, 1);
		}
		afe_memif_enable(AFE_MEM_UL2, 1);
	} else {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "No interconnect settings for input %u!", input);
		return -EIO;
	}
	i2s_data->last_rx_buf = 0;

	return 0;
}

static void i2s_rx_start(struct audio_priv *i2s_data)
{
	if (i2s_data->rx_params->ring_cnt == 2) {
		afe_i2s_in_enable(AFE_I2S_IN_2, 1);
		afe_i2s_in_enable(AFE_I2S_IN_1, 1);
		schedule_delayed_work(&(i2s_data->rx_sync_search_work), msecs_to_jiffies(1000));
	} else if (i2s_data->rx_params->ring_cnt == 1) {
		afe_i2s_in_enable(AFE_I2S_IN_2, 1);
	} else {
		bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Could not start RX: invalid number of input streams %u.", i2s_data->rx_params->ring_cnt);
	}
	lla_register_dev(&(i2s_data->rx_lla_dev));
	i2s_data->rx_shared->flags |= LLA_SHARED_FLAG_RUNNING;
}

static u32 i2s_isr(struct audio_priv *i2s_data, u32 status)
{
	u32 cleared = 0;
	u32 now_ptr, i;
	if (status & AFE_IRQ1_IRQ) {
		u32 buf_idx, num_completed = 0;
		afe_memif_pointer(AFE_MEM_DLMCH, &now_ptr);
		now_ptr -= i2s_data->tx_shared->buffers.p;
		buf_idx = now_ptr / i2s_data->tx_shared->buffer_len;
		if (i2s_data->last_tx_buf > buf_idx) {
			for (i = i2s_data->last_tx_buf; i < i2s_data->tx_shared->buffer_cnt; i++) {
				i2s_data->slices.i2s[i].state = BUF_SLICE_DONE;
				num_completed++;
			}
			i2s_data->last_tx_buf = 0;
			i2s_data->last_tx_ptr = 0;
		}
		for (i = i2s_data->last_tx_buf; i < buf_idx; i++) {
			i2s_data->slices.i2s[i].state = BUF_SLICE_DONE;
			num_completed++;
		}
		if (num_completed < COMPLETION_HIST_SIZE) {
			i2s_data->tx_completion_hist[num_completed]++;
		} else {
			i2s_data->tx_completion_hist[COMPLETION_HIST_SIZE - 1]++;
		}
		if ((num_completed == 0) && ((now_ptr - i2s_data->last_tx_ptr) >= MTK_DMA_PTR_SLOP(i2s_data->tx_shared->buffer_len))) {
			num_completed = 1;
			buf_idx++;
		}
		if (num_completed) {
			u32 completion_time = lla_clock_read();
			i2s_data->last_tx_buf = buf_idx;
			i2s_data->tx_shared->complete_time = completion_time;
			dmb();
			i2s_data->tx_shared->buffers_complete += num_completed;
			dmb();
			i2s_data->tx_shared->complete_time2 = completion_time;
		} else {
			i2s_data->zero_comp_cnt++;
		}
		i2s_data->last_tx_ptr = now_ptr;
		if (!(i2s_data->tx_irq_cnt % 16)) {
			tasklet_schedule(&(i2s_data->erase_task));
		}
		wake_up_interruptible(&(i2s_data->tx_poll_event));
		i2s_data->tx_irq_cnt++;
		cleared |= AFE_IRQ1_IRQ;
	}
	if (status & AFE_IRQ2_IRQ) {
		u32 buf_idx, num_completed = 0;

		if (i2s_data->rx_params->ring_cnt == 2) {
			afe_memif_pointer(AFE_MEM_UL1, &now_ptr);
		} else {
			afe_memif_pointer(AFE_MEM_UL2, &now_ptr);
		}

		now_ptr -= i2s_data->rx_shared->buffers.p;
		buf_idx = now_ptr / i2s_data->rx_shared->buffer_len;
		if (i2s_data->last_rx_buf > buf_idx) {
			num_completed = (i2s_data->rx_shared->buffer_cnt - i2s_data->last_rx_buf) + buf_idx;
		} else if (i2s_data->last_rx_buf < buf_idx) {
			num_completed = (buf_idx - i2s_data->last_rx_buf);
		}
		if (num_completed < COMPLETION_HIST_SIZE) {
			i2s_data->rx_completion_hist[num_completed]++;
		} else {
			i2s_data->rx_completion_hist[COMPLETION_HIST_SIZE - 1]++;
		}
		if (num_completed) {
			u32 completion_time = lla_clock_read();
			i2s_data->rx_shared->complete_time = completion_time;
			dmb();
			i2s_data->rx_shared->buffers_complete += num_completed;
			dmb();
			i2s_data->rx_shared->complete_time2 = completion_time;
		}
		i2s_data->last_rx_buf = buf_idx;
		i2s_data->last_rx_ptr = now_ptr;
		wake_up_interruptible(&(i2s_data->rx_poll_event));
		i2s_data->rx_irq_cnt++;
		cleared |= AFE_IRQ2_IRQ;
	}

	return cleared;
}

int i2s_init(struct device *dev, struct proc_dir_entry *parent)
{
	int ret = 0;
	struct audio_priv *i2s_data = devm_kzalloc(dev, sizeof(struct audio_priv), GFP_KERNEL);
	if (!i2s_data) {
		bb_log_dev(dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to allocate memory for I2S peripheral.");
		return -ENOMEM;
	}
	i2s_data->dev = dev;
	i2s_data->parent = parent;
	i2s_data->tx_params = audio_get_params(LLA_DEV_TYPE_DAC);
	i2s_data->rx_params = audio_get_params(LLA_DEV_TYPE_MIC);
	if (!i2s_data->rx_params->enable) {
		i2s_data->rx_params = audio_get_params(LLA_DEV_TYPE_LINE_IN);
	}

	if (i2s_data->rx_params->enable) {
		init_waitqueue_head(&(i2s_data->rx_poll_event));
		if (i2s_data->rx_params->ring_cnt > 1) {
			INIT_DELAYED_WORK(&(i2s_data->rx_sync_search_work), i2s_rx_scan_magic_word);
		}
		if (i2s_rx_lla_init(i2s_data)) {
			return -1;
		}
	}
	if (i2s_data->tx_params->enable) {
		init_waitqueue_head(&(i2s_data->tx_poll_event));
		if (i2s_tx_lla_init(i2s_data)) {
			return -1;
		}
	}

	i2s_data->irq_callback = i2s_isr;
	i2s_data->do_zero = i2s_tx_do_zero;
	i2s_data->exit = i2s_exit;
	audio_register_dev(i2s_data);

	if (i2s_data->rx_params->enable) {
		struct audio_irq_config irq2_config = {
			.mode		= fs_enum(i2s_data->rx_params->virtual_fps),
			.init_val	= 128,
		};
		if (i2s_data->rx_params->ring_cnt == 2) {
			i2s_rx_setup(i2s_data, 1);
			i2s_rx_setup(i2s_data, 0);
		} else if (i2s_data->rx_params->ring_cnt == 1) {
			i2s_rx_setup(i2s_data, 1);
		} else {
			bb_log_dev(dev, BB_MOD_LLA, BB_LVL_ERR, "Invalid number of input streams %u.", i2s_data->rx_params->ring_cnt);
		}
		i2s_rx_start(i2s_data);
		audio_irq_configurate(IRQ_AFE_IRQ2, &irq2_config);
		audio_irq_enable(IRQ_AFE_IRQ2, 1);
	}
	if (i2s_data->tx_params->enable) {
		struct audio_irq_config irq1_config = {
			.mode		= fs_enum(i2s_data->tx_params->frames_per_second),
			.init_val	= 128,
		};
		i2s_data->last_erase = 0;
		i2s_tx_zero_init(i2s_data);
		i2s_tx_start(i2s_data);
		audio_irq_configurate(IRQ_AFE_IRQ1, &irq1_config);
		audio_irq_enable(IRQ_AFE_IRQ1, 1);
	}
	i2s_procfs_init(i2s_data);

	bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Registered I2S");
	return ret;
}

void i2s_exit(struct audio_priv *i2s_data)
{
	i2s_procfs_exit(i2s_data);
	if (i2s_data->tx_params->enable) {
		audio_irq_enable(IRQ_AFE_IRQ1, 0);
		tasklet_kill(&(i2s_data->erase_task));
		lla_unregister_dev(&(i2s_data->tx_lla_dev));
		lla_free_shared_buffers(&(i2s_data->tx_lla_dev));
	}
	if (i2s_data->rx_params->enable) {
		audio_irq_enable(IRQ_AFE_IRQ2, 0);
		lla_unregister_dev(&(i2s_data->rx_lla_dev));
		lla_free_shared_buffers(&(i2s_data->rx_lla_dev));
	}
	audio_unregister_dev(i2s_data);
	bb_log_dev(i2s_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Removed I2S");
}
