/*
 * Copyright (c) 2017 Sonos, Inc.
 * This file exists to bridge the gap between the HDMI DRM driver
 * and the LLA. It sets up the HDMI periperal for SPDIF audio output.
 */
#include <linux/hdmi.h>
#include <linux/module.h>
#include <linux/of_platform.h>
#include <linux/proc_fs.h>
#include <sound/hdmi-codec.h>

#include "blackbox.h"
#include "lla.h"
#include "mt8521p_audio.h"
#include "mt8521p_hdmi_codec.h"

static int hdmi_codec_procfs_show(struct seq_file *m, void *v)
{
	struct hdmi_codec_pdata *hcd = m->private;
	seq_printf(m, "HDMI Codec config: %s %s, %d max channels\n",
		(hcd->i2s ? "I2S" : ""),
		(hcd->spdif ? "SPDIF" : ""),
		hcd->max_i2s_channels);
	return 0;
}

static int hdmi_codec_procfs_open(struct inode *inode, struct file *file)
{
	return single_open(file, hdmi_codec_procfs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_hdmi_codec_ops = {
	.owner = THIS_MODULE,
	.open = hdmi_codec_procfs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int hdmi_codec_procfs_init(struct hdmi_codec_pdata *hcd)
{
	struct proc_dir_entry *file;
	file = proc_create_data("driver/hdmi_codec", S_IRUGO, NULL, &procfs_hdmi_codec_ops, hcd);
	if (!file) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "Failed to create HDMI codec procfs file.");
		return -1;
	}
	return 0;
}

static void hdmi_codec_procfs_exit(void)
{
	remove_proc_entry("driver/hdmi_codec", NULL);
}

static int hdmi_codec_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct hdmi_codec_pdata *hcd = pdev->dev.platform_data;
	struct device *dev = &pdev->dev;
	struct audio_params *tx_params = audio_get_params(LLA_DEV_TYPE_SPDIF_TX);

	struct hdmi_codec_daifmt fmt = {
		.fmt			= HDMI_SPDIF,
		.bit_clk_master		= 0,
		.frame_clk_master	= 0,
	};
	struct hdmi_codec_params params = {
		.cea			= {
			.channels	= tx_params->channels_per_frame,
		},
		.sample_rate		= tx_params->frames_per_second,
		.sample_width		= tx_params->channel_len * 8,
		.channels		= tx_params->channels_per_frame,
	};

	bb_log_dev(dev, BB_MOD_LLA, BB_LVL_INFO, "HDMI codec probe");

	if (hcd->ops->audio_startup) {
		ret = hcd->ops->audio_startup(dev->parent, hcd->data);
		if (ret) {
			bb_log_dev(dev, BB_MOD_LLA, BB_LVL_ERR, "Error starting HDMI audio! %d", ret);
			return ret;
		}
	} else {
		bb_log_dev(dev, BB_MOD_LLA, BB_LVL_DEBUG, "No audio startup function.");
	}

	if (hcd->ops->hw_params) {
		bb_log_dev(dev, BB_MOD_LLA, BB_LVL_DEBUG, "HW params: %u Hz, %d bit, %d channels", params.sample_rate, params.sample_width, params.cea.channels);
		ret = hcd->ops->hw_params(dev->parent, hcd->data, &fmt, &params);
		if (ret) {
			bb_log_dev(dev, BB_MOD_LLA, BB_LVL_ERR, "Error setting HDMI params! %d", ret);
		}
	}

	hdmi_codec_procfs_init(hcd);
	return ret;
}

static int hdmi_codec_remove(struct platform_device *pdev)
{
	struct hdmi_codec_pdata *hcd = pdev->dev.platform_data;
	struct device *dev = &pdev->dev;

	hdmi_codec_procfs_exit();
	if (hcd->ops->audio_shutdown) {
		hcd->ops->audio_shutdown(dev->parent, hcd->data);
	}
	return 0;
}

static struct platform_driver hdmi_codec_driver = {
	.driver = {
		.name = HDMI_CODEC_DRV_NAME,
	},
	.probe = hdmi_codec_probe,
	.remove = hdmi_codec_remove,
};

int hdmi_codec_init(struct device *dev, struct proc_dir_entry *parent)
{
	(void)dev;
	(void)parent;
	bb_log_dev(dev, BB_MOD_LLA, BB_LVL_INFO, "Starting HDMI codec");
	return platform_driver_register(&hdmi_codec_driver);
}
