/*
 * Copyright (c) 2016-2017, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * This file is the central control for the various MT8521p audio functions
 */
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/of_platform.h>
#include <linux/of.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>

#include "mt8521p-afe.h"
#include "mt8521p-aud-gpio.h"
#include "mt8521p-afe-clk.h"
#include "blackbox.h"

#include "mt8521p_hdmi_codec.h"
#include "mt8521p_i2s.h"
#include "mt8521p_spdif.h"

#include "mt8521p_audio.h"

#define DRV_NAME	"mtk_audio"

struct audio_mgr_priv {
	void __iomem		*base_addr;
	int			irq;
	struct audio_params	params[LLA_DEV_TYPE_MAX];
	struct resource		*res;
	struct device		*dev;
	struct proc_dir_entry	*proc_dir;
};

static struct audio_priv		*audio_cb_head = NULL;
static struct audio_mgr_priv		*aud_mgr_data = NULL;

int (*audio_init_funcs[])(struct device *, struct proc_dir_entry *) = {
	i2s_init,
	spdif_init,
	hdmi_codec_init,
	NULL
};

void audio_register_dev(struct audio_priv *cb_struct)
{
	struct audio_priv *cb_ptr = audio_cb_head;
	cb_struct->next = NULL;
	if (!audio_cb_head) {
		audio_cb_head = cb_struct;
		return;
	}

	while (cb_ptr->next) {
		cb_ptr = cb_ptr->next;
	}
	cb_ptr->next = cb_struct;
}

void audio_unregister_dev(struct audio_priv *cb_struct)
{
	struct audio_priv *cb_ptr = audio_cb_head;
	if (!audio_cb_head) {
		return;
	}

	while (cb_ptr && (cb_ptr->next != cb_struct)) {
		cb_ptr = cb_ptr->next;
	}
	if (cb_ptr) {
		cb_ptr->next = cb_struct->next;
	}
}

void audio_tx_zero_work(unsigned long data)
{
	struct audio_priv *aud_data = (struct audio_priv *)data;
	u32 now_idx, erase_idx, zero_num;
	dma_addr_t erase_ptr = (aud_data->last_erase * aud_data->tx_shared->buffer_len) + aud_data->tx_shared->buffers.p;
	now_idx = aud_data->last_tx_buf;
	erase_idx = aud_data->last_erase;
	if (now_idx == erase_idx) {
		return;
	} else if (now_idx < erase_idx) {
		zero_num = (aud_data->tx_shared->buffer_cnt - erase_idx) + now_idx;
	} else {
		zero_num = now_idx - erase_idx;
	}
	if (aud_data->do_zero) {
		aud_data->do_zero(aud_data, erase_idx, zero_num);
	} else {
		bb_log_dev(aud_data->dev, BB_MOD_LLA, BB_LVL_WARNING, "No work to do for zero.");
	}
	dma_sync_single_for_device(aud_data->dev, erase_ptr, zero_num * aud_data->tx_shared->buffer_len, DMA_TO_DEVICE);
}

struct audio_params *audio_get_params(enum lla_dev_type type)
{
	struct audio_mgr_priv *audio_data = aud_mgr_data;
	if (type > LLA_DEV_TYPE_MAX) {
		bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Invalid LLA type %u.", type);
		return NULL;
	}

	return &(audio_data->params[type]);
}

static irqreturn_t audio_isr(int irq_num, void *dev_id)
{
	struct audio_priv *cb_ptr = audio_cb_head;
	u32 status = afe_irq_status();
	u32 cleared = 0;
	irqreturn_t ret = IRQ_HANDLED;
	while (cb_ptr) {
		if (cb_ptr->irq_callback) {
			cleared |= cb_ptr->irq_callback(cb_ptr, status);
		}
		cb_ptr = cb_ptr->next;
	}
	afe_irq_clear(status);
	return ret;
}

static int audio_procfs_show(struct seq_file *m, void *v)
{
	seq_printf(m, "Sonos MT8521p Audio Peripheral Driver\n");
	seq_printf(m, "write: echo <reg>=<value>\n");
	seq_printf(m, "read:  echo read <reg>\n");
	return 0;
}

static int audio_procfs_open(struct inode *inode, struct file *file)
{
	return single_open(file, audio_procfs_show, PDE_DATA(inode));
}

static int audio_procfs_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	struct audio_mgr_priv *audio_data = PDE_DATA(file_inode(file));
	unsigned long reg;
	unsigned long val;
	char read = 0;
	char *reg_str;
	char *val_str;
	char buf[200];
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}
	val_str = buf;
	reg_str = strsep(&val_str, "=");
	if (!strncmp(reg_str, "read ", 4)) {
		read = 1;
		reg_str += 5;
	}
	if (kstrtoul(reg_str, 0, &reg)) {
		bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
		return count;
	}
	if (read) {
		goto do_read;
	}
	if (kstrtoul(val_str, 0, &val)) {
		bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
		return count;
	}
	writel(val, audio_data->base_addr + reg);
do_read:
	bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_INFO, "%04lx: %08x", reg, readl(audio_data->base_addr + reg));
	return count;
}

static const struct file_operations procfs_audio_ops = {
	.owner = THIS_MODULE,
	.open = audio_procfs_open,
	.read = seq_read,
	.write = audio_procfs_write,
	.llseek = seq_lseek,
	.release = single_release,
};

static int audio_procfs_init(struct audio_mgr_priv *audio_data)
{
	struct proc_dir_entry *file;
	audio_data->proc_dir = proc_mkdir("driver/audio", NULL);
	if (!(audio_data->proc_dir)) {
		bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to create procfs directory.");
		return -1;
	}
	file = proc_create_data("regs", S_IWUGO, audio_data->proc_dir, &procfs_audio_ops, audio_data);
	if (!file) {
		bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to create procfs file.");
		return -1;
	}
	return 0;
}

static void audio_procfs_exit(struct audio_mgr_priv *audio_data)
{
	remove_proc_entry("regs", audio_data->proc_dir);
	remove_proc_entry("driver/audio", NULL);
}
static int audio_probe(struct platform_device *pdev)
{
	int idx, ret = 0;
	struct audio_mgr_priv *audio_data = NULL;
	struct device_node *child;
	u32 type;

	audio_data = devm_kzalloc(&(pdev->dev), sizeof(struct audio_mgr_priv), GFP_KERNEL);
	if (!audio_data) {
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "Failed to allocate memory for audio peripheral.");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, audio_data);
	audio_data->dev = &(pdev->dev);
	aud_mgr_data = audio_data;

	audio_data->res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (IS_ERR(audio_data->res)) {
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "Failed to get resource! %ld", PTR_ERR(audio_data->res));
	}

	audio_data->base_addr = devm_ioremap_resource(&pdev->dev, audio_data->res);
	if (IS_ERR(audio_data->base_addr)) {
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "IOmap failed! %ld", PTR_ERR(audio_data->base_addr));
	}
	mt8521p_gpio_probe(audio_data->dev);
	mt_afe_platform_init(audio_data->dev);
	afe_enable(1);
	audio_data->irq = mt_afe_get_afe_irq_id();
	if (request_irq(audio_data->irq, audio_isr, IRQF_TRIGGER_LOW, "Audio IRQ", (void *)audio_data)) {
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "IRQ get failed!");
	}

	audio_procfs_init(audio_data);

	for_each_available_child_of_node(pdev->dev.of_node, child) {
		ret = of_property_read_u32(child, "lla-type", &type);
		if (ret) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "%p type read returned %d", child, ret);
			continue;
		}
		audio_data->params[type].enable = 1;
		audio_data->params[type].lla_type = type;
		of_property_read_u32(child, "frames-per-second", &(audio_data->params[type].frames_per_second));
		of_property_read_u32(child, "channel-len", &(audio_data->params[type].channel_len));
		of_property_read_u32(child, "channel-mask", &(audio_data->params[type].channel_mask));
		of_property_read_u32(child, "channels-per-frame", &(audio_data->params[type].channels_per_frame));
		of_property_read_u32(child, "frames-per-buffer", &(audio_data->params[type].frames_per_buffer));
		of_property_read_u32(child, "buffer-cnt", &(audio_data->params[type].buffer_cnt));
		if (of_property_read_u32(child, "ring-cnt", &(audio_data->params[type].ring_cnt))) {
			audio_data->params[type].ring_cnt = 1;
		}
		if (of_property_read_u32(child, "virtual-fps", &(audio_data->params[type].virtual_fps))) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_DEBUG, "No virtual FPS for type %u.", type);
			audio_data->params[type].virtual_fps = audio_data->params[type].frames_per_second;
		}
		audio_data->params[type].invert_fs = of_property_read_bool(child, "invert-fs");
		audio_data->params[type].uses_mclk = of_property_read_bool(child, "uses-mclk");
		audio_data->params[type].sync_tx_rx = of_property_read_bool(child, "sync-tx-rx");
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_DEBUG, "type %u fps %u len %u mask %#x cpf %u fpb %u bufs %u rings %u",
				type,
				audio_data->params[type].frames_per_second,
				audio_data->params[type].channel_len,
				audio_data->params[type].channel_mask,
				audio_data->params[type].channels_per_frame,
				audio_data->params[type].frames_per_buffer,
				audio_data->params[type].buffer_cnt,
				audio_data->params[type].ring_cnt);
	}

	for (idx = 0; audio_init_funcs[idx]; idx++) {
		ret = audio_init_funcs[idx](audio_data->dev, audio_data->proc_dir);
		if (ret) {
			bb_log(BB_MOD_LLA, BB_LVL_ERR, "Failed to init device %d, error %d", idx, ret);
		}
	}

	bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Registered audio dispatcher");
	return ret;
}

static int audio_remove(struct platform_device *pdev)
{
	struct audio_mgr_priv *audio_data = platform_get_drvdata(pdev);
	struct audio_priv *cb_ptr = audio_cb_head;
	while (cb_ptr) {
		if (cb_ptr->exit) {
			cb_ptr->exit(cb_ptr);
		} else {
			bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_WARNING, "Device registered without exit function! May be leaking resources.");
		}
		cb_ptr = cb_ptr->next;
	}
	audio_procfs_exit(audio_data);
	free_irq(audio_data->irq, audio_data);
	afe_enable(0);
	mt_afe_platform_deinit(audio_data->dev);
	bb_log_dev(audio_data->dev, BB_MOD_LLA, BB_LVL_INFO, "Removed audio dispatcher");
	return 0;
}

static const struct of_device_id audio_ids[] = {
	{ .compatible = "sonos,mt8521p-lla-driver", },
	{ }
};

static struct platform_driver audio_driver = {
	.probe = audio_probe,
	.remove = audio_remove,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = audio_ids,
	},
};

int audio_init(void)
{
	return platform_driver_register(&audio_driver);
}

void audio_exit(void)
{
	platform_driver_unregister(&audio_driver);
}
