/*
 * Copyright (c) 2016-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */
#include <linux/version.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/ktime.h>
#include <linux/hrtimer.h>
#include <linux/dma-mapping.h>
#include <linux/proc_fs.h>
#include <linux/delay.h>
#include <linux/kthread.h>
#include <asm/pgtable.h>

#include "blackbox.h"
#include "sonos_device.h"
#include "lla.h"

#define PROCFS_DIR	"driver/lla"

static int devno = -1;

#define LLA_DEV_MSG_PRINT(ADEV, LEVEL, FORMAT, ARG...) bb_log(BB_MOD_LLA, LEVEL, "%s(%d): "FORMAT, ADEV->name, ADEV->type, ## ARG)
#define LLA_DEV_MSG_INFO(ADEV, FORMAT, ARG...) LLA_DEV_MSG_PRINT(ADEV, BB_LVL_INFO, FORMAT, ## ARG)
#define LLA_DEV_MSG_WARN(ADEV, FORMAT, ARG...) LLA_DEV_MSG_PRINT(ADEV, BB_LVL_WARNING, FORMAT, ## ARG)
#define LLA_DEV_MSG_ERROR(ADEV, FORMAT, ARG...) LLA_DEV_MSG_PRINT(ADEV, BB_LVL_ERR, FORMAT, ## ARG)
#define LLA_DEV_MSG_DBG(ADEV, FORMAT, ARG...) bb_log_dbg(BB_MOD_LLA, "%s(%d): "FORMAT, ADEV->name, ADEV->type, ## ARG)

#define ASSERT_ADEV_SET(ADEV) \
	if (ADEV == NULL) { \
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "%s() called without device association", __FUNCTION__); \
		return -EIO; \
	} \
	LLA_DEV_MSG_DBG(ADEV, "")

#if defined(SONOS_ARCH_ATTR_SOC_IS_MT8521P)
#include "mt8521p_audio.h"
#include "mt8521p_timer.h"

int (*lla_init_funcs[])(void) = {
	mtk_gpt_init,
	audio_init,
	NULL
};
void (*lla_exit_funcs[])(void) = {
	audio_exit,
	mtk_gpt_exit,
	NULL
};

#elif defined(SONOS_ARCH_ATTR_SOC_IS_IMX6)
#include "audmux.h"
#include "epit.h"
#include "esai.h"
#include "spdif.h"
#include "ssi.h"
#include "vcxo.h"

int (*lla_init_funcs[])(void) = {
	vcxo_init,
	epit_init,
	audmux_init,
	ssi_init,
	esai_init,
	imx_spdif_init,
	NULL
};
void (*lla_exit_funcs[])(void) = {
	imx_spdif_exit,
	esai_exit,
	audmux_exit,
	ssi_exit,
	epit_exit,
	vcxo_exit,
	NULL
};

#endif

struct lla_dev *lla_dev_list[LLA_DEV_TYPE_MAX];

struct cdev lla_chr_dev;

struct lla_hw_clk *hw_clock_dev;

u64 ioctl_profile_timestamp = 0;

struct proc_dir_entry *procfs_lla_dir = NULL;

struct task_struct *sync_watchdog_task = NULL;
int sync_count = 0;
#define SYNC_WATCHDOG_THRESHOLD	(1)

inline s32 lla_get_sync_offset(
	struct lla_dev *adev)
{
	return (s32)((s64)(adev->shared->buffers_complete) - (s64)(adev->sync_dev->shared->buffers_complete));
}

static int lla_sync_watchdog_thread(void *data)
{
	int curr_dev;

	while (!kthread_should_stop()) {
		msleep(1000);

		for (curr_dev = 0; curr_dev < LLA_DEV_TYPE_MAX; curr_dev++) {
			if (lla_dev_list[curr_dev] && IS_LLA_DEV_SYNC_PRIMARY(lla_dev_list[curr_dev])) {
				s32 offset = lla_get_sync_offset(lla_dev_list[curr_dev]);
				if ((offset < -SYNC_WATCHDOG_THRESHOLD) || (SYNC_WATCHDOG_THRESHOLD < offset)) {
					LLA_DEV_MSG_WARN(lla_dev_list[curr_dev], "sync divergence (offset=%d)!!", offset);
					lla_dev_list[curr_dev]->shared->flags |= LLA_SHARED_FLAG_ERROR;
				}
			}
		}
	}

	return 0;
}

static inline const char *type2str(enum lla_dev_type type)
{
	switch (type) {
		case LLA_DEV_TYPE_DAC:
			return "dac";
		case LLA_DEV_TYPE_MIC:
			return "mic";
		case LLA_DEV_TYPE_SPDIF_RX:
			return "spdif_rx";
		case LLA_DEV_TYPE_SPDIF_TX:
			return "spdif_tx";
		case LLA_DEV_TYPE_LINE_IN:
			return "line_in";
		case LLA_DEV_TYPE_UNKNOWN:
		case LLA_DEV_TYPE_MAX:
		default:
			return "UNKNOWN";
	}
	return "??";
}

static int procfs_lladev_show(struct seq_file *m, void *v)
{
	struct lla_dev *adev = m->private;
	const char *type_str = type2str(adev->type);

	seq_printf(m, "Name      = %s\n", adev->name);
	seq_printf(m, "Type      = %d (%s)\n", adev->type, type_str);
	seq_printf(m, "Ref Count = %d\n", adev->ref);

	seq_printf(m, "Sync\n");
	if (adev->sync_dev) {
		seq_printf(m, "\tRole              = %s\n", IS_LLA_DEV_SYNC_PRIMARY(adev) ? "primary" : "secondary");
		seq_printf(m, "\tPeer              = %d-%s\n", adev->sync_dev->type, type2str(adev->sync_dev->type));
		seq_printf(m, "\tOffset            = %d\n", lla_get_sync_offset(adev));
	} else {
		seq_printf(m, "\t<DISABLED>\n");
	}

	seq_printf(m, "Shared\n");
	if (adev->shared) {
		struct lla_shared *shared = adev->shared;

		seq_printf(m, "\tType              = %d (%s)\n", shared->type, type_str);

		seq_printf(m, "\tFlags             = 0x%x ( %s%s%s%s%s%s)\n",
			shared->flags,
			IS_LLA_SHARED_RX(shared) ? "RX " : "TX ",
			(shared->flags & LLA_SHARED_FLAG_NEED_COMMIT_TX) ? "NEED_COMMIT_TX " : "",
			(shared->flags & LLA_SHARED_FLAG_RUNNING) ? "RUNNING " : "",
			(shared->flags & LLA_SHARED_FLAG_ACTIVE) ? "ACTIVE " : "",
			(shared->flags & LLA_SHARED_FLAG_ERROR) ? "ERROR " : "",
			(shared->flags & LLA_SHARED_FLAG_CLK_COUNTS_UP) ? "CLK_COUNTS_UP" : "CLK_COUNTS_DOWN");

		seq_printf(m, "\tRing Count        = %d\n", shared->ring_cnt);
		seq_printf(m, "\tBuffer Length     = %d\n", shared->buffer_len);
		seq_printf(m, "\tBuffer Count      = %d (0x%08x)\n", shared->buffer_cnt, shared->buffers_to_index_mask);
		seq_printf(m, "\tBuffers Total     = %d\n", shared->buffers_total);
		seq_printf(m, "\tFrames/Buffer     = %d (1 << %d)\n", shared->frames_per_buffer, shared->buffers_to_frames_shift);
		seq_printf(m, "\tFrame Length      = %d\n", shared->frame_len);
		seq_printf(m, "\tChannels/Frame    = %d\n", shared->channels_per_frame);
		seq_printf(m, "\tChannel Mask      = 0x%08x\n", shared->channel_mask);
		seq_printf(m, "\tChannel Length    = %d\n", shared->channel_len);
		seq_printf(m, "\tSample Rate       = %d\n", shared->sample_rate);
		seq_printf(m, "\tBuffers Completed = %llu\n", shared->buffers_complete);
		seq_printf(m, "\tCompleted Time    = %u\n", shared->complete_time);
		seq_printf(m, "\tSMP Guard Time    = %u\n", shared->complete_time2);
		seq_printf(m, "\tPoll Number       = %llu\n", shared->poll_num);
		seq_printf(m, "\tTx Underflows     = %d\n", shared->tx_underflows);
		seq_printf(m, "\tTx Latency        = %d\n", shared->tx_latency);
		seq_printf(m, "\tHw Clock Rate     = %d\n", shared->hw_clock_rate);
		seq_printf(m, "\tHw Ticks/Frame    = %d\n", shared->hw_ticks_per_frame);
	}
	else {
		seq_printf(m, "\t<NOT ALLOCATED>\n");
	}

	return 0;
}

static int procfs_lladev_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_lladev_show, PDE_DATA(inode));
}

static const struct file_operations procfs_lladev_ops = {
	.owner = THIS_MODULE,
	.open = procfs_lladev_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int lla_open(struct inode *inode, struct file *filp)
{
	bb_log_dbg(BB_MOD_LLA, "");
	return 0;
}

static int lla_release(struct inode *inode, struct file *filp)
{
	struct lla_dev *adev = filp->private_data;
	int ret = 0;
	bb_log_dbg(BB_MOD_LLA, "");

	if (adev) {
		bb_log_dbg(BB_MOD_LLA, "adev=%s(%d)", adev->name, adev->type);
		adev->ref--;
		if (adev->ops->release) {
			ret = adev->ops->release(filp, adev->context);
		}
	}

	return ret;
}

static long lla_ioctl(struct file *filp, unsigned int num, unsigned long param)
{
	struct lla_dev *adev = filp->private_data;
	int ret = 0;

	bb_log_dbg(BB_MOD_LLA, "%s(cmd=%u, param=%lu)", __FUNCTION__, _IOC_NR(num), param);

	if ((num != LLA_IOCTL_SET_DEVICE) &&
	    (num != LLA_IOCTL_TEST_SYNC_DEV_ENABLE) &&
	    (num != LLA_IOCTL_TEST_SYNC_DEV_DISABLE)) {
		ASSERT_ADEV_SET(adev);
	}

	switch (num) {
		case LLA_IOCTL_SET_DEVICE:
			{
				enum lla_dev_type type = param;

				if (adev) {
					bb_log(BB_MOD_LLA, BB_LVL_ERR, "select device called more than once");
					return -EACCES;
				}
				if ((type <= LLA_DEV_TYPE_UNKNOWN) || (type >= LLA_DEV_TYPE_MAX)) {
					bb_log(BB_MOD_LLA, BB_LVL_ERR, "requested invalid lla device type (%d)", type);
					return -EINVAL;
				}
				if (lla_dev_list[type] == NULL) {
					bb_log(BB_MOD_LLA, BB_LVL_ERR, "requested unregistered lla device type (%d)", type);
					return -ENODEV;
				}
				if (IS_LLA_DEV_IN_USE(lla_dev_list[type])) {
					bb_log(BB_MOD_LLA, BB_LVL_ERR, "requested device type already in use (%d)", type);
					return -EBUSY;
				}

				filp->private_data = lla_dev_list[type];
				adev = lla_dev_list[type];
				adev->ref++;

				if (adev->ops->attach) {
					ret = adev->ops->attach(filp, adev->context);
				}

				LLA_DEV_MSG_DBG(adev, "attached");
			}
			break;

		case LLA_IOCTL_COMMIT_TX_BUFFER:
			if (adev->ops->commit) {
				ret = adev->ops->commit(param, adev->context);

				if ((ret == 0) && IS_LLA_DEV_SYNC_PRIMARY(adev) && adev->sync_dev->ops->commit) {
					ret = adev->sync_dev->ops->commit(param, adev->sync_dev->context);
				}
			}
			break;

		case LLA_IOCTL_GET_TIME:
			ret = lla_clock_read();
			break;

		case LLA_IOCTL_SET_TX_LATENCY:
			if (adev->shared->flags & LLA_SHARED_FLAG_RX) {
				LLA_DEV_MSG_ERROR(adev, "Tried to call SET_TX_LATENCY on an RX device.");
				ret = -EINVAL;
			} else {
				adev->shared->tx_latency = param;

				if (IS_LLA_DEV_SYNC_PRIMARY(adev)) {
					adev->sync_dev->shared->tx_latency = param;
				}
			}
			break;

		case LLA_IOCTL_SET_POLL_NUM:
			adev->shared->poll_num = param;

			if (IS_LLA_DEV_SYNC_PRIMARY(adev)) {
				adev->sync_dev->shared->poll_num = param;
			}
			break;

		case LLA_IOCTL_GET_CSB:
			if (adev->ops->get_csb) {
				struct lla_csb csb;
				memset(csb.data, 0x00, sizeof(csb.data));

				ret = adev->ops->get_csb(&csb, adev->context);

				if (!ret) {
					ret = copy_to_user((struct lla_csb*)param, &csb, sizeof(csb));
				}
			}
			else {
				ret = -EPERM;
			}
			break;

		case LLA_IOCTL_ACK_RESTART:
			adev->shared->flags &= ~LLA_SHARED_FLAG_RESTARTED;

			if (IS_LLA_DEV_SYNC_PRIMARY(adev)) {
				adev->sync_dev->shared->flags &= ~LLA_SHARED_FLAG_RESTARTED;
			}
			break;

		case LLA_IOCTL_TEST_SYNC_DEV_ENABLE:
		case LLA_IOCTL_TEST_SYNC_DEV_DISABLE:
			{
				enum lla_dev_type type = param;

				if ((type <= LLA_DEV_TYPE_UNKNOWN) || (type >= LLA_DEV_TYPE_MAX)) {
					bb_log(BB_MOD_LLA, BB_LVL_ERR, "requested invalid lla device type (%d)", type);
					return -EINVAL;
				}
				if (lla_dev_list[type] == NULL) {
					bb_log(BB_MOD_LLA, BB_LVL_ERR, "requested unregistered lla device type (%d)", type);
					return -ENODEV;
				}
				if (!IS_LLA_DEV_SYNC_SECONDARY(lla_dev_list[type])) {
					bb_log(BB_MOD_LLA, BB_LVL_ERR, "requested lla device is not sync secondary (%d)", type);
					return -EPERM;
				}

				lla_dev_list[type]->ref = (num == LLA_IOCTL_TEST_SYNC_DEV_DISABLE);
				bb_log(BB_MOD_LLA, BB_LVL_INFO, "test access to sync secondary device (%d) %s", type,
					lla_dev_list[type]->ref ? "disabled" : "enabled");
			}
			break;

		default:
			ret = -EPERM;
			break;
	}

	return ret;
}

static unsigned int lla_poll(struct file *filp, struct poll_table_struct *table)
{
	struct lla_dev *adev = filp->private_data;
	int err = -ENOSYS;
	ASSERT_ADEV_SET(adev);
	if (adev->ops->poll) {
		err = adev->ops->poll(filp, table, adev->context);
	}
	return err;
}

static int lla_mmap(struct file *filp, struct vm_area_struct *vma)
{
	struct lla_dev *adev = filp->private_data;
	unsigned long vma_offset = 0;
	int err = 0;
	ASSERT_ADEV_SET(adev);

	if ((adev->shared == NULL)  || (adev->shared->buffers.k == NULL)) {
		LLA_DEV_MSG_ERROR(adev, "no shared buffers to mmap");
		return -EIO;
	}

	if (vma->vm_pgoff != 0) {
		LLA_DEV_MSG_ERROR(adev, "invalid mmap page offset (%lu)", vma->vm_pgoff);
		return -EFAULT;
	}

	err = lla_map_to_user(
			vma,
			&(adev->shared->ref),
			adev->shared->ref_size,
			&vma_offset,
			false);
	if (err) {
		LLA_DEV_MSG_ERROR(adev, "failed to remap shared structure (%d)", err);
		return err;
	}

#ifndef SONOS_ARCH_ATTR_LACKS_HIGH_RES_CLOCK
	err = lla_map_to_user(
			vma,
			&(adev->shared->hw_clock),
			sizeof(u32),
			&vma_offset,
			true);
	if (err) {
		LLA_DEV_MSG_ERROR(adev, "failed to remap clock (%d)", err);
		return err;
	}
#endif

	err = lla_map_to_user(
			vma,
			&(adev->shared->buffers),
			adev->shared->buffers_total,
			&vma_offset,
			false);
	if (err) {
		LLA_DEV_MSG_ERROR(adev, "failed to remap audio buffers (%d)", err);
		return err;
	}

	return 0;
}

struct file_operations lla_fops =
{
	.open		= lla_open,
	.release	= lla_release,
	.unlocked_ioctl	= lla_ioctl,
	.poll		= lla_poll,
	.mmap		= lla_mmap,
};


int lla_map_to_user(
	struct vm_area_struct *vma,
	struct lla_ptr *ptr,
	unsigned long size,
	unsigned long *vma_offset,
	bool uncached)
{
	int err;
	unsigned long pfn;
	pgprot_t protections;

	if (*vma_offset == 0) {
		*vma_offset = vma->vm_start;
	}

	pfn = (ptr->p) >> PAGE_SHIFT;

	ptr->u = (void*)(*vma_offset + ((ptr->p) & ~PAGE_MASK));

	if (uncached) {
		protections = pgprot_noncached(vma->vm_page_prot);
	}
	else {
		protections = pgprot_writecombine(vma->vm_page_prot);
	}

	bb_log_dbg(BB_MOD_LLA, "remap ptr=%p/%p/%08lx pfn=%08lx size=%ld pages=%ld vma_offset=%08lx",
		ptr->k, ptr->u, ptr->p, pfn, size, (roundup(size, PAGE_SIZE) / PAGE_SIZE), *vma_offset);

	err = remap_pfn_range(vma, *vma_offset, pfn, size, protections);
	if (err) {
		return err;
	}

	*vma_offset += roundup(size, PAGE_SIZE);
	return 0;
}

inline static u32 _round_down_to_pow2(u32 val)
{
	val |= val >> 1;
	val |= val >> 2;
	val |= val >> 4;
	val |= val >> 8;
	val |= val >> 16;
	return val - (val >> 1);
}

inline static u32 _get_exp_of_pow2(u32 val)
{
	int num = 1;
	if (val == 0) return 32;
	if ((val & 0x0000FFFF) == 0) {
		num += 16;
		val = val >> 16;
	}
	if ((val & 0x000000FF) == 0) {
		num += 8;
		val = val >> 8;
	}
	if ((val & 0x0000000F) == 0) {
		num += 4;
		val = val >> 4;
	}
	if ((val & 0x00000003) == 0) {
		num += 2;
		val = val >> 2;
	}
	return num - (val & 1);
}

#ifdef SONOS_ARCH_ATTR_LACKS_HIGH_RES_CLOCK
#define LLA_LOW_RES_CLOCK_RATE		1014300
#define LLA_LOW_RES_CLOCK_DIV		(NSEC_PER_SEC / LLA_LOW_RES_CLOCK_RATE)
#endif

struct lla_shared *lla_allocate_shared_buffers(
		struct lla_dev *adev,
		u32 channel_len,
		u32 channels_per_frame,
		u32 frames_per_buffer,
		u32 buffer_cnt,
		u32 ring_cnt,
		struct lla_ptr *audio_buf)
{
	dma_addr_t phys_addr = 0;
	size_t alloc_size = round_up(sizeof(struct lla_shared), PAGE_SIZE);

	adev->shared = dma_alloc_coherent(adev->dev, alloc_size, &phys_addr, GFP_KERNEL);
	if (adev->shared == NULL) {
		LLA_DEV_MSG_ERROR(adev, "shared allocation failed (size = %zu)", alloc_size);
		return NULL;
	}
	adev->shared->ref.k = adev->shared;
	adev->shared->ref.p = phys_addr;
	adev->shared->ref_size = alloc_size;

	LLA_DEV_MSG_DBG(adev, "shared allocation complete (shared = %zub)", alloc_size);

	adev->shared->type			= adev->type;
	adev->shared->flags			= 0;
	adev->shared->channel_len		= channel_len;
	adev->shared->channels_per_frame	= channels_per_frame;
	adev->shared->frame_len			= adev->shared->channel_len * adev->shared->channels_per_frame;
	adev->shared->frames_per_buffer		= _round_down_to_pow2(frames_per_buffer);
	adev->shared->buffers_to_frames_shift	= _get_exp_of_pow2(adev->shared->frames_per_buffer);
	adev->shared->buffer_len		= adev->shared->frames_per_buffer * adev->shared->frame_len;
	adev->shared->buffer_cnt		= _round_down_to_pow2(buffer_cnt);
	adev->shared->buffers_complete		= adev->shared->buffer_cnt;
	adev->shared->buffer_start_offset	= 0;
	adev->shared->ring_cnt			= ring_cnt;
	adev->shared->buffers_to_index_mask	= adev->shared->buffer_cnt - 1;
	adev->shared->buffers_total		= adev->shared->buffer_len * adev->shared->buffer_cnt * adev->shared->ring_cnt;
	adev->shared->channel_mask		= (1 << channels_per_frame) - 1;

#ifdef SONOS_ARCH_ATTR_LACKS_HIGH_RES_CLOCK
	LLA_DEV_MSG_WARN(adev, "High-res time tracking is disabled! Using system clock for timekeeping.");
	adev->shared->hw_clock_rate		= LLA_LOW_RES_CLOCK_RATE;
	adev->shared->flags			|= LLA_SHARED_FLAG_CLK_COUNTS_UP;
#else
	if (!hw_clock_dev) {
		LLA_DEV_MSG_ERROR(adev, "No high-res timer has been registered! Cannot setup shared memory region.");
		goto alloc_error;
	}
	adev->shared->hw_clock.p		= hw_clock_dev->reg_addr;
	adev->shared->hw_clock.k		= NULL;
	adev->shared->hw_clock_rate		= hw_clock_dev->rate;
	if (hw_clock_dev->counts_up) {
		adev->shared->flags		|= LLA_SHARED_FLAG_CLK_COUNTS_UP;
	}
#endif

	if (audio_buf) {
		adev->shared->buffers.k = audio_buf->k;
		adev->shared->buffers.p = audio_buf->p;
		LLA_DEV_MSG_DBG(adev, "linking pre-allocated audio buffers");
	} else if (IS_LLA_DEV_SYNC_SECONDARY(adev)) {
		if ((adev->sync_dev->shared->channel_len != channel_len) ||
		    (adev->sync_dev->shared->channels_per_frame != channels_per_frame) ||
		    (adev->sync_dev->shared->frames_per_buffer != frames_per_buffer) ||
		    (adev->sync_dev->shared->buffer_cnt != buffer_cnt) ||
		    (adev->sync_dev->shared->ring_cnt != ring_cnt)) {
			LLA_DEV_MSG_ERROR(adev, "synced device audio properties mistmatch primary");
			goto alloc_error;
		}

		adev->shared->buffers.k = adev->sync_dev->shared->buffers.k;
		adev->shared->buffers.p = adev->sync_dev->shared->buffers.p;

		LLA_DEV_MSG_DBG(adev, "synced device linking primary audio buffers");
	} else {
		adev->shared->buffers.k = dma_alloc_coherent(NULL, adev->shared->buffers_total, (dma_addr_t*)&(adev->shared->buffers.p), GFP_KERNEL);
		if (adev->shared->buffers.k == NULL) {
			LLA_DEV_MSG_ERROR(adev, "audio buffers allocation failed (size = %u)", adev->shared->buffers_total);
			goto alloc_error;
		}

		LLA_DEV_MSG_DBG(adev, "audio buffer allocation complete (buffers = %ub)", adev->shared->buffers_total);
	}

	return adev->shared;
alloc_error:
	lla_free_shared_buffers(adev);
	return NULL;
}

void lla_free_shared_buffers(struct lla_dev *adev)
{
	if (adev->shared != NULL) {
		if (adev->shared->buffers.k != NULL) {
			dma_free_coherent(NULL, adev->shared->buffers_total, adev->shared->buffers.k, adev->shared->buffers.p);
		}
		dma_free_coherent(adev->dev, sizeof(struct lla_shared), adev->shared->ref.k, adev->shared->ref.p);
	}
	adev->shared = NULL;
}

void lla_update_sample_rate(
	struct lla_dev *adev,
	u32 sample_rate)
{
	if (sample_rate) {
		if (sample_rate != adev->shared->sample_rate) {

			adev->shared->hw_ticks_per_frame = adev->shared->hw_clock_rate / sample_rate;
		}
	}
	else {
		adev->shared->hw_ticks_per_frame = 0;
	}

	adev->shared->sample_rate = sample_rate;

	if (sample_rate > adev->shared->hw_clock_rate) {
		LLA_DEV_MSG_WARN(adev, "sample rate exceeds hw clock (%d > %d)", sample_rate, adev->shared->hw_ticks_per_frame);
		adev->shared->hw_clock_rate = 0;
		return;
	}
}

int lla_init_clk(
	struct lla_hw_clk *hwclk,
	const char *name,
	phys_addr_t reg_addr,
	u32 rate,
	u32 counts_up,
	struct lla_hw_clk_ops *ops,
	void *context)
{
	memset(hwclk, 0, sizeof(struct lla_hw_clk));
	hwclk->name = name;
	hwclk->reg_addr = reg_addr;
	hwclk->rate = rate;
	hwclk->counts_up = counts_up;
	hwclk->ops = ops;
	hwclk->context = context;

	bb_log_dbg(BB_MOD_LLA, "clock device %s initialized", hwclk->name);
	return 0;
}

int lla_register_clk(struct lla_hw_clk *hwclk)
{
	if (hw_clock_dev != NULL) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "clock device %s already registered", hw_clock_dev->name);
		return -EBUSY;
	}

	hw_clock_dev = hwclk;

	return 0;
}

void lla_unregister_clk(struct lla_hw_clk *hwclk)
{
	int idx;

	for (idx = 0; idx < LLA_DEV_TYPE_MAX; idx++) {
		if (lla_dev_list[idx]) {
			bb_log(BB_MOD_LLA, BB_LVL_WARNING, "unregister clock device %s while device drivers active", hwclk->name);
			break;
		}
	}

	if (hw_clock_dev != hwclk) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "clock device %s is not registered", hwclk->name);
		return;
	}

	hw_clock_dev = NULL;
	bb_log_dbg(BB_MOD_LLA, "clock device %s unregistered", hwclk->name);
}

u32 lla_clock_read(void)
{
#ifdef SONOS_ARCH_ATTR_LACKS_HIGH_RES_CLOCK
	u64 result = ktime_to_ns(ktime_get());
	u32 rem = do_div(result, LLA_LOW_RES_CLOCK_DIV);
	(void)rem;
	return (u32)(result);
#else
	if (hw_clock_dev && hw_clock_dev->ops->read) {
		return hw_clock_dev->ops->read(hw_clock_dev->context);
	} else {
		return (-1);
	}
#endif
}

int lla_init_dev(
		struct lla_dev *adev,
		struct device *dev,
		const char *name,
		enum lla_dev_type type,
		struct lla_ops *ops,
		void *context)
{
	if ((type < 0) || (type >= LLA_DEV_TYPE_MAX)) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "invalid lla device type (%d)", type);
		return -ENODEV;
	}

	memset(adev, 0, sizeof(struct lla_dev));
	adev->context = context;
	adev->name = name;
	adev->type = type;
	adev->ops = ops;
	adev->dev = dev;

	adev->sync_dev = NULL;
	adev->is_primary = 0;

	LLA_DEV_MSG_DBG(adev, "device initialized");
	return 0;
}

int lla_sync_to_dev(
	struct lla_dev *adev,
	enum lla_dev_type primary)
{
	if ((primary < 0) || (primary >= LLA_DEV_TYPE_MAX)) {
		LLA_DEV_MSG_ERROR(adev, "invalid lla device type for sync primary (%d)", primary);
		return -ENODEV;
	}

	if (lla_dev_list[primary] == NULL) {
		LLA_DEV_MSG_ERROR(adev, "unregistered lla device type for sync primary (%d)", primary);
		return -ENODEV;
	}

	if (IS_LLA_SHARED_RX(lla_dev_list[primary]->shared)) {
		LLA_DEV_MSG_ERROR(adev, "cannot sync to rx device (%d)", primary);
		return -EIO;
	}

	lla_dev_list[primary]->sync_dev = adev;
	lla_dev_list[primary]->is_primary = 1;

	adev->sync_dev = lla_dev_list[primary];
	adev->is_primary = 0;

	adev->ref++;

	sync_count++;

	LLA_DEV_MSG_DBG(adev, "device synced with %s(%d)", adev->sync_dev->name, adev->sync_dev->type);
	return 0;
}

int lla_register_dev(struct lla_dev *adev)
{
	if (lla_dev_list[adev->type] != NULL) {
		LLA_DEV_MSG_ERROR(adev, "device type already registered");
		return -EBUSY;
	}

	lla_dev_list[adev->type] = adev;

	snprintf(adev->filename, sizeof(adev->filename), "%d-%s", adev->type, type2str(adev->type));
	adev->filename[sizeof(adev->filename) - 1] = '\0';

	adev->procfs = proc_create_data(adev->filename, (S_IRUSR | S_IRGRP | S_IROTH), procfs_lla_dir, &procfs_lladev_ops, adev);
	if (!adev->procfs) {
		LLA_DEV_MSG_WARN(adev, "failed to create procfs file %s/%s", PROCFS_DIR, adev->filename);
	}

	LLA_DEV_MSG_DBG(adev, "device registered");
	return 0;
}

void lla_unregister_dev(struct lla_dev *adev)
{
	if (IS_LLA_DEV_SYNC_PRIMARY(adev)) {
		LLA_DEV_MSG_WARN(adev, "unregister sync primary before secondary %s(%d)", adev->sync_dev->name, adev->sync_dev->type);
	}
	if (IS_LLA_DEV_SYNC_SECONDARY(adev)) {
		struct lla_dev *primary = adev->sync_dev;

		adev->shared->buffers.k = NULL;
		adev->shared->buffers.p = 0ul;

		adev->sync_dev->sync_dev = NULL;
		adev->sync_dev = NULL;
		adev->ref = 0;

		LLA_DEV_MSG_DBG(adev, "device unsynced with %s(%d)", primary->name, primary->type);
		(void)primary;
	}

	if (IS_LLA_DEV_IN_USE(adev)) {
		LLA_DEV_MSG_WARN(adev, "unregister while userspace client still associated");
	}

	lla_dev_list[adev->type] = NULL;

	if (adev->procfs) {
		remove_proc_entry(adev->filename, procfs_lla_dir);
	}

	LLA_DEV_MSG_DBG(adev, "device unregistered");
}

int __init lla_init_module(void)
{
	int err;
	int idx;
	struct device *class_dev = NULL;

	 devno = MKDEV(LLA_MAJOR_AUDIODEV, 0);

#ifdef CONFIG_DEVTMPFS
	err = alloc_chrdev_region(&devno, 0, 1, LLA_NAME);
#else
	err = register_chrdev_region(devno, 1, LLA_NAME);
#endif
	if (err) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "unable to register LLA character nodes (%d)", err);
		return err;
	}

	cdev_init(&lla_chr_dev, &lla_fops);
	lla_chr_dev.owner = THIS_MODULE;
	err = cdev_add(&lla_chr_dev, devno, 1);
	if (err) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "failed to add lla chrdev %d:0 (%d)", MAJOR(devno), err);
		goto out_cdev_err;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, LLA_NAME);
	if (IS_ERR(class_dev)) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "Error creating lla class.");
		cdev_del(&lla_chr_dev);
		err = PTR_ERR(class_dev);
		goto out_cdev_err;
	}

	procfs_lla_dir = proc_mkdir(PROCFS_DIR, NULL);
	if (!procfs_lla_dir) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs dir %s", PROCFS_DIR);
		goto out_proc_err;
	}

	hw_clock_dev = NULL;
	for (idx = 0; idx < LLA_DEV_TYPE_MAX; idx++) {
		lla_dev_list[idx] = NULL;
	}

	for(idx = 0; lla_init_funcs[idx]; idx++) {
		err = lla_init_funcs[idx]();
		if (err) {
			bb_log(BB_MOD_LLA, BB_LVL_ERR, "failed to init device type %d", idx);
			goto out_err;
		}
	}

	if (sync_count) {
		sync_watchdog_task = kthread_run(&lla_sync_watchdog_thread, NULL, "LLA Sync Watchdog");
		if (IS_ERR(sync_watchdog_task)) {
			bb_log(BB_MOD_LLA, BB_LVL_ERR, "sync watchdog did not start (%ld)", PTR_ERR(sync_watchdog_task));
			goto out_err;
		}
	}

	bb_log(BB_MOD_LLA, BB_LVL_INFO, "Registered for %d:0 (%d device types)", MAJOR(devno), LLA_DEV_TYPE_MAX);
	return 0;

out_err:
	for (; idx >= 0; --idx) {
		if (lla_exit_funcs[idx]) {
			lla_exit_funcs[idx]();
		}
	}
	proc_remove(procfs_lla_dir);
out_proc_err:
	sonos_device_destroy(devno);
	cdev_del(&lla_chr_dev);
out_cdev_err:
	unregister_chrdev_region(devno, 1);
	return err;
}
module_init(lla_init_module);

void __exit lla_exit_module(void)
{
	int idx = 0;

	if (sync_watchdog_task) {
		kthread_stop(sync_watchdog_task);
	}

	for(; lla_exit_funcs[idx]; idx++) {
		lla_exit_funcs[idx]();
	}
	proc_remove(procfs_lla_dir);
	sonos_device_destroy(devno);
	cdev_del(&lla_chr_dev);
	unregister_chrdev_region(devno, 1);
}
module_exit(lla_exit_module);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Low-latency Audio Driver");
MODULE_LICENSE("GPL");
