/*
 * Copyright (c) 2016-2017, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for VCXO with IMX PWM pin
 */

#include "linux/module.h"
#include <linux/moduleparam.h>
#include <asm/delay.h>
#include <asm/uaccess.h>
#include <linux/cdev.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/of_platform.h>
#include <linux/pwm.h>

#include "blackbox.h"
#include "vcxo_dev.h"

static struct pwm_device *sonos_vcxo;

static uint32_t vcxo_frequency;
static uint32_t vcxo_speed;

struct cdev vcxo_chr_dev;
static int devno = -1;

static int vcxo_open(struct inode *inodep, struct file *filp);
static int vcxo_release(struct inode *inodep, struct file *filp);
static long vcxo_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);
static ssize_t vcxo_read(struct file *filp, char *data, size_t len, loff_t *ppos);
static ssize_t vcxo_write(struct file *file, const char *data, size_t len, loff_t *ppos);

static int vcxo_proc_init(void);

struct file_operations vcxo_fops =
{
	.open		= vcxo_open,
	.release	= vcxo_release,
	.unlocked_ioctl = vcxo_ioctl,
	.read		= vcxo_read,
	.write		= vcxo_write,
};

static int vcxo_update(int freq, int speed)
{
	int period_ns = 1000000000 / freq;
	int duty_ns = (period_ns * (VCXO_MAX_RATE - speed)) / VCXO_MAX_RATE;
	int error = pwm_config(sonos_vcxo, duty_ns, period_ns);
	if (error) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "VCXO config error %d", error);
		return error;
	}
	vcxo_frequency = freq;
	vcxo_speed = speed;
	return 0;
}

static int vcxo_probe(struct platform_device *pdev)
{
	int error;
	struct device_node *node = pdev->dev.of_node;

	devno = MKDEV(VCXO_MAJOR_NUMBER, 0);

	bb_log(BB_MOD_LLA, BB_LVL_INFO, "Starting Sonos PWM VCXO Control driver");

#ifdef CONFIG_DEVTMPFS
	error = alloc_chrdev_region(&devno, 0, 1, VCXO_DEVICE_NAME);
#else
	error = register_chrdev_region(devno, 1, VCXO_DEVICE_NAME);
#endif
	if (error) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "VCXO: Couldn't register character device region (%d).", error);
		return error;
	}

	cdev_init(&vcxo_chr_dev, &vcxo_fops);
	vcxo_chr_dev.owner = THIS_MODULE;
	error = cdev_add(&vcxo_chr_dev, devno, 1);
	if (error) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "VCXO: Couldn't add character device (%d).", error);
		goto vcxo_chrdev;
	}

	vcxo_proc_init();

	sonos_vcxo = of_pwm_get(node, "vcxo");
	if (IS_ERR(sonos_vcxo)) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "of_pwm_get(vcxo) returned %ld", PTR_ERR(sonos_vcxo));
		return PTR_ERR(sonos_vcxo);
	}
	error = pwm_enable(sonos_vcxo);
	if (error) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "VCXO enable error %d", error);
		return error;
	}
	return vcxo_update(VCXO_FREQUENCY, VCXO_NOMINAL_RATE);

vcxo_chrdev:
	unregister_chrdev_region(devno, 1);
	return error;
}

static ssize_t vcxo_proc_write(struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	char *keyword;
	s32 result;

	if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';

		keyword = "speed=";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			int speed = 0, ret = 0;
			char *start = (char *)buf + strlen(keyword);
			if ((ret = kstrtoint(start, 0, &speed))) {
				bb_log(BB_MOD_LLA, BB_LVL_WARNING, "%s: kstrtoint value parse failed with error %d", __func__, ret);
			} else if ((speed <= 0) || (speed >= VCXO_MAX_RATE)) {
				bb_log(BB_MOD_LLA, BB_LVL_WARNING, "invalid speed %d", speed);
			} else {
				vcxo_update(vcxo_frequency, speed);
			}
			return count;
		}
		keyword = "frequency=";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			int val = 0, ret = 0;
			char *start = (char *)buf + strlen(keyword);
			if ((ret = kstrtoint(start, 0, &val))) {
				bb_log(BB_MOD_LLA, BB_LVL_WARNING, "%s: kstrtoint value parse failed with error %d", __func__, ret);
			} else if ((val <= 1) || (val >= 50000)) {
				bb_log(BB_MOD_LLA, BB_LVL_WARNING, "invalid frequency %d", val);
			} else {
				vcxo_update(val, vcxo_speed);
			}
			return count;
		}
		result = count;
	}
	return result;
}

static int vcxo_show(struct seq_file *m, void *v)
{
	seq_printf(m, "VCXO speed %5d\n", vcxo_speed);
	seq_printf(m, "VCXO max   %5d\n", VCXO_MAX_RATE);
	seq_printf(m, "VCXO frequency %d\n", vcxo_frequency);
	return 0;
}

const static struct seq_operations vcxo_op = {
	.show		= vcxo_show
};

static int vcxo_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, vcxo_show, PDE_DATA(inode));
}

static struct file_operations vcxo_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= vcxo_proc_open,
	.write		= vcxo_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define VCXO_PROCFS_FILE "driver/vcxo"

static int vcxo_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create(VCXO_PROCFS_FILE, 0666, NULL, &vcxo_proc_ops);
	if (!entry)
		return -EIO;

	return 0;
}

static void vcxo_proc_remove(void)
{
	remove_proc_entry(VCXO_PROCFS_FILE, NULL);
}

static int vcxo_remove(struct platform_device *pdev)
{
	vcxo_proc_remove();
	pwm_put(sonos_vcxo);
	cdev_del(&vcxo_chr_dev);
	unregister_chrdev_region(devno, 1);
	return 0;
}

const static struct of_device_id vcxo_ids[] = {
	{ .compatible = "sonos,pwm-vcxo", },
	{}
};

static struct platform_driver vcxo_driver = {
	.probe = vcxo_probe,
	.remove = vcxo_remove,
	.driver = {
		.name = "Sonos PWM VCXO Driver",
		.owner = THIS_MODULE,
		.of_match_table = vcxo_ids,
	},
};

static int vcxo_open(struct inode *inodep, struct file *filp)
{
	return 0;
}

static int vcxo_release(struct inode *inodep, struct file *filp)
{
	return 0;
}

static long vcxo_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	long ret = 0;

	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = VCXO_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case 1:
	{
		uint32_t max = VCXO_MAX_RATE;
		if (copy_to_user((uint32_t *)arg, &max, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	default:
		bb_log(BB_MOD_LLA, BB_LVL_WARNING, "Unrecognized IOCTL %d", _IOC_NR(cmd));
		return -EINVAL;
	}
	return ret;
}

static ssize_t vcxo_read(struct file *filp, char *data, size_t len, loff_t *ppos)
{
	if (len < sizeof(vcxo_speed)) {
		return -EINVAL;
	}

	if (copy_to_user(data, &vcxo_speed, sizeof(vcxo_speed))) {
		return -EACCES;
	}
	return sizeof(vcxo_speed);
}

static ssize_t vcxo_write(struct file *file, const char *data, size_t len, loff_t *ppos)
{
	int error;
	uint32_t speed;

	if (len < sizeof(vcxo_speed)) {
		return -EINVAL;
	}
	if (copy_from_user(&speed, data, sizeof(speed))) {
		return -EACCES;
	}
	if (speed > VCXO_MAX_RATE) {
		bb_log(BB_MOD_LLA, BB_LVL_WARNING, "invalid VCXO %d, max %d", speed, VCXO_MAX_RATE);
		return -EINVAL;
	}
	error = vcxo_update(vcxo_frequency, speed);
	if (error) {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "VCXO_SET_VCXO IOCTL failed with %d.", error);
		return -EINVAL;
	}
	return sizeof(vcxo_speed);
}

int vcxo_init(void)
{
	int error;

	error = platform_driver_register(&vcxo_driver);

	return error;
}

void vcxo_exit(void)
{
	platform_driver_unregister(&vcxo_driver);
}
