/*
 * Copyright (c) 2014-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Copyright 2008-2013 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/version.h>
#include <linux/clk.h>
#include <linux/wait.h>
#include <linux/dmaengine.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/kernel.h>
#include <linux/dma-mapping.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,9,0)
#include <linux/busfreq-imx6.h>
#endif
#include <linux/ktime.h>
#include <linux/hrtimer.h>
#include <linux/math64.h>
#include <asm/barrier.h>

#include "blackbox.h"
#include "lla.h"
#include "ssi.h"

#define DRV_NAME		"ssi"

struct ssi_regs {
	u32 stx0;
	u32 stx1;
	u32 srx0;
	u32 srx1;
	u32 scr;
	u32 sisr;
	u32 sier;
	u32 stcr;
	u32 srcr;
	u32 stccr;
	u32 srccr;
	u32 sfcsr;
	u32 str;
	u32 sor;
	u32 sacnt;
	u32 sacadd;
	u32 sacdat;
	u32 satag;
	u32 stmsk;
	u32 srmsk;
	u32 saccst;
	u32 saccen;
	u32 saccdis;
};

struct ssi_dma {
	struct dma_chan			*chan;
	struct dma_async_tx_descriptor	*desc;
	dma_cookie_t			cookie;
	wait_queue_head_t		poll_event;
};

struct ssi_params {
	u32 enable;
	u32 lla_type;
	u32 frames_per_second;
	u32 channel_len;
	u32 channel_mask;
	u32 channels_per_frame;
	u32 frames_per_buffer;
	u32 buffer_cnt;
	struct _div {
		u32 div2;
		u32 sr;
		u32 pm;
	} dividers;
	u32 watermark_words;
	u8 invert_fs;
	u8 tdm_master;
	u8 i2s_mode;
	u8 right_align;
};

#define SSI_DEV_NAME_MAX	16

struct ssi {
	u32				dev_id;
	char				dev_name[SSI_DEV_NAME_MAX];

	struct ssi_regs __iomem	*regs;
	struct resource			*res;
	int				irq;

	struct clk			*ssi_clk;
	struct clk			*baud_clk;

	struct ssi_params		rx_params;
	struct ssi_dma			rx_dma;
	struct lla_dev			rx_lla_dev;
	struct lla_shared		*rx_shared;

	struct timer_list		dma_watchdog;

	struct _ssi_procfs {
		char			dir_name[32];
		struct proc_dir_entry	*dir;
		struct proc_dir_entry	*regs;
		struct proc_dir_entry	*irqs;
		struct proc_dir_entry	*dma;
		struct proc_dir_entry	*snoop;
		struct proc_dir_entry	*fail;
	} procfs;

	struct _ssi_stats {
		u32			irq_overrun0;
		u32			irq_overrun1;

		u64			rx_start;
		u64			rx_stop;
	} stats;

	struct platform_device		*pdev;
	struct device			*dev;
};

#define PROCFS_PERM_READ	(S_IRUSR | S_IRGRP | S_IROTH)
#define PROCFS_PERM_WRITE	(S_IWUSR | S_IWGRP | S_IWOTH)

#define PROCFS_DIR		"driver/"DRV_NAME
#define PROCFS_REGS_FILE	"regs"
#define PROCFS_IRQS_FILE	"irqs"
#define PROCFS_DMA_FILE		"dma"
#define PROCFS_SNOOP_FILE	"snoop"
#define PROCFS_FAIL_FILE	"fail"

#define DMA_WATCHDOG_DURATION	(HZ)

#if LINUX_VERSION_CODE > KERNEL_VERSION(4,9,0)
enum bus_freq_mode {
	BUS_FREQ_HIGH,
	BUS_FREQ_MED,
	BUS_FREQ_AUDIO,
	BUS_FREQ_LOW,
};
#endif

#ifdef IMX_SSI_REG_DEBUG

static inline u32 _read_n_print(const char *func, int line, void *reg_addr, const char *reg_name)
{
	u32 val = readl(reg_addr);
	printk("[READ]  %08x <-- %-5s (%s:%d)\n", val, reg_name, func, line);
	return val;
}

#define SSI_READ(R) _read_n_print(__FUNCTION__, __LINE__, &(ssi->regs->R), #R)

#define SSI_WRITE(V, R) \
	do { \
		printk("[WRITE] %08x --> %-5s (%s:%d)\n", V, #R, __FUNCTION__, __LINE__); \
		writel(V, &(ssi->regs->R)); \
	} while (0)

#else
#define SSI_READ(R)		readl(&(ssi->regs->R))
#define SSI_WRITE(V, R)		writel(V, &(ssi->regs->R))
#endif

#ifdef IMX_SSI_DUMP
#define SSI_REG_PRINT(R) \
	pr_info("SSI%d_%-8s 0x%08x\n", ssi->dev_id, #R, readl(&(ssi->regs->R)));

#define SSI_DUMP() \
	do { \
		pr_info("dump @ %s:%d\n", __FUNCTION__, __LINE__); \
		SSI_REG_PRINT(scr); \
		SSI_REG_PRINT(sisr); \
		SSI_REG_PRINT(sier); \
		SSI_REG_PRINT(stcr); \
		SSI_REG_PRINT(srcr); \
		SSI_REG_PRINT(stccr); \
		SSI_REG_PRINT(srccr); \
		SSI_REG_PRINT(sfcsr); \
		SSI_REG_PRINT(str); \
		SSI_REG_PRINT(sor); \
		SSI_REG_PRINT(sacnt); \
		SSI_REG_PRINT(sacadd); \
		SSI_REG_PRINT(sacdat); \
		SSI_REG_PRINT(satag); \
		SSI_REG_PRINT(stmsk); \
		SSI_REG_PRINT(srmsk); \
		SSI_REG_PRINT(saccst); \
		SSI_REG_PRINT(saccen); \
		SSI_REG_PRINT(saccdis); \
	} while (0)
#else
#define SSI_DUMP()
#endif

#ifdef SNOOP_RX

static int procfs_snoop_show(struct seq_file *m, void *v)
{
	struct ssi *ssi = m->private;
	seq_write(m, ssi->rx_shared->buffers.k, ssi->rx_shared->buffer_len);
	return 0;
}

static int procfs_snoop_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_snoop_show, PDE_DATA(inode));
}

static const struct file_operations procfs_snoop_ops = {
	.owner = THIS_MODULE,
	.open = procfs_snoop_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};
#endif

static void procfs_dma_show_single_stats(
		struct seq_file *m,
		const char *pfx,
		u64 start,
		u64 stop,
		struct lla_shared *shared)
{
	u64 elapsed = 0;
	u64 elapsed_sec;

	if (start) {
		if (!stop) {
			elapsed = ktime_to_ns(ktime_get()) - start;
		}
		else {
			elapsed = stop - start;
		}
	}
	elapsed_sec = div64_u64(elapsed, 1000000000);

	seq_printf(m, "%s DMAed Buffers    = %llu\n", pfx, shared->buffers_complete);
	seq_printf(m, "%s Total Bytes      = %llu\n", pfx, (shared->buffers_complete * shared->buffer_len));
	if (elapsed_sec) {
		seq_printf(m, "%s bytes/sec        = %llu\n", pfx, div64_u64(((shared->buffers_complete - 1) * shared->buffer_len), elapsed_sec));
	}
	else {
		seq_printf(m, "%s bytes/sec        = n/a\n", pfx);
	}
	seq_printf(m, "%s Duration (ns)    = %llu\n", pfx, elapsed);
	if (shared->buffers_complete > 1) {
		seq_printf(m, "%s ns/buffer        = %llu\n", pfx, div64_u64(elapsed, (shared->buffers_complete - 1)));
	}
	else {
		seq_printf(m, "%s ns/buffer        = n/a\n", pfx);
	}
}

static int procfs_dma_show(struct seq_file *m, void *v)
{
	struct ssi *ssi = m->private;
	procfs_dma_show_single_stats(m, "Rx", ssi->stats.rx_start, ssi->stats.rx_stop, ssi->rx_shared);

	return 0;
}

static int procfs_dma_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_dma_show, PDE_DATA(inode));
}

static const struct file_operations procfs_dma_ops = {
	.owner = THIS_MODULE,
	.open = procfs_dma_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int procfs_regs_show(struct seq_file *m, void *v)
{
	struct ssi *ssi = m->private;

	#define _SSI_PROCFS_REG_SHOW(reg) seq_printf(m, "[%03x] SSI%d_%-7s = %08x\n", (&(ssi->regs->reg) - (u32*)ssi->regs) * sizeof(u32), ssi->dev_id, #reg, SSI_READ(reg))

	seq_printf(m, "%08x-%08x\n", ssi->res->start, ssi->res->end);
	_SSI_PROCFS_REG_SHOW(scr);
	_SSI_PROCFS_REG_SHOW(sisr);
	_SSI_PROCFS_REG_SHOW(sier);
	_SSI_PROCFS_REG_SHOW(stcr);
	_SSI_PROCFS_REG_SHOW(srcr);
	_SSI_PROCFS_REG_SHOW(stccr);
	_SSI_PROCFS_REG_SHOW(srccr);
	_SSI_PROCFS_REG_SHOW(sfcsr);
	_SSI_PROCFS_REG_SHOW(str);
	_SSI_PROCFS_REG_SHOW(sor);
	_SSI_PROCFS_REG_SHOW(sacnt);
	_SSI_PROCFS_REG_SHOW(sacadd);
	_SSI_PROCFS_REG_SHOW(sacdat);
	_SSI_PROCFS_REG_SHOW(satag);
	_SSI_PROCFS_REG_SHOW(stmsk);
	_SSI_PROCFS_REG_SHOW(srmsk);
	_SSI_PROCFS_REG_SHOW(saccst);
	_SSI_PROCFS_REG_SHOW(saccen);
	_SSI_PROCFS_REG_SHOW(saccdis);

	return 0;
}

static int procfs_regs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_regs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_regs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_regs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int procfs_irqs_show(struct seq_file *m, void *v)
{
	struct ssi *ssi = m->private;

#define _IRQ_DISABLED_STR(reg, bit) (reg & bit) ? "" : "(disabled)"

	seq_printf(m, "IRQ #%d\n", ssi->irq);

	seq_printf(m, "RX 0 Overruns:        %lu\n", (unsigned long)ssi->stats.irq_overrun0);
	seq_printf(m, "RX 1 Overruns:        %lu\n", (unsigned long)ssi->stats.irq_overrun1);
	return 0;
}

static int procfs_irqs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_irqs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_irqs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_irqs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

#ifdef TEST_FAILURES

static int test_fail_rx_dma_hang = 0;

static int procfs_fail_show(struct seq_file *m, void *v)
{
	return 0;
}

static int procfs_fail_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_fail_show, PDE_DATA(inode));
}

static ssize_t procfs_fail_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	struct ssi *ssi = PDE_DATA(file_inode(file));
	char cmd[32];
	char *newline;
	ssize_t result = count;

	if ((count < 0) || (count >= sizeof(cmd))) {
		result = -EIO;
	}
	else if (copy_from_user(&cmd, buffer, count)) {
		result = -EFAULT;
	}
	else {
		cmd[count] = '\0';

		newline = strchr(cmd, '\n');
		if (newline != NULL) {
			*newline = '\0';
		}

		if (strcmp(cmd, "rx-dma-hang") == 0) {
			bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx DMA hang");
			test_fail_rx_dma_hang++;
		}

		if (strcmp(cmd, "rx-dma-stop") == 0) {
			bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx DMA stop");
			dmaengine_terminate_all(ssi->rx_dma.chan);
		}

		if (strcmp(cmd, "rx-lla-error") == 0) {
			bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx LLA error flag");
			ssi->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
			ssi->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		}
	}

	return result;
}

static const struct file_operations procfs_fail_ops = {
	.owner = THIS_MODULE,
	.open = procfs_fail_open,
	.read = seq_read,
	.write = procfs_fail_write,
	.llseek = seq_lseek,
	.release = single_release,
};
#endif

static void procfs_remove(struct ssi *ssi)
{
#ifdef TEST_FAILURES
	if (ssi->procfs.fail) {
		remove_proc_entry(PROCFS_FAIL_FILE, ssi->procfs.dir);
	}
#endif
#ifdef SNOOP_RX
	if (ssi->procfs.snoop) {
		remove_proc_entry(PROCFS_SNOOP_FILE, ssi->procfs.dir);
	}
#endif
	if (ssi->procfs.dma) {
		remove_proc_entry(PROCFS_DMA_FILE, ssi->procfs.dir);
	}
	if (ssi->procfs.regs) {
		remove_proc_entry(PROCFS_REGS_FILE, ssi->procfs.dir);
	}
	if (ssi->procfs.irqs) {
		remove_proc_entry(PROCFS_IRQS_FILE, ssi->procfs.dir);
	}
	if (ssi->procfs.dir) {
		remove_proc_entry(ssi->procfs.dir_name, NULL);
	}
}

static int procfs_init(struct ssi *ssi)
{
	snprintf(ssi->procfs.dir_name, sizeof(ssi->procfs.dir_name), PROCFS_DIR "%d", ssi->dev_id);

	ssi->procfs.dir = proc_mkdir(ssi->procfs.dir_name, NULL);
	if (!ssi->procfs.dir) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs dir %s", ssi->procfs.dir_name);
		goto failed_procfs;
	}

	ssi->procfs.regs = proc_create_data(PROCFS_REGS_FILE, PROCFS_PERM_READ, ssi->procfs.dir, &procfs_regs_ops, ssi);
	if (!ssi->procfs.regs) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_REGS_FILE);
		goto failed_procfs;
	}

	ssi->procfs.irqs = proc_create_data(PROCFS_IRQS_FILE, PROCFS_PERM_READ, ssi->procfs.dir, &procfs_irqs_ops, ssi);
	if (!ssi->procfs.irqs) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_IRQS_FILE);
		goto failed_procfs;
	}

	ssi->procfs.dma = proc_create_data(PROCFS_DMA_FILE, PROCFS_PERM_READ, ssi->procfs.dir, &procfs_dma_ops, ssi);
	if (!ssi->procfs.dma) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_DMA_FILE);
		goto failed_procfs;
	}

#ifdef SNOOP_RX
	ssi->procfs.snoop = proc_create_data(PROCFS_SNOOP_FILE, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), ssi->procfs.dir, &procfs_snoop_ops, ssi);
	if (!ssi->procfs.snoop) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_SNOOP_FILE);
		goto failed_procfs;
	}
#endif

#ifdef TEST_FAILURES
	ssi->procfs.fail = proc_create_data(PROCFS_FAIL_FILE, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), ssi->procfs.dir, &procfs_fail_ops, ssi);
	if (!ssi->procfs.fail) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_FAIL_FILE);
		goto failed_procfs;
	}
#endif

	return 0;
failed_procfs:
	procfs_remove(ssi);
	return -1;
}

static irqreturn_t ssi_isr(int irq, void *p)
{
	struct ssi *ssi = p;
	irqreturn_t status = IRQ_NONE;
	u32 val;

	val = SSI_READ(sisr);
	if (val & SSI_SISR_ROE0) {
		ssi->stats.irq_overrun0++;

		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: RX 0 Overrun (SFCSR dump: %08x)", SSI_READ(sfcsr));

		val = SSI_READ(scr);
		val &= ~SSI_SCR_RE & ~SSI_SCR_SSIEN;
		SSI_WRITE(val, scr);

		ssi->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		ssi->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		ssi->stats.rx_stop = ktime_to_ns(ktime_get());

		dmaengine_terminate_all(ssi->rx_dma.chan);

		status = IRQ_HANDLED;
	}
	if (val & SSI_SISR_ROE1) {
		ssi->stats.irq_overrun1++;

		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: RX 1 Overrun (SFCSR dump: %08x)", SSI_READ(sfcsr));

		val = SSI_READ(scr);
		val &= ~SSI_SCR_RE & ~SSI_SCR_SSIEN;
		SSI_WRITE(val, scr);

		ssi->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		ssi->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		ssi->stats.rx_stop = ktime_to_ns(ktime_get());

		dmaengine_terminate_all(ssi->rx_dma.chan);

		status = IRQ_HANDLED;
	}
	if (status != IRQ_HANDLED) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "I don't know what to do! %x", val);
		status = IRQ_HANDLED;
	}
	return status;
}

static void ssi_rx_init(struct ssi *ssi)
{
	u32 reg, scr_val = 0, sier_val = 0, srcr_val = 0, srccr_val = 0, sfcsr_val = 0;

	reg = SSI_READ(scr);
	reg &= ~SSI_SCR_SSIEN;
	SSI_WRITE(reg, scr);

	scr_val |= SSI_SCR_TCH_EN
		| SSI_SCR_NET;
	if (ssi->rx_params.i2s_mode && ssi->rx_params.tdm_master) {
		scr_val |= SSI_SCR_I2S_MODE_MSTR;
	}
	SSI_WRITE(scr_val, scr);
	sier_val |= SSI_SIER_RDMAE | SSI_SIER_RIE
		| SSI_SIER_ROE1_EN | SSI_SIER_ROE0_EN;
	SSI_WRITE(sier_val, sier);
	srcr_val |= SSI_SRCR_RXEXT
		| SSI_SRCR_RXBIT0
		| SSI_SRCR_RFEN1
		| SSI_SRCR_RFEN0
		| SSI_SRCR_RSCKP;
	if (ssi->rx_params.tdm_master) {
		srcr_val |= SSI_SRCR_RFDIR | SSI_SRCR_RXDIR;
	}
	if (ssi->rx_params.i2s_mode) {
		srcr_val |= SSI_SRCR_REFS | SSI_SRCR_RFSI;
	}
	SSI_WRITE(srcr_val, srcr);
	srccr_val |= SSI_SRCCR_WL(24)
		| SSI_SRCCR_DC(ssi->rx_params.channels_per_frame - 1)
		| SSI_SRCCR_PM(ssi->rx_params.dividers.pm);
	if (ssi->rx_params.dividers.div2) {
		srccr_val |= SSI_SRCCR_DIV2;
	}
	if (ssi->rx_params.dividers.sr) {
		srccr_val |= SSI_SRCCR_PSR;
	}
	SSI_WRITE(srccr_val, srccr);
	sfcsr_val |= SSI_SFCSR_RFWM0(ssi->rx_params.watermark_words)
		| SSI_SFCSR_RFWM1(ssi->rx_params.watermark_words);
	SSI_WRITE(sfcsr_val, sfcsr);

	reg = SSI_READ(scr);
	reg |= SSI_SCR_SSIEN;
	SSI_WRITE(reg, scr);

	SSI_WRITE(~(ssi->rx_params.channel_mask), srmsk);

	reg = SSI_READ(sfcsr);
	if ((reg & SSI_SFCSR_RFCNT1(0xf)) || (reg & SSI_SFCSR_RFCNT0(0xf))) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "Data in RX FIFO! SFCSR: %x", reg);
	}
	reg = SSI_READ(sor);
	reg |= SSI_SOR_RX_CLR;
	SSI_WRITE(reg, sor);

	bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_DEBUG, "SSI%d init complete.", ssi->dev_id);
}

static unsigned int lla_ssi_rx_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	unsigned int mask = 0;
	struct ssi *ssi = context;

	poll_wait(filp, &(ssi->rx_dma.poll_event), table);

	if (ssi->rx_shared->poll_num < ssi->rx_shared->buffers_complete) {
		mask |= POLLIN | POLLRDNORM;
	}

	return mask;
}

static struct lla_ops lla_rx_ops = {
	.poll	= lla_ssi_rx_poll,
};

static int lla_rx_setup(struct ssi *ssi)
{
	if (lla_init_dev(&(ssi->rx_lla_dev), ssi->dev,
				  (const char *)&(ssi->dev_name),
				  ssi->rx_params.lla_type,
				  &lla_rx_ops, ssi))
	{
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "RX LLA init failed.");
		return -1;
	}

	ssi->rx_shared = lla_allocate_shared_buffers(
					&(ssi->rx_lla_dev),
					ssi->rx_params.channel_len,
					ssi->rx_params.channels_per_frame,
					ssi->rx_params.frames_per_buffer,
					ssi->rx_params.buffer_cnt, 1, NULL);
	if (ssi->rx_shared == NULL) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "RX LLA buffer alloc failed.");
		return -1;
	}

	ssi->rx_shared->flags |= LLA_SHARED_FLAG_RX;
	ssi->rx_shared->channel_mask = ssi->rx_params.channel_mask;

	lla_update_sample_rate(&(ssi->rx_lla_dev), ssi->rx_params.frames_per_second);

	bb_log_dbg_dev(ssi->dev, BB_MOD_LLA, "RX LLA setup complete.");
	return 0;
}

static void ssi_rx_dma_watchdog(unsigned long param)
{
	struct ssi *ssi = (struct ssi*)param;

	if (ssi->rx_shared->buffers_complete == 0) {
		ssi->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		ssi->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "ASSERT: %s DMA failed to start!", ssi->dev_name);
	}
}

static void ssi_rx_dma_callback(void *param)
{
	struct ssi *ssi = param;
	u32 buffer_idx;
	dma_addr_t buffer_p;
	u32 completion_time;

#ifdef TEST_FAILURES
	if (test_fail_rx_dma_hang) {
		mdelay(((ssi->rx_shared->buffer_cnt + 1) * ssi->rx_shared->frames_per_buffer) /
			(ssi->rx_params.frames_per_second / 1000));
		test_fail_rx_dma_hang = 0;
	}
#endif

	if (!(ssi->rx_shared->flags & LLA_SHARED_FLAG_RUNNING)) {
		ssi->rx_shared->flags |= LLA_SHARED_FLAG_RUNNING;
		ssi->stats.rx_start = ktime_to_ns(ktime_get());
	}

	buffer_idx = LLA_BUFFER_NUM_TO_IDX(ssi->rx_shared, ssi->rx_shared->buffers_complete);
	buffer_p = ssi->rx_shared->buffers.p + (buffer_idx * ssi->rx_shared->buffer_len);


	completion_time = lla_clock_read();
	ssi->rx_shared->complete_time = completion_time;
	dmb();

	ssi->rx_shared->buffers_complete++;
	dmb();
	ssi->rx_shared->complete_time2 = completion_time;

	wake_up_interruptible(&(ssi->rx_dma.poll_event));
}

static int ssi_rx_dma_setup(struct ssi *ssi)
{
	int ret;
	struct dma_slave_config slave_config = {
		.direction = DMA_DEV_TO_MEM,
		.src_addr = (ssi->res->start + SSI_SRX0),
		.src_addr_width = DMA_SLAVE_BUSWIDTH_4_BYTES,
		.src_maxburst = ssi->rx_params.watermark_words,
	};

	init_waitqueue_head(&(ssi->rx_dma.poll_event));

	ssi->rx_dma.chan = dma_request_slave_channel(ssi->dev, "rx");
	if (ssi->rx_dma.chan == NULL) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "RX DMA request failed.");
		return -1;
	}

	ret = dmaengine_slave_config(ssi->rx_dma.chan, &slave_config);
	if (ret) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to configure DMA engine for RX (%d).", ret);
		return -1;
	}

	ssi->rx_dma.desc = dmaengine_prep_dma_cyclic(
				ssi->rx_dma.chan,
				ssi->rx_shared->buffers.p,
				ssi->rx_shared->buffers_total,
				ssi->rx_shared->buffer_len,
				slave_config.direction,
				0);
	if (!ssi->rx_dma.desc) {
		bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to prepare cyclic RX DMA.");
		return -1;
	}

	ssi->rx_dma.desc->callback = ssi_rx_dma_callback;
	ssi->rx_dma.desc->callback_param = (void *)ssi;
	ssi->rx_dma.cookie = dmaengine_submit(ssi->rx_dma.desc);

	dma_async_issue_pending(ssi->rx_dma.chan);

	bb_log_dbg_dev(ssi->dev, BB_MOD_LLA, "SSI%d RX DMA setup complete.", ssi->dev_id);
	return 0;
}

static void ssi_rx_start(struct ssi *ssi)
{
	u32 reg = SSI_READ(scr);
	reg |= SSI_SCR_RE;
	SSI_WRITE(reg, scr);

	ssi->stats.rx_start = 0;
	ssi->stats.rx_stop = 0;

	lla_register_dev(&(ssi->rx_lla_dev));

	bb_log_dev(ssi->dev, BB_MOD_LLA, BB_LVL_DEBUG, "RX Running");
}

static int ssi_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct device_node *child;
	struct ssi *ssi;
	int ret = 0;

	ssi = devm_kzalloc(&pdev->dev, sizeof(*ssi), GFP_KERNEL);
	if (!ssi) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "mem allocation failed");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, ssi);
	ssi->pdev = pdev;
	ssi->dev = &(pdev->dev);

	if (of_property_read_u32(np, "dev-id", &(ssi->dev_id))) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Could not get device ID.");
		return -ENODEV;
	}
	snprintf((char *)&(ssi->dev_name), SSI_DEV_NAME_MAX, DRV_NAME"%d", ssi->dev_id);
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_DEBUG, "I am %s.", ssi->dev_name);

	ssi->res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (IS_ERR(ssi->res)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not determine device resources");
		return PTR_ERR(ssi->res);
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "iomem range %08x-%08x", ssi->res->start, ssi->res->end);

	ssi->regs = devm_ioremap_resource(&pdev->dev, ssi->res);
	if (IS_ERR(ssi->regs)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not map device resources");
		return PTR_ERR(ssi->regs);
	}

	ssi->irq = platform_get_irq(pdev, 0);
	if (ssi->irq == NO_IRQ) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "no irq for node %s", np->full_name);
		return ssi->irq;
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "irq %d", ssi->irq);

	ret = devm_request_irq(&pdev->dev, ssi->irq, ssi_isr, 0, ssi->dev_name, ssi);
	if (ret) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not claim irq %u: %d", ssi->irq, ret);
		return ret;
	}

	ssi->ssi_clk = devm_clk_get(&pdev->dev, "ipg");
	if (IS_ERR(ssi->ssi_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get ipg clock");
		return PTR_ERR(ssi->ssi_clk);
	}
	clk_prepare_enable(ssi->ssi_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "clock rate %ld", clk_get_rate(ssi->ssi_clk));

	ssi->baud_clk = devm_clk_get(&pdev->dev, "baud");
	if (IS_ERR(ssi->baud_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get baud clock");
		ret = PTR_ERR(ssi->baud_clk);
		goto failed_probe;
	}
	clk_prepare_enable(ssi->baud_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "baud clock rate %ld", clk_get_rate(ssi->baud_clk));

	for_each_available_child_of_node(pdev->dev.of_node, child) {
		u32 type;
		struct ssi_params *params = NULL;
		ret = of_property_read_u32(child, "lla-type", &type);
		if (ret) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "%p type read returned %d", child, ret);
			continue;
		}
#ifndef CONFIG_SONOS_DIAGS
		if (of_property_read_bool(child, "diags-only")) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_INFO, "Disabled in production builds.");
			goto failed_probe;
		}
#endif
		params = &(ssi->rx_params);
		params->enable = 1;
		params->lla_type = type;
		of_property_read_u32(child, "frames-per-second", &(params->frames_per_second));
		of_property_read_u32(child, "channel-len", &(params->channel_len));
		of_property_read_u32(child, "channel-mask", &(params->channel_mask));
		of_property_read_u32(child, "channels-per-frame", &(params->channels_per_frame));
		of_property_read_u32(child, "frames-per-buffer", &(params->frames_per_buffer));
		of_property_read_u32(child, "buffer-cnt", &(params->buffer_cnt));
		of_property_read_u32_array(child, "dividers", (u32 *)&(params->dividers), 3);
		of_property_read_u32(child, "watermark-words", &(params->watermark_words));
		params->invert_fs = of_property_read_bool(child, "invert-fs");
		params->tdm_master = of_property_read_bool(child, "tdm-master");
		params->i2s_mode = of_property_read_bool(child, "i2s-mode");
		params->right_align = of_property_read_bool(child, "right-align");
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_DEBUG, "type %u fps %u len %u mask %#x cpf %u fpb %u bufs %u divs %u %u %u",
				type,
				params->frames_per_second,
				params->channel_len,
				params->channel_mask,
				params->channels_per_frame,
				params->frames_per_buffer,
				params->buffer_cnt,
				params->dividers.div2, params->dividers.sr, params->dividers.pm);
	}

	ssi_rx_init(ssi);

	if (lla_rx_setup(ssi)) {
		ret = -1;
		goto failed_probe;
	}
	if (ssi_rx_dma_setup(ssi)) {
		ret = -1;
		goto failed_probe;
	}

	init_timer(&(ssi->dma_watchdog));
	ssi->dma_watchdog.function = ssi_rx_dma_watchdog;
	ssi->dma_watchdog.data = (unsigned long)ssi;

	ssi_rx_start(ssi);

	ret = procfs_init(ssi);
	if (ret) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Failed to create procfiles.");
		procfs_remove(ssi);
	}

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "registered");

	return ret;
failed_probe:
	if (ssi->ssi_clk) {
		clk_disable_unprepare(ssi->ssi_clk);
	}
	if (ssi->baud_clk) {
		clk_disable_unprepare(ssi->baud_clk);
	}

	return ret;
}

static int ssi_remove(struct platform_device *pdev)
{
	struct ssi *ssi = platform_get_drvdata(pdev);

	procfs_remove(ssi);

	del_timer(&(ssi->dma_watchdog));

	disable_irq(ssi->irq);

	lla_unregister_dev(&(ssi->rx_lla_dev));
	if (ssi->rx_dma.chan != NULL) {
		dma_release_channel(ssi->rx_dma.chan);
	}
	lla_free_shared_buffers(&(ssi->rx_lla_dev));

	SSI_WRITE(~SSI_SCR_RE & ~SSI_SCR_SSIEN, scr);

	clk_disable_unprepare(ssi->baud_clk);
	clk_disable_unprepare(ssi->ssi_clk);

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "unregistered");
	return 0;
}

static int ssi_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct ssi *ssi = platform_get_drvdata(pdev);
	u32 reg = SSI_READ(scr);
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "suspend");
	reg &= ~SSI_SCR_RE;
	SSI_WRITE(reg, scr);
	return 0;
}

static int ssi_resume(struct platform_device *pdev)
{
	struct ssi *ssi = platform_get_drvdata(pdev);
	u32 reg = SSI_READ(scr);
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "resume");
	reg |= SSI_SCR_RE;
	SSI_WRITE(reg, scr);
	return 0;
}

static const struct of_device_id ssi_ids[] = {
	{ .compatible = "fsl,imx6sx-ssi", },
	{ .compatible = "fsl,imx6q-ssi", },
	{}
};

static struct platform_driver ssi_driver = {
	.probe = ssi_probe,
	.remove = ssi_remove,
	.suspend = ssi_suspend,
	.resume = ssi_resume,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = ssi_ids,
	},
};

int ssi_init(void)
{
	return platform_driver_register(&ssi_driver);
}

void ssi_exit(void)
{
	platform_driver_unregister(&ssi_driver);
}
