/*
 * Copyright (c) 2014-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Copyright (C) 2013 Freescale Semiconductor, Inc.
 *
 * This file is licensed under the terms of the GNU General Public License
 * version 2.  This program  is licensed "as is" without any warranty of any
 * kind, whether express or implied.
 */

#include <linux/version.h>
#include <linux/clk.h>
#include <linux/wait.h>
#include <linux/dmaengine.h>
#include <linux/platform_data/dma-imx.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/kernel.h>
#include <linux/dma-mapping.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,9,0)
#include <linux/busfreq-imx6.h>
#endif
#include <linux/ktime.h>
#include <linux/hrtimer.h>
#include <linux/math64.h>
#include <asm/barrier.h>
#include <linux/sonos_kernel.h>

#include "blackbox.h"
#include "lla.h"
#include "spdif.h"

#define DRV_NAME		"imx_spdif"

#define PROCFS_PERM_READ	(S_IRUSR | S_IRGRP | S_IROTH)
#define PROCFS_PERM_WRITE	(S_IWUSR | S_IWGRP | S_IWOTH)

#define PROCFS_DIR		"driver/"DRV_NAME
#define PROCFS_REGS_FILE	"regs"
#define PROCFS_IRQS_FILE	"irqs"
#define PROCFS_DMA_FILE		"dma"
#define PROCFS_RX_FILE		"rx"
#define PROCFS_FAIL_FILE	"fail"

#define SPDIF_RX_ERROR_IRQS \
	(SPDIF_SIESC_SYM_ERR | \
	 SPDIF_SIESC_BIT_ERR | \
	 SPDIF_SIESC_VAL_NO_GOOD)

#define SPDIF_RX_ERROR_IRQ_RATELIMIT	(HZ / 10)

#define SPDIF_RX_DPLL_LOCK_STABILITY_DELAY		(HZ / 20)
#define SPDIF_RX_DPLL_LOCK_STABILITY_POLL_INTERVAL	(HZ / 200)
#define SPDIF_RX_DPLL_LOCK_STABILITY_FREQ_TOLERANCE	(10)
#define SPDIF_RX_DPLL_LOCK_STABILITY_SUCCESS_LIMIT	(2)
#define SPDIF_RX_DPLL_LOCK_STABILITY_WARNING_LIMIT	(30)

#define UNUSED_PARAM(x)		(void)x

struct imx_spdif_regs {
	u32 scr;
	u32 srcd;
	u32 srpc;
	u32 sie;
	u32 sis_sic;
	u32 srl;
	u32 srr;
	u32 srcsh;
	u32 srcsl;
	u32 sru;
	u32 srq;
	u32 stl;
	u32 str;
	u32 stcsch;
	u32 stcscl;
	u8 reserved0[0x8];
	u32 srfm;
	u8 reserved1[0x8];
	u32 stc;
};

struct spdif_dma {
	struct dma_chan			*chan;
	struct dma_async_tx_descriptor	*desc;
	dma_cookie_t			cookie;
	u32				control_offset;
	wait_queue_head_t		poll_event;
};

#define SPDIF_CHAN_STATUS_SIZE	6

struct spdif_params {
	u32 enable;
	u32 lla_type;
	u32 lla_sync_to_dev;
	u32 frames_per_second;
	u8 channel_status[SPDIF_CHAN_STATUS_SIZE];
	u32 channel_len;
	u32 channel_mask;
	u32 channels_per_frame;
	u32 frames_per_buffer;
	u32 buffer_cnt;
	u32 clk_div;
};

#define SPDIF_WORDS_PER_FIFO	16

struct imx_spdif {
	struct imx_spdif_regs __iomem	*regs;
	struct resource			*res;
	int				irq;

	struct clk			*spdif_clk;
	struct clk			*dma_clk;
	struct clk			*rx_clk;
	struct clk			*tx_clk;
	struct clk			*ipg_clk;
	unsigned long			ipg_clk_freq;

	struct spdif_params		tx_params;
	struct spdif_dma		tx_dma;
	struct lla_dev			tx_lla_dev;
	struct lla_shared		*tx_shared;
	struct lla_ptr			tx_silence;

	struct spdif_params		rx_params;
	struct spdif_dma		rx_dma;
	struct lla_dev			rx_lla_dev;
	struct lla_shared		*rx_shared;

	spinlock_t			rx_csb_lock;
	u32				rx_csb_valid;
	u8				rx_csb_data[6];

	u32				rx_dpll_locked;
	u32				rx_dpll_lock_last_rate;
	u16				rx_dpll_lock_stable_pass;
	u16				rx_dpll_lock_stable_polls;
	struct timer_list		rx_dpll_lock_timer;

	struct timer_list		rx_error_timer;

	struct _spdif_procfs {
		struct proc_dir_entry	*dir;
		struct proc_dir_entry	*regs;
		struct proc_dir_entry	*irqs;
		struct proc_dir_entry	*dma;
		struct proc_dir_entry	*rx;
		struct proc_dir_entry	*rate;
		struct proc_dir_entry	*snoop;
		struct proc_dir_entry	*fail;
	} procfs;

	struct _spdif_stats {
		u32	irq_lock;
		u32	irq_tx_un_ov;
		u32	irq_tx_resyn;
		u32	irq_cnew;
		u32	irq_val_no_good;
		u32	irq_sym_err;
		u32	irq_bit_err;
		u32	irq_u_rx_ful;
		u32	irq_u_rx_ov;
		u32	irq_q_rx_ful;
		u32	irq_q_rx_ov;
		u32	irq_u_q_sync;
		u32	irq_u_q_err;
		u32	irq_rx_fifo_un_ov;
		u32	irq_rx_fifo_resyn;
		u32	irq_lock_loss;
		u32	irq_tx_em;
		u32	irq_rx_fifo_ful;
		u64	tx_start;
		u64	tx_stop;
		u64	rx_start;
		u64	rx_start_buffers;
		u64	rx_stable;
		u64	rx_error;
		u64	rx_stop;
	} stats;

#ifdef RATE_TRACKING

#define	RATE_TRACKING_SIZE	64
	struct _spdif_rate_tracking {
		u32	count;
		u64	time[RATE_TRACKING_SIZE];
		u8	state[RATE_TRACKING_SIZE];
		u32	bnum[RATE_TRACKING_SIZE];
		u32	rate[RATE_TRACKING_SIZE];
	} rate_tracking;
#endif

	struct platform_device		*pdev;
	struct device			*dev;
};

#define SPDIF_FIFO_WATERMARK_WORDS   (SPDIF_WORDS_PER_FIFO / 2)

#define SPDIF_TX_REG_MASK_SCR \
	~(SPDIF_SCR_TXFIFO_AUTOSYNC_MASK | \
	  SPDIF_SCR_TXFIFO_FSEL_MASK | \
	  SPDIF_SCR_TXFIFO_CTRL_MASK | \
	  SPDIF_SCR_DMA_TX_EN_MASK | \
	  SPDIF_SCR_VAL_MASK | \
	  SPDIF_SCR_TXSEL_MASK)
#define SPDIF_TX_REG_VALUE_SCR \
	(SPDIF_SCR_TXFIFO_AUTOSYNC | \
	 SPDIF_SCR_TXFIFO_FSEL_IF8 | \
	 SPDIF_SCR_TXFIFO_CTRL_NORMAL | \
	 SPDIF_SCR_DMA_TX_EN | \
	 SPDIF_SCR_VAL_CLEAR | \
	 SPDIF_SCR_TXSEL_NORMAL)

#define SPDIF_TX_REG_MASK_SIESC \
	~(SPDIF_SIESC_TX_UN_OV | SPDIF_SIESC_TX_RESYN | SPDIF_SIESC_TX_EM)
#define SPDIF_TX_REG_VALUE_SIESC \
	(SPDIF_SIESC_TX_UN_OV | SPDIF_SIESC_TX_RESYN)

#define SPDIF_TX_REG_VALUE_STC \
	(SPDIF_STC_SYSCLK_NO_SIGNAL | SPDIF_STC_TXCLK_SRC_ESAI_HCKT | \
	 SPDIF_STC_TXCLK_ALL_EN)

#define SPDIF_RX_REG_MASK_SCR \
	~(SPDIF_SCR_RXFIFO_CTL_MASK | \
	  SPDIF_SCR_RXFIFO_OFF_MASK | \
	  SPDIF_SCR_RXFIFO_RST_MASK | \
	  SPDIF_SCR_RXFIFO_FSEL_MASK | \
	  SPDIF_SCR_RXFIFO_AUTOSYNC_MASK | \
	  SPDIF_SCR_USRC_SEL_MASK)
#define SPDIF_RX_REG_VALUE_SCR \
	(SPDIF_SCR_RXFIFO_FSEL_IF8 | \
	 SPDIF_SCR_RXFIFO_AUTOSYNC | \
	 SPDIF_SCR_RXFIFO_RST | \
	 SPDIF_SCR_DMA_RX_EN)

#define SPDIF_RX_REG_VALUE_SRCD 0

#define SPDIF_RX_REG_MASK_SRPC \
	~(SPDIF_SRPC_CLKSRC_SEL_MASK | SPDIF_SRPC_GAIN_SEL_MASK)
#define SPDIF_RX_REG_VALUE_SRPC \
	(SPDIF_SRPC_CLKSRC_SEL_RX_OR_TX | SPDIF_SRPC_GAIN_SEL_8X)

#define SPDIF_RX_REG_MASK_SIESC \
	~(SPDIF_SIESC_LOCK | \
	  SPDIF_SIESC_CNEW | SPDIF_SIESC_VAL_NO_GOOD | \
	  SPDIF_SIESC_SYM_ERR | SPDIF_SIESC_BIT_ERR | \
	  SPDIF_SIESC_U_RX_FUL | SPDIF_SIESC_U_RX_OV | \
	  SPDIF_SIESC_Q_RX_FUL | SPDIF_SIESC_Q_RX_OV | \
	  SPDIF_SIESC_U_Q_SYNC | SPDIF_SIESC_U_Q_ERR | \
	  SPDIF_SIESC_RX_FIFO_UN_OV | SPDIF_SIESC_RX_FIFO_RESYN | \
	  SPDIF_SIESC_LOCK_LOSS | SPDIF_SIESC_RX_FIFO_FUL)
#define SPDIF_RX_REG_VALUE_SIESC \
	(SPDIF_SIESC_LOCK | SPDIF_SIESC_LOCK_LOSS | \
	 SPDIF_SIESC_RX_FIFO_UN_OV | SPDIF_SIESC_RX_FIFO_RESYN)

#ifdef IMX_SPDIF_REG_DEBUG

static inline u32 _read_n_print(const char *func, int line, void *reg_addr, const char *reg_name)
{
	u32 val = readl(reg_addr);
	printk("[READ]  %08x <-- %-8s (%s:%d)\n", val, reg_name, func, line);
	return val;
}

#define SPDIF_READ(R) _read_n_print(__FUNCTION__, __LINE__, &(spdif->regs->R), #R)

#define SPDIF_WRITE(V, R) \
	do { \
		printk("[WRITE] %08x --> %-8s (%s:%d)\n", V, #R, __FUNCTION__, __LINE__); \
		writel(V, &(spdif->regs->R)); \
	} while (0)

#else
#define SPDIF_READ(R)		readl(&(spdif->regs->R))
#define SPDIF_WRITE(V, R)	writel(V, &(spdif->regs->R))
#endif


#ifdef IMX_SPDIF_DUMP
#define SPDIF_REG_PRINT(R) \
	pr_info("SPDIF_%-8s 0x%08x\n", #R, readl(&(spdif->regs->R)));

#define SPDIF_DUMP() \
	do { \
		pr_info("dump @ %s:%d\n", __FUNCTION__, __LINE__); \
		SPDIF_REG_PRINT(scr); \
		SPDIF_REG_PRINT(srcd); \
		SPDIF_REG_PRINT(srpc); \
		SPDIF_REG_PRINT(sie); \
		SPDIF_REG_PRINT(sis_sic); \
		SPDIF_REG_PRINT(srl); \
		SPDIF_REG_PRINT(srr); \
		SPDIF_REG_PRINT(srcsh); \
		SPDIF_REG_PRINT(srcsl); \
		SPDIF_REG_PRINT(sru); \
		SPDIF_REG_PRINT(srq); \
		SPDIF_REG_PRINT(stl); \
		SPDIF_REG_PRINT(str); \
		SPDIF_REG_PRINT(stcsch); \
		SPDIF_REG_PRINT(stcscl); \
		SPDIF_REG_PRINT(srfm); \
		SPDIF_REG_PRINT(stc); \
	} while (0)
#else
#define SPDIF_DUMP()
#endif

static int lla_spdif_get_csb(struct lla_csb *csb, void *context);

static int procfs_regs_show(struct seq_file *m, void *v)
{
	struct imx_spdif *spdif = m->private;

	#define _SPDIF_PROCFS_REG_SHOW(reg) seq_printf(m, "[%03x] SPDIF_%-8s = %08x\n", (&(spdif->regs->reg) - (u32*)spdif->regs) * sizeof(u32), #reg, SPDIF_READ(reg))

	seq_printf(m, "%08x-%08x\n", spdif->res->start, spdif->res->end);
	_SPDIF_PROCFS_REG_SHOW(scr);
	_SPDIF_PROCFS_REG_SHOW(srcd);
	_SPDIF_PROCFS_REG_SHOW(srpc);
	_SPDIF_PROCFS_REG_SHOW(sie);
	_SPDIF_PROCFS_REG_SHOW(sis_sic);
#if 0
	_SPDIF_PROCFS_REG_SHOW(srl);
	_SPDIF_PROCFS_REG_SHOW(srr);
#endif
	_SPDIF_PROCFS_REG_SHOW(srcsh);
	_SPDIF_PROCFS_REG_SHOW(srcsl);
	_SPDIF_PROCFS_REG_SHOW(sru);
	_SPDIF_PROCFS_REG_SHOW(srq);

#if 0
	_SPDIF_PROCFS_REG_SHOW(stl);
	_SPDIF_PROCFS_REG_SHOW(str);
#endif
	_SPDIF_PROCFS_REG_SHOW(stcsch);
	_SPDIF_PROCFS_REG_SHOW(stcscl);
	_SPDIF_PROCFS_REG_SHOW(srfm);
	_SPDIF_PROCFS_REG_SHOW(stc);

	return 0;
}

#define GET_TIME_NSEC()		ktime_to_ns(ktime_get())

static inline u64 get_elapsed_nsec(u64 event_nsec)
{
	return GET_TIME_NSEC() - event_nsec;
}

static inline u32 get_elapsed_sec(u64 event_nsec)
{
	return (u32)div64_u64(get_elapsed_nsec(event_nsec), NSEC_PER_SEC);
}

static int procfs_regs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_regs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_regs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_regs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int procfs_irqs_show(struct seq_file *m, void *v)
{
	struct imx_spdif *spdif = m->private;
	u32 sie_val = SPDIF_READ(sie);
	char *ratelimit_str = (timer_pending(&(spdif->rx_error_timer))) ? "(limited)" : "";

	#define _IRQ_DISABLED_STR(reg, bit) (reg & bit) ? "" : "(disabled)"

	seq_printf(m, "IRQ #%d\n", spdif->irq);

	seq_printf(m, "Tx FIFO Empty                = %d %s\n", spdif->stats.irq_tx_em, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_TX_EM));
	seq_printf(m, "Tx FIFO Under/Overrun        = %d %s\n", spdif->stats.irq_tx_un_ov, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_TX_UN_OV));
	seq_printf(m, "Tx FIFO Resync               = %d %s\n", spdif->stats.irq_tx_resyn, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_TX_RESYN));
	seq_printf(m, "Rx Lock                      = %d %s\n", spdif->stats.irq_lock, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_LOCK));
	seq_printf(m, "Rx Lost Lock                 = %d %s\n", spdif->stats.irq_lock_loss, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_LOCK_LOSS));
	seq_printf(m, "Rx Control Channel Change    = %d %s\n", spdif->stats.irq_cnew, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_CNEW));

	seq_printf(m, "Rx Invalid                   = %d %s %s\n", spdif->stats.irq_val_no_good, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_VAL_NO_GOOD),
			ratelimit_str);
	seq_printf(m, "Rx Illegal Symbol            = %d %s %s\n", spdif->stats.irq_sym_err, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_SYM_ERR),
			ratelimit_str);
	seq_printf(m, "Rx Bit Error                 = %d %s %s\n", spdif->stats.irq_bit_err, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_BIT_ERR),
			ratelimit_str);

	seq_printf(m, "Rx U Channel Reg Full        = %d %s\n", spdif->stats.irq_u_rx_ful, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_U_RX_FUL));
	seq_printf(m, "Rx U Channel Reg Overrun     = %d %s\n", spdif->stats.irq_u_rx_ov, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_U_RX_OV));
	seq_printf(m, "Rx Q Channel Reg Full        = %d %s\n", spdif->stats.irq_q_rx_ful, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_Q_RX_FUL));
	seq_printf(m, "Rx Q Channel Reg Overrun     = %d %s\n", spdif->stats.irq_q_rx_ov, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_Q_RX_OV));
	seq_printf(m, "Rx U/Q Channel Sync Found    = %d %s\n", spdif->stats.irq_u_q_sync, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_U_Q_SYNC));
	seq_printf(m, "Rx U/Q Channel Framing Error = %d %s\n", spdif->stats.irq_u_q_err, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_U_Q_ERR));
	seq_printf(m, "Rx FIFO Full                 = %d %s\n", spdif->stats.irq_rx_fifo_ful, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_RX_FIFO_FUL));
	seq_printf(m, "Rx FIFO Under/Overrun        = %d %s\n", spdif->stats.irq_rx_fifo_un_ov, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_RX_FIFO_UN_OV));
	seq_printf(m, "Rx FIFO Resync               = %d %s\n", spdif->stats.irq_rx_fifo_resyn, _IRQ_DISABLED_STR(sie_val, SPDIF_SIESC_RX_FIFO_RESYN));

	return 0;
}

static int procfs_irqs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_irqs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_irqs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_irqs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static void procfs_dma_show_single_stats(
		struct seq_file *m,
		const char *pfx,
		u64 start,
		u64 stop,
		struct lla_shared *shared)
{
	u64 elapsed = 0;
	u64 elapsed_sec;

	if (start) {
		if (!stop) {
			elapsed = ktime_to_ns(ktime_get()) - start;
		}
		else {
			elapsed = stop - start;
		}
	}
	elapsed_sec = div64_u64(elapsed, NSEC_PER_SEC);

	seq_printf(m, "%s DMAed Buffers    = %llu\n", pfx, shared->buffers_complete);
	seq_printf(m, "%s Total Bytes      = %llu\n", pfx, (shared->buffers_complete * shared->buffer_len));
	if (elapsed_sec) {
		seq_printf(m, "%s bytes/sec        = %llu\n", pfx, div64_u64(((shared->buffers_complete - 1) * shared->buffer_len), elapsed_sec));
	}
	else {
		seq_printf(m, "%s bytes/sec        = n/a\n", pfx);
	}
	seq_printf(m, "%s Duration (ns)    = %llu\n", pfx, elapsed);
	if (shared->buffers_complete > 1) {
		seq_printf(m, "%s ns/buffer        = %llu\n", pfx, div64_u64(elapsed, (shared->buffers_complete - 1)));
	}
	else {
		seq_printf(m, "%s ns/buffer        = n/a\n", pfx);
	}
}

static int procfs_dma_show(struct seq_file *m, void *v)
{
	struct imx_spdif *spdif = m->private;

	if (spdif->tx_params.enable) {
		procfs_dma_show_single_stats(m, "Tx", spdif->stats.tx_start, spdif->stats.tx_stop, spdif->tx_shared);
	}
	if (spdif->rx_params.enable) {
		procfs_dma_show_single_stats(m, "Rx", spdif->stats.rx_start, spdif->stats.rx_stop, spdif->rx_shared);
	}

	return 0;
}

static int procfs_dma_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_dma_show, PDE_DATA(inode));
}

static const struct file_operations procfs_dma_ops = {
	.owner = THIS_MODULE,
	.open = procfs_dma_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static u32 spdif_get_rxclk_rate(struct imx_spdif *spdif)
{
	u64 tmpval64;
	u32 freqmeas, srpc_val, gainsel;
	const u32 gainsel_tbl[] = {24, 16, 12, 8, 6, 4, 3};

	if (!spdif->rx_dpll_locked) {
		return 0;
	}

	freqmeas = SPDIF_READ(srfm);
	srpc_val = SPDIF_READ(srpc);
	gainsel = (srpc_val & SPDIF_SRPC_GAIN_SEL_MASK) >> SPDIF_SRPC_GAIN_SEL_OFFSET;

	if (gainsel > (sizeof(gainsel_tbl) / sizeof(gainsel_tbl[0]))) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_WARNING, "invalid gainsel in SRPC register (%x)", gainsel);
		return 0;
	}

	tmpval64 = spdif->ipg_clk_freq * (u64)freqmeas;
	do_div(tmpval64, gainsel_tbl[gainsel] * 1024);
	do_div(tmpval64, 128 * 1024);


	return ((int)tmpval64);
}

static void spdif_read_chan_status(struct imx_spdif *spdif, int valid)
{
	spin_lock(&spdif->rx_csb_lock);

	if (valid) {
		u32 hi_val = SPDIF_READ(srcsh);
		u32 lo_val = SPDIF_READ(srcsl);

		spdif->rx_csb_data[0] = (hi_val >> 16) & 0xFF;
		spdif->rx_csb_data[1] = (hi_val >> 8) & 0xFF;
		spdif->rx_csb_data[2] = hi_val & 0xFF;

		spdif->rx_csb_data[3] = (lo_val >> 16) & 0xFF;
		spdif->rx_csb_data[4] = (lo_val >> 8) & 0xFF;
		spdif->rx_csb_data[5] = lo_val & 0xFF;
	}

	spdif->rx_csb_valid = valid;

	spin_unlock(&spdif->rx_csb_lock);
}

static int procfs_rx_show(struct seq_file *m, void *v)
{
	struct imx_spdif *spdif = m->private;
	u32 rxclk_rate;
	struct lla_csb csb;

	seq_printf(m, "Status                = %s\n", spdif->rx_dpll_locked ? "LOCKED" : "no signal");

	rxclk_rate = spdif_get_rxclk_rate(spdif);
	seq_printf(m, "Clock Rate (hz)       = %d\n", rxclk_rate);

	seq_printf(m, "Channel Status        = ");
	if (lla_spdif_get_csb(&csb, spdif) == 0) {
		seq_printf(m, "%6ph\n", csb.data);
	}
	else {
		seq_printf(m, "n/a\n");
	}

	seq_printf(m, "Stability Time (msec) = ");
	if (spdif->stats.rx_stable) {
		seq_printf(m, "%llu\n", div64_u64((spdif->stats.rx_stable - spdif->stats.rx_start), NSEC_PER_MSEC));
	}
	else {
		seq_printf(m, "n/a\n");
	}

	seq_printf(m, "Stability Polls       = %d\n", spdif->rx_dpll_lock_stable_polls);

	seq_printf(m, "Sec Since Last Lock   = %u\n", get_elapsed_sec(spdif->stats.rx_start));
	seq_printf(m, "Sec Since Last Error  = %u\n", get_elapsed_sec(spdif->stats.rx_error));

	return 0;
}

static int procfs_rx_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_rx_show, PDE_DATA(inode));
}

static const struct file_operations procfs_rx_ops = {
	.owner = THIS_MODULE,
	.open = procfs_rx_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

#ifdef RATE_TRACKING

#define PROCFS_RATE_FILE	"rate"

static int procfs_rate_show(struct seq_file *m, void *v)
{
	struct imx_spdif *spdif = m->private;
	int idx;

	for (idx = 0; idx < spdif->rate_tracking.count; idx++) {
		seq_printf(m, "%d: usec=%llu bnum=%u rate=%u %s\n",
			idx,
			div64_u64((spdif->rate_tracking.time[idx] - spdif->stats.rx_start), NSEC_PER_USEC),
			spdif->rate_tracking.bnum[idx],
			spdif->rate_tracking.rate[idx],
			(spdif->rate_tracking.state[idx]) ? "RUNNING" : "");
	}

	return 0;
}

static int procfs_rate_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_rate_show, PDE_DATA(inode));
}

static const struct file_operations procfs_rate_ops = {
	.owner = THIS_MODULE,
	.open = procfs_rate_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static inline void rate_tracking_add(struct imx_spdif *spdif, u32 sample_rate)
{
	if ( (spdif->rate_tracking.count < RATE_TRACKING_SIZE) &&
	     (sample_rate != spdif->rate_tracking.rate[ spdif->rate_tracking.count - 1 ]) )
	{
		spdif->rate_tracking.state[spdif->rate_tracking.count] =
			spdif->rx_shared->flags & LLA_SHARED_FLAG_RUNNING;

		spdif->rate_tracking.time[spdif->rate_tracking.count] =
			GET_TIME_NSEC();

		spdif->rate_tracking.bnum[spdif->rate_tracking.count] =
			(u32)(spdif->rx_shared->buffers_complete - spdif->stats.rx_start_buffers);

		spdif->rate_tracking.rate[spdif->rate_tracking.count] =
			sample_rate;

		spdif->rate_tracking.count++;
	}
}

static inline void rate_tracking_reset(struct imx_spdif *spdif)
{
	spdif->rate_tracking.count = 0;
}

#endif

#ifdef SNOOP_RX

#define PROCFS_SNOOP_FILE	"snoop"

static int procfs_snoop_show(struct seq_file *m, void *v)
{
	struct imx_spdif *spdif = m->private;
	seq_write(m, spdif->rx_shared->buffers.k, spdif->rx_shared->buffer_len);
	return 0;
}

static int procfs_snoop_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_snoop_show, PDE_DATA(inode));
}

static const struct file_operations procfs_snoop_ops = {
	.owner = THIS_MODULE,
	.open = procfs_snoop_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};
#endif

#ifdef TEST_FAILURES

#define SPDIF_WORST_CASE_SAMPLE_RATE	8000
static int test_fail_rx_dma_hang = 0;
static int test_fail_rate_zero = 0;

static int procfs_fail_show(struct seq_file *m, void *v)
{
	return 0;
}

static int procfs_fail_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_fail_show, PDE_DATA(inode));
}

static ssize_t procfs_fail_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	struct imx_spdif *spdif = PDE_DATA(file_inode(file));
	char cmd[32];
	char *newline;
	ssize_t result = count;

	if ((count < 0) || (count >= sizeof(cmd))) {
		result = -EIO;
	}
	else if (copy_from_user(&cmd, buffer, count)) {
		result = -EFAULT;
	}
	else {
		cmd[count] = '\0';

		newline = strchr(cmd, '\n');
		if (newline != NULL) {
			*newline = '\0';
		}

		if (strcmp(cmd, "dma-hang") == 0) {
			bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx DMA hang");
			test_fail_rx_dma_hang++;
		}

		if (strcmp(cmd, "dma-stop") == 0) {
			if (spdif->rx_dma.chan) {
				bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx DMA stop");
				dmaengine_terminate_all(spdif->rx_dma.chan);
			} else {
				bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "No Rx DMA");
			}
		}

		if (strcmp(cmd, "lla-error") == 0) {
			if (spdif->rx_shared) {
				bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx LLA error flag");
				spdif->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
				spdif->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
			} else {
				bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "No Rx LLA");
			}
		}

		if (strcmp(cmd, "rate-zero") == 0) {
			bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Forcing sample rate zero");
			test_fail_rate_zero++;
		}
	}

	return result;
}

static const struct file_operations procfs_fail_ops = {
	.owner = THIS_MODULE,
	.open = procfs_fail_open,
	.read = seq_read,
	.write = procfs_fail_write,
	.llseek = seq_lseek,
	.release = single_release,
};
#endif

static void procfs_remove(struct imx_spdif *spdif)
{
#ifdef TEST_FAILURES
	if (spdif->procfs.fail) {
		remove_proc_entry(PROCFS_FAIL_FILE, spdif->procfs.dir);
	}
#endif
#ifdef SNOOP_RX
	if (spdif->procfs.snoop) {
		remove_proc_entry(PROCFS_SNOOP_FILE, spdif->procfs.dir);
	}
#endif
#ifdef RATE_TRACKING
	if (spdif->procfs.rate) {
		remove_proc_entry(PROCFS_RATE_FILE, spdif->procfs.dir);
	}
#endif
	if (spdif->procfs.rx) {
		remove_proc_entry(PROCFS_RX_FILE, spdif->procfs.dir);
	}
	if (spdif->procfs.dma) {
		remove_proc_entry(PROCFS_DMA_FILE, spdif->procfs.dir);
	}
	if (spdif->procfs.regs) {
		remove_proc_entry(PROCFS_REGS_FILE, spdif->procfs.dir);
	}
	if (spdif->procfs.irqs) {
		remove_proc_entry(PROCFS_IRQS_FILE, spdif->procfs.dir);
	}
	if (spdif->procfs.dir) {
		remove_proc_entry(PROCFS_DIR, NULL);
	}
}

static int procfs_init(struct imx_spdif *spdif)
{
	spdif->procfs.dir = proc_mkdir(PROCFS_DIR, NULL);
	if (!spdif->procfs.dir) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs dir %s", PROCFS_DIR);
		goto failed_procfs;
	}

	spdif->procfs.regs = proc_create_data(PROCFS_REGS_FILE, PROCFS_PERM_READ, spdif->procfs.dir, &procfs_regs_ops, spdif);
	if (!spdif->procfs.regs) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_REGS_FILE);
		goto failed_procfs;
	}

	spdif->procfs.irqs = proc_create_data(PROCFS_IRQS_FILE, PROCFS_PERM_READ, spdif->procfs.dir, &procfs_irqs_ops, spdif);
	if (!spdif->procfs.irqs) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_IRQS_FILE);
		goto failed_procfs;
	}

	spdif->procfs.dma = proc_create_data(PROCFS_DMA_FILE, PROCFS_PERM_READ, spdif->procfs.dir, &procfs_dma_ops, spdif);
	if (!spdif->procfs.dma) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_DMA_FILE);
		goto failed_procfs;
	}

	if (spdif->rx_params.enable) {
		spdif->procfs.rx = proc_create_data(PROCFS_RX_FILE, PROCFS_PERM_READ, spdif->procfs.dir, &procfs_rx_ops, spdif);
		if (!spdif->procfs.rx) {
			bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_RX_FILE);
			goto failed_procfs;
		}
	}

#ifdef RATE_TRACKING
	spdif->procfs.rate = proc_create_data(PROCFS_RATE_FILE, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), spdif->procfs.dir, &procfs_rate_ops, spdif);
	if (!spdif->procfs.rate) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_RATE_FILE);
		goto failed_procfs;
	}
#endif

#ifdef SNOOP_RX
	spdif->procfs.snoop = proc_create_data(PROCFS_SNOOP_FILE, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), spdif->procfs.dir, &procfs_snoop_ops, spdif);
	if (!spdif->procfs.snoop) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_SNOOP_FILE);
		goto failed_procfs;
	}
#endif

#ifdef TEST_FAILURES
	spdif->procfs.fail = proc_create_data(PROCFS_FAIL_FILE, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), spdif->procfs.dir, &procfs_fail_ops, spdif);
	if (!spdif->procfs.fail) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_FAIL_FILE);
		goto failed_procfs;
	}
#endif

	return 0;
failed_procfs:
	procfs_remove(spdif);
	return -1;
}

static void spdif_rx_dpll_lock_timer(unsigned long data)
{
	struct imx_spdif *spdif = (struct imx_spdif *)data;
	u32 sample_rate;
	s32 rate_delta;

	disable_irq(spdif->irq);

	if (spdif->rx_dpll_locked) {

		bb_log_dbg_dev(spdif->dev, BB_MOD_LLA, "lock stability test");
		spdif->rx_dpll_lock_stable_polls++;

		sample_rate = spdif_get_rxclk_rate(spdif);
		rate_delta = (s32)spdif->rx_dpll_lock_last_rate - (s32)sample_rate;

		if ((rate_delta >= -SPDIF_RX_DPLL_LOCK_STABILITY_FREQ_TOLERANCE) &&
		    (rate_delta <= SPDIF_RX_DPLL_LOCK_STABILITY_FREQ_TOLERANCE)) {
			bb_log_dbg_dev(spdif->dev, BB_MOD_LLA, "lock stable pass %d -> %d (%d)", spdif->rx_dpll_lock_last_rate, sample_rate, rate_delta);
			spdif->rx_dpll_lock_stable_pass++;
		}
		else {
			bb_log_dbg_dev(spdif->dev, BB_MOD_LLA, "lock stable fail %d -> %d (%d)", spdif->rx_dpll_lock_last_rate, sample_rate, rate_delta);
			spdif->rx_dpll_lock_stable_pass = 0;
		}

		if (spdif->rx_dpll_lock_stable_pass >= SPDIF_RX_DPLL_LOCK_STABILITY_SUCCESS_LIMIT) {
			spdif->rx_shared->flags |= LLA_SHARED_FLAG_RUNNING;
			spdif->stats.rx_stable = GET_TIME_NSEC();
			bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_INFO, "lock stable @ %dhz", sample_rate);
		}
		else {
			spdif->rx_dpll_lock_last_rate = sample_rate;
			mod_timer(&(spdif->rx_dpll_lock_timer), jiffies + SPDIF_RX_DPLL_LOCK_STABILITY_POLL_INTERVAL);

			if (spdif->rx_dpll_lock_stable_polls == SPDIF_RX_DPLL_LOCK_STABILITY_WARNING_LIMIT) {
				bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_WARNING, "lock not stabilizing %d -> %d", spdif->rx_dpll_lock_last_rate, sample_rate);
			}
		}
	}

	enable_irq(spdif->irq);
}

static void spdif_irq_dpll_lock(struct imx_spdif *spdif)
{
	u32 sie_val = SPDIF_READ(sie);
	u32 scr_val = SPDIF_READ(scr);

	spdif->rx_dpll_locked = !!(SPDIF_READ(srpc) & SPDIF_SRPC_LOCK);
	bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_INFO, "RX dpll %s ", spdif->rx_dpll_locked ? "locked" : "loss lock");

	if (!spdif->rx_dpll_locked) {

		sie_val &= ~SPDIF_RX_ERROR_IRQS;

		sie_val &= ~SPDIF_SIESC_CNEW;

		scr_val |= SPDIF_SCR_RXFIFO_RST;

		spdif->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		spdif->rx_shared->flags |= LLA_SHARED_FLAG_RESTARTED;
		spdif->stats.rx_stop = GET_TIME_NSEC();

		wake_up_interruptible(&spdif->rx_dma.poll_event);
	}
	else {

		sie_val |= SPDIF_RX_ERROR_IRQS;

		sie_val |= SPDIF_SIESC_CNEW;

		scr_val &= ~(SPDIF_SCR_RXFIFO_RST);

		spdif->rx_dpll_lock_last_rate = 0;
		spdif->rx_dpll_lock_stable_pass = 0;
		spdif->rx_dpll_lock_stable_polls = 0;
		mod_timer(&(spdif->rx_dpll_lock_timer), jiffies + SPDIF_RX_DPLL_LOCK_STABILITY_DELAY);

		spdif->stats.rx_start = GET_TIME_NSEC();
		spdif->stats.rx_start_buffers = spdif->rx_shared->buffers_complete;
		spdif->stats.rx_stop = 0;
		spdif->stats.rx_stable = 0;

#ifdef RATE_TRACKING
		rate_tracking_reset(spdif);
#endif
	}

	spdif_read_chan_status(spdif, 0);

	SPDIF_WRITE(sie_val, sie);
	SPDIF_WRITE(scr_val, scr);
}

static void spdif_rx_error_timer(unsigned long data)
{
	struct imx_spdif *spdif = (struct imx_spdif *)data;
	u32 sie_val;

	disable_irq(spdif->irq);

	SPDIF_WRITE(SPDIF_RX_ERROR_IRQS, sis_sic);

	if (spdif->rx_dpll_locked) {
		sie_val = SPDIF_READ(sie);
		sie_val |= SPDIF_RX_ERROR_IRQS;
		SPDIF_WRITE(sie_val, sie);
	}

	enable_irq(spdif->irq);
}

static void spdif_irq_rx_error(struct imx_spdif *spdif)
{
	u32 sie_val = SPDIF_READ(sie);

	spdif->stats.rx_error = GET_TIME_NSEC();

	sie_val &= ~(SPDIF_RX_ERROR_IRQS);
	SPDIF_WRITE(sie_val, sie);

	mod_timer(&(spdif->rx_error_timer), jiffies + SPDIF_RX_ERROR_IRQ_RATELIMIT);
}

static int spdif_softreset(struct imx_spdif *spdif)
{
	u32 cycles = 0;
	u32 val = SPDIF_READ(scr);

	val |= SPDIF_SCR_SOFT_RESET;
	SPDIF_WRITE(val, scr);

	do {
		cycles++;
	} while ((SPDIF_READ(scr) & SPDIF_SCR_SOFT_RESET) && (cycles < 1000));

	bb_log_dbg_dev(spdif->dev, BB_MOD_LLA, "reset %s (%d cycles)", (cycles) ? "complete" : "failed", cycles);

	if (!cycles) {
		return -EBUSY;
	}
	return 0;
}


static u32 spdif_irq_status_clear(struct imx_spdif *spdif)
{
	u32 sis_expected = SPDIF_READ(sis_sic) & SPDIF_READ(sie);
	SPDIF_WRITE(sis_expected, sis_sic);
	return sis_expected;
}

static irqreturn_t imx_spdif_isr(int irq, void *p)
{
	struct imx_spdif *spdif = p;
	u32 sis_val = spdif_irq_status_clear(spdif);


	if (sis_val & SPDIF_SIESC_LOCK) {
		spdif->stats.irq_lock++;
		spdif_irq_dpll_lock(spdif);
	}

	if (sis_val & SPDIF_SIESC_LOCK_LOSS) {
		spdif->stats.irq_lock_loss++;
		spdif_irq_dpll_lock(spdif);
	}


	if (sis_val & SPDIF_SIESC_CNEW) {
		spdif_read_chan_status(spdif, 1);
		spdif->stats.irq_cnew++;
	}

	if (sis_val & SPDIF_SIESC_VAL_NO_GOOD) {
		spdif->stats.irq_val_no_good++;
		spdif_irq_rx_error(spdif);
	}

	if (sis_val & SPDIF_SIESC_SYM_ERR) {
		spdif->stats.irq_sym_err++;
		spdif_irq_rx_error(spdif);
	}

	if (sis_val & SPDIF_SIESC_BIT_ERR) {
		spdif->stats.irq_bit_err++;
		spdif_irq_rx_error(spdif);
	}

	if (sis_val & SPDIF_SIESC_U_Q_SYNC) {
		spdif->stats.irq_u_q_sync++;
	}

	if (sis_val & SPDIF_SIESC_U_Q_ERR) {
		spdif->stats.irq_u_q_err++;
		SPDIF_READ(sru);
		SPDIF_READ(srq);
	}


	if (sis_val & SPDIF_SIESC_TX_RESYN) {
		spdif->stats.irq_tx_resyn++;
	}

	if (sis_val & SPDIF_SIESC_TX_EM) {
		spdif->stats.irq_tx_em++;
	}

	if (sis_val & SPDIF_SIESC_RX_FIFO_RESYN) {
		spdif->stats.irq_rx_fifo_resyn++;
	}

	if (sis_val & SPDIF_SIESC_RX_FIFO_FUL) {
		spdif->stats.irq_rx_fifo_ful++;
	}

	if (sis_val & SPDIF_SIESC_U_RX_FUL) {
		spdif->stats.irq_u_rx_ful++;
		SPDIF_READ(sru);
	}

	if (sis_val & SPDIF_SIESC_Q_RX_FUL) {
		spdif->stats.irq_q_rx_ful++;
		SPDIF_READ(srq);
	}


	if (sis_val & SPDIF_SIESC_TX_UN_OV) {
		uint32_t scr = SPDIF_READ(scr);
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: Tx FIFO Under/Overrun");
		spdif->stats.irq_tx_un_ov++;

		scr &= ~(SPDIF_SCR_TXSEL_MASK);
		SPDIF_WRITE(scr, scr);
		spdif_softreset(spdif);
		spdif->tx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		spdif->tx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		spdif->stats.tx_stop = GET_TIME_NSEC();

		dmaengine_terminate_all(spdif->tx_dma.chan);
	}

	if (sis_val & SPDIF_SIESC_RX_FIFO_UN_OV) {
		uint32_t scr = SPDIF_READ(scr);
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: Rx FIFO Under/Overrun");
		spdif->stats.irq_rx_fifo_un_ov++;

		scr |= SPDIF_SCR_RXFIFO_RST | SPDIF_SCR_RXFIFO_OFF;
		SPDIF_WRITE(scr, scr);
		spdif_softreset(spdif);
		spdif->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		spdif->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		spdif->stats.rx_stop = GET_TIME_NSEC();

		dmaengine_terminate_all(spdif->rx_dma.chan);
	}

	if (sis_val & SPDIF_SIESC_U_RX_OV) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: Rx U Channel Overrun");
		spdif->stats.irq_u_rx_ov++;
	}

	if (sis_val & SPDIF_SIESC_Q_RX_OV) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: Rx Q Channel Overrun");
		spdif->stats.irq_q_rx_ov++;
	}

	return IRQ_HANDLED;
}

static unsigned int lla_spdif_tx_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	unsigned int mask = 0;
	struct imx_spdif *spdif = context;

	poll_wait(filp, &(spdif->tx_dma.poll_event), table);

	if (spdif->tx_shared->poll_num < (spdif->tx_shared->buffers_complete + spdif->tx_shared->tx_latency)) {
		mask |= POLLOUT | POLLWRNORM;
	}

	return mask;
}

int lla_spdif_tx_commit(unsigned long buffer_idx, void *context)
{
	struct imx_spdif *spdif = context;
	dma_addr_t buffer_p = spdif->tx_shared->buffers.p + (buffer_idx * spdif->tx_shared->buffer_len);

	if (buffer_idx >= spdif->tx_shared->buffer_cnt) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "buffer index out of bounds (%lu)", buffer_idx);
		return -EFAULT;
	}

	sdma_sonos_swap_data_pointer(spdif->tx_dma.chan, buffer_idx, buffer_p);

	return 0;
}

static struct lla_ops lla_tx_ops = {
	.poll	= lla_spdif_tx_poll,
	.commit	= lla_spdif_tx_commit,
};

static int lla_tx_setup(struct imx_spdif *spdif)
{
	if (lla_init_dev(&(spdif->tx_lla_dev), spdif->dev, DRV_NAME"_tx", LLA_DEV_TYPE_SPDIF_TX, &lla_tx_ops, spdif)) {
		return -1;
	}

	if (spdif->tx_params.lla_sync_to_dev != LLA_DEV_TYPE_UNKNOWN) {
		if (lla_sync_to_dev(&(spdif->tx_lla_dev), spdif->tx_params.lla_sync_to_dev)) {
			return -1;
		}
	}

	spdif->tx_shared = lla_allocate_shared_buffers(
			&(spdif->tx_lla_dev),
			spdif->tx_params.channel_len,
			spdif->tx_params.channels_per_frame,
			spdif->tx_params.frames_per_buffer,
			spdif->tx_params.buffer_cnt,
			1, NULL);
	if (spdif->tx_shared == NULL) {
		return -1;
	}

	spdif->tx_shared->flags |= LLA_SHARED_FLAG_TX | LLA_SHARED_FLAG_NEED_COMMIT_TX;
	spdif->tx_shared->tx_latency = spdif->tx_params.buffer_cnt;
	spdif->tx_shared->channel_mask = spdif->tx_params.channel_mask;

	lla_update_sample_rate(&(spdif->tx_lla_dev), spdif->tx_params.frames_per_second);

	return 0;
}

static void spdif_tx_dma_callback(void *param)
{
	struct imx_spdif *spdif = param;
	u32 completion_time;
	u32 buffer_idx;
	dma_addr_t original_ptr;

	if (!(spdif->tx_shared->flags & (LLA_SHARED_FLAG_RUNNING | LLA_SHARED_FLAG_ERROR))) {
		spdif->tx_shared->flags |= LLA_SHARED_FLAG_RUNNING;
		spdif->stats.tx_start = ktime_to_ns(ktime_get());
	}

	buffer_idx = LLA_BUFFER_NUM_TO_IDX(spdif->tx_shared, spdif->tx_shared->buffers_complete);
	original_ptr = sdma_sonos_swap_data_pointer(spdif->tx_dma.chan, buffer_idx, spdif->tx_silence.p);

	if ((spdif->tx_shared->flags & LLA_SHARED_FLAG_ACTIVE)
	    && (original_ptr == spdif->tx_silence.p)) {
		spdif->tx_shared->flags &= ~LLA_SHARED_FLAG_ACTIVE;
	} else if (!(spdif->tx_shared->flags & LLA_SHARED_FLAG_ACTIVE)
		   && (original_ptr != spdif->tx_silence.p)) {
		spdif->tx_shared->flags |= LLA_SHARED_FLAG_ACTIVE;
	}

	if ( (IS_LLA_DEV_IN_USE( &(spdif->tx_lla_dev))) && (original_ptr == spdif->tx_silence.p) ) {
		spdif->tx_shared->tx_underflows++;
	}


	completion_time = lla_clock_read();
	spdif->tx_shared->complete_time = completion_time;
	dmb();

	spdif->tx_shared->buffers_complete++;
	dmb();
	spdif->tx_shared->complete_time2 = completion_time;

	wake_up_interruptible(&(spdif->tx_dma.poll_event));
}

static int spdif_tx_dma_setup(struct imx_spdif *spdif)
{
	int ret;
	int buffer_idx;
	struct dma_slave_config slave_config = {
		.direction = DMA_MEM_TO_DEV,
		.dst_addr = (spdif->res->start + SPDIF_STL),
		.dst_addr_width = DMA_SLAVE_BUSWIDTH_4_BYTES,
		.dst_maxburst = SPDIF_FIFO_WATERMARK_WORDS,
	};

	init_waitqueue_head(&(spdif->tx_dma.poll_event));

	spdif->tx_dma.chan = dma_request_slave_channel(spdif->dev, "tx");
	if (spdif->tx_dma.chan == NULL) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "tx dma request failed");
		return -1;
	}

	ret = dmaengine_slave_config(spdif->tx_dma.chan, &slave_config);
	if (ret) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to config dmaengine for tx (%d)", ret);
		return -1;
	}

	spdif->tx_dma.desc = dmaengine_prep_dma_cyclic(
			spdif->tx_dma.chan,
			spdif->tx_shared->buffers.p,
			spdif->tx_shared->buffers_total,
			spdif->tx_shared->buffer_len,
			slave_config.direction,
			0);
	if (!spdif->tx_dma.desc) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to prepare cyclic dma");
		return -1;
	}

	spdif->tx_dma.desc->callback = spdif_tx_dma_callback;
	spdif->tx_dma.desc->callback_param = (void *)spdif;
	spdif->tx_dma.cookie = dmaengine_submit(spdif->tx_dma.desc);

	spdif->tx_silence.k = dma_alloc_coherent(NULL, spdif->tx_shared->buffer_len, (dma_addr_t*)&(spdif->tx_silence.p), GFP_KERNEL);
	if (spdif->tx_silence.k == NULL) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "silence allocation failed (size = %u)", spdif->tx_shared->buffer_len);
		return -1;
	}
	spdif->tx_silence.u = NULL;
	memset(spdif->tx_silence.k, 0x00, spdif->tx_shared->buffer_len);

	for (buffer_idx = 0; buffer_idx < spdif->tx_params.buffer_cnt; buffer_idx++) {
		sdma_sonos_swap_data_pointer(spdif->tx_dma.chan, buffer_idx, spdif->tx_silence.p);
	}

	dma_async_issue_pending(spdif->tx_dma.chan);

	return 0;
}

static int spdif_tx_dev_init(struct imx_spdif *spdif)
{
	u32 val;

	val = SPDIF_READ(scr);
	val &= SPDIF_TX_REG_MASK_SCR;
	val |= SPDIF_TX_REG_VALUE_SCR;
	SPDIF_WRITE(val, scr);

	val = spdif->tx_params.channel_status[0] << 16;
	val |= spdif->tx_params.channel_status[1] << 8;
	val |= spdif->tx_params.channel_status[2];
	SPDIF_WRITE(val, stcsch);
	val = spdif->tx_params.channel_status[3] << 16;
	val |= spdif->tx_params.channel_status[4] << 8;
	val |= spdif->tx_params.channel_status[5];
	SPDIF_WRITE(val, stcscl);

	val = SPDIF_READ(sie);
	val &= SPDIF_TX_REG_MASK_SIESC;
	val |= SPDIF_TX_REG_VALUE_SIESC;
	SPDIF_WRITE(val, sie);

	val = SPDIF_TX_REG_VALUE_STC;

	if (spdif->tx_params.clk_div) {
		val |= SPDIF_STC_TXCLK_DIV(spdif->tx_params.clk_div);
	} else {
		val |= SPDIF_STC_TXCLK_DIV(1);
	}

	SPDIF_WRITE(val, stc);

	return 0;
}

static unsigned int lla_spdif_rx_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	unsigned int mask = 0;
	struct imx_spdif *spdif = context;


	poll_wait(filp, &(spdif->rx_dma.poll_event), table);

	if ((spdif->rx_shared->poll_num < spdif->rx_shared->buffers_complete) &&
	    (spdif->rx_shared->flags & LLA_SHARED_FLAG_RUNNING)) {
		mask |= POLLIN | POLLRDNORM;
	}

	return mask;
}

static int lla_spdif_get_csb(struct lla_csb *csb, void *context)
{
	struct imx_spdif *spdif = context;
	int ret = -EAGAIN;

	spin_lock_irq(&spdif->rx_csb_lock);
	if (spdif->rx_csb_valid) {
		memcpy(csb->data, spdif->rx_csb_data, sizeof(spdif->rx_csb_data));
		ret = 0;
	}
	spin_unlock_irq(&spdif->rx_csb_lock);

	csb->len = sizeof(spdif->rx_csb_data);
	return ret;
}

static struct lla_ops lla_rx_ops = {
	.poll		= lla_spdif_rx_poll,
	.get_csb	= lla_spdif_get_csb,
};

static int lla_rx_setup(struct imx_spdif *spdif)
{
	if (lla_init_dev(&(spdif->rx_lla_dev), spdif->dev, DRV_NAME"_rx", LLA_DEV_TYPE_SPDIF_RX, &lla_rx_ops, spdif)) {
		return -1;
	}
	spdif->rx_shared = lla_allocate_shared_buffers(
			&(spdif->rx_lla_dev),
			spdif->rx_params.channel_len,
			spdif->rx_params.channels_per_frame,
			spdif->rx_params.frames_per_buffer,
			spdif->rx_params.buffer_cnt,
			1, NULL);
	if (spdif->rx_shared == NULL) {
		return -1;
	}

	spdif->rx_shared->flags |= LLA_SHARED_FLAG_RX;
	spdif->rx_shared->channel_mask = spdif->rx_params.channel_mask;

	lla_update_sample_rate(&(spdif->rx_lla_dev), 0);

	return 0;
}

static void spdif_rx_dma_callback(void *param)
{
	struct imx_spdif *spdif = param;
	u32 sample_rate;
	u32 completion_time;

#ifdef TEST_FAILURES
	if (test_fail_rx_dma_hang && spdif->rx_dpll_locked) {
		mdelay( ((spdif->rx_shared->buffer_cnt + 1) * spdif->rx_shared->frames_per_buffer) /
			 (SPDIF_WORST_CASE_SAMPLE_RATE / 1000));
		test_fail_rx_dma_hang = 0;
	}
#endif

	sample_rate = spdif_get_rxclk_rate(spdif);
	lla_update_sample_rate(&(spdif->rx_lla_dev), sample_rate);

#ifdef RATE_TRACKING
	rate_tracking_add(spdif, sample_rate);
#endif
#ifdef TEST_FAILURES
	if (test_fail_rate_zero) {
		lla_update_sample_rate(&(spdif->rx_lla_dev), 0);
	}
#endif


	completion_time = lla_clock_read();
	spdif->rx_shared->complete_time = completion_time;
	dmb();

	spdif->rx_shared->buffers_complete++;
	dmb();
	spdif->rx_shared->complete_time2 = completion_time;

	wake_up_interruptible(&(spdif->rx_dma.poll_event));
}

static int spdif_rx_dma_setup(struct imx_spdif *spdif)
{
	int ret;
	struct dma_slave_config slave_config = {
		.direction = DMA_DEV_TO_MEM,
		.src_addr = (spdif->res->start + SPDIF_SRL),
		.src_addr_width = DMA_SLAVE_BUSWIDTH_4_BYTES,
		.src_maxburst = SPDIF_FIFO_WATERMARK_WORDS,
	};

	init_waitqueue_head(&(spdif->rx_dma.poll_event));

	spdif->rx_dma.chan = dma_request_slave_channel(spdif->dev, "rx");
	if (spdif->rx_dma.chan == NULL) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "rx dma request failed");
		return -1;
	}

	ret = dmaengine_slave_config(spdif->rx_dma.chan, &slave_config);
	if (ret) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to config dmaengine for rx (%d)", ret);
		return -1;
	}

	spdif->rx_dma.desc = dmaengine_prep_dma_cyclic(
			spdif->rx_dma.chan,
			spdif->rx_shared->buffers.p,
			spdif->rx_shared->buffers_total,
			spdif->rx_shared->buffer_len,
			slave_config.direction,
			0);
	if (!spdif->rx_dma.desc) {
		bb_log_dev(spdif->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to prepare cyclic dma");
		return -1;
	}

	spdif->rx_dma.desc->callback = spdif_rx_dma_callback;
	spdif->rx_dma.desc->callback_param = (void *)spdif;
	spdif->rx_dma.cookie = dmaengine_submit(spdif->rx_dma.desc);

	dma_async_issue_pending(spdif->rx_dma.chan);

	return 0;
}

static int spdif_rx_dev_init(struct imx_spdif *spdif)
{
	u32 val;

	val = SPDIF_READ(scr);
	val &= SPDIF_RX_REG_MASK_SCR;
	val |= SPDIF_RX_REG_VALUE_SCR;
	SPDIF_WRITE(val, scr);

	SPDIF_WRITE(SPDIF_RX_REG_VALUE_SRCD, srcd);

	val = SPDIF_READ(srpc);
	val &= SPDIF_RX_REG_MASK_SRPC;
	val |= SPDIF_RX_REG_VALUE_SRPC;
	SPDIF_WRITE(val, srpc);

	val = SPDIF_READ(sie);
	val &= SPDIF_RX_REG_MASK_SIESC;
	val |= SPDIF_RX_REG_VALUE_SIESC;
	SPDIF_WRITE(val, sie);

	return 0;
}

static int imx_spdif_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct device_node *child;
	struct imx_spdif *spdif;
	int ret = 0;

	spdif = devm_kzalloc(&pdev->dev, sizeof(*spdif), GFP_KERNEL);
	if (!spdif) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "mem allocation failed");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, spdif);
	spdif->pdev = pdev;
	spdif->dev = &(pdev->dev);

	spdif->res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (IS_ERR(spdif->res)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not determine device resources");
		return PTR_ERR(spdif->res);
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "iomem range %08x-%08x", spdif->res->start, spdif->res->end);

	spdif->regs = devm_ioremap_resource(&pdev->dev, spdif->res);
	if (IS_ERR(spdif->regs)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not map device resources");
		return PTR_ERR(spdif->regs);
	}

	spin_lock_init(&spdif->rx_csb_lock);

	spdif->irq = platform_get_irq(pdev, 0);
	if (spdif->irq == NO_IRQ) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "no irq for node %s", np->full_name);
		return spdif->irq;
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "irq %d", spdif->irq);

	ret = devm_request_irq(&pdev->dev, spdif->irq, imx_spdif_isr, 0, DRV_NAME, spdif);
	if (ret) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not claim irq %u: %d", spdif->irq, ret);
		return ret;
	}

	spdif->spdif_clk = devm_clk_get(&pdev->dev, "core");
	if (IS_ERR(spdif->spdif_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get core clock");
		return PTR_ERR(spdif->spdif_clk);
	}
	clk_prepare_enable(spdif->spdif_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "clock rate %ld", clk_get_rate(spdif->spdif_clk));

	spdif->dma_clk = devm_clk_get(&pdev->dev, "dma");
	if (IS_ERR(spdif->dma_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get dma clock");
		ret = PTR_ERR(spdif->dma_clk);
		goto failed_probe;
	}
	clk_prepare_enable(spdif->dma_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "dma clock rate %ld", clk_get_rate(spdif->dma_clk));

	spdif->ipg_clk = devm_clk_get(&pdev->dev, "ipg");
	if (IS_ERR(spdif->ipg_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get IPG clock");
		ret = PTR_ERR(spdif->ipg_clk);
		goto failed_probe;
	}
	clk_prepare_enable(spdif->ipg_clk);
	spdif->ipg_clk_freq = clk_get_rate(spdif->ipg_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "IPG clock rate %ld", spdif->ipg_clk_freq);


	spdif->rx_clk = devm_clk_get(&pdev->dev, "rxtx1");
	if (IS_ERR(spdif->rx_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get rx clock");
		ret = PTR_ERR(spdif->rx_clk);
		goto failed_probe;
	}
	clk_prepare_enable(spdif->rx_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "rx clock rate %ld", clk_get_rate(spdif->rx_clk));

	spdif->tx_clk = devm_clk_get(&pdev->dev, "rxtx4");
	if (IS_ERR(spdif->tx_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get tx clock");
		ret = PTR_ERR(spdif->tx_clk);
		goto failed_probe;
	}
	clk_prepare_enable(spdif->tx_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "tx clock rate %ld", clk_get_rate(spdif->tx_clk));

	for_each_available_child_of_node(pdev->dev.of_node, child) {
		u32 type;
		struct spdif_params *params = NULL;
		ret = of_property_read_u32(child, "lla-type", &type);
		if (ret) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "%p type read returned %d", child, ret);
			continue;
		}
		if (type == LLA_DEV_TYPE_SPDIF_TX) {
			params = &(spdif->tx_params);
		} else {
			params = &(spdif->rx_params);
		}
		params->enable = 1;
		params->lla_type = type;
		params->lla_sync_to_dev = LLA_DEV_TYPE_UNKNOWN;
		of_property_read_u32(child, "lla-sync-to-dev", &(params->lla_sync_to_dev));
		of_property_read_u32(child, "frames-per-second", &(params->frames_per_second));
		of_property_read_u32(child, "channel-len", &(params->channel_len));
		of_property_read_u32(child, "channel-mask", &(params->channel_mask));
		of_property_read_u32(child, "channels-per-frame", &(params->channels_per_frame));
		of_property_read_u32(child, "frames-per-buffer", &(params->frames_per_buffer));
		of_property_read_u32(child, "buffer-cnt", &(params->buffer_cnt));
		of_property_read_u32(child, "clk-div", &(params->clk_div));
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_DEBUG, "type %u len %u mask %#x cpf %u fpb %u bufs %u",
				type,
				params->channel_len,
				params->channel_mask,
				params->channels_per_frame,
				params->frames_per_buffer,
				params->buffer_cnt);

		if (type == LLA_DEV_TYPE_SPDIF_TX) {
			int len;
			const void *data = of_get_property(child, "channel-status", &len);
			if (data) {
				if (len > sizeof(params->channel_status)) {
					bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_WARNING,
						   "TX CSB truncate (%d > %d)",
						   len, sizeof(params->channel_status));
					len = sizeof(params->channel_status);
				}
				memcpy(params->channel_status, data, len);
			}
			else {
				bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_WARNING, "TX CSB not found");
				memset(params->channel_status, 0, sizeof(params->channel_status));
			}
		}
	}

	ret = procfs_init(spdif);
	if (ret) {
		goto failed_probe;
	}

	if (spdif_softreset(spdif)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed soft reset");
		goto failed_probe;
	}

	if (spdif->tx_params.enable) {
		ret = lla_tx_setup(spdif);
		if (ret) {
			goto failed_probe;
		}

		ret = spdif_tx_dma_setup(spdif);
		if (ret) {
			goto failed_probe;
		}

		ret = spdif_tx_dev_init(spdif);
		if (ret) {
			goto failed_probe;
		}

		lla_register_dev(&(spdif->tx_lla_dev));
	}

	if (spdif->rx_params.enable) {
		init_timer(&(spdif->rx_error_timer));
		spdif->rx_error_timer.function = spdif_rx_error_timer;
		spdif->rx_error_timer.data = (unsigned long)spdif;

		init_timer(&(spdif->rx_dpll_lock_timer));
		spdif->rx_dpll_lock_timer.function = spdif_rx_dpll_lock_timer;
		spdif->rx_dpll_lock_timer.data = (unsigned long)spdif;

		ret = lla_rx_setup(spdif);
		if (ret) {
			goto failed_probe;
		}

		ret = spdif_rx_dma_setup(spdif);
		if (ret) {
			goto failed_probe;
		}

		ret = spdif_rx_dev_init(spdif);
		if (ret) {
			goto failed_probe;
		}

		lla_register_dev(&(spdif->rx_lla_dev));
	}

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "registered");

	return ret;
failed_probe:
	if (spdif->tx_clk) {
		clk_disable_unprepare(spdif->tx_clk);
	}
	if (spdif->rx_clk) {
		clk_disable_unprepare(spdif->rx_clk);
	}
	if (spdif->dma_clk) {
		clk_disable_unprepare(spdif->dma_clk);
	}
	if (spdif->spdif_clk) {
		clk_disable_unprepare(spdif->spdif_clk);
	}
	return ret;
}

static int imx_spdif_remove(struct platform_device *pdev)
{
	struct imx_spdif *spdif = platform_get_drvdata(pdev);

	procfs_remove(spdif);

	disable_irq(spdif->irq);

	if (spdif->tx_params.enable) {
		lla_unregister_dev(&(spdif->tx_lla_dev));
		dma_release_channel(spdif->tx_dma.chan);
		lla_free_shared_buffers(&(spdif->tx_lla_dev));
	}

	if (spdif->rx_params.enable) {
		del_timer(&(spdif->rx_dpll_lock_timer));
		del_timer(&(spdif->rx_error_timer));

		lla_unregister_dev(&(spdif->rx_lla_dev));
		dma_release_channel(spdif->rx_dma.chan);
		lla_free_shared_buffers(&(spdif->rx_lla_dev));
	}

	spdif_softreset(spdif);

	clk_disable_unprepare(spdif->tx_clk);
	clk_disable_unprepare(spdif->rx_clk);
	clk_disable_unprepare(spdif->dma_clk);
	clk_disable_unprepare(spdif->spdif_clk);

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "unregistered");
	return 0;
}

static int imx_spdif_suspend(struct platform_device *pdev, pm_message_t state)
{
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "suspend");
	return 0;
}

static int imx_spdif_resume(struct platform_device *pdev)
{
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "resume");
	return 0;
}

static const struct of_device_id imx_spdif_ids[] = {
	{ .compatible = "fsl,imx6sx-spdif", },
	{}
};

static struct platform_driver imx_spdif_driver = {
	.probe = imx_spdif_probe,
	.remove = imx_spdif_remove,
	.suspend = imx_spdif_suspend,
	.resume = imx_spdif_resume,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = imx_spdif_ids,
	},
};

int imx_spdif_init(void)
{
	return platform_driver_register(&imx_spdif_driver);
}

void imx_spdif_exit(void)
{
	platform_driver_unregister(&imx_spdif_driver);
}
