/*
 * Copyright (c) 2014-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Copyright 2008-2013 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */


#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/kernel.h>
#include <linux/dma-mapping.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,9,0)
#include <linux/busfreq-imx6.h>
#endif
#include <linux/ktime.h>
#include <linux/hrtimer.h>
#include <linux/math64.h>
#include <linux/clk.h>
#include <linux/wait.h>
#include <linux/dmaengine.h>
#include <linux/platform_data/dma-imx.h>

#include <linux/sonos_kernel.h>
#include <asm/barrier.h>

#include "blackbox.h"
#include "lla.h"
#include "esai.h"
#include "event_queue_api.h"
#include "sonos_lock.h"
#include "mdp.h"

#define DRV_NAME		"esai"

#define PROCFS_PERM_READ	(S_IRUSR | S_IRGRP | S_IROTH)
#define PROCFS_PERM_WRITE	(S_IWUSR | S_IWGRP | S_IWOTH)

#define PROCFS_DIR		"driver/"DRV_NAME
#define PROCFS_REGS_FILE	"regs"
#define PROCFS_IRQS_FILE	"irqs"
#define PROCFS_DMA_FILE		"dma"
#define PROCFS_SNOOP_FILE	"snoop"
#define PROCFS_FAIL_FILE	"fail"

#define DMA_WATCHDOG_DURATION	(HZ)

struct esai_regs {
	u32 etdr;
	u32 erdr;
	u32 ecr;
	u32 esr;
	u32 tfcr;
	u32 tfsr;
	u32 rfcr;
	u32 rfsr;
	u8 reserved0[0x60];
	u32 tx[6];
	u32 tsr;
	u8 reserved1[0x4];
	u32 rx[4];
	u8 reserved2[0x1c];
	u32 saisr;
	u32 saicr;
	u32 tcr;
	u32 tccr;
	u32 rcr;
	u32 rccr;
	u32 tsma;
	u32 tsmb;
	u32 rsma;
	u32 rsmb;
	u8 reserved3[0x4];
	u32 prrc;
	u32 pcrc;
};

struct esai_dma {
	struct dma_chan			*chan;
	struct dma_async_tx_descriptor	*desc;
	dma_cookie_t			cookie;
	wait_queue_head_t		poll_event;
};

struct esai_params {
	u32 enable;
	u32 lla_type;
	u32 frames_per_second;
	u32 channel_len;
	u32 channel_mask;
	u32 channels_per_frame;
	u32 frames_per_buffer;
	u32 buffer_cnt;
	struct _div {
		u32 fp;
		u32 sr;
		u32 pm;
	} dividers;
	u32 watermark_frames;
	u32 active_lane_mask;
	u8 invert_fs;
	u8 tdm_master;
	u8 i2s_mode;
	u8 clock_only;
	u8 right_align;
	u8 sync_tx_rx;
	u8 nonzero_silence;
};

struct esai {
	struct esai_regs __iomem	*regs;
	struct resource			*res;
	int				irq;

	struct clk			*esai_clk;
	struct clk			*dma_clk;
	struct clk			*extal_clk;

	struct lla_ptr			tx_silence;

	struct esai_dma			tx_dma;
	struct esai_dma			rx_dma;
	struct timer_list		dma_watchdog;

	struct esai_params		tx_params;
	struct esai_params		rx_params;

	struct lla_dev			tx_lla_dev;
	struct lla_dev			rx_lla_dev;

	struct lla_shared		*tx_shared;
	struct lla_shared		*rx_shared;

	struct _esai_procfs {
		struct proc_dir_entry	*dir;
		struct proc_dir_entry	*regs;
		struct proc_dir_entry	*irqs;
		struct proc_dir_entry	*dma;
		struct proc_dir_entry	*snoop;
		struct proc_dir_entry	*fail;
	} procfs;

	struct _esai_stats {
		u32			irq_rx_overflow;
		u32			irq_rx_even_full;
		u32			irq_rx_full;
		u32			irq_rx_last_slot;
		u32			irq_tx_underrun;
		u32			irq_tx_even_empty;
		u32			irq_tx_empty;
		u32			irq_tx_last_slot;

		u64			tx_start;
		u64			tx_stop;

		u64			rx_start;
		u64			rx_stop;
	} stats;

	struct platform_device		*pdev;
	struct device			*dev;

	struct event_reg		*clocks_ready_event_reg;
};

#if LINUX_VERSION_CODE > KERNEL_VERSION(4,9,0)
enum bus_freq_mode {
	BUS_FREQ_HIGH,
	BUS_FREQ_MED,
	BUS_FREQ_AUDIO,
	BUS_FREQ_LOW,
};
static void request_bus_freq(enum bus_freq_mode mode)
{
	return;
}

static void release_bus_freq(enum bus_freq_mode mode)
{
	return;
}
#endif

#ifdef IMX_ESAI_REG_DEBUG

static inline u32 _read_n_print(const char *func, int line, void *reg_addr, const char *reg_name)
{
	u32 val = readl(reg_addr);
	printk("[READ]  %08x <-- %-5s (%s:%d)\n", val, reg_name, func, line);
	return val;
}

#define ESAI_READ(R) _read_n_print(__FUNCTION__, __LINE__, &(esai->regs->R), #R)

#define ESAI_WRITE(V, R) \
	do { \
		printk("[WRITE] %08x --> %-5s (%s:%d)\n", V, #R, __FUNCTION__, __LINE__); \
		writel(V, &(esai->regs->R)); \
	} while (0)

#else
#define ESAI_READ(R)		readl(&(esai->regs->R))
#define ESAI_WRITE(V, R)	writel(V, &(esai->regs->R))
#endif


#ifdef IMX_ESAI_DUMP
#define ESAI_REG_PRINT(R) \
	pr_info("ESAI_%-8s 0x%08x\n", #R, readl(&(esai->regs->R)));

#define ESAI_DUMP() \
	do { \
		pr_info("dump @ %s:%d\n", __FUNCTION__, __LINE__); \
		ESAI_REG_PRINT(ecr); \
		ESAI_REG_PRINT(esr); \
		ESAI_REG_PRINT(tfcr); \
		ESAI_REG_PRINT(tfsr); \
		ESAI_REG_PRINT(rfcr); \
		ESAI_REG_PRINT(rfsr); \
		ESAI_REG_PRINT(tsr); \
		ESAI_REG_PRINT(saisr); \
		ESAI_REG_PRINT(saicr); \
		ESAI_REG_PRINT(tcr); \
		ESAI_REG_PRINT(tccr); \
		ESAI_REG_PRINT(rcr); \
		ESAI_REG_PRINT(rccr); \
		ESAI_REG_PRINT(tsma); \
		ESAI_REG_PRINT(tsmb); \
		ESAI_REG_PRINT(rsma); \
		ESAI_REG_PRINT(rsmb); \
		ESAI_REG_PRINT(prrc); \
		ESAI_REG_PRINT(pcrc); \
	} while (0)
#else
#define ESAI_DUMP()
#endif


static int procfs_regs_show(struct seq_file *m, void *v)
{
	struct esai *esai = m->private;

	#define _ESAI_PROCFS_REG_SHOW(reg) seq_printf(m, "[%03x] ESAI_%-6s = %08x\n", (&(esai->regs->reg) - (u32*)esai->regs) * sizeof(u32), #reg, ESAI_READ(reg))

	seq_printf(m, "%08x-%08x\n", esai->res->start, esai->res->end);
	_ESAI_PROCFS_REG_SHOW(ecr);
	_ESAI_PROCFS_REG_SHOW(esr);
	_ESAI_PROCFS_REG_SHOW(tfcr);
	_ESAI_PROCFS_REG_SHOW(tfsr);
	_ESAI_PROCFS_REG_SHOW(rfcr);
	_ESAI_PROCFS_REG_SHOW(rfsr);
	_ESAI_PROCFS_REG_SHOW(tsr);
	_ESAI_PROCFS_REG_SHOW(saisr);
	_ESAI_PROCFS_REG_SHOW(saicr);
	_ESAI_PROCFS_REG_SHOW(tcr);
	_ESAI_PROCFS_REG_SHOW(tccr);
	_ESAI_PROCFS_REG_SHOW(rcr);
	_ESAI_PROCFS_REG_SHOW(rccr);
	_ESAI_PROCFS_REG_SHOW(tsma);
	_ESAI_PROCFS_REG_SHOW(tsmb);
	_ESAI_PROCFS_REG_SHOW(rsma);
	_ESAI_PROCFS_REG_SHOW(rsmb);
	_ESAI_PROCFS_REG_SHOW(prrc);
	_ESAI_PROCFS_REG_SHOW(pcrc);

	return 0;
}

static int procfs_regs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_regs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_regs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_regs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int procfs_irqs_show(struct seq_file *m, void *v)
{
	struct esai *esai = m->private;
	u32 rcr = ESAI_READ(rcr);
	u32 tcr = ESAI_READ(tcr);

	#define _IRQ_DISABLED_STR(reg, bit) (reg & bit) ? "" : "(disabled)"

	seq_printf(m, "IRQ #%d\n", esai->irq);

	seq_printf(m, "Rx Overflow        = %d %s\n", esai->stats.irq_rx_overflow, _IRQ_DISABLED_STR(rcr, ESAI_RCR_REIE));
	seq_printf(m, "Rx Full Even Data  = %d %s\n", esai->stats.irq_rx_even_full, _IRQ_DISABLED_STR(rcr, ESAI_RCR_REDIE));
	seq_printf(m, "Rx Full            = %d %s\n", esai->stats.irq_rx_full, _IRQ_DISABLED_STR(rcr, ESAI_RCR_RIE));
	seq_printf(m, "Rx Last Slot       = %d %s\n", esai->stats.irq_rx_last_slot, _IRQ_DISABLED_STR(rcr, ESAI_RCR_RLIE));
	seq_printf(m, "Tx Underflow       = %d %s\n", esai->stats.irq_tx_underrun, _IRQ_DISABLED_STR(tcr, ESAI_TCR_TEIE));
	seq_printf(m, "Tx Empty Even Data = %d %s\n", esai->stats.irq_tx_even_empty, _IRQ_DISABLED_STR(tcr, ESAI_TCR_TEDIE));
	seq_printf(m, "Tx Empty           = %d %s\n", esai->stats.irq_tx_empty, _IRQ_DISABLED_STR(tcr, ESAI_TCR_TIE));
	seq_printf(m, "Tx Last Slot       = %d %s\n", esai->stats.irq_tx_last_slot, _IRQ_DISABLED_STR(tcr, ESAI_TCR_TLIE));

	return 0;
}

static int procfs_irqs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_irqs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_irqs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_irqs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static void procfs_dma_show_single_stats(
		struct seq_file *m,
		const char *pfx,
		u64 start,
		u64 stop,
		struct lla_shared *shared)
{
	u64 elapsed = 0;
	u64 elapsed_sec;

	if (start) {
		if (!stop) {
			elapsed = ktime_to_ns(ktime_get()) - start;
		}
		else {
			elapsed = stop - start;
		}
	}
	elapsed_sec = div64_u64(elapsed, 1000000000);

	seq_printf(m, "%s DMAed Buffers    = %llu\n", pfx, shared->buffers_complete);
	seq_printf(m, "%s Total Bytes      = %llu\n", pfx, (shared->buffers_complete * shared->buffer_len));
	if (elapsed_sec) {
		seq_printf(m, "%s bytes/sec        = %llu\n", pfx, div64_u64(((shared->buffers_complete - 1) * shared->buffer_len), elapsed_sec));
	}
	else {
		seq_printf(m, "%s bytes/sec        = n/a\n", pfx);
	}
	seq_printf(m, "%s Duration (ns)    = %llu\n", pfx, elapsed);
	if (shared->buffers_complete > 1) {
		seq_printf(m, "%s ns/buffer        = %llu\n", pfx, div64_u64(elapsed, (shared->buffers_complete - 1)));
	}
	else {
		seq_printf(m, "%s ns/buffer        = n/a\n", pfx);
	}
}

static int procfs_dma_show(struct seq_file *m, void *v)
{
	struct esai *esai = m->private;

	if (esai->tx_params.enable) {
		procfs_dma_show_single_stats(m, "Tx", esai->stats.tx_start, esai->stats.tx_stop, esai->tx_shared);
	}
	if (esai->rx_params.enable && !esai->rx_params.clock_only) {
		procfs_dma_show_single_stats(m, "Rx", esai->stats.rx_start, esai->stats.rx_stop, esai->rx_shared);
	}

	return 0;
}

static int procfs_dma_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_dma_show, PDE_DATA(inode));
}

static const struct file_operations procfs_dma_ops = {
	.owner = THIS_MODULE,
	.open = procfs_dma_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

#ifdef SNOOP_RX

#define PROCFS_SNOOP_FILE	"snoop"

static int procfs_snoop_show(struct seq_file *m, void *v)
{
	struct esai *esai = m->private;
	seq_write(m, esai->rx_shared->buffers.k, esai->rx_shared->buffer_len);
	return 0;
}

static int procfs_snoop_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_snoop_show, PDE_DATA(inode));
}

static const struct file_operations procfs_snoop_ops = {
	.owner = THIS_MODULE,
	.open = procfs_snoop_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};
#endif


static int test_fail_tx_dma_hang = 0;
static int test_fail_rx_dma_hang = 0;

static int procfs_fail_show(struct seq_file *m, void *v)
{
	seq_printf(m, "Force ESAI failures by writing to this file\n");
	seq_printf(m, "Available Commands:\n\n");
	seq_printf(m, "    [tx-dma-hang | rx-dma-hang]   -- dma-hang\n");
	seq_printf(m, "                                     The next dma callback will stall the CPU until\n");
	seq_printf(m, "                                     the DMA engine has wrapped the entire ring buffer.\n");
	seq_printf(m, "    [tx-dma-stop | rx-dma-stop]   -- dma-stop\n");
	seq_printf(m, "                                     Stops all DMA.  Should cause a FIFO underflow/overflow.\n");
	seq_printf(m, "    [tx-lla-error | rx-lla-error] -- lla-error\n");
	seq_printf(m, "                                     This flags a false error in the shared structure which\n");
	seq_printf(m, "                                     will be seen by the userspace client.\n");
	return 0;
}

static int procfs_fail_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_fail_show, PDE_DATA(inode));
}

static ssize_t procfs_fail_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	struct esai *esai = PDE_DATA(file_inode(file));
	char cmd[32];
	char *newline;
	ssize_t result = count;

	if ((count < 0) || (count >= sizeof(cmd))) {
		result = -EIO;
	}
	else if (copy_from_user(&cmd, buffer, count)) {
		result = -EFAULT;
	}
	else {
		cmd[count] = '\0';

		newline = strchr(cmd, '\n');
		if (newline != NULL) {
			*newline = '\0';
		}

		if (strcmp(cmd, "tx-dma-hang") == 0) {
			bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Tx DMA hang");
			test_fail_tx_dma_hang++;
		}
		if (strcmp(cmd, "rx-dma-hang") == 0) {
			bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx DMA hang");
			test_fail_rx_dma_hang++;
		}

		if (strcmp(cmd, "tx-dma-stop") == 0) {
			if (esai->tx_dma.chan) {
				bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Tx DMA stop");
				dmaengine_terminate_all(esai->tx_dma.chan);
			} else {
				bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "No Tx DMA");
			}
		}
		if (strcmp(cmd, "rx-dma-stop") == 0) {
			if (esai->rx_dma.chan) {
				bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx DMA stop");
				dmaengine_terminate_all(esai->rx_dma.chan);
			} else {
				bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "No Rx DMA");
			}
		}

		if (strcmp(cmd, "tx-lla-error") == 0) {
			if (esai->tx_shared) {
				bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Tx LLA error flag");
				esai->tx_shared->flags |= LLA_SHARED_FLAG_ERROR;
				esai->tx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
			} else {
				bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "No Tx LLA");
			}
		}
		if (strcmp(cmd, "rx-lla-error") == 0) {
			if (esai->rx_shared) {
				bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "INJECTING FAILURE: Rx LLA error flag");
				esai->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
				esai->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
			} else {
				bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "No Rx LLA");
			}
		}
	}

	return result;
}

static const struct file_operations procfs_fail_ops = {
	.owner = THIS_MODULE,
	.open = procfs_fail_open,
	.read = seq_read,
	.write = procfs_fail_write,
	.llseek = seq_lseek,
	.release = single_release,
};

static void procfs_remove(struct esai *esai)
{
	if (esai->procfs.fail) {
		remove_proc_entry(PROCFS_FAIL_FILE, esai->procfs.dir);
	}

#ifdef SNOOP_RX
	if (esai->procfs.snoop) {
		remove_proc_entry(PROCFS_SNOOP_FILE, esai->procfs.dir);
	}
#endif
	if (esai->procfs.dma) {
		remove_proc_entry(PROCFS_DMA_FILE, esai->procfs.dir);
	}
	if (esai->procfs.regs) {
		remove_proc_entry(PROCFS_REGS_FILE, esai->procfs.dir);
	}
	if (esai->procfs.irqs) {
		remove_proc_entry(PROCFS_IRQS_FILE, esai->procfs.dir);
	}
	if (esai->procfs.dir) {
		remove_proc_entry(PROCFS_DIR, NULL);
	}
}

static int procfs_init(struct esai *esai)
{
	esai->procfs.dir = proc_mkdir(PROCFS_DIR, NULL);
	if (!esai->procfs.dir) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs dir %s", PROCFS_DIR);
		goto failed_procfs;
	}

	esai->procfs.regs = proc_create_data(PROCFS_REGS_FILE, PROCFS_PERM_READ, esai->procfs.dir, &procfs_regs_ops, esai);
	if (!esai->procfs.regs) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_REGS_FILE);
		goto failed_procfs;
	}

	esai->procfs.irqs = proc_create_data(PROCFS_IRQS_FILE, PROCFS_PERM_READ, esai->procfs.dir, &procfs_irqs_ops, esai);
	if (!esai->procfs.irqs) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_IRQS_FILE);
		goto failed_procfs;
	}

	esai->procfs.dma = proc_create_data(PROCFS_DMA_FILE, PROCFS_PERM_READ, esai->procfs.dir, &procfs_dma_ops, esai);
	if (!esai->procfs.dma) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_DMA_FILE);
		goto failed_procfs;
	}

#ifdef SNOOP_RX
	esai->procfs.snoop = proc_create_data(PROCFS_SNOOP_FILE, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), esai->procfs.dir, &procfs_snoop_ops, esai);
	if (!esai->procfs.snoop) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_SNOOP_FILE);
		goto failed_procfs;
	}
#endif

	if(is_mdp_authorized(MDP_AUTH_FLAG_DBG_UI_ENABLE)) {
		esai->procfs.fail = proc_create_data(PROCFS_FAIL_FILE, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), esai->procfs.dir, &procfs_fail_ops, esai);
		if (!esai->procfs.fail) {
			bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_FAIL_FILE);
			goto failed_procfs;
		}
		bb_log(BB_MOD_LLA, BB_LVL_INFO, "ESAI failure test points enabled");
	}

	return 0;
failed_procfs:
	procfs_remove(esai);
	return -1;
}

static void rx_drain_fifo(struct esai *esai)
{
	int read_cnt;

	for (read_cnt = 0; read_cnt < ESAI_WORDS_PER_FIFO; read_cnt++) {
		ESAI_READ(rx[0]);

		if (! (ESAI_READ(esr) & (ESAI_ESR_RFF & ESAI_ESR_RDE))) {
			return;
		}
	}

	if (ESAI_READ(esr) & (ESAI_ESR_RFF & ESAI_ESR_RDE)) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to drain rx fifo (ESR=%08x)", ESAI_READ(esr));
	}
}

static irqreturn_t esai_isr(int irq, void *p)
{
	struct esai *esai = p;
	u32 saisr, esr, val;
	irqreturn_t status = IRQ_NONE;

	saisr = ESAI_READ(saisr);
	esr = ESAI_READ(esr);

	bb_log_dbg_dev(esai->dev, BB_MOD_LLA, "IRQ: ESR=%08x SAISR=%08x", esr, saisr);

	if (esr & ESAI_ESR_RDE) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: Rx Overrun Error");
		esai->stats.irq_rx_overflow++;

#ifndef SONOS_ARCH_BOOTLEG
		val = ESAI_READ(rcr);
		val &= ESAI_RCR_RE_MASK;
		ESAI_WRITE(val, rcr);
#endif

		val = ESAI_READ(rfcr);
		val |= ESAI_RFCR_RFR;
		val &= ~ESAI_RFCR_RFEN;
		ESAI_WRITE(val, rfcr);
		val &= ~ESAI_RFCR_RFR;
		ESAI_WRITE(val, rfcr);

		rx_drain_fifo(esai);

		esai->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		esai->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		esai->stats.rx_stop = ktime_to_ns(ktime_get());

#ifndef SONOS_ARCH_BOOTLEG
		dmaengine_terminate_all(esai->rx_dma.chan);
#endif

		status = IRQ_HANDLED;
	}

	if (esr & ESAI_ESR_RED) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "IRQ: Rx Even-Data Register Full");
		esai->stats.irq_rx_even_full++;
	}

	if (esr & ESAI_ESR_RD) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "IRQ: Rx Data Register Full");
		esai->stats.irq_rx_full++;
	}

	if (esr & ESAI_ESR_RLS) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "Rx Last Slot");
		esai->stats.irq_rx_last_slot++;
	}

	if (esr & ESAI_ESR_TDE) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: Tx Underrun Error");
		esai->stats.irq_tx_underrun++;

#ifndef SONOS_ARCH_BOOTLEG
		val = ESAI_READ(tcr);
		val &= ESAI_TCR_TE_MASK;
		ESAI_WRITE(val, tcr);
#endif

		val = ESAI_READ(tfcr);
		val |= ESAI_TFCR_TFR;
		val &= ~ESAI_TFCR_TFEN;
		ESAI_WRITE(val, tfcr);
		val &= ~ESAI_TFCR_TFR;
		ESAI_WRITE(val, tfcr);

		esai->tx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		esai->tx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		esai->stats.tx_stop = ktime_to_ns(ktime_get());
#ifndef SONOS_ARCH_BOOTLEG
		dmaengine_terminate_all(esai->tx_dma.chan);
#endif

		ESAI_WRITE(0x0, tsr);

		status = IRQ_HANDLED;
	}

	if (esr & ESAI_ESR_TLS) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "IRQ: Tx Last Slot");
		esai->stats.irq_tx_last_slot++;
	}

	if (esr & ESAI_ESR_TED) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "IRQ: Tx Even-Data Register Empty");
		esai->stats.irq_tx_even_empty++;
	}

	if (esr & ESAI_ESR_TD) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_INFO, "IRQ: Tx Data Register Empty");
		esai->stats.irq_tx_empty++;
	}

	if (status == IRQ_NONE) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "ASSERT: Unhandled interrupt!");
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ: ESR=%08x SAISR=%08x", esr, saisr);
	}

	return status;
}

static void esai_dma_watchdog(unsigned long param)
{
	struct esai *esai = (struct esai*)param;


	if (esai->tx_params.enable &&
	    (esai->tx_shared->buffers_complete == 0)) {
		esai->tx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		esai->tx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "ASSERT: TX DMA failed to start!");
	}

	if (esai->rx_params.enable && !esai->rx_params.clock_only &&
	    (esai->rx_shared->buffers_complete == 0)) {
		esai->rx_shared->flags |= LLA_SHARED_FLAG_ERROR;
		esai->rx_shared->flags &= ~(LLA_SHARED_FLAG_RUNNING);
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "ASSERT: RX DMA failed to start!");
	}
}

static void esai_clocks_ready_callback(void *param, enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
	struct esai *esai = (struct esai *)param;
	if ((source == HWEVTQSOURCE_AMP) && (info == HWEVTQINFO_AMP_LO_RAIL)) {
		mod_timer(&(esai->dma_watchdog), jiffies + DMA_WATCHDOG_DURATION);
		event_unregister_event(&esai->clocks_ready_event_reg);
	} else {
		bb_log(BB_MOD_LLA, BB_LVL_ERR, "ASSERT: %s: bad callback source %s, info %s", __func__,
			   event_queue_get_source_string(source), event_queue_get_event_string(info));
	}
}

static void esai_start_dma_watchdog(struct esai *esai)
{
	if (!esai->tx_params.tdm_master) {
		esai->clocks_ready_event_reg = event_register_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_LO_RAIL, esai_clocks_ready_callback, esai, "clocks-ready");
		if (esai->clocks_ready_event_reg == NULL) {
			bb_log(BB_MOD_LLA, BB_LVL_ERR, "ASSERT: clock ready registration failed");
		}
	} else {
		mod_timer(&(esai->dma_watchdog), jiffies + DMA_WATCHDOG_DURATION);
	}
}

static int tx_write_frames(struct esai *esai, int num_frames, u32 value)
{
	int word;
	int frame;

	for (frame = 0; frame < num_frames; frame++) {
		for (word = 0; word < esai->tx_params.channels_per_frame; word++) {
			ESAI_WRITE(value, etdr);
		}
	}

	return (ESAI_READ(esr) & ESAI_ESR_TFE);
}

static void esai_tx_init(struct esai *esai)
{
	u32 tfcr_val, tccr_val = 0, tcr_val = 0;
	u32 active_lanes = 0;
	u32 cpf = 0;
	u32 wmark = ESAI_WORDS_PER_FIFO - (esai->tx_params.channels_per_frame * esai->tx_params.watermark_frames);
	int i;
	for (i = 0; i < sizeof(esai->tx_params.active_lane_mask) * 8; i++) {
		if (esai->tx_params.active_lane_mask & (1 << i)) {
			active_lanes++;
		}
	}

	cpf = esai->tx_params.channels_per_frame / active_lanes;

	tfcr_val = ESAI_READ(tfcr);
	tfcr_val |= ESAI_TFCR_TFR;
	ESAI_WRITE(tfcr_val, tfcr);

	tccr_val |= ESAI_TCCR_THCKP | ESAI_TCCR_TCKP
		| ESAI_TCCR_TDC(cpf - 1)
		| ESAI_TCCR_TFP(esai->tx_params.dividers.fp)
		| ESAI_TCCR_TPM(esai->tx_params.dividers.pm);
	if (!esai->tx_params.dividers.sr) {
		tccr_val |= ESAI_TCCR_TPSR_BYPASS;
	}
	if (esai->tx_params.tdm_master) {
		tccr_val |= ESAI_TCCR_THCKD | ESAI_TCCR_TCKD | ESAI_TCCR_TFSD;
	}
	if (esai->tx_params.invert_fs) {
		tccr_val |= ESAI_TCCR_TFSP;
	}
	ESAI_WRITE(tccr_val, tccr);

	ESAI_WRITE(0x00, tsmb);
	ESAI_WRITE(0x00, tsma);

	tfcr_val = ESAI_TFCR_TIEN
		| ESAI_TFCR_TFWM(wmark)
		| (esai->tx_params.active_lane_mask << 2);

	if (esai->tx_params.right_align) {
		tfcr_val |= ESAI_WORD_LEN_24;
	} else {
		tfcr_val |= ESAI_WORD_LEN_32;
	}

	ESAI_WRITE(tfcr_val, tfcr);

	tcr_val |= ESAI_TCR_TEIE
		| ESAI_TCR_PADC
		| ESAI_TCR_TSWS_STL32_WDL24
		| ESAI_TCR_TMOD_NETWORK
		| esai->tx_params.active_lane_mask;
	if (esai->tx_params.i2s_mode) {
		tcr_val |= ESAI_TCR_TFSR;
	}

	if (esai->tx_params.right_align) {
		tcr_val |= ESAI_TCR_TWA;
	}

	ESAI_WRITE(tcr_val, tcr);

	tfcr_val = ESAI_READ(tfcr);
	tfcr_val |= ESAI_TFCR_TFEN;
	ESAI_WRITE(tfcr_val, tfcr);

	tx_write_frames(esai, (ESAI_WORDS_PER_FIFO / esai->tx_params.channels_per_frame), 0x0);

	bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_DEBUG, "Tx init complete");
}

unsigned int esai_tx_lla_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	unsigned int mask = 0;
	struct esai *esai = context;


	poll_wait(filp, &(esai->tx_dma.poll_event), table);

	if (esai->tx_shared->poll_num < (esai->tx_shared->buffers_complete + esai->tx_shared->tx_latency)) {
		mask |= POLLOUT | POLLWRNORM;
	}

	return mask;
}

int esai_tx_lla_commit(unsigned long buffer_idx, void *context)
{
	struct esai *esai = context;
	dma_addr_t buffer_p = esai->tx_shared->buffers.p + (buffer_idx * esai->tx_shared->buffer_len);

	if (buffer_idx >= esai->tx_shared->buffer_cnt) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "buffer index out of bounds (%lu)", buffer_idx);
		return -EFAULT;
	}

	sdma_sonos_swap_data_pointer(esai->tx_dma.chan, buffer_idx, buffer_p);

	return 0;
}

static void esai_tx_dma_callback(void *param)
{
	struct esai *esai = param;
	u32 completion_time;

	if (test_fail_tx_dma_hang) {
		mdelay( ((esai->tx_shared->buffer_cnt + 1) * esai->tx_shared->frames_per_buffer) /
			 (esai->tx_params.frames_per_second / 1000));
		test_fail_tx_dma_hang = 0;
	}

	if (!(esai->tx_shared->flags & LLA_SHARED_FLAG_RUNNING)) {
		esai->tx_shared->flags |= LLA_SHARED_FLAG_RUNNING;
		esai->stats.tx_start = ktime_to_ns(ktime_get());
	}

	if (!esai->stats.irq_tx_underrun) {
		u32 buffer_idx;
		dma_addr_t original_ptr;

		buffer_idx = LLA_BUFFER_NUM_TO_IDX(esai->tx_shared, esai->tx_shared->buffers_complete);
		original_ptr = sdma_sonos_swap_data_pointer(esai->tx_dma.chan, buffer_idx, esai->tx_silence.p);

		if ((esai->tx_shared->flags & LLA_SHARED_FLAG_ACTIVE)
		    && (original_ptr == esai->tx_silence.p)) {
			esai->tx_shared->flags &= ~LLA_SHARED_FLAG_ACTIVE;
		} else if (!(esai->tx_shared->flags & LLA_SHARED_FLAG_ACTIVE)
		           && (original_ptr != esai->tx_silence.p)) {
			esai->tx_shared->flags |= LLA_SHARED_FLAG_ACTIVE;
		}

		if ( (IS_LLA_DEV_IN_USE( &(esai->tx_lla_dev))) && (original_ptr == esai->tx_silence.p) ) {
			esai->tx_shared->tx_underflows++;
		}


		completion_time = lla_clock_read();
		esai->tx_shared->complete_time = completion_time;
		dmb();

		esai->tx_shared->buffers_complete++;
		dmb();
		esai->tx_shared->complete_time2 = completion_time;
	}

	wake_up_interruptible(&(esai->tx_dma.poll_event));
}

static struct lla_ops lla_tx_ops = {
	.poll	= esai_tx_lla_poll,
	.commit	= esai_tx_lla_commit,
};

static int esai_tx_lla_setup(struct esai *esai)
{
	if (lla_init_dev(&(esai->tx_lla_dev), esai->dev, DRV_NAME"_tx", esai->tx_params.lla_type, &lla_tx_ops, esai)) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "tx lla init failed");
		return -1;
	}

	esai->tx_shared = lla_allocate_shared_buffers(
			&(esai->tx_lla_dev),
			esai->tx_params.channel_len,
			esai->tx_params.channels_per_frame,
			esai->tx_params.frames_per_buffer,
			esai->tx_params.buffer_cnt, 1, NULL);
	if (esai->tx_shared == NULL) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "tx lla buffer allocation failed");
		return -1;
	}

	esai->tx_shared->flags |= LLA_SHARED_FLAG_TX | LLA_SHARED_FLAG_NEED_COMMIT_TX;
	esai->tx_shared->tx_latency = esai->tx_params.buffer_cnt;
	esai->tx_shared->channel_mask = esai->tx_params.channel_mask;

	lla_update_sample_rate(&(esai->tx_lla_dev), esai->tx_params.frames_per_second);

	bb_log_dbg_dev(esai->dev, BB_MOD_LLA, "tx lla setup complete");
	return 0;
}

static int esai_tx_dma_setup(struct esai *esai)
{
	int ret;
	int buffer_idx;
	u32 wmark = ESAI_WORDS_PER_FIFO - (esai->tx_params.channels_per_frame * esai->tx_params.watermark_frames);
	struct dma_slave_config slave_config = {
		.direction = DMA_MEM_TO_DEV,
		.dst_addr = (esai->res->start + ESAI_ETDR),
		.dst_addr_width = DMA_SLAVE_BUSWIDTH_4_BYTES,
		.dst_maxburst = wmark,
	};

	init_waitqueue_head(&(esai->tx_dma.poll_event));

	esai->tx_dma.chan = dma_request_slave_channel(esai->dev, "tx");
	if (esai->tx_dma.chan == NULL) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "tx dma request failed");
		return -1;
	}

	ret = dmaengine_slave_config(esai->tx_dma.chan, &slave_config);
	if (ret) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to config dmaengine for tx (%d)", ret);
		return -1;
	}

	esai->tx_dma.desc = dmaengine_prep_dma_cyclic(
			esai->tx_dma.chan,
			esai->tx_shared->buffers.p,
			esai->tx_shared->buffers_total,
			esai->tx_shared->buffer_len,
			slave_config.direction,
			0);
	if (!esai->tx_dma.desc) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to prepare cyclic tx dma");
		return -1;
	}

	esai->tx_dma.desc->callback = esai_tx_dma_callback;
	esai->tx_dma.desc->callback_param = (void *)esai;
	esai->tx_dma.cookie = dmaengine_submit(esai->tx_dma.desc);

	esai->tx_silence.k = dma_alloc_coherent(NULL, esai->tx_shared->buffer_len, (dma_addr_t*)&(esai->tx_silence.p), GFP_KERNEL);
	if (esai->tx_silence.k == NULL) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "silence allocation failed (size = %u)", esai->tx_shared->buffer_len);
		return -1;
	}
	esai->tx_silence.u = NULL;

	if (esai->tx_params.nonzero_silence) {
		memset(esai->tx_silence.k, 0xFF, esai->tx_shared->buffer_len);
	} else {
		memset(esai->tx_silence.k, 0x00, esai->tx_shared->buffer_len);
	}

	for (buffer_idx = 0; buffer_idx < esai->tx_params.buffer_cnt; buffer_idx++) {
		sdma_sonos_swap_data_pointer(esai->tx_dma.chan, buffer_idx, esai->tx_silence.p);
	}

	dma_async_issue_pending(esai->tx_dma.chan);

	bb_log_dbg_dev(esai->dev, BB_MOD_LLA, "tx dma setup complete");
	return 0;
}

static void esai_tx_start(struct esai *esai)
{
	ESAI_WRITE(0x00, tsmb);
	ESAI_WRITE(((1 << esai->tx_params.channels_per_frame) - 1), tsma);

	esai->stats.tx_start = 0;
	esai->stats.tx_stop = 0;

	lla_register_dev(&(esai->tx_lla_dev));

	bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_DEBUG, "tx tdm started");
}

static void esai_rx_init(struct esai *esai)
{
	u32 rfcr_val, rccr_val = 0, rcr_val = 0;
	u32 wmark = ESAI_WORDS_PER_FIFO - (esai->rx_params.channels_per_frame * esai->rx_params.watermark_frames);

	rfcr_val = ESAI_READ(rfcr);
	rfcr_val |= ESAI_RFCR_RFR;
	ESAI_WRITE(rfcr_val, rfcr);

	rccr_val |= ESAI_RCCR_RHCKP | ESAI_RCCR_RCKP
		| ESAI_RCCR_RDC(esai->rx_params.channels_per_frame - 1)
		| ESAI_RCCR_RFP(esai->rx_params.dividers.fp)
		| ESAI_RCCR_RPM(esai->rx_params.dividers.pm);
	if (!esai->rx_params.dividers.sr) {
		rccr_val |= ESAI_RCCR_RPSR_BYPASS;
	}
	if (esai->rx_params.tdm_master) {
		rccr_val |= ESAI_RCCR_RHCKD | ESAI_RCCR_RCKD | ESAI_RCCR_RFSD;
	}
	if (esai->rx_params.invert_fs) {
		rccr_val |= ESAI_RCCR_RFSP;
	}
	ESAI_WRITE(rccr_val, rccr);

	ESAI_WRITE(0x00, rsmb);
	ESAI_WRITE(0x00, rsma);

	rfcr_val = ESAI_RFCR_REXT
		| ESAI_RFCR_RFWM(wmark)
		| (esai->rx_params.active_lane_mask << 2);
	if (esai->rx_params.right_align) {
		rfcr_val |= ESAI_WORD_LEN_24;
	} else {
		rfcr_val |= ESAI_WORD_LEN_32;
	}
	ESAI_WRITE(rfcr_val, rfcr);

	rcr_val |= ESAI_RCR_REIE
		| ESAI_RCR_RSWS_STL32_WDL24
		| ESAI_RCR_RMOD_NETWORK
		| esai->rx_params.active_lane_mask;
	if (esai->rx_params.i2s_mode) {
		rcr_val |= ESAI_RCR_RFSR;
	}
	ESAI_WRITE(rcr_val, rcr);

	if (!esai->rx_params.clock_only) {
		rfcr_val = ESAI_READ(rfcr);
		rfcr_val |= ESAI_RFCR_RFEN;
		ESAI_WRITE(rfcr_val, rfcr);
	}

	bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_DEBUG, "Rx init complete");
}

unsigned int esai_rx_lla_poll(struct file *filp, struct poll_table_struct *table, void *context)
{
	unsigned int mask = 0;
	struct esai *esai = context;


	poll_wait(filp, &(esai->rx_dma.poll_event), table);

	if (esai->rx_shared->poll_num < esai->rx_shared->buffers_complete) {
		mask |= POLLIN | POLLRDNORM;
	}

	return mask;
}

static void esai_rx_dma_callback(void *param)
{
	struct esai *esai = param;
	u32 completion_time;

	if (test_fail_rx_dma_hang) {
		mdelay( ((esai->rx_shared->buffer_cnt + 1) * esai->rx_shared->frames_per_buffer) /
			 (esai->rx_params.frames_per_second / 1000));
		test_fail_rx_dma_hang = 0;
	}

	if (!(esai->rx_shared->flags & LLA_SHARED_FLAG_RUNNING)) {
		esai->rx_shared->flags |= LLA_SHARED_FLAG_RUNNING;
		esai->stats.rx_start = ktime_to_ns(ktime_get());
	}


	completion_time = lla_clock_read();
	esai->rx_shared->complete_time = completion_time;
	dmb();

	esai->rx_shared->buffers_complete++;
	dmb();
	esai->rx_shared->complete_time2 = completion_time;

	wake_up_interruptible(&(esai->rx_dma.poll_event));
}

static struct lla_ops lla_rx_ops = {
	.poll	= esai_rx_lla_poll,
};

static int esai_rx_lla_setup(struct esai *esai)
{
	if (lla_init_dev(&(esai->rx_lla_dev), esai->dev, DRV_NAME"_rx", esai->rx_params.lla_type, &lla_rx_ops, esai)) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "rx lla init failed");
		return -1;
	}

	esai->rx_shared = lla_allocate_shared_buffers(
			&(esai->rx_lla_dev),
			esai->rx_params.channel_len,
			esai->rx_params.channels_per_frame,
			esai->rx_params.frames_per_buffer,
			esai->rx_params.buffer_cnt, 1, NULL);
	if (esai->rx_shared == NULL) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "rx lla buffer allocation failed");
		return -1;
	}

	esai->rx_shared->flags |= LLA_SHARED_FLAG_RX;
	esai->rx_shared->channel_mask = esai->rx_params.channel_mask;

	lla_update_sample_rate(&(esai->rx_lla_dev), esai->rx_params.frames_per_second);

	bb_log_dbg_dev(esai->dev, BB_MOD_LLA, "rx lla setup complete");
	return 0;
}

static int esai_rx_dma_setup(struct esai *esai)
{
	int ret;
	u32 wmark = ESAI_WORDS_PER_FIFO - (esai->rx_params.channels_per_frame * esai->rx_params.watermark_frames);
	struct dma_slave_config slave_config = {
		.direction = DMA_DEV_TO_MEM,
		.src_addr = (esai->res->start + ESAI_ERDR),
		.src_addr_width = DMA_SLAVE_BUSWIDTH_4_BYTES,
		.src_maxburst = wmark,
	};

	init_waitqueue_head(&(esai->rx_dma.poll_event));

	esai->rx_dma.chan = dma_request_slave_channel(esai->dev, "rx");
	if (esai->rx_dma.chan == NULL) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "rx dma request failed");
		return -1;
	}

	ret = dmaengine_slave_config(esai->rx_dma.chan, &slave_config);
	if (ret) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to config dmaengine for rx (%d)", ret);
		return -1;
	}

	esai->rx_dma.desc = dmaengine_prep_dma_cyclic(
			esai->rx_dma.chan,
			esai->rx_shared->buffers.p,
			esai->rx_shared->buffers_total,
			esai->rx_shared->buffer_len,
			slave_config.direction,
			0);
	if (!esai->rx_dma.desc) {
		bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to prepare cyclic rx dma");
		return -1;
	}

	esai->rx_dma.desc->callback = esai_rx_dma_callback;
	esai->rx_dma.desc->callback_param = (void *)esai;
	esai->rx_dma.cookie = dmaengine_submit(esai->rx_dma.desc);

	dma_async_issue_pending(esai->rx_dma.chan);

	bb_log_dbg_dev(esai->dev, BB_MOD_LLA, "rx dma setup complete");
	return 0;
}

static void esai_rx_start(struct esai *esai)
{
	ESAI_WRITE(0x00, rsmb);
	ESAI_WRITE(((1 << esai->rx_params.channels_per_frame) - 1), rsma);

	esai->stats.rx_start = 0;
	esai->stats.rx_stop = 0;

	lla_register_dev(&(esai->rx_lla_dev));

	bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_DEBUG, "rx tdm started");
}

static int esai_dev_init(struct esai *esai)
{
	u32 saicr, ecr;

	request_bus_freq(BUS_FREQ_HIGH);

	ESAI_WRITE(ESAI_ECR_ERST, ecr);
	ESAI_WRITE(ESAI_ECR_ESAIEN, ecr);
	bb_log_dbg_dev(esai->dev, BB_MOD_LLA, "core reset and logic clock enabled");

	ecr = ESAI_READ(ecr);
	ecr |= ESAI_ECR_ETI | ESAI_ECR_ERI;
	ESAI_WRITE(ecr, ecr);

	if (esai->tx_params.enable) {
		esai_tx_init(esai);
	}
	if (esai->rx_params.enable) {
		esai_rx_init(esai);
	}

	if (esai->tx_params.sync_tx_rx) {
		saicr = ESAI_READ(saicr);
		saicr |= ESAI_SAICR_SYNC;
		ESAI_WRITE(saicr, saicr);
	}

	ESAI_WRITE(ESAI_GPIO_ESAI, prrc);
	ESAI_WRITE(ESAI_GPIO_ESAI, pcrc);

	bb_log_dev(esai->dev, BB_MOD_LLA, BB_LVL_DEBUG, "init complete");

	if (esai->tx_params.enable) {
		if (esai_tx_lla_setup(esai) || esai_tx_dma_setup(esai)) {
			return -1;
		}
	}
	if (esai->rx_params.enable && !esai->rx_params.clock_only) {
		if (esai_rx_lla_setup(esai) || esai_rx_dma_setup(esai)) {
			return -1;
		}
	}
	init_timer(&(esai->dma_watchdog));
	esai->dma_watchdog.function = esai_dma_watchdog;
	esai->dma_watchdog.data = (unsigned long)esai;

	if (esai->tx_params.enable) {
		esai_tx_start(esai);
	}
	if (esai->rx_params.enable && !esai->rx_params.clock_only) {
		esai_rx_start(esai);
	}

	esai_start_dma_watchdog(esai);

	return 0;
}

static int esai_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct device_node *child;
	struct esai *esai;
	int ret = 0;

	esai = devm_kzalloc(&pdev->dev, sizeof(*esai), GFP_KERNEL);
	if (!esai) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "mem allocation failed");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, esai);
	esai->pdev = pdev;
	esai->dev = &(pdev->dev);

	esai->res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (IS_ERR(esai->res)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not determine device resources");
		return PTR_ERR(esai->res);
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "iomem range %08x-%08x", esai->res->start, esai->res->end);

	esai->regs = devm_ioremap_resource(&pdev->dev, esai->res);
	if (IS_ERR(esai->regs)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not map device resources");
		return PTR_ERR(esai->regs);
	}

	esai->irq = platform_get_irq(pdev, 0);
	if (esai->irq == NO_IRQ) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "no irq for node %s", np->full_name);
		return esai->irq;
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "irq %d", esai->irq);

	ret = devm_request_irq(&pdev->dev, esai->irq, esai_isr, 0, DRV_NAME, esai);
	if (ret) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not claim irq %u: %d", esai->irq, ret);
		return ret;
	}

	esai->esai_clk = devm_clk_get(&pdev->dev, "core");
	if (IS_ERR(esai->esai_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get core clock");
		return PTR_ERR(esai->esai_clk);
	}
	clk_prepare_enable(esai->esai_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "clock rate %ld", clk_get_rate(esai->esai_clk));

	esai->dma_clk = devm_clk_get(&pdev->dev, "dma");
	if (IS_ERR(esai->dma_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get dma clock");
		ret = PTR_ERR(esai->dma_clk);
		goto failed_probe;
	}
	clk_prepare_enable(esai->dma_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "dma clock rate %ld", clk_get_rate(esai->dma_clk));

	esai->extal_clk = devm_clk_get(&pdev->dev, "extal");
	if (IS_ERR(esai->extal_clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get extal clock");
		ret = PTR_ERR(esai->extal_clk);
		goto failed_probe;
	}
	clk_prepare_enable(esai->extal_clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "extal clock rate %ld", clk_get_rate(esai->extal_clk));

	for_each_available_child_of_node(pdev->dev.of_node, child) {
		u32 type;
		struct esai_params *params = NULL;
		ret = of_property_read_u32(child, "lla-type", &type);
		if (ret) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "%p type read returned %d", child, ret);
			continue;
		}
		if (type == LLA_DEV_TYPE_DAC) {
			params = &(esai->tx_params);
		} else {
			params = &(esai->rx_params);
		}
		params->enable = 1;
		params->lla_type = type;
		of_property_read_u32(child, "frames-per-second", &(params->frames_per_second));
		of_property_read_u32(child, "channel-len", &(params->channel_len));
		of_property_read_u32(child, "channel-mask", &(params->channel_mask));
		of_property_read_u32(child, "channels-per-frame", &(params->channels_per_frame));
		of_property_read_u32(child, "frames-per-buffer", &(params->frames_per_buffer));
		of_property_read_u32(child, "buffer-cnt", &(params->buffer_cnt));
		of_property_read_u32_array(child, "dividers", (u32 *)&(params->dividers), 3);
		if (of_property_read_u32(child, "active-lane-mask", &(params->active_lane_mask))) {
			params->active_lane_mask = 0x01;
		}
		of_property_read_u32(child, "watermark-frames", &(params->watermark_frames));
		params->invert_fs = of_property_read_bool(child, "invert-fs");
		params->tdm_master = of_property_read_bool(child, "tdm-master");
		params->i2s_mode = of_property_read_bool(child, "i2s-mode");
		params->clock_only = of_property_read_bool(child, "clock-only");
		params->right_align = of_property_read_bool(child, "right-align");
		params->sync_tx_rx = of_property_read_bool(child, "sync-tx-rx");
		params->nonzero_silence = of_property_read_bool(child, "nonzero-silence");
		if (params->clock_only && type == LLA_DEV_TYPE_DAC) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_WARNING, "clock_only is unused for TX");
		}
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_DEBUG, "type %u fps %u len %u mask %#x cpf %u fpb %u bufs %u divs %u %u %u lanes %#x",
				type,
				params->frames_per_second,
				params->channel_len,
				params->channel_mask,
				params->channels_per_frame,
				params->frames_per_buffer,
				params->buffer_cnt,
				params->dividers.fp, params->dividers.sr, params->dividers.pm,
				params->active_lane_mask);
	}

	ret = procfs_init(esai);
	if (ret) {
		goto failed_probe;
	}

	ret = esai_dev_init(esai);
	if (ret) {
		goto failed_probe;
	}

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "registered");

	return ret;
failed_probe:
	procfs_remove(esai);
	if (esai->extal_clk) {
		clk_disable_unprepare(esai->extal_clk);
	}
	if (esai->dma_clk) {
		clk_disable_unprepare(esai->dma_clk);
	}
	if (esai->esai_clk) {
		clk_disable_unprepare(esai->esai_clk);
	}
	return ret;
}

static int esai_remove(struct platform_device *pdev)
{
	struct esai *esai = platform_get_drvdata(pdev);

	if (!esai->tx_params.tdm_master && esai->clocks_ready_event_reg != NULL) {
		event_unregister_event(&esai->clocks_ready_event_reg);
	}

	procfs_remove(esai);

	del_timer(&(esai->dma_watchdog));

	disable_irq(esai->irq);

	if (esai->rx_params.enable && !esai->rx_params.clock_only) {
		lla_unregister_dev(&(esai->rx_lla_dev));
		dma_release_channel(esai->rx_dma.chan);
		lla_free_shared_buffers(&(esai->rx_lla_dev));
	}
	if (esai->tx_params.enable) {
		lla_unregister_dev(&(esai->tx_lla_dev));
		dma_release_channel(esai->tx_dma.chan);
		lla_free_shared_buffers(&(esai->tx_lla_dev));
	}

	ESAI_WRITE(0, ecr);

	clk_disable_unprepare(esai->extal_clk);
	clk_disable_unprepare(esai->dma_clk);
	clk_disable_unprepare(esai->esai_clk);

	release_bus_freq(BUS_FREQ_HIGH);

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "unregistered");
	return 0;
}

static int esai_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct esai *esai = platform_get_drvdata(pdev);
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "suspend");
	ESAI_WRITE(0x0, tsma);
	ESAI_WRITE(0x0, tsmb);
	return 0;
}

static int esai_resume(struct platform_device *pdev)
{
	struct esai *esai = platform_get_drvdata(pdev);
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "resume");
	ESAI_WRITE(0x00, tsmb);
	ESAI_WRITE(((1 << esai->tx_params.channels_per_frame) - 1), tsma);
	return 0;
}

static const struct of_device_id esai_ids[] = {
	{ .compatible = "fsl,imx6q-esai", },
	{}
};

static struct platform_driver esai_driver = {
	.probe = esai_probe,
	.remove = esai_remove,
	.suspend = esai_suspend,
	.resume = esai_resume,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = esai_ids,
	},
};

int esai_init(void)
{
	return platform_driver_register(&esai_driver);
}

void esai_exit(void)
{
	platform_driver_unregister(&esai_driver);
}
