/*
 * Copyright (c) 2015-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/version.h>
#include <linux/clk.h>
#include <linux/wait.h>
#include <linux/dmaengine.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,9,0)
#include <linux/busfreq-imx6.h>
#endif

#include "blackbox.h"
#include "lla.h"
#include "epit.h"

#define DRV_NAME		"epit"

struct epit_regs {
	u32 cr;
	u32 sr;
	u32 lr;
	u32 cmpr;
	u32 cnr;
};

#define EPIT_CR_VAL (\
		EPIT_CR_CLKSRC_REF_HIGH |	 \
		 \
		 \
		EPIT_CR_WAITEN |		 \
		 \
		 \
		 \
		EPIT_CR_PRESC(0) |		 \
		 \
		 \
		EPIT_CR_ENMOD |			 \
		EPIT_CR_EN			 \
)

#define EPIT_DEV_NAME_MAX	16
#define EPIT_DIR_NAME_MAX	32

struct epit {
	u32				dev_id;
	char				dev_name[EPIT_DEV_NAME_MAX];

	struct epit_regs __iomem	*regs;
	struct resource			*res;
	int				irq;
	struct clk			*clk;

	struct clk			*src_clk;
	unsigned long			src_clk_rate;

	bool				for_lla;
	struct lla_hw_clk		lla_clk_src;

	struct _epit_procfs {
		char			dir_name[EPIT_DIR_NAME_MAX];
		struct proc_dir_entry	*dir;
		struct proc_dir_entry	*regs;
		struct proc_dir_entry	*hz;
	} procfs;

	struct platform_device		*pdev;
	struct device			*dev;
};

#define PROCFS_PERM_READ	(S_IRUSR | S_IRGRP | S_IROTH)
#define PROCFS_PERM_WRITE	(S_IWUSR | S_IWGRP | S_IWOTH)

#define PROCFS_DIR		"driver/"DRV_NAME
#define PROCFS_REGS_FILE	"regs"
#define PROCFS_HZ_FILE		"hz"

#define EPIT_READ(R)		readl(&(epit->regs->R))
#define EPIT_WRITE(V, R)	writel(V, &(epit->regs->R))


static int procfs_hz_show(struct seq_file *m, void *v)
{
	struct epit *epit = m->private;
	seq_printf(m, "%lu\n", epit->src_clk_rate);
	return 0;
}

static int procfs_hz_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_hz_show, PDE_DATA(inode));
}

static const struct file_operations procfs_hz_ops = {
	.owner = THIS_MODULE,
	.open = procfs_hz_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int procfs_regs_show(struct seq_file *m, void *v)
{
	struct epit *epit = m->private;

	#define _EPIT_PROCFS_REG_SHOW(reg) seq_printf(m, "[%03x] EPIT%d_%-7s = %08x\n", (&(epit->regs->reg) - (u32*)epit->regs) * sizeof(u32), epit->dev_id, #reg, EPIT_READ(reg))

	seq_printf(m, "%08x-%08x\n", epit->res->start, epit->res->end);
	_EPIT_PROCFS_REG_SHOW(cr);
	_EPIT_PROCFS_REG_SHOW(sr);
	_EPIT_PROCFS_REG_SHOW(lr);
	_EPIT_PROCFS_REG_SHOW(cmpr);
	_EPIT_PROCFS_REG_SHOW(cnr);

	return 0;
}

static int procfs_regs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_regs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_regs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_regs_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static void procfs_remove(struct epit *epit)
{
	if (epit->procfs.regs) {
		remove_proc_entry(PROCFS_REGS_FILE, epit->procfs.dir);
	}
	if (epit->procfs.hz) {
		remove_proc_entry(PROCFS_HZ_FILE, epit->procfs.dir);
	}
	if (epit->procfs.dir) {
		remove_proc_entry(epit->procfs.dir_name, NULL);
	}
}

static int procfs_init(struct epit *epit)
{
	snprintf(epit->procfs.dir_name, sizeof(epit->procfs.dir_name), PROCFS_DIR "%d", epit->dev_id);

	epit->procfs.dir = proc_mkdir(epit->procfs.dir_name, NULL);
	if (!epit->procfs.dir) {
		bb_log_dev(epit->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs dir %s", PROCFS_DIR);
		goto failed_procfs;
	}

	epit->procfs.regs = proc_create_data(PROCFS_REGS_FILE, PROCFS_PERM_READ, epit->procfs.dir, &procfs_regs_ops, epit);
	if (!epit->procfs.regs) {
		bb_log_dev(epit->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_REGS_FILE);
		goto failed_procfs;
	}

	epit->procfs.hz = proc_create_data(PROCFS_HZ_FILE, PROCFS_PERM_READ, epit->procfs.dir, &procfs_hz_ops, epit);
	if (!epit->procfs.hz) {
		bb_log_dev(epit->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_HZ_FILE);
		goto failed_procfs;
	}

	return 0;
failed_procfs:
	procfs_remove(epit);
	return -1;
}

static irqreturn_t epit_isr(int irq, void *p)
{
	struct epit *epit = p;
	irqreturn_t status = IRQ_NONE;

	bb_log_dev(epit->dev, BB_MOD_LLA, BB_LVL_ERR, "IRQ %d! WHAT DO I DO?!", irq);
	return status;
}

static u32 epit_lla_clock_read(void *context)
{
	struct epit *epit = context;
	return EPIT_READ(cnr);
}

static struct lla_hw_clk_ops lla_clk_ops = {
	.read	= epit_lla_clock_read,
};

static int epit_lla_clock_setup(struct epit *epit)
{
	if (lla_init_clk(&(epit->lla_clk_src), epit->dev_name, (epit->res->start + EPIT_CNR), epit->src_clk_rate, 0, &lla_clk_ops, epit)) {
		return -1;
	}

	if (lla_register_clk(&(epit->lla_clk_src))) {
		return -1;
	}
	return 0;
}

static int epit_dev_init(struct epit *epit)
{
	char *clk_name;

	switch (EPIT_CR_VAL & EPIT_CR_CLKSRC_MASK) {
		case EPIT_CR_CLKSRC_REF_LOW:
			clk_name = "low";
			break;
		case EPIT_CR_CLKSRC_REF_HIGH:
			clk_name = "high";
			break;
		case EPIT_CR_CLKSRC_PERIPHERAL:
			clk_name = "per";
			break;
		case EPIT_CR_CLKSRC_OFF:
		default:
			clk_name = "dummy";
			break;
	}

	epit->src_clk = devm_clk_get(epit->dev, clk_name);
	if (IS_ERR(epit->src_clk)) {
		bb_log_dev(epit->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get %s clock", clk_name);
		return PTR_ERR(epit->src_clk);
	}
	clk_prepare_enable(epit->src_clk);
	epit->src_clk_rate = clk_get_rate(epit->src_clk);
	bb_log_dbg_dev(epit->dev, BB_MOD_LLA, "source clock '%s' rate %ld", clk_name, epit->src_clk_rate);

	EPIT_WRITE(EPIT_CR_VAL, cr);

	return 0;
}

static int epit_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct epit *epit;
	int ret = 0;

	epit = devm_kzalloc(&pdev->dev, sizeof(*epit), GFP_KERNEL);
	if (!epit) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "mem allocation failed");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, epit);
	epit->pdev = pdev;
	epit->dev = &(pdev->dev);

	if (of_property_read_u32(np, "dev-id", &(epit->dev_id))) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "Could not get device ID.");
		return -ENODEV;
	}
	snprintf(epit->dev_name, sizeof(epit->dev_name), DRV_NAME"%d", epit->dev_id);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "I am %s.", epit->dev_name);

	epit->res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (IS_ERR(epit->res)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not determine device resources");
		return PTR_ERR(epit->res);
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "iomem range %08x-%08x", epit->res->start, epit->res->end);

	epit->regs = devm_ioremap_resource(&pdev->dev, epit->res);
	if (IS_ERR(epit->regs)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not map device resources");
		return PTR_ERR(epit->regs);
	}

	epit->irq = platform_get_irq(pdev, 0);
	if (epit->irq == NO_IRQ) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "no irq for node %s", np->full_name);
		return epit->irq;
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "irq %d", epit->irq);

	ret = devm_request_irq(&pdev->dev, epit->irq, epit_isr, 0, epit->dev_name, epit);
	if (ret) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not claim irq %u: %d", epit->irq, ret);
		return ret;
	}

	epit->clk = devm_clk_get(&pdev->dev, "epit");
	if (IS_ERR(epit->clk)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to get epit clock");
		return PTR_ERR(epit->clk);
	}
	clk_prepare_enable(epit->clk);
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "clock rate %ld", clk_get_rate(epit->clk));

	ret = procfs_init(epit);
	if (ret) {
		goto failed_probe;
	}

	ret = epit_dev_init(epit);
	if (ret) {
		goto failed_probe;
	}

	epit->for_lla = of_property_read_bool(epit->dev->of_node, "lla-clock");
	if (epit->for_lla) {
		ret = epit_lla_clock_setup(epit);
		if (ret) {
			goto failed_probe;
		}
	}

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "registered");
	return ret;
failed_probe:
	clk_disable_unprepare(epit->src_clk);
	clk_disable_unprepare(epit->clk);
	return ret;
}

static int epit_remove(struct platform_device *pdev)
{
	struct epit *epit = platform_get_drvdata(pdev);

	procfs_remove(epit);

	disable_irq(epit->irq);

	lla_unregister_clk(&(epit->lla_clk_src));

	EPIT_WRITE(0, cr);

	clk_disable_unprepare(epit->src_clk);
	clk_disable_unprepare(epit->clk);

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "unregistered");
	return 0;
}

static int epit_suspend(struct platform_device *pdev, pm_message_t state)
{
	return 0;
}

static int epit_resume(struct platform_device *pdev)
{
	return 0;
}

static const struct of_device_id epit_ids[] = {
	{ .compatible = "fsl,imx6sx-epit", },
	{ .compatible = "fsl,imx6q-epit", },
	{}
};

static struct platform_driver epit_driver = {
	.probe = epit_probe,
	.remove = epit_remove,
	.suspend = epit_suspend,
	.resume = epit_resume,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = epit_ids,
	},
};

int epit_init(void)
{
	return platform_driver_register(&epit_driver);
}

void epit_exit(void)
{
	platform_driver_unregister(&epit_driver);
}
