/*
 * Copyright (c) 2014-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The Audio MUX is a TDM routing peripheral in the IMX6 which
 * manages audio streams to and from the SSIs.  This driver just configures
 * the state of the AUDMUX ports during init.  It has no other processing.
 *
 * Copyright 2012 Freescale Semiconductor, Inc.
 * Copyright 2012 Linaro Ltd.
 * Copyright 2009 Pengutronix, Sascha Hauer <s.hauer@pengutronix.de>
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/version.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/ctype.h>
#include <linux/string.h>

#include "blackbox.h"
#include "audmux.h"

#define DRV_NAME		"audmux"

struct audmux_regs {
	struct _port_reg {
		u32 ptcr;
		u32 pdcr;
	} port[IMX_AUDMUX_PORTS];
};

struct audmux {
	struct audmux_regs __iomem	*regs;
	struct resource			*res;

	struct _audmux_procfs {
		struct proc_dir_entry	*dir;
		struct proc_dir_entry	*regs;
	} procfs;

	struct platform_device		*pdev;
	struct device			*dev;
};

#define PROCFS_PERM_READ	(S_IRUSR | S_IRGRP | S_IROTH)
#define PROCFS_PERM_WRITE	(S_IWUSR | S_IWGRP | S_IWOTH)

#define PROCFS_DIR		"driver/"DRV_NAME
#define PROCFS_REGS_FILE	"regs"

#define AUDMUX_READ(P, R)	readl(&(audmux->regs->port[(P-1)].R))
#define AUDMUX_WRITE(V, P, R)	writel(V, &(audmux->regs->port[(P-1)].R))

static void audmux_set_ptcr(struct audmux *audmux, int port, u32 val)
{
	u32 reg;

	reg = AUDMUX_READ(port, ptcr);
	reg &= IMX_AUDMUX_PTCR_SYN;
	reg |= val & ~(IMX_AUDMUX_PTCR_SYN);
	AUDMUX_WRITE(reg, port, ptcr);
	AUDMUX_WRITE(val, port, ptcr);
}

static inline void audmux_set_pdcr(struct audmux *audmux, int port, u32 val)
{
	AUDMUX_WRITE(val, port, pdcr);
}

static inline void audmux_set_port(struct audmux *audmux, int port, u32 ptcr_val, u32 pdcr_val)
{
	audmux_set_ptcr(audmux, port, ptcr_val);
	audmux_set_pdcr(audmux, port, pdcr_val);
}

static int procfs_regs_show(struct seq_file *m, void *v)
{
	struct audmux *audmux = m->private;
	int port_num;

	#define _AUDMUX_PROCFS_REG_SHOW(pnum, reg) seq_printf(m, "[%03x] AUDMUX_%s%d = %08x\n", (&(audmux->regs->port[(pnum-1)].reg) - (u32*)audmux->regs) * sizeof(u32), #reg, pnum, AUDMUX_READ(pnum, reg))

	seq_printf(m, "%08x-%08x\n", audmux->res->start, audmux->res->end);
	for (port_num = 1; port_num <= IMX_AUDMUX_PORTS; port_num++) {
		_AUDMUX_PROCFS_REG_SHOW(port_num, ptcr);
		_AUDMUX_PROCFS_REG_SHOW(port_num, pdcr);
	}

	return 0;
}

static ssize_t procfs_regs_write(struct file *file, const char __user *buffer, size_t count, loff_t *ppos)
{
	struct audmux *audmux = PDE_DATA(file_inode(file));
	char cmd[32];
	char *newline;
	char *regname;
	char *strval;
	int set_ptcr = -1;
	int port;
	unsigned long val;

	if ((count < 0) || (count >= sizeof(cmd))) {
		return -EIO;
	}
	if (copy_from_user(&cmd, buffer, count)) {
		return -EFAULT;
	}

	cmd[count] = '\0';

	newline = strchr(cmd, '\n');
	if (newline != NULL) {
		*newline = '\0';
	}

	regname = strstr(cmd, "ptcr");
	if (regname != NULL) {
		set_ptcr = 1;
	}
	else {
		regname = strstr(cmd, "pdcr");
		if (regname != NULL) {
			set_ptcr = 0;
		}
		else {
			bb_log_dev(audmux->dev, BB_MOD_LLA, BB_LVL_WARNING, "invalid register");
			goto giveup;
		}
	}

	port = regname[4] - '0';
	if ((port < 1) || (port > 7)) {
		bb_log_dev(audmux->dev, BB_MOD_LLA, BB_LVL_WARNING, "invalid port");
		goto giveup;
	}

	strval = strchr(regname, '=');
	if (strval == NULL) {
		bb_log_dev(audmux->dev, BB_MOD_LLA, BB_LVL_WARNING, "missing assignment");
		goto giveup;
	}

	do {
		strval++;
		if (*strval == '\0') {
			bb_log_dev(audmux->dev, BB_MOD_LLA, BB_LVL_WARNING, "missing value");
			goto giveup;
		}
	} while (isspace(*strval));

#if LINUX_VERSION_CODE < KERNEL_VERSION(4,9,0)
	if (strict_strtoul(strval, 16, &val))
#else
	if (kstrtoul(strval, 16, &val))
#endif
	{
		bb_log_dev(audmux->dev, BB_MOD_LLA, BB_LVL_WARNING, "bad value");
		goto giveup;
	}

	if (set_ptcr) {
		audmux_set_ptcr(audmux, port, val);
	} else {
		audmux_set_pdcr(audmux, port, val);
	}

	bb_log_dbg_dev(audmux->dev, BB_MOD_LLA, "Set %s%d to %08lx", set_ptcr ? "ptcr" : "pdcr", port, val);

giveup:
	return count;
}

static int procfs_regs_open(struct inode *inode, struct file *file)
{
	return single_open(file, procfs_regs_show, PDE_DATA(inode));
}

static const struct file_operations procfs_regs_ops = {
	.owner = THIS_MODULE,
	.open = procfs_regs_open,
	.read = seq_read,
	.write = procfs_regs_write,
	.llseek = seq_lseek,
	.release = single_release,
};

static void procfs_remove(struct audmux *audmux)
{
	if (audmux->procfs.regs) {
		remove_proc_entry(PROCFS_REGS_FILE, audmux->procfs.dir);
	}
	if (audmux->procfs.dir) {
		remove_proc_entry(PROCFS_DIR, NULL);
	}
}

static int procfs_init(struct audmux *audmux)
{
	audmux->procfs.dir = proc_mkdir(PROCFS_DIR, NULL);
	if (!audmux->procfs.dir) {
		bb_log_dev(audmux->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs dir %s/n", PROCFS_DIR);
		goto failed_procfs;
	}

	audmux->procfs.regs = proc_create_data(PROCFS_REGS_FILE, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), audmux->procfs.dir, &procfs_regs_ops, audmux);
	if (!audmux->procfs.regs) {
		bb_log_dev(audmux->dev, BB_MOD_LLA, BB_LVL_ERR, "failed to create procfs file %s", PROCFS_REGS_FILE);
		goto failed_procfs;
	}

	return 0;
failed_procfs:
	procfs_remove(audmux);
	return -1;
}

static int audmux_probe(struct platform_device *pdev)
{
	struct audmux *audmux;
	struct device_node *child;

	audmux = devm_kzalloc(&pdev->dev, sizeof(*audmux), GFP_KERNEL);
	if (!audmux) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "mem allocation failed");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, audmux);
	audmux->pdev = pdev;
	audmux->dev = &(pdev->dev);

	audmux->res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (IS_ERR(audmux->res)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not determine device resources");
		return PTR_ERR(audmux->res);
	}
	bb_log_dbg_dev(&pdev->dev, BB_MOD_LLA, "iomem range %08x-%08x", audmux->res->start, audmux->res->end);

	audmux->regs = devm_ioremap_resource(&pdev->dev, audmux->res);
	if (IS_ERR(audmux->regs)) {
		bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_ERR, "could not map device resources");
		return PTR_ERR(audmux->regs);
	}

	if (procfs_init(audmux)) {
		return -1;
	}

	for_each_available_child_of_node(pdev->dev.of_node, child) {
		int ret;
		u32 port = 0, data_port = 0;
		u32 ptcr = 0, pdcr = 0;
		ret = of_property_read_u32(child, "port-id", &port);
		if (ret) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "%p port id read returned %d", child, ret);
			continue;
		}
		if (of_property_read_bool(child, "sync-tx-rx")) {
			ptcr |= IMX_AUDMUX_PTCR_SYN;
		}
		if (of_property_read_u32 (child, "data-port", &data_port)) {
			bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "AUDMUX %d: no data-port specified!", port);
			continue;
		}
		pdcr |= IMX_AUDMUX_PDCR_RXDSEL(data_port - 1);
		if (of_property_read_bool(child, "clock-tx")) {
			u32 clock_port;
			if (of_property_read_u32(child, "clock-port", &clock_port)) {
				bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_ERR, "AUDMUX %d: clock-tx set but no clock-port specified.", port);
				continue;
			}
			clock_port += 7;
			ptcr |= IMX_AUDMUX_PTCR_TFSDIR
				| IMX_AUDMUX_PTCR_TFSEL(clock_port)
				| IMX_AUDMUX_PTCR_TCLKDIR
				| IMX_AUDMUX_PTCR_TCSEL(clock_port)
				| IMX_AUDMUX_PTCR_RFSDIR
				| IMX_AUDMUX_PTCR_RFSEL(clock_port)
				| IMX_AUDMUX_PTCR_RCLKDIR
				| IMX_AUDMUX_PTCR_RCSEL(clock_port);
		}
		audmux_set_port(audmux, port, ptcr, pdcr);
		bb_log_dev(&(pdev->dev), BB_MOD_LLA, BB_LVL_DEBUG, "AUDMUX %d: Set up mux with data from port %d, ptcr %#010x.", port, data_port, ptcr);
	}

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "registered");
	return 0;
}

static int audmux_remove(struct platform_device *pdev)
{
	struct audmux *audmux = platform_get_drvdata(pdev);

	procfs_remove(audmux);

	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "unregistered");
	return 0;
}

static int audmux_suspend(struct platform_device *pdev, pm_message_t state)
{
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "suspend");
	return 0;
}

static int audmux_resume(struct platform_device *pdev)
{
	bb_log_dev(&pdev->dev, BB_MOD_LLA, BB_LVL_INFO, "resume");
	return 0;
}

static const struct of_device_id audmux_ids[] = {
	{ .compatible = "fsl,imx6sx-audmux", },
	{ .compatible = "fsl,imx6q-audmux", },
	{}
};

static struct platform_driver audmux_driver = {
	.probe		= audmux_probe,
	.remove		= audmux_remove,
	.suspend	= audmux_suspend,
	.resume		= audmux_resume,
	.driver	= {
		.name	= DRV_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = audmux_ids,
	}
};

int audmux_init(void)
{
	return platform_driver_register(&audmux_driver);
}

void audmux_exit(void)
{
	platform_driver_unregister(&audmux_driver);
}
