/*
 * Copyright (c) 2016-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * SONOS LED control module
 */

#include <asm/uaccess.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/kthread.h>
#include <linux/jiffies.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/semaphore.h>
#include <linux/seq_file.h>
#include <linux/workqueue.h>
#include <linux/scatterlist.h>
#include <asm/uaccess.h>
#include "event_queue_api.h"
#include "hwevent_queue_api.h"
#include "blackbox.h"
#include "sonos_device.h"
#include <linux/crc32.h>

#include "ledctl.h"

static int devno = -1;

#define CUR		priv_data->cur_index
#define NEXT		((priv_data->cur_index + 1) % 3)
#define QUEUED		((priv_data->cur_index + 2) % 3)

struct ledctl_data {
	struct ledctl_hw_ops	*chip_ops;
	struct delayed_work	worker;
	struct led_step		*step;
	struct led_pattern	patterns[3];
	struct semaphore	*lock;
	struct workqueue_struct	*led_work;
	struct hwevtq_reg	*hwevt_reg[2];
	uint8_t			cur_index;
	enum feedback_state	feedback_active;
	uint16_t		feedback_delay;
	struct led_step		feedback_color;
	uint32_t		ref_count;
	struct cdev		chr_dev;
};

static struct led_step slot_one_steps[LED_CTL_MAX_STEPS] = {{ 0 }};
static struct led_step slot_two_steps[LED_CTL_MAX_STEPS] = {{ 0 }};
static struct led_step slot_three_steps[LED_CTL_MAX_STEPS] = {{ 0 }};
static DEFINE_SEMAPHORE(led_lock);
static struct ledctl_data global_data = {
	.chip_ops		= NULL,
	.step			= NULL,
	.lock			= &led_lock,
	.cur_index		= 0,
	.feedback_active	= FB_OFF,
	.feedback_delay		= 64,
	.feedback_color		= { 0xff, 0xff, 0xff },
};

static int ledctl_proc_init(void);
static void ledctl_proc_exit(void);

#define MAX(a, b)	((a) > (b) ? (a) : (b))

inline int ledctl_max(struct led_step *step)
{
	return MAX(step->r, MAX(step->g, step->b));
}
EXPORT_SYMBOL(ledctl_max);

inline int ledctl_is_white(struct led_step *step)
{
	return (step->r == step->g && step->r == step->b && step->r != 0x00);
}
EXPORT_SYMBOL(ledctl_is_white);

struct led_step ledctl_diff(struct led_step *step1, struct led_step *step2)
{
	struct led_step diff;
	diff.r = (step1->r > step2->r ? (step1->r - step2->r) : (step2->r - step1->r));
	diff.g = (step1->g > step2->g ? (step1->g - step2->g) : (step2->g - step1->g));
	diff.b = (step1->b > step2->b ? (step1->b - step2->b) : (step2->b - step1->b));
	return diff;
}
EXPORT_SYMBOL(ledctl_diff);

inline int ledctl_same_color(struct led_step *step1, struct led_step *step2)
{
	return ((step1->r == step2->r) && (step1->g == step2->g) && (step1->b == step2->b));
}
EXPORT_SYMBOL(ledctl_same_color);

inline int ledctl_is_same(struct led_step *step1, struct led_step *step2)
{
	return (ledctl_same_color(step1, step2) && (step1->fade == step2->fade) && (step1->hold_time == step2->hold_time));
}

int ledctl_patcpy(struct led_pattern *dest, struct led_pattern *src)
{
	int i = 0;
	struct led_step *step = dest->step_array;

	if (src->num_steps > LED_CTL_MAX_STEPS) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Too many steps! (%u)", src->num_steps);
		return -E2BIG;
	}
	memset(dest->step_array, 0x00, LED_CTL_MAX_STEPS);
	memcpy(&(dest->cksum), &(src->cksum), sizeof(dest->cksum));
	dest->flags = src->flags;
	dest->repeats = src->repeats;
	dest->num_steps = src->num_steps;
	if (dest->num_steps == 0) {
		bb_log_dbg(BB_MOD_LEDCTL, "No steps. This is allowed, but a no-op.");
		return 0;
	}
	for (step = dest->step_array; step < dest->step_array + dest->num_steps; step++,i++) {
		if (src->step_array[i].reserved) {
			bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "%u in reserved field of step %d! Assuming corrupted array or incorrect num_steps (%u).",
			        src->step_array[i].reserved, i, dest->num_steps);
			return -EFAULT;
		}
		memcpy(step, &src->step_array[i], sizeof(struct led_step));
		if (global_data.chip_ops->scale) {
			global_data.chip_ops->scale(step);
		}
	}
	bb_log_dbg(BB_MOD_LEDCTL, "Copied pattern to internal array. Num steps: %u. Flags: %#x", dest->num_steps, dest->flags);
	return 0;
}

static void ledctl_swap_patterns(struct ledctl_data *priv_data)
{
	priv_data->patterns[CUR].num_steps = 0;
	down(priv_data->lock);
	bb_log_dbg(BB_MOD_LEDCTL, "Swapping patterns. Cur %u, Next %u.", CUR, NEXT);
	priv_data->cur_index = NEXT;
	up(priv_data->lock);
	priv_data->step = priv_data->patterns[CUR].step_array;
}

void ledctl_worker(struct work_struct *d)
{
	struct delayed_work *delayed = container_of(d, struct delayed_work, work);
	struct ledctl_data *priv_data = container_of(delayed, struct ledctl_data, worker);
	uint16_t hold = 0;
	uint16_t fade = 0;

	if (priv_data->patterns[NEXT].num_steps && !(priv_data->patterns[NEXT].flags & LED_CTL_ENQUEUE)) {
		ledctl_swap_patterns(priv_data);
	}
	if (!priv_data->step) {
		bb_log_dbg(BB_MOD_LEDCTL, "Initializing step.");
		priv_data->step = priv_data->patterns[CUR].step_array;
	}
	if (priv_data->feedback_active == FB_SCHED) {
		fade = priv_data->step->fade;
		priv_data->step->fade = 80;
	}
	if (priv_data->chip_ops) {
		hold = priv_data->chip_ops->update(priv_data->step);
	}
	if (priv_data->feedback_active == FB_SCHED) {
		priv_data->step->fade = fade;
		priv_data->feedback_active = FB_OFF;
	}
	if (priv_data->patterns[CUR].num_steps > 1 &&
	    priv_data->step + 1 < priv_data->patterns[CUR].step_array + priv_data->patterns[CUR].num_steps) {
		priv_data->step++;
	} else if (priv_data->patterns[CUR].repeats > 1) {
		priv_data->step = priv_data->patterns[CUR].step_array;
		priv_data->patterns[CUR].repeats--;
		bb_log_dbg(BB_MOD_LEDCTL, "%u repeats left.", priv_data->patterns[CUR].repeats);
	} else if (priv_data->patterns[CUR].repeats == 0 && priv_data->patterns[CUR].num_steps > 1) {
		bb_log_dbg(BB_MOD_LEDCTL, "Start again!");
		if (priv_data->patterns[NEXT].num_steps && priv_data->patterns[NEXT].flags & LED_CTL_ENQUEUE) {
			ledctl_swap_patterns(priv_data);
		} else {
			priv_data->step = priv_data->patterns[CUR].step_array;
		}
	} else {
		bb_log_dbg(BB_MOD_LEDCTL, "Pattern over.");
		if (!priv_data->patterns[NEXT].num_steps) {
			priv_data->patterns[CUR].num_steps = 0;
			return;
		} else {
			ledctl_swap_patterns(priv_data);
		}
	}
	mod_delayed_work(priv_data->led_work, &priv_data->worker, msecs_to_jiffies(hold));
}

static int ledctl_open(struct inode *inodep, struct file *filp)
{
	global_data.ref_count++;
	return 0;
}

static ssize_t ledctl_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset)
{
	struct led_pattern pat;
	struct led_step step_array[LED_CTL_MAX_STEPS];
	uint32_t pattern_crc;
	if (size != sizeof(struct led_pattern)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Tried to write something that is not a pattern. Unsupported.");
		return -1;
	}
	if (copy_from_user(&pat, buffer, size)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Copy pattern from user failed!");
		return -1;
	}
	if (pat.num_steps > LED_CTL_MAX_STEPS) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Too many steps in pattern! %d", (int)pat.num_steps);
		return -1;
	}
	if (copy_from_user(&step_array, pat.step_array, ((pat.num_steps)*sizeof(pat.step_array[0])))) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Copy steps from user failed!");
		return -1;
	}
	pat.step_array = step_array;
	pattern_crc = crc32(0xffffffff, step_array, (pat.num_steps)*sizeof(pat.step_array[0])) ^ 0xffffffff;
	pattern_crc = crc32(pattern_crc, &pat.flags, sizeof(pat.flags)) ^ 0xffffffff;
	pat.cksum = crc32(pattern_crc, &pat.repeats, sizeof(pat.repeats)) ^ 0xffffffff;
	ledctl_update(&pat);
	return size;
}

static long ledctl_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = LEDCTL_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case 1:
	{
		uint8_t on = 1;
		if (copy_from_user(&on, (uint8_t *)arg, sizeof(uint8_t))) {
			return -EACCES;
		}
		if (on) {
			global_data.feedback_active = FB_OFF;
		} else {
			global_data.feedback_active = FB_STOP;
		}
		break;
	}
	case 2:
	{
		struct led_step new_color;
		if (copy_from_user(&new_color, (struct led_step *)arg, sizeof(struct led_step))) {
			return -EACCES;
		}
		global_data.feedback_color = new_color;
		break;
	}
	default:
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Unrecognized IOCTL command %u.", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}

static int ledctl_release(struct inode *inodep, struct file *filp)
{
	global_data.ref_count--;
	return 0;
}

const struct file_operations ledctl_fops = {
	.open		= ledctl_open,
	.write		= ledctl_write,
	.unlocked_ioctl	= ledctl_ioctl,
	.release	= ledctl_release,
};

#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
void ledctl_feedback_callback(void *param, enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
	struct led_step step;
	if (!global_data.chip_ops) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "LED controller not initialized.");
		return;
	}
	if (global_data.feedback_active == FB_STOP) {
		bb_log_dbg(BB_MOD_LEDCTL, "LED Event Feedback disabled by userspace.");
		return;
	}
	if (info == HWEVTQINFO_PRESSED) {
		switch (source) {
		case HWEVTQSOURCE_CAPZONEA:
		case HWEVTQSOURCE_CAPZONEB:
		case HWEVTQSOURCE_CAPZONEC:
			global_data.feedback_active = FB_ON;
			step = global_data.feedback_color;
			cancel_delayed_work_sync(&global_data.worker);
			if (global_data.chip_ops->scale) {
				global_data.chip_ops->scale(&step);
			}
			if (global_data.chip_ops->update) {
				global_data.chip_ops->update(&step);
			}
			break;
		case HWEVTQSOURCE_CAPZONEM:
			break;
		default:
			bb_log_dbg(BB_MOD_LEDCTL, "Got PRESSED event from something not a cap zone.");
			break;
		}
	} else if (global_data.feedback_active && info == HWEVTQINFO_RELEASED) {
		switch (source) {
		case HWEVTQSOURCE_CAPZONEA:
		case HWEVTQSOURCE_CAPZONEB:
		case HWEVTQSOURCE_CAPZONEC:
			cancel_delayed_work_sync(&global_data.worker);
			if (global_data.step) {
				if (!queue_delayed_work(global_data.led_work, &global_data.worker,
				    msecs_to_jiffies(global_data.feedback_delay))) {
					bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Unable to re-queue work after sig detail interrupt.");
				}
				global_data.feedback_active = FB_SCHED;
			} else {
				struct led_step steps[2] = {
					[0]	= {0xff, 0xff, 0xff, 0, 200, 600},
					[1]	= {0x00, 0x00, 0x00, 0, 600, 900}
				};
				struct led_pattern pattern = {
					.flags		= LED_CTL_INTERRUPT,
					.repeats	= 0,
					.num_steps	= 2,
					.step_array	= steps
				};
				global_data.feedback_active = FB_OFF;
				ledctl_update(&pattern);
			}
			break;
		case HWEVTQSOURCE_CAPZONEM:
			break;
		default:
			bb_log_dbg(BB_MOD_LEDCTL, "Got RELEASED event from something not a cap zone.");
			break;
		}
	}
}
#endif

void ledctl_hw_register(struct ledctl_hw_ops *ops)
{
	if (global_data.chip_ops) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "HW has already been registered, Quit");
		return;
	}
	global_data.chip_ops = ops;
}
EXPORT_SYMBOL(ledctl_hw_register);

void ledctl_hw_unregister(void)
{
	global_data.chip_ops = NULL;
}
EXPORT_SYMBOL(ledctl_hw_unregister);

int __init ledctl_init(void)
{
	int ret = 0;
	struct device *class_dev = NULL;

	devno = MKDEV(LEDCTL_MAJOR_NUMBER, 0);

#ifdef CONFIG_DEVTMPFS
	ret = alloc_chrdev_region(&devno, 0, 1, LEDCTL_DEVICE_NAME);
#else
	ret = register_chrdev_region(devno, 1, LEDCTL_DEVICE_NAME);
#endif
	if (ret) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Couldn't register character device region (%d).", ret);
		return ret;
	}
	cdev_init(&global_data.chr_dev, &ledctl_fops);
	global_data.chr_dev.owner = THIS_MODULE;

	ret = cdev_add(&global_data.chr_dev, devno, 1);
	if (ret) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Couldn't add character device (%d).", ret);
		unregister_chrdev_region(devno, 1);
		return ret;
	}

	global_data.led_work = create_singlethread_workqueue("ledctl");
	INIT_DELAYED_WORK(&global_data.worker, ledctl_worker);
	bb_log(BB_MOD_LEDCTL, BB_LVL_DEBUG, "Work queue creation complete.");
	global_data.patterns[0].step_array = slot_one_steps;
	global_data.patterns[1].step_array = slot_two_steps;
	global_data.patterns[2].step_array = slot_three_steps;
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	global_data.hwevt_reg[0] = hwevtq_register_info(HWEVTQINFO_PRESSED, ledctl_feedback_callback, NULL, "LED Feedback");
        global_data.hwevt_reg[1] = hwevtq_register_info(HWEVTQINFO_RELEASED, ledctl_feedback_callback, NULL, "LED Feedback");
#endif

	ret = ledctl_proc_init();
	if (ret) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Could not initialize procfs.");
	}

	class_dev = sonos_device_create(NULL, devno, NULL, LEDCTL_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Invalid pointer to the Sonos device class struct!");
		ret = PTR_ERR(class_dev);
		ledctl_proc_exit();
		hwevtq_unregister_event(&global_data.hwevt_reg[0]);
		hwevtq_unregister_event(&global_data.hwevt_reg[1]);
		cancel_delayed_work_sync(&global_data.worker);
		destroy_workqueue(global_data.led_work);
		cdev_del(&global_data.chr_dev);
		unregister_chrdev_region(devno, 1);
		return ret;
	}

	bb_log(BB_MOD_LEDCTL, BB_LVL_INFO, "Registered LED control at %d.", MAJOR(devno));
	return ret;
}

void __exit ledctl_exit(void)
{
	hwevtq_unregister_event(&global_data.hwevt_reg[0]);
	hwevtq_unregister_event(&global_data.hwevt_reg[1]);
	cancel_delayed_work_sync(&global_data.worker);
	destroy_workqueue(global_data.led_work);
	cdev_del(&global_data.chr_dev);
	unregister_chrdev_region(devno, 1);
	ledctl_proc_exit();
	sonos_device_destroy(devno);
}

int ledctl_update(struct led_pattern *to_pattern)
{
	struct ledctl_data *priv_data = &global_data;
	int index = 0, cmp_idx = -1, ret = 0, cksum_cmp = 1;
	if (!priv_data->chip_ops) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_DEBUG, "LED controller not initialized.");
		return -ENODEV;
	}
	down(priv_data->lock);
	index = CUR;
	if (to_pattern->flags & (LED_CTL_OVERRIDE_FEEDBACK | LED_CTL_INTERRUPT)) {
		if (priv_data->patterns[CUR].num_steps != 0 && priv_data->patterns[NEXT].num_steps == 0) {
			cmp_idx = CUR;
			cksum_cmp = memcmp(&(priv_data->patterns[CUR].cksum), &(to_pattern->cksum), sizeof(to_pattern->cksum));
		}
	} else if (priv_data->patterns[QUEUED].num_steps != 0) {
		cmp_idx = QUEUED;
		cksum_cmp = memcmp(&(priv_data->patterns[QUEUED].cksum), &(to_pattern->cksum), sizeof(to_pattern->cksum));
	} else if (priv_data->patterns[NEXT].num_steps != 0) {
		cmp_idx = NEXT;
		cksum_cmp = memcmp(&(priv_data->patterns[NEXT].cksum), &(to_pattern->cksum), sizeof(to_pattern->cksum));
	} else if (priv_data->patterns[CUR].num_steps != 0) {
		cmp_idx = CUR;
		cksum_cmp = memcmp(&(priv_data->patterns[CUR].cksum), &(to_pattern->cksum), sizeof(to_pattern->cksum));
	}
	up(priv_data->lock);
	if (cksum_cmp == 0) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_DEBUG, "LED DUP pattern [f:0x%X|ns:%u] curIdx:%d cmpIdx:%d",
				(unsigned int)to_pattern->flags, (unsigned int)to_pattern->num_steps,
				index, cmp_idx);
		return 0;
	} else {
		bb_log_dbg(BB_MOD_LEDCTL, "LED NEW pattern [f:0x%X|ns:%u] curIdx:%d cmpIdx:%d",
				(unsigned int)to_pattern->flags, (unsigned int)to_pattern->num_steps,
				index, cmp_idx);
	}
	if (to_pattern->flags & LED_CTL_OVERRIDE_FEEDBACK) {
		priv_data->feedback_active = FB_OFF;
		cancel_delayed_work_sync(&priv_data->worker);
		down(priv_data->lock);
		index = NEXT;
		priv_data->patterns[QUEUED].num_steps = 0;
	} else if (to_pattern->flags & LED_CTL_INTERRUPT) {
		if (priv_data->feedback_active == FB_SCHED) {
			priv_data->feedback_active = FB_OFF;
		}
		cancel_delayed_work_sync(&priv_data->worker);
		down(priv_data->lock);
		index = NEXT;
		priv_data->patterns[QUEUED].num_steps = 0;
	} else {
		down(priv_data->lock);
		index = NEXT;
		if (priv_data->patterns[index].num_steps) {
			index = QUEUED;
			if (priv_data->patterns[index].num_steps) {
				bb_log_dbg(BB_MOD_LEDCTL, "Overwriting queued pattern.");
			}
		}
	}
	ret = ledctl_patcpy(&priv_data->patterns[index], to_pattern);
	if (ret) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Attempt to copy LED pattern returned %d.", ret);
	} else {
		bb_log_dbg(BB_MOD_LEDCTL, "update ledctl_patcpy(%d)", index);
	}
	up(priv_data->lock);
	if ((priv_data->feedback_active <= 0) && !delayed_work_pending(&priv_data->worker)) {
		if (!queue_delayed_work(priv_data->led_work, &priv_data->worker, 0)) {
			bb_log_dbg(BB_MOD_LEDCTL, "Worker already queued.");
		}
		bb_log_dbg(BB_MOD_LEDCTL, "LET'S GOOOOOOOO!");
	}
	return ret;
}

const char *ledctl_feedback_state_str(enum feedback_state state)
{
	switch (state) {
	case FB_STOP:
		return "disabled";
	case FB_OFF:
		return "off";
	case FB_ON:
		return "on";
	case FB_SCHED:
		return "scheduled";
	default:
		return "unknown";
	}
}

static int ledctl_proc_show(struct seq_file *m, void *v)
{
	struct ledctl_data *priv_data = (struct ledctl_data *)m->private;
	int i = 0, j = 0;
	seq_printf(m, "Reference count: %u\n", priv_data->ref_count);
	seq_printf(m, "Feedback: %s\n", ledctl_feedback_state_str(priv_data->feedback_active));
	seq_printf(m, "Feedback delay: %u\n", priv_data->feedback_delay);
	seq_printf(m, "Active pattern: %d\n", priv_data->cur_index);
	if (priv_data->step) {
		seq_printf(m, "Active step %p: %02x%02x%02x f: %ums h: %ums\n", priv_data->step,
				priv_data->step->r, priv_data->step->g, priv_data->step->b,
				priv_data->step->fade, priv_data->step->hold_time);
	}
	for (j = 0; j < 3; j++) {
		if (!priv_data->patterns[j].num_steps) {
			continue;
		}
		seq_printf(m, "Pattern %d: %u repeats, %u steps.\n", j, priv_data->patterns[j].repeats, priv_data->patterns[j].num_steps);
		seq_printf(m, "+---+--------+-----------+-----------+\n");
		seq_printf(m, "| # | Color  | Fade (ms) | Hold (ms) |\n");
		for (i = 0; i < priv_data->patterns[j].num_steps; i++) {
			struct led_step *step = &priv_data->patterns[j].step_array[i];
			seq_printf(m, "+---+--------+-----------+-----------+\n");
			seq_printf(m, "| %d | %02x%02x%02x |    %6u |    %6u |\n", i + 1,
						step->r, step->g, step->b,
						step->fade,
						step->hold_time);
		}
		seq_printf(m, "+---+--------+-----------+-----------+\n\n");
	}
	return 0;
}

static int ledctl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ledctl_proc_show, PDE_DATA(inode));
}

static ssize_t ledctl_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	struct ledctl_data *priv_data = PDE_DATA(file_inode(file));
	char buf[256];
	char *val;

	if (count > 255) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	}
	buf[count] = '\0';
	val = strchr(buf, '=');
	if (strncmp(buf, "feedback-delay", 14) == 0) {
		unsigned long delay = priv_data->feedback_delay;
		if ( val && kstrtoul(val + 1, 0, &delay)) {
			bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Failed to parse delay value: %s", val);
		} else {
			priv_data->feedback_delay = (uint16_t)delay;
			bb_log(BB_MOD_LEDCTL, BB_LVL_INFO, "Setting feedback delay to %u.", priv_data->feedback_delay);
		}
	} else {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Unknown command: %s", buf);
	}

	return count;
}


struct file_operations ledctl_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ledctl_proc_open,
	.write		= ledctl_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

static int ledctl_proc_init(void)
{
	struct proc_dir_entry *entry;
	entry = proc_create_data("driver/ledctl", 0666, NULL, &ledctl_proc_ops, &global_data);
	if (!entry) {
		return -EIO;
	}
	return 0;
}

static void ledctl_proc_exit(void)
{
	remove_proc_entry("driver/ledctl", NULL);
}


module_init(ledctl_init);
module_exit(ledctl_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for Sonos LED subsystem");
MODULE_LICENSE("GPL");
