/*
 * Copyright (c) 2022, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#include <asm/byteorder.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include "blackbox.h"
#include "ledctl.h"
#include "is31fl3195.h"

#define LED_SLOW_BREATHING_FADE_IN_TIME_MS          900
#define LED_SLOW_BREATHING_FADE_OUT_TIME_MS         1500

struct is31fl3195_state {
	struct i2c_client *client;
	struct mutex lock;
	struct led_step last_step;
	u16 white_fade_out_time;
	u16 rgb_fade_out_time;
	u8 first_update;
};

static int is31fl3195_proc_init(struct is31fl3195_state *state);
static void is31fl3195_proc_remove(void);
struct is31fl3195_state *is31fl3195_global_state = NULL;

#define LM_PRINT(lvl, msg, args...)	bb_log_dev(&(state->client->dev), BB_MOD_LEDCTL, lvl, msg, ##args)
#define LM_DEBUG(msg, args...)		bb_log_dbg_dev(&(state->client->dev), BB_MOD_LEDCTL, msg, ##args)
#define LM_INFO(msg, args...)		LM_PRINT(BB_LVL_INFO, msg, ##args)
#define LM_WARNING(msg, args...)	LM_PRINT(BB_LVL_WARNING, msg, ##args)
#define LM_ERR(msg, args...)		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "IS31FL3195: "msg, ##args)

int is31fl3195_read(struct is31fl3195_state *state, u8 addr, u8 *val)
{
	int ret = 0;
	static u32 err_count = 0;
	ret = i2c_smbus_read_byte_data(state->client, addr);
	if (ret < 0 && !(err_count % 5)) {
		LM_ERR("I2C read error %d (attempt %u)", ret, err_count++);
	} else {
		*val = ret;
		ret = 0;
	}
	return ret;
}

int is31fl3195_read_block(struct is31fl3195_state *state, u8 addr, u8 *val, u8 len)
{
	u8 *start = val;
	int ret = 0;
	static u32 err_count = 0;

	for(; val < start + len; val++,addr++) {
		ret = i2c_smbus_read_byte_data(state->client, addr);
		if (ret < 0 && !(err_count % 5)) {
			LM_ERR("I2C read error %d (attempt %u)", ret, err_count++);
			break;
		} else {
			*val = ret;
			ret = 0;
		}
	}
	return ret;
}

int is31fl3195_write(struct is31fl3195_state *state, u8 addr, u8 val)
{
	int ret = 0;
	static u32 err_count = 0;
	ret = i2c_smbus_write_byte_data(state->client, addr, val);
	if (ret < 0 && !(err_count % 5)) {
		LM_ERR("I2C write error %d (attempt %u)", ret, err_count++);
	}
	return ret;
}

static inline struct is31fl3195_state *is31fl3195_get_state(void)
{
	return is31fl3195_global_state;
}

static const u16 is31fl3195_code_to_fade_time[] = { 30, 130, 260, 380,
						510, 770, 1040, 1600,
						2100, 2600, 3100, 4200,
						5200, 6200, 7300, 8300 };

static u8 is1fl3195_fade_time_to_code(u16 fade_ms)
{
	u8 code;

	if (fade_ms > 8300) {
		fade_ms = 8300;
	}

	if (fade_ms < 30) {
		code = 0;
	} else if (fade_ms < 260) {
		code = 1;
	} else if (fade_ms < 380) {
		code = 2;
	} else if (fade_ms < 510) {
		code = 3;
	} else if (fade_ms < 770) {
		code = 4;
	} else if (fade_ms < 1040) {
		code = 5;
	} else if (fade_ms < 1600) {
		code = 6;
	} else if (fade_ms < 4200) {
		code = 7 + ((fade_ms - 1600) / 500);
	} else if (fade_ms < 7300) {
		code = 11 + ((fade_ms - 4200) / 1000);
	} else {
		code = 14 + ((fade_ms - 7300) / 1000);
	}

	return code;
}

static u16 is31fl3195_make_pattern(struct is31fl3195_state *state, struct led_step *step,
					u8 ts_and_t1_reg, u8 t2_and_t3_reg)
{
	u8 fade_in_code = 0;
	u8 fade_out_code = 0;
	u16 fade_out_time = step->fade;

	if (step->fade) {
		fade_in_code = is1fl3195_fade_time_to_code(step->fade);

		if (step->fade == LED_SLOW_BREATHING_FADE_IN_TIME_MS) {
			fade_out_time = LED_SLOW_BREATHING_FADE_OUT_TIME_MS;
			fade_out_code = is1fl3195_fade_time_to_code(fade_out_time);
		} else if (step->fade == LED_SLOW_BREATHING_FADE_OUT_TIME_MS) {
			fade_out_time = LED_SLOW_BREATHING_FADE_IN_TIME_MS;
			fade_out_code = is1fl3195_fade_time_to_code(fade_out_time);
		} else {
			fade_out_code = fade_in_code;
		}
	}

	is31fl3195_write(state, ts_and_t1_reg, fade_in_code << IS31FL3195_T1_SHIFT);

	is31fl3195_write(state, t2_and_t3_reg, fade_out_code << IS31FL3195_T3_SHIFT);

	return fade_out_time;
}

static int is31fl3195_init_regs(struct is31fl3195_state *state)
{
	int ret = 0;

	ret = is31fl3195_write(state, IS31FL3195_REG_RESET,
				IS31FL3195_RESET);
	if (ret) {
		LM_ERR("Failed to reset chip, abort (code %d).", ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_SHUTDOWN_CONTROL,
				IS31FL3195_SHUTDOWN_CONTROL);
	if (ret) {
		LM_ERR("Failed to enable chip, abort (code %d).", ret);
		return ret;
	}


	ret = is31fl3195_write(state, IS31FL3195_REG_OPERATING_MODE,
				IS31FL3195_OPERATING_MODE);
	if (ret) {
		LM_ERR("Failed to set chip's operating mode, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_CHARGE_PUMP_MODE,
				IS31FL3195_CHARGE_PUMP_MODE);
	if (ret) {
		LM_ERR("Failed to set chip's charge pump mode to auto, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_CHARGE_PUMP_HR,
				IS31FL3195_CHARGE_PUMP_HR);
	if (ret) {
		LM_ERR("Failed to set chip's charge pump headroom, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_CURRENT_BAND,
				IS31FL3195_CURRENT_BAND);
	if (ret) {
		LM_ERR("Failed to set chip's max output current, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_P1_COLOR_CYCLES,
				IS31FL3195_SINGLE_COLOR_CYCLE);
	if (ret) {
		LM_ERR("Failed to set RGB (P1) Pattern 1 Color Cycle Register, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_P4_COLOR_CYCLES,
				IS31FL3195_SINGLE_COLOR_CYCLE);
	if (ret) {
		LM_ERR("Failed to set White (P4) Pattern Color Cycle Register, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_P1_LOOP, IS31FL3195_DONT_LOOP);
	if (ret) {
		LM_ERR("Failed to configure RGB (P1) Pattern to not loop, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_P4_LOOP, IS31FL3195_DONT_LOOP);
	if (ret) {
		LM_ERR("Failed to configure White (P4) Pattern to not loop, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_P1_NXT, IS31FL3195_NXT_INIT);
	if (ret) {
		LM_ERR("Failed to configure RGB (P1) Pattern NXT, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_P4_NXT, IS31FL3195_NXT_INIT);
	if (ret) {
		LM_ERR("Failed to configure White (P4) Pattern NXT, abort (code %d).",
			ret);
		return ret;
	}

	ret = is31fl3195_write(state, IS31FL3195_REG_HOLD_FUNCTION,
				IS31FL3195_HOLD_P1_AT_T2_BIT |
				IS31FL3195_HOLD_P4_AT_T2_BIT);
	if (ret) {
		LM_ERR("Failed to configure chip's Hold Function Register, abort (code %d).",
			ret);
		return ret;
	}

	return ret;
}

static void is31fl3195_init_state(struct is31fl3195_state *state, u8 uboot_is_breathing)
{
	state->first_update = uboot_is_breathing;
}

static void is31fl3195_set_white(struct is31fl3195_state *state, u8 white)
{
	is31fl3195_write(state, IS31FL3195_REG_P4_COLOR_1_W, white);
}

static void is31fl3195_set_rgb(struct is31fl3195_state *state,
				u8 red, u8 green, u8 blue)
{
	is31fl3195_write(state, IS31FL3195_REG_P1_COLOR_1_R, red);
	is31fl3195_write(state, IS31FL3195_REG_P1_COLOR_1_G, green);
	is31fl3195_write(state, IS31FL3195_REG_P1_COLOR_1_B, blue);
}


int is31fl3195_update(struct led_step *step)
{
	int ret;
	struct is31fl3195_state *state = is31fl3195_get_state();
	struct led_step diff = *step;

	u8 white_pattern_state = 0;
	u8 rgb_pattern_state = 0;
	u8 new_hold_function = 0;

	int need_color_update = 0;
	int need_white_pattern_update = 0;
	int need_rgb_pattern_update = 0;
	int need_hold_function_update = 0;

	if (!state) {
		LM_ERR("State uninitialized. Device probe has failed or not taken place.");
		return -1;
	}

	mutex_lock(&(state->lock));

	if (state->first_update) {
		LM_DEBUG("First update, disable u-boot white breathing");
		is31fl3195_init_regs(state);
		state->first_update = 0;

	}

	is31fl3195_read(state, IS31FL3195_REG_P4_STATE, &white_pattern_state);
	is31fl3195_read(state, IS31FL3195_REG_P1_STATE, &rgb_pattern_state);
	is31fl3195_read(state, IS31FL3195_REG_HOLD_FUNCTION, &new_hold_function);

	if (ledctl_same_color(step, &(state->last_step))) {
		LM_DEBUG("New step is the same color as current step.");
		goto up_out;
	}

	if ((ledctl_is_white(step) && ledctl_is_white(&(state->last_step))) ||
		(!ledctl_is_white(step) && !ledctl_is_white(&(state->last_step)))) {
		diff = ledctl_diff(step, &(state->last_step));
		diff.fade = step->fade;
		diff.hold_time = step->hold_time;
		LM_DEBUG("diff %02x%02x%02x", diff.r, diff.g, diff.b);
	}

	if (ledctl_is_white(&diff) && ledctl_is_white(step)) {
		if (step->fade && !ledctl_is_white(&(state->last_step))) {
			LM_DEBUG("Fading-in White LED from 00 to %02x in %u ms", step->r, step->fade);

			if (!(new_hold_function & IS31FL3195_HOLD_P4_ENABLE_BIT)) {
				new_hold_function |= IS31FL3195_HOLD_P4_ENABLE_BIT;
				need_hold_function_update++;
			}

			state->white_fade_out_time = is31fl3195_make_pattern(state, step,
										IS31FL3195_REG_P4_SET_TS_T1,
										IS31FL3195_REG_P4_SET_T2_T3);
			need_white_pattern_update++;

			is31fl3195_set_white(state, step->r);
			need_color_update++;
		} else {
			if (step->fade && ledctl_is_white(&(state->last_step))) {
				LM_DEBUG("Warning: Last White Step was %02x, but hardware only fades to or from 00",
						state->last_step.r);
			}

			LM_DEBUG("Setting White Brightness to %02x", step->r);

			if (!(new_hold_function & IS31FL3195_HOLD_P4_ENABLE_BIT)) {
				new_hold_function |= IS31FL3195_HOLD_P4_ENABLE_BIT;
				need_hold_function_update++;

				state->white_fade_out_time = is31fl3195_make_pattern(state, step,
											IS31FL3195_REG_P4_SET_TS_T1,
											IS31FL3195_REG_P4_SET_T2_T3);
				need_white_pattern_update++;
			}

			is31fl3195_set_white(state, step->r);
			need_color_update++;
		}


	} else if (ledctl_is_white(step)) {


	} else if (ledctl_is_white(&(state->last_step))) {

		if (step->fade) {
			if (state->white_fade_out_time != step->fade) {
				LM_DEBUG("White LED fade-out time preset is %u ms but requested fade out is %u ms",
					state->white_fade_out_time, step->fade);
			}

			if (state->white_fade_out_time > 0) {
				LM_DEBUG("Fading-out White LED from %02x to 00 in %u ms", state->last_step.r, state->white_fade_out_time);
				state->white_fade_out_time = 0;
				new_hold_function &= ~IS31FL3195_HOLD_P4_ENABLE_BIT;
				need_hold_function_update++;
			} else {
				LM_DEBUG("A fade-out preset for the White Pattern is not available, so turn off White LED");
				is31fl3195_set_white(state, 0);
				need_color_update++;
			}

		} else {
			LM_DEBUG("This step is not White, so turn off White LED");
			is31fl3195_set_white(state, 0);
			need_color_update++;
		}
	}

	if (ledctl_max(step) && ledctl_max(&diff) && !ledctl_is_white(step)) {
		int last_step_was_rgb = ledctl_max(&state->last_step) &&
					!ledctl_is_white(&(state->last_step));
		if (step->fade && !last_step_was_rgb) {
			LM_DEBUG("Fading-in RGB LEDs from 000000 to %02x%02x%02x in %u ms",
					step->r, step->g, step->b, step->fade);


			if (!(new_hold_function & IS31FL3195_HOLD_P1_ENABLE_BIT)) {
				new_hold_function |= IS31FL3195_HOLD_P1_ENABLE_BIT;
				need_hold_function_update++;
			}

			state->rgb_fade_out_time = is31fl3195_make_pattern(state, step,
										IS31FL3195_REG_P1_SET_TS_T1,
										IS31FL3195_REG_P1_SET_T2_T3);
			need_rgb_pattern_update++;

			is31fl3195_set_rgb(state, step->r, step->g, step->b);
			need_color_update++;
		} else {
			if (step->fade && last_step_was_rgb) {
				LM_DEBUG("Warning: Last RGB Step was %02x%02x%02x, but hardware only fades to or from 000000",
						state->last_step.r, state->last_step.g, state->last_step.b);
			}

			LM_DEBUG("Setting RGB Color to %02x%02x%02x",
				 step->r, step->g, step->b);

			if (!(new_hold_function & IS31FL3195_HOLD_P1_ENABLE_BIT)) {
				new_hold_function |= IS31FL3195_HOLD_P1_ENABLE_BIT;
				need_hold_function_update++;

				state->rgb_fade_out_time = is31fl3195_make_pattern(state, step,
											IS31FL3195_REG_P1_SET_TS_T1,
											IS31FL3195_REG_P1_SET_T2_T3);
				need_rgb_pattern_update++;
			}

			is31fl3195_set_rgb(state, step->r, step->g, step->b);
			need_color_update++;
		}
	} else if (ledctl_max(step) && !ledctl_is_white(step)) {

	} else if (ledctl_max(&state->last_step) && !ledctl_is_white(&(state->last_step))) {

		if (step->fade) {
			if (state->rgb_fade_out_time != step->fade) {
				LM_DEBUG("RGB LED fade-out time preset is %u ms but requested fade out is %u ms",
					state->rgb_fade_out_time, step->fade);
			}

			if (state->rgb_fade_out_time > 0) {
				LM_DEBUG("Fading-out RGB LEDs from %02x%02x%02x to 000000 in %u ms",
					state->last_step.r, state->last_step.g, state->last_step.b,
					state->rgb_fade_out_time);
				state->rgb_fade_out_time = 0;
				new_hold_function &= ~IS31FL3195_HOLD_P1_ENABLE_BIT;
				need_hold_function_update++;
			} else {
				LM_DEBUG("A fade-out preset for the White Pattern is not available, so turn off White LED");
				is31fl3195_set_rgb(state, 0, 0, 0);
				need_color_update++;
			}
		} else {
			LM_DEBUG("This step is not RGB so turn off RGB LEDs");
			is31fl3195_set_rgb(state, 0, 0, 0);
			need_color_update++;
		}
	}


	if (need_hold_function_update) {
		ret = is31fl3195_write(state, IS31FL3195_REG_HOLD_FUNCTION, new_hold_function);
		if (ret) {
			LM_ERR("Failed to update chip's Hold Function Register (code %d).",
				ret);
		}
	}

	if (need_white_pattern_update) {
		ret = is31fl3195_write(state, IS31FL3195_REG_P4_UPDATE, IS31FL3195_PATTERN_UPDATE);
		if (ret) {
			LM_ERR("Failed to update chip's White (P4) Pattern Registers (code %d).",
				ret);
		}
	}

	if (need_rgb_pattern_update) {
		ret = is31fl3195_write(state, IS31FL3195_REG_P1_UPDATE, IS31FL3195_PATTERN_UPDATE);
		if (ret) {
			LM_ERR("Failed to update chip's RGB (P1) Pattern Registers (code %d).",
				ret);
		}
	}

	if (need_color_update) {
		ret = is31fl3195_write(state, IS31FL3195_REG_COLOR_UPDATE, IS31FL3195_COLOR_UPDATE);
		if (ret) {
			LM_ERR("Failed to update chip's Color Registers (code %d).",
				ret);
		}
	}

	state->last_step = *step;
up_out:
	mutex_unlock(&(state->lock));
	LM_DEBUG("**************************");
	return step->hold_time + step->fade;
}

const uint8_t is31fl3195_white_intensity_map[] = {
	0x00, 0x00, 0x01, 0x01, 0x02, 0x02, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x05, 0x05, 0x06, 0x06,
	0x07, 0x07, 0x08, 0x08, 0x08, 0x08, 0x09, 0x09, 0x0A, 0x0A, 0x0B, 0x0B, 0x0B, 0x0B, 0x0C, 0x0C,
	0x0D, 0x0D, 0x0E, 0x0E, 0x0E, 0x0E, 0x0F, 0x0F, 0x10, 0x10, 0x11, 0x11, 0x11, 0x11, 0x12, 0x12,
	0x13, 0x13, 0x14, 0x14, 0x14, 0x14, 0x15, 0x15, 0x16, 0x16, 0x17, 0x17, 0x18, 0x18, 0x18, 0x18,
	0x19, 0x19, 0x1A, 0x1A, 0x1B, 0x1B, 0x1B, 0x1B, 0x1C, 0x1C, 0x1D, 0x1D, 0x1E, 0x1E, 0x1F, 0x1F,
	0x20, 0x20, 0x20, 0x20, 0x21, 0x21, 0x22, 0x22, 0x23, 0x23, 0x24, 0x24, 0x24, 0x24, 0x25, 0x25,
	0x26, 0x26, 0x27, 0x27, 0x28, 0x28, 0x29, 0x29, 0x2A, 0x2A, 0x2A, 0x2A, 0x2B, 0x2B, 0x2C, 0x2C,
	0x2D, 0x2D, 0x2E, 0x2E, 0x2E, 0x2E, 0x2F, 0x2F, 0x30, 0x30, 0x31, 0x31, 0x32, 0x32, 0x33, 0x33,
	0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3E, 0x3F, 0x40, 0x41, 0x42,
	0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51,
	0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 0x60, 0x61,
	0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71,
	0x73, 0x75, 0x77, 0x79, 0x7B, 0x7D, 0x7F, 0x81, 0x83, 0x85, 0x87, 0x89, 0x8C, 0x8E, 0x90, 0x92,
	0x94, 0x96, 0x99, 0x9B, 0x9D, 0xA0, 0xA2, 0xA4, 0xA7, 0xA9, 0xAB, 0xAE, 0xB0, 0xB2, 0xB4, 0xB7,
	0xB9, 0xBB, 0xBE, 0xC0, 0xC2, 0xC5, 0xC7, 0xCA, 0xCC, 0xCE, 0xD1, 0xD3, 0xD5, 0xD8, 0xDB, 0xDD,
	0xE0, 0xE2, 0xE5, 0xE7, 0xEA, 0xEC, 0xEF, 0xF1, 0xF4, 0xF6, 0xF9, 0xFC, 0xFE, 0xFF, 0xFF, 0xFF
};

const uint8_t is31fl3195_red_intensity_map[] = {
	0x00, 0x00, 0x02, 0x02, 0x03, 0x03, 0x04, 0x04, 0x05, 0x05, 0x06, 0x06, 0x07, 0x07, 0x08, 0x08,
	0x09, 0x09, 0x0A, 0x0A, 0x0B, 0x0B, 0x0C, 0x0C, 0x0D, 0x0D, 0x0D, 0x0D, 0x0E, 0x0E, 0x0F, 0x0F,
	0x10, 0x10, 0x11, 0x11, 0x12, 0x12, 0x13, 0x13, 0x14, 0x14, 0x15, 0x15, 0x16, 0x16, 0x17, 0x17,
	0x18, 0x18, 0x19, 0x19, 0x1A, 0x1A, 0x1A, 0x1A, 0x1B, 0x1B, 0x1C, 0x1C, 0x1D, 0x1D, 0x1E, 0x1E,
	0x1F, 0x1F, 0x20, 0x20, 0x21, 0x21, 0x22, 0x22, 0x22, 0x22, 0x23, 0x23, 0x24, 0x24, 0x25, 0x25,
	0x26, 0x26, 0x27, 0x27, 0x28, 0x28, 0x29, 0x29, 0x2A, 0x2A, 0x2A, 0x2A, 0x2B, 0x2B, 0x2C, 0x2C,
	0x2D, 0x2D, 0x2E, 0x2E, 0x2F, 0x2F, 0x30, 0x30, 0x31, 0x31, 0x32, 0x32, 0x32, 0x32, 0x33, 0x33,
	0x34, 0x34, 0x35, 0x35, 0x36, 0x36, 0x37, 0x37, 0x38, 0x38, 0x39, 0x39, 0x39, 0x39, 0x3A, 0x3A,
	0x3B, 0x3C, 0x3C, 0x3D, 0x3E, 0x3F, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x46, 0x47,
	0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4D, 0x4E, 0x4E, 0x4F, 0x50, 0x51, 0x53, 0x54, 0x54, 0x55,
	0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5C, 0x5D, 0x5E, 0x5E, 0x5F, 0x60, 0x61, 0x62, 0x63,
	0x64, 0x65, 0x66, 0x67, 0x68, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72,
	0x73, 0x75, 0x76, 0x79, 0x7A, 0x7B, 0x7E, 0x7F, 0x81, 0x82, 0x85, 0x86, 0x88, 0x8A, 0x8B, 0x8E,
	0x8F, 0x92, 0x93, 0x94, 0x97, 0x98, 0x9A, 0x9B, 0x9D, 0x9F, 0xA1, 0xA3, 0xA4, 0xA6, 0xA8, 0xA9,
	0xAB, 0xAE, 0xAF, 0xB1, 0xB2, 0xB4, 0xB6, 0xB7, 0xB9, 0xBB, 0xBD, 0xBF, 0xC0, 0xC1, 0xC4, 0xC5,
	0xC6, 0xC9, 0xCA, 0xCC, 0xCE, 0xCF, 0xD1, 0xD4, 0xD5, 0xD6, 0xD8, 0xDA, 0xDC, 0xDD, 0xE0, 0xE2
};

const uint8_t is31fl3195_green_intensity_map[] = {
	0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
	0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x08, 0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
	0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0C, 0x0C, 0x0C, 0x0C, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D, 0x0D,
	0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x11, 0x11,
	0x11, 0x11, 0x11, 0x11, 0x12, 0x12, 0x12, 0x12, 0x13, 0x13, 0x13, 0x13, 0x14, 0x14, 0x14, 0x14,
	0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x16, 0x16, 0x16, 0x16, 0x17, 0x17, 0x17, 0x17, 0x18, 0x18,
	0x18, 0x18, 0x19, 0x19, 0x1A, 0x1B, 0x1B, 0x1C, 0x1C, 0x1C, 0x1D, 0x1D, 0x1E, 0x1E, 0x1F, 0x1F,
	0x20, 0x20, 0x21, 0x21, 0x22, 0x22, 0x23, 0x23, 0x24, 0x24, 0x25, 0x25, 0x26, 0x26, 0x27, 0x27,
	0x28, 0x28, 0x29, 0x29, 0x2A, 0x2A, 0x2B, 0x2B, 0x2C, 0x2C, 0x2D, 0x2D, 0x2E, 0x2E, 0x2F, 0x30,
	0x30, 0x31, 0x32, 0x32, 0x33, 0x33, 0x34, 0x35, 0x35, 0x36, 0x36, 0x37, 0x38, 0x38, 0x39, 0x39,
	0x3A, 0x3B, 0x3C, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x44, 0x45, 0x46, 0x47, 0x48, 0x4A, 0x4B, 0x4C,
	0x4E, 0x4F, 0x50, 0x52, 0x53, 0x54, 0x56, 0x57, 0x58, 0x5A, 0x5C, 0x5D, 0x5E, 0x5F, 0x61, 0x62,
	0x63, 0x65, 0x66, 0x68, 0x69, 0x6B, 0x6C, 0x6E, 0x6F, 0x71, 0x73, 0x74, 0x75, 0x77, 0x78, 0x7A,
	0x7B, 0x7D, 0x7F, 0x81, 0x82, 0x84, 0x85, 0x87, 0x88, 0x8A, 0x8C, 0x8D, 0x8F, 0x91, 0x92, 0x95
};

const uint8_t is31fl3195_blue_intensity_map[] = {
	0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04,
	0x04, 0x04, 0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x06, 0x07, 0x07, 0x07, 0x07, 0x08, 0x08,
	0x08, 0x08, 0x09, 0x09, 0x09, 0x09, 0x0A, 0x0A, 0x0B, 0x0B, 0x0B, 0x0B, 0x0C, 0x0C, 0x0C, 0x0C,
	0x0D, 0x0D, 0x0D, 0x0D, 0x0E, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F, 0x10, 0x10, 0x10, 0x10, 0x11, 0x11,
	0x11, 0x11, 0x12, 0x12, 0x13, 0x13, 0x13, 0x13, 0x14, 0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x15,
	0x16, 0x16, 0x16, 0x16, 0x17, 0x17, 0x18, 0x18, 0x18, 0x18, 0x19, 0x19, 0x1A, 0x1A, 0x1A, 0x1A,
	0x1B, 0x1B, 0x1B, 0x1B, 0x1C, 0x1C, 0x1D, 0x1D, 0x1E, 0x1E, 0x1E, 0x1E, 0x1F, 0x1F, 0x20, 0x20,
	0x20, 0x20, 0x21, 0x21, 0x22, 0x22, 0x22, 0x22, 0x23, 0x23, 0x24, 0x24, 0x24, 0x24, 0x25, 0x25,
	0x25, 0x25, 0x26, 0x27, 0x31, 0x29, 0x29, 0x2A, 0x2B, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x2F, 0x30,
	0x30, 0x31, 0x31, 0x32, 0x33, 0x34, 0x34, 0x35, 0x36, 0x37, 0x38, 0x38, 0x39, 0x3A, 0x3B, 0x3C,
	0x3C, 0x3C, 0x3D, 0x3E, 0x3F, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x43, 0x44, 0x44, 0x45, 0x45, 0x46,
	0x47, 0x48, 0x49, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x4F, 0x4F, 0x50, 0x51, 0x52, 0x53,
	0x55, 0x57, 0x58, 0x59, 0x5B, 0x5D, 0x5F, 0x61, 0x62, 0x63, 0x65, 0x67, 0x69, 0x6B, 0x6C, 0x6E,
	0x70, 0x72, 0x73, 0x75, 0x77, 0x79, 0x7B, 0x7C, 0x7E, 0x80, 0x82, 0x83, 0x85, 0x88, 0x8A, 0x8B,
	0x8D, 0x8F, 0x91, 0x92, 0x95, 0x97, 0x99, 0x9A, 0x9C, 0x9F, 0xA1, 0xA2, 0xA4, 0xA7, 0xA8, 0xAB,
	0xAD, 0xAF, 0xB0, 0xB3, 0xB5, 0xB8, 0xB9, 0xBC, 0xBE, 0xBF, 0xC1, 0xC4, 0xC6, 0xC8, 0xCA, 0xCE
};

int is31fl3195_scale(struct led_step *step)
{
	if (ledctl_is_white(step)) {
		step->r = is31fl3195_white_intensity_map[step->r];
		step->g = is31fl3195_white_intensity_map[step->g];
		step->b = is31fl3195_white_intensity_map[step->b];
	} else {
		if (step->r) {
			step->r = is31fl3195_red_intensity_map[step->r];
		}

		if (step->g) {
			step->g = is31fl3195_green_intensity_map[step->g];
		}

		if (step->b) {
			step->b = is31fl3195_blue_intensity_map[step->b];
		}
	}
	return 0;
}

static struct ledctl_hw_ops ops = {
	.update		= is31fl3195_update,
	.scale		= is31fl3195_scale,
};

static int is31fl3195_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	u8 uboot_is_breathing = 0;
	u8 client_address;
	u8 p4_state;
	u8 p4_nxt;

	struct is31fl3195_state *state = devm_kzalloc(&(client->dev), sizeof(struct is31fl3195_state), GFP_KERNEL);
	if (!state) {
		LM_ERR("Memory allocation failed");
		return -ENOMEM;
	}
	dev_set_drvdata(&(client->dev), state);
	mutex_init(&(state->lock));
	is31fl3195_global_state = state;

	mutex_lock(&(state->lock));

	state->client = client;
	ret = is31fl3195_read(state, IS31FL3195_REG_PRODUCT_ID, &client_address);
	if (ret) {
		LM_ERR("Error %d reading Product ID register, abort.", ret);
		mutex_unlock(&(state->lock));
		is31fl3195_global_state = (struct is31fl3195_state *)NULL;
		return -ENODEV;
	} else if (client_address != (client->addr << 1)) {
		LM_ERR("Device is not an IS31FL3195.  Expected PRODUCT ID %02x, found %02x.", client->addr << 1, client_address);
		mutex_unlock(&(state->lock));
		is31fl3195_global_state = (struct is31fl3195_state *)NULL;
		return -ENODEV;
	}

	LM_INFO("Found IS31FL3195 at i2c address %02x", client->addr);

	ret = is31fl3195_read(state, IS31FL3195_REG_P4_STATE, &p4_state);
	if (!ret) {
		ret = is31fl3195_read(state, IS31FL3195_REG_P4_NXT, &p4_nxt);
	}

	if (!ret && (p4_state & IS31FL3195_PATTERN_RUNNING_BIT) &&
			((p4_nxt >> IS31FL3195_NXT_MTPLT_SHIFT) == 0)) {
		LM_DEBUG("Chip initialized by u-boot, skipping reg init");
		uboot_is_breathing = 1;
	} else {
		ret = is31fl3195_init_regs(state);
		if (ret) {
			LM_ERR("Error %d initializing chip.", ret);
		}
	}

	is31fl3195_init_state(state, uboot_is_breathing);

	mutex_unlock(&(state->lock));

	ret = is31fl3195_proc_init(state);
	if (ret) {
		LM_ERR("Procfs init failed");
	}

	ledctl_hw_register(&ops);

	return ret;
}

static int is31fl3195_remove(struct i2c_client *client)
{
	struct is31fl3195_state *state = dev_get_drvdata(&(client->dev));

	ledctl_hw_unregister();
	is31fl3195_proc_remove();
	is31fl3195_write(state, IS31FL3195_REG_RESET, IS31FL3195_RESET);
	LM_INFO("LED controller removed.");
	return 0;
}

static const char *is31fl3195_ts_names[] = { "TS", "T1", "T2", "T3", "TP", "T4",
						"Unknown", "Unknown" };

static const u8 pattern_state_reg_addrs[] = { IS31FL3195_REG_P1_STATE,
						IS31FL3195_REG_P2_STATE,
						IS31FL3195_REG_P3_STATE,
						IS31FL3195_REG_P4_STATE };


static const u8 pattern_ts_t1_reg_addrs[] = { IS31FL3195_REG_P1_SET_TS_T1,
						IS31FL3195_REG_P2_SET_TS_T1,
						IS31FL3195_REG_P3_SET_TS_T1,
						IS31FL3195_REG_P4_SET_TS_T1 };

static const u8 pattern_t2_t3_reg_addrs[] =  { IS31FL3195_REG_P1_SET_T2_T3,
						IS31FL3195_REG_P2_SET_T2_T3,
						IS31FL3195_REG_P3_SET_T2_T3,
						IS31FL3195_REG_P4_SET_T2_T3 };

static int is31fl3195_show(struct seq_file *m, void *v)
{
	int i;
	u8 regs[IS31FL3195_NUM_REGS];
	struct is31fl3195_state *state = (struct is31fl3195_state *)m->private;
	int ret = 0;

	seq_printf(m, "Lumissil IS31FL3195\n\n");

	ret = is31fl3195_read_block(state, 0x00, regs, IS31FL3195_NUM_REGS);

	if (ret < 0) {
		seq_printf(m, "Failed to read registers (ret = %d)\n", ret);
		return ret;
	}

	for (i = 0; i < sizeof(pattern_state_reg_addrs); i++) {
		u8 pattern_state = regs[pattern_state_reg_addrs[i]];
		u8 pattern_t1_code = regs[pattern_ts_t1_reg_addrs[i]] >> IS31FL3195_T1_SHIFT;
		u8 pattern_t3_code = regs[pattern_t2_t3_reg_addrs[i]] >> IS31FL3195_T3_SHIFT;

		seq_printf(m, "Pattern %d State:\n", i + 1);
		if (pattern_state & IS31FL3195_PATTERN_RUNNING_BIT) {
			seq_printf(m, "\tTime State: %s\n",
				is31fl3195_ts_names[pattern_state & IS31FL3195_TIME_STATE_MASK]);
			seq_printf(m, "\tT1 (Fade-In)  Time Code: 0x%01x (%u ms)\n",
				pattern_t1_code, is31fl3195_code_to_fade_time[pattern_t1_code]);
			seq_printf(m, "\tT3 (Fade-Out) Time Code: 0x%01x (%u ms)\n",
				pattern_t3_code, is31fl3195_code_to_fade_time[pattern_t3_code]);
			seq_printf(m, "\tColor State: 0x%01X\n",
					(pattern_state >> IS31FL3195_COLOR_STATE_SHIFT) &
					IS31FL3195_COLOR_STATE_MASK);
		} else {
			seq_printf(m, "\tNot running\n");
		}
	}

	seq_printf(m, "\nRegister dump:");
	for (i = 0; i < IS31FL3195_NUM_REGS; i++) {
		if (!(i % 4)) {
			seq_printf(m, "\n%#04x:  ", i);
		}
		seq_printf(m, "%02x\t", regs[i]);
	}

	seq_printf(m, "\n");
	return ret;
}

static ssize_t is31fl3195_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct is31fl3195_state *state = PDE_DATA(file_inode(filp));
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "reg", 3) == 0) {
		unsigned long reg;
		unsigned long val;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 16, &reg)) {
			LM_WARNING("Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 16, &val)) {
			LM_WARNING("Cannot parse value %s.", val_str);
			return count;
		}
		is31fl3195_write(state, reg, val);
		is31fl3195_read(state, reg, (u8 *)&val);
		LM_INFO("Wrote register %lu, read back %lx.", reg, val);
	}

	return count;
}

const static struct seq_operations is31fl3195_op = {
	.show		= is31fl3195_show
};

static int is31fl3195_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, is31fl3195_show, PDE_DATA(inode));
}

static struct file_operations is31fl3195_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= is31fl3195_proc_open,
	.write		= is31fl3195_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define IS31FL3195_PROCFS_FILE "driver/is31fl3195"

static int is31fl3195_proc_init(struct is31fl3195_state *state)
{
	struct proc_dir_entry *entry;
	entry = proc_create_data(IS31FL3195_PROCFS_FILE, 0666, NULL, &is31fl3195_proc_ops, state);
	if (!entry) {
		return -EIO;
	}

	return 0;
}

static void is31fl3195_proc_remove(void)
{
	remove_proc_entry(IS31FL3195_PROCFS_FILE, NULL);
}

static const struct i2c_device_id is31fl3195_id[] = {
	{ "is31fl3195", 0},
	{ }
};
MODULE_DEVICE_TABLE(is31fl3195_i2c, is31fl3195_id);

static struct of_device_id is31fl3195_ids[] = {
	{ .compatible = "lumissil,is31fl3195"},
	{ }
};

static struct i2c_driver is31fl3195_i2c_driver = {
	.driver = {
		.name		= "is31fl3195",
		.owner		= THIS_MODULE,
		.of_match_table	= is31fl3195_ids,
	},
	.id_table	= is31fl3195_id,
	.probe		= is31fl3195_probe,
	.remove		= is31fl3195_remove
};

module_i2c_driver(is31fl3195_i2c_driver);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for HW using IS31FL3195 to control the LEDs");
MODULE_LICENSE("GPL");
