/*
 * Copyright (c) 2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/cdev.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include "blackbox.h"
#include "sonos_device.h"
#include "ir_common.h"
#include "ir_rcvr.h"
#include "mtk_m4.h"

#define IR_PROC_FILE            "driver/"IR_DEVICE_NAME
#define IR_PROC_PERM            (S_IRUSR | S_IRGRP | S_IROTH | S_IWUSR)

#define IR_MSG_PRINT(LEVEL, FORMAT, ARG...) bb_log(BB_MOD_IR, LEVEL, FORMAT, ## ARG)
#define IR_MSG_INFO(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_INFO, FORMAT, ## ARG)
#define IR_MSG_WARN(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_WARNING, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define IR_MSG_ERROR(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_ERR, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define IR_MSG_DEBUG(FORMAT, ARG...) bb_log_dbg(BB_MOD_IR, FORMAT, ## ARG)

struct ir_private_data {
	int ir_gpio;
	bool ir_ready;
	bool record_stat;
	uint32_t user_count;
	uint32_t timeout;
	uint32_t runt_duration;
	uint32_t ir_len;
	uint32_t ir_data[IR_SYMBOL_MAX_NUM_EDGES];
	ir_message_t msg;
	struct semaphore ir_msg_sem;
};

static int devno = -1;

static int __init ir_driver_init(void);
static void __exit ir_driver_exit(void);
static int ir_receive_thread(void *data);
static int ir_proc_init(void);
static int ir_proc_show(struct seq_file *m, void *v);
static int ir_proc_open(struct inode *inode, struct  file *file);
static ssize_t ir_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data);
static int ir_open(struct inode *inodep, struct file *filp);
static int ir_release(struct inode *inodep, struct file *filp);
static int ir_probe(struct platform_device *pdev);
static int ir_remove(struct platform_device *pdev);
static ssize_t ir_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset);
static ssize_t ir_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset);
static loff_t ir_llseek(struct file *filp, loff_t offset, int seek_type);
static unsigned int ir_poll(struct file *filp, struct poll_table_struct *table);

DEFINE_RWLOCK(ir_rwlock);
DECLARE_WAIT_QUEUE_HEAD(read_queue);

struct task_struct *ir_task;
struct cdev ir_chr_dev;
static struct ir_private_data ir_priv = {
	.user_count = 0,
	.record_stat = 0,
	.ir_len = 0,
	.timeout = TIMEOUT_INITIAL_VAL,
	.ir_ready = 0,
	.runt_duration = RUNT_INITIAL_VAL,
};

irqreturn_t ir_gpio_dummy_isr(int irq, void *data)
{
	return IRQ_NONE;
}

static void ir_msg_recv_callback(uint8_t *buf)
{
	memcpy(&ir_priv.msg, buf, sizeof(ir_message_t));
	up(&ir_priv.ir_msg_sem);
}


static int ir_receive_thread(void *data)
{
	int i;
	int ret = 0;

	IR_MSG_INFO("IR receive thread started");

	while (!kthread_should_stop()) {
		if (!down_interruptible(&ir_priv.ir_msg_sem)) {
			switch(ir_priv.msg.msg_type) {
				case IR_SYMBOL_DATA:
					IR_MSG_DEBUG("IR Symbol Received!");
					write_lock(&ir_rwlock);
					ir_priv.ir_len = ir_priv.msg.msg_len;
					for(i=0; i< ir_priv.msg.msg_len; i++) {
						ir_priv.ir_data[i] = ir_priv.msg.msg_data[i];
					}
					ir_priv.ir_ready = 1;
					if(ir_priv.user_count > 0) {
						wake_up(&read_queue);
					}
					write_unlock(&ir_rwlock);
					break;

				default:
					IR_MSG_ERROR("Invalid Message (%d) received from M4!", ir_priv.msg.msg_type);
					break;
			}
		}
	}
	return ret;
}


static int ir_proc_show(struct seq_file *m, void *v)
{
	int i = 0;
	bool ir_ready;
	bool record_stat;
	uint32_t temp_len;
	uint32_t timeout;
	uint32_t runt_duration;
	uint32_t temp_buff[IR_SYMBOL_MAX_NUM_EDGES];
	struct ir_private_data *priv_data = (struct ir_private_data *)m->private;

	read_lock(&ir_rwlock);
	ir_ready = priv_data->ir_ready;
	record_stat = priv_data->record_stat;
	temp_len = priv_data->ir_len;
	timeout = priv_data->timeout;
	runt_duration = priv_data->runt_duration;
	for(i = 0; i < temp_len; i++) {
		temp_buff[i] = priv_data->ir_data[i];
	}
	read_unlock(&ir_rwlock);

	if(record_stat == IR_RECORD_ON) {
		if(ir_ready) {
			seq_printf(m, "replay=");
			for(i=0; i<temp_len; i++) {
				seq_printf(m, "%08X,", temp_buff[i]);
			}
		} else {
			seq_printf(m, "NO IR Data!\n");
		}
	} else {
		seq_printf(m, "IR Timeout: %uus\n", timeout);
		seq_printf(m, "IR Runt filter duration: %uus\n", runt_duration);
		if(ir_ready) {
			seq_printf(m, "IR Data: ");
			for(i=0; i<temp_len; i++) {
				seq_printf(m, "%08X,", temp_buff[i]);
			}
			seq_printf(m, "\n");
		} else {
			seq_printf(m, "NO IR Data!\n");
		}
	}

	write_lock(&ir_rwlock);
	priv_data->ir_ready = 0;
	write_unlock(&ir_rwlock);
	return 0;
}


static int ir_proc_open(struct inode *inode, struct  file *file)
{
	return single_open(file, ir_proc_show, PDE_DATA(inode));
}


static ssize_t ir_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	struct ir_private_data *priv_data = PDE_DATA(file_inode(file));
	char *temp = NULL;
	char *buf = NULL;
	char *tok = NULL;
	int i = 0;
	int ret;
	unsigned long value;
	struct ir_message msg;

	buf = (char *)kmalloc(sizeof(char)*2048, GFP_KERNEL);
	if (buf == NULL) {
		IR_MSG_ERROR("Failed to allocate memory for buffer");
		ret = -ENOMEM;
		goto err_out;
	}

	if (count > 2047) {
		IR_MSG_ERROR("The string is too long");
		ret = -EIO;
		goto err_out;
	} else if (copy_from_user(buf, buffer, count)) {
		ret = -EFAULT;
		goto err_out;
	}

	buf[count] = '\0';
	temp = strchr(buf, '=');
	if(temp == NULL) {
		IR_MSG_ERROR("Invalid format for proc command");
		IR_MSG_ERROR("Use <command>=<value(s)>");
		ret = -EINVAL;
		goto err_out;
	}
	temp++;

	if (strncmp(buf, "timeout", 7) == 0) {
		if (kstrtoul(temp, 0, &value)) {
			IR_MSG_ERROR("Failed to parse the value: %s", temp);
			ret = -EINVAL;
			goto err_out;
		} else {
			write_lock(&ir_rwlock);
			priv_data->timeout = (uint32_t)value;
			write_unlock(&ir_rwlock);
			msg.msg_type = IR_TIMEOUT;
			msg.msg_len = 1;
			msg.msg_data[0] = (uint32_t)value;
			ret = cm4_ipc_send_packet(IPC_PKT_CT1, (uint8_t *)&msg, sizeof(msg));
			if (ret) {
				IR_MSG_ERROR("Sending message to M4 failed");
				ret = -EFAULT;
				goto err_out;
			}
			IR_MSG_INFO("Setting timeout to %u us", priv_data->timeout);
		}
	} else if (strncmp(buf, "runt", 4) == 0) {
		if (kstrtoul(temp, 0, &value)) {
			IR_MSG_ERROR("Failed to parse the value: %s", temp);
			ret = -EINVAL;
			goto err_out;
		} else {
			write_lock(&ir_rwlock);
			priv_data->runt_duration = (uint32_t)value;
			write_unlock(&ir_rwlock);
			msg.msg_type = IR_RUNT_DURATION;
			msg.msg_len = 1;
			msg.msg_data[0] = (uint32_t)value;
			ret = cm4_ipc_send_packet(IPC_PKT_CT1, (uint8_t *)&msg, sizeof(msg));
			if (ret) {
				IR_MSG_ERROR("Sending message to M4 failed");
				ret  = -EFAULT;
				goto err_out;
			}
			IR_MSG_INFO("Setting runt filter duration to %u us", priv_data->runt_duration);
		}
	} else if (strncmp(buf, "record", 6) == 0) {
		if (strncmp(temp, "on", 2) == 0) {
			write_lock(&ir_rwlock);
			priv_data->record_stat = IR_RECORD_ON;
			write_unlock(&ir_rwlock);
			msg.msg_data[0] = IR_RECORD_ON;
			IR_MSG_INFO("IR record mode ON. The IR data received will be raw edge timestamps");
		} else if (strncmp(temp, "off", 3) == 0) {
			write_lock(&ir_rwlock);
			priv_data->record_stat = IR_RECORD_OFF;
			write_unlock(&ir_rwlock);
			msg.msg_data[0] = IR_RECORD_OFF;
			IR_MSG_INFO("IR record mode OFF. The IR data received will be normal");
		} else {
			IR_MSG_ERROR("Failed to parse the command: %s", temp);
			ret = -EINVAL;
			goto err_out;
		}
		msg.msg_type = IR_RECORD_EDGES;
		msg.msg_len = 1;
		ret = cm4_ipc_send_packet(IPC_PKT_CT1, (uint8_t *)&msg, sizeof(msg));
		if (ret) {
			IR_MSG_ERROR("Sending message to M4 failed");
			ret = -EFAULT;
			goto err_out;
		}
	} else if (strncmp(buf, "replay", 6) == 0) {
		msg.msg_len = 0;
		for (i = 0; i < IR_SYMBOL_MAX_NUM_EDGES; i++) {
			tok = strsep(&temp, ",");
			if (temp == NULL){
				break;
			}
			if (kstrtoul(tok, 16, &value)) {
				IR_MSG_ERROR("Failed to parse the value: %s", tok);
				ret = -EINVAL;
				goto err_out;
			} else {
				msg.msg_data[i] = (uint32_t)value;
				msg.msg_len++;
			}
		}

		IR_MSG_INFO("Message length = %d",msg.msg_len);
		msg.msg_type = IR_REPLAY_EDGES;
		ret = cm4_ipc_send_packet(IPC_PKT_CT1, (uint8_t *)&msg, sizeof(msg));
		if (ret) {
			IR_MSG_ERROR("Sending message to M4 failed");
			ret = -EFAULT;
			goto err_out;
		}
		IR_MSG_INFO("IR in replay mode. Will replay the edges provided");
	}else {
		IR_MSG_ERROR("Unknown command: %s", buf);
		ret = -EINVAL;
		goto err_out;
	}

	ret = count;
err_out:
	kfree(buf);
	return ret;
}


static const struct file_operations ir_proc_fops = {
	.owner = THIS_MODULE,
	.open = ir_proc_open,
	.write = ir_proc_write,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static int ir_proc_init(void)
{
	struct proc_dir_entry *ir_proc_entry;
	ir_proc_entry = proc_create_data(IR_PROC_FILE, IR_PROC_PERM, NULL, &ir_proc_fops, &ir_priv);
	if (!ir_proc_entry) {
		return -EIO;
	}
	return 0;
}


static int ir_open(struct inode *inodep, struct file *filp)
{
	write_lock(&ir_rwlock);
	ir_priv.user_count++;
	write_unlock(&ir_rwlock);
	return 0;
}


static ssize_t ir_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset)
{
	int i;
	int ret;
	char kbuf[IR_SYMBOL_MAX_NUM_EDGES];
	uint32_t temp_len;

	read_lock(&ir_rwlock);
	temp_len = ir_priv.ir_len;
	read_unlock(&ir_rwlock);

	if (size < IR_SYMBOL_MAX_NUM_EDGES) {
		IR_MSG_ERROR("Invalid size! Read must be at least %d bytes (1 max-length symbol).", IR_SYMBOL_MAX_NUM_EDGES);
		return -EINVAL;
	}

	read_lock(&ir_rwlock);
	for (i = 0; i < temp_len; i++) {
		kbuf[i] = (char)ir_priv.ir_data[i];
	}
	read_unlock(&ir_rwlock);

	ret = copy_to_user(buffer, kbuf, temp_len);
	if (ret) {
		IR_MSG_ERROR("Failed to copy buffer from kernel to userspace");
		return ret;
	}

	IR_MSG_DEBUG("IR symbol read by userspace process");

	write_lock(&ir_rwlock);
	ir_priv.ir_ready = 0;
	write_unlock(&ir_rwlock);
	return temp_len;
}

static long ir_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = IR_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			IR_MSG_ERROR("copy_to_user() failed");
			return -EACCES;
		}
		break;
	}
	default:
		IR_MSG_ERROR("Unrecognized IOCTL command %u.", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}


static unsigned int ir_poll(struct file *filp, struct poll_table_struct *table)
{
	unsigned int mask = 0;
	read_lock(&ir_rwlock);
	if (ir_priv.ir_ready) {
		IR_MSG_DEBUG("IR Data ready to be read!");
		mask |= (POLLIN | POLLRDNORM);
	}
	read_unlock(&ir_rwlock);
	poll_wait(filp, &read_queue, table);
	return mask;
}
static ssize_t ir_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset)
{
	IR_MSG_ERROR("Writing to the device file not implemented yet!");
	return -EINVAL;
}


static int ir_release(struct inode *inodep, struct file *filp)
{
	write_lock(&ir_rwlock);
	ir_priv.user_count--;
	write_unlock(&ir_rwlock);
	return 0;
}


static loff_t ir_llseek(struct file *filp, loff_t offset, int seek_type)
{
	IR_MSG_ERROR("Seek not supported in this driver!");
	return -EINVAL;
}

static const struct file_operations ir_fops =
{
	.open		= ir_open,
	.read		= ir_read,
	.poll		= ir_poll,
	.write		= ir_write,
	.llseek		= ir_llseek,
	.release	= ir_release,
	.unlocked_ioctl	= ir_ioctl,
};



static int ir_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct ir_message msg;
	struct device_node *np = pdev->dev.of_node;
	struct device *class_dev = NULL;

	ir_priv.ir_gpio = of_get_named_gpio(np, "ir-gpio", 0);
	if (!gpio_is_valid(ir_priv.ir_gpio)) {
		IR_MSG_ERROR("Failed to get the IR GPIO pin(%d)", ir_priv.ir_gpio);
		return ir_priv.ir_gpio;
	}

	if (gpio_request_one(ir_priv.ir_gpio, GPIOF_IN, "IR rcvr pin")) {
		IR_MSG_ERROR("Failed to request IR GPIO pin");
		return -EFAULT;
	}
	if (request_irq(gpio_to_irq(ir_priv.ir_gpio), ir_gpio_dummy_isr, IRQ_TYPE_EDGE_FALLING, "IR GPIO", NULL)) {
		IR_MSG_ERROR("Failed to request IR GPIO interrupt");
		gpio_free(ir_priv.ir_gpio);
		return -EFAULT;
	}

	msg.msg_type = IR_DRVR_READY;
	msg.msg_len = 0;
	ret = cm4_ipc_send_packet(IPC_PKT_CT1, (uint8_t *)&msg, sizeof(msg));
	if (ret) {
		IR_MSG_ERROR("Failed to send message to CM4(%d)", ret);
		return ret;
	}

	ret = ir_proc_init();
	if (ret) {
		IR_MSG_ERROR("Setting up proc file system failed");
		return ret;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, IR_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		IR_MSG_ERROR("Invalid pointer to the Sonos device struct!");
		ret = PTR_ERR(class_dev);
		remove_proc_entry(IR_PROC_FILE, NULL);
		return ret;
	}

	sema_init(&ir_priv.ir_msg_sem, 0);

	ir_task = kthread_run(&ir_receive_thread, NULL, "IR-RX");
	if (IS_ERR(ir_task)) {
		IR_MSG_ERROR("Starting kthread failed");
		sonos_device_destroy(devno);
		remove_proc_entry(IR_PROC_FILE, NULL);
		return -EFAULT;
	}

	cm4_register_ipc_rcv_callback(ir_msg_recv_callback, IPC_PKT_CT1);

	return ret;
}


static int ir_remove(struct platform_device *pdev)
{
	struct ir_message msg;

	msg.msg_type = IR_DRVR_AWAY;
	msg.msg_len = 0;
	if (cm4_ipc_send_packet(IPC_PKT_CT1, (uint8_t *)&msg, sizeof(msg))) {
		IR_MSG_ERROR("Failed to send message to M4");
	}

	free_irq(gpio_to_irq(ir_priv.ir_gpio), NULL);
	gpio_free(ir_priv.ir_gpio);

	up(&ir_priv.ir_msg_sem);
	cm4_unregister_ipc_rcv_callback(IPC_PKT_CT1);
	kthread_stop(ir_task);
	remove_proc_entry(IR_PROC_FILE, NULL);
	sonos_device_destroy(devno);

	return 0;
}

static const struct of_device_id ir_dt_ids[] = {
	{ .compatible = "sonos,ir-rcvr", },
	{  }
};
MODULE_DEVICE_TABLE(of, ir_dt_ids);

static struct platform_driver ir_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "IR Driver",
		.of_match_table = ir_dt_ids,
		},
	.probe = ir_probe,
	.remove = ir_remove,
};


static int __init ir_driver_init(void)
{
	int ret;

	devno = MKDEV(IR_MAJOR_NUMBER, 0);

#ifdef CONFIG_DEVTMPFS
	ret = alloc_chrdev_region(&devno, 0, 1, IR_DEVICE_NAME);
#else
	ret = register_chrdev_region(devno, 1, IR_DEVICE_NAME);
#endif
	if (ret) {
		IR_MSG_ERROR("Failed to register IR character node (%d)", ret);
		return ret;
	}

	cdev_init(&ir_chr_dev, &ir_fops);
	ir_chr_dev.owner = THIS_MODULE;

	ret = cdev_add(&ir_chr_dev, devno, 1);
	if (ret) {
		IR_MSG_ERROR("Failed to add IR chrdev %d:0 (%d)", MAJOR(devno), ret);
		goto cdev_err;
	}

	ret = platform_driver_register(&ir_driver);
	if (ret) {
		IR_MSG_ERROR("Failed to register IR Receiver driver");
		goto plat_err;
	} else {
		IR_MSG_INFO("Registered IR Receiver driver with major number: %d", MAJOR(devno));
	}

	return 0;

plat_err:
	cdev_del(&ir_chr_dev);
cdev_err:
	unregister_chrdev_region(devno, 1);

	return ret;
}


static void __exit ir_driver_exit(void)
{
	platform_driver_unregister(&ir_driver);
	cdev_del(&ir_chr_dev);
	unregister_chrdev_region(devno, 1);
	IR_MSG_INFO("Unloaded IR Receiver driver");
}

module_init(ir_driver_init);
module_exit(ir_driver_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("IR Receiver driver");
MODULE_LICENSE("GPL");
