/*
 * Copyright (c) 2014-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#ifndef LLA_H
#define LLA_H

#include <linux/types.h>
#include <linux/poll.h>
#include <asm-generic/ioctl.h>

#define LLA_NAME		"llaudio"
#define LLA_DEVICE_NODE		"/dev/"LLA_NAME
#define LLA_MAJOR_AUDIODEV	125


struct lla_csb {
	__u8	data[24];
	__u8	len;
};

enum lla_dev_type {
	LLA_DEV_TYPE_UNKNOWN = -1,
	LLA_DEV_TYPE_DAC = 0,
	LLA_DEV_TYPE_MIC,
	LLA_DEV_TYPE_SPDIF_RX,
	LLA_DEV_TYPE_SPDIF_TX,
	LLA_DEV_TYPE_LINE_IN,
	LLA_DEV_TYPE_MAX,
};

#define LLA_IOCTL_SET_DEVICE		_IO(LLA_MAJOR_AUDIODEV, 1)
#define LLA_IOCTL_COMMIT_TX_BUFFER	_IO(LLA_MAJOR_AUDIODEV, 2)
#define LLA_IOCTL_GET_CSB		_IOR(LLA_MAJOR_AUDIODEV, 3, struct lla_csb)
#define LLA_IOCTL_GET_TIME		_IO(LLA_MAJOR_AUDIODEV, 4)
#define LLA_IOCTL_SET_TX_LATENCY	_IO(LLA_MAJOR_AUDIODEV, 5)
#define LLA_IOCTL_SET_POLL_NUM		_IO(LLA_MAJOR_AUDIODEV, 6)
#define LLA_IOCTL_ACK_RESTART		_IO(LLA_MAJOR_AUDIODEV, 7)

#define LLA_IOCTL_TEST_SYNC_DEV_ENABLE	_IOW(LLA_MAJOR_AUDIODEV, 100, enum lla_dev_type)
#define LLA_IOCTL_TEST_SYNC_DEV_DISABLE	_IOW(LLA_MAJOR_AUDIODEV, 101, enum lla_dev_type)

#define LLA_SHARED_FLAG_TX		0x00
#define LLA_SHARED_FLAG_RX		0x01
#define LLA_SHARED_FLAG_NEED_COMMIT_TX	0x02
#define LLA_SHARED_FLAG_RUNNING		0x04
#define LLA_SHARED_FLAG_ACTIVE		0x08
#define LLA_SHARED_FLAG_ERROR		0x10
#define LLA_SHARED_FLAG_CLK_COUNTS_UP	0x20
#define LLA_SHARED_FLAG_RESTARTED	0x40

#define IS_LLA_SHARED_RX(SHR)	((SHR)->flags & LLA_SHARED_FLAG_RX)
#define IS_LLA_SHARED_TX(SHR)	(!(IS_LLA_SHARED_RX(SHR)))

struct lla_ptr {
	void		*k;
	void		*u;
	unsigned long	p;
};

struct lla_shared {
	struct lla_ptr	ref;
	__u32		ref_size;

	__s32		type;
	__u32		flags;

	struct lla_ptr	buffers;
	__u32		buffer_len;
	__u32		buffer_cnt;
	__u32		buffer_start_offset;
	__u32		ring_cnt;
	__u32		buffers_total;

	__u32		frames_per_buffer;
	__u32		frame_len;
	__u32		channels_per_frame;
	__u32		channel_len;
	__u32		channel_mask;

	__u32		buffers_to_frames_shift;
	__u32		buffers_to_index_mask;

	volatile __u32	sample_rate;
	volatile __u64	buffers_complete;
	volatile __u32	complete_time;
	volatile __u32  complete_time2;
	volatile __u64	poll_num;
	volatile __u32	tx_underflows;
	volatile __u32	tx_latency;

	struct lla_ptr	hw_clock;
	__u32		hw_clock_rate;
	__u32		hw_ticks_per_frame;
};

#define _LLA_DUMP_SHARED(shared, func, pfx) \
{ \
	func(pfx "Dump of lla_shared (%p)\n", shared); \
	func(pfx "\t ref                     = %p/%p/%08lx\n", shared->ref.k, shared->ref.u, shared->ref.p); \
	func(pfx "\t ref_size                = %u\n", shared->ref_size); \
	\
	func(pfx "\t type                    = %d\n", shared->type); \
	func(pfx "\t flags                   = %08x\n", shared->flags); \
	\
	func(pfx "\t buffers                 = %p/%p/%08lx\n", shared->buffers.k, shared->buffers.u, shared->buffers.p); \
	func(pfx "\t buffer_len              = %u\n", shared->buffer_len); \
	func(pfx "\t buffer_cnt              = %u\n", shared->buffer_cnt); \
	func(pfx "\t buffer_start_offset     = %u\n", shared->buffer_start_offset); \
	func(pfx "\t buffers_total           = %u\n", shared->buffers_total); \
	\
	func(pfx "\t ring_cnt                = %u\n", shared->ring_cnt); \
	\
	func(pfx "\t frames_per_buffer       = %u\n", shared->frames_per_buffer); \
	func(pfx "\t frame_len               = %u\n", shared->frame_len); \
	func(pfx "\t channels_per_frame      = %u\n", shared->channels_per_frame); \
	func(pfx "\t channel_len             = %u\n", shared->channel_len); \
	func(pfx "\t channel_mask            = %08x\n", shared->channel_mask); \
	\
	func(pfx "\t buffers_to_frames_shift = %u\n", shared->buffer_to_frames_shift); \
	func(pfx "\t buffer_to_index_mask    = %u\n", shared->buffer_to_index_mask); \
	\
	func(pfx "\t sample_rate             = %llu\n", shared->sample_rate); \
	func(pfx "\t buffers_complete        = %llu\n", shared->buffers_complete); \
	func(pfx "\t complete_time           = %u\n", shared->complete_time); \
	func(pfx "\t complete_time2          = %u\n", shared->complete_time2); \
	func(pfx "\t poll_num                = %llu\n", shared->poll_num); \
	func(pfx "\t tx_underflows           = %u\n", shared->tx_underflows); \
	func(pfx "\t tx_latency              = %u\n", shared->tx_latency); \
	\
	func(pfx "\t hw_clock                = %p/%p/%08lx\n", shared->hw_clock.k, shared->hw_clock.u, shared->hw_clock.p); \
	func(pfx "\t hw_clock_rate           = %u\n", shared->hw_clock_rate); \
	func(pfx "\t hw_ticks_per_frame      = %u\n", shared->hw_ticks_per_frame); \
}

#define LLA_BUFFER_NUM_TO_FRAMES(shared, buffer_num)	((buffer_num) << shared->buffers_to_frames_shift)

#define LLA_FRAMES_TO_BUFFER_NUM(shared, frames)	((frames) >> shared->buffers_to_frames_shift)

#define LLA_BUFFER_NUM_TO_IDX(shared, buffer_num)	((buffer_num) & shared->buffers_to_index_mask)


#ifdef __KERNEL__
#define LLA_DUMP_SHARED(shared) _LLA_DUMP_SHARED(shared, printk, KERN_INFO)
#else
#define LLA_DUMP_SHARED(shared) _LLA_DUMP_SHARED(shared, printf, "")
#endif

#ifdef __KERNEL__

struct lla_ops {
	int (*attach) (struct file *filp, void *context);
	int (*release) (struct file *filp, void *context);
	unsigned int (*poll) (struct file *filp, struct poll_table_struct *table, void *context);
	int (*commit) (unsigned long buffer_idx, void *context);
	int (*get_csb) (struct lla_csb *csb, void *context);
};

struct lla_dev {
	const char		*name;
	enum lla_dev_type	type;
	int			ref;
	struct lla_ops		*ops;
	void			*context;
	struct lla_shared	*shared;
	char			filename[32];
	struct device		*dev;
	struct proc_dir_entry	*procfs;

	struct lla_dev		*sync_dev;
	int			is_primary;
};

struct lla_hw_clk_ops {
	u32 (*read) (void *context);
};

struct lla_hw_clk {
	const char		*name;
	phys_addr_t		reg_addr;
	u32			rate;
	u32			counts_up;
	struct lla_hw_clk_ops	*ops;
	void			*context;
};

#define IS_LLA_DEV_IN_USE(DEV)	((DEV)->ref)

#define IS_LLA_DEV_SYNC_PRIMARY(DEV)	((DEV)->sync_dev && (DEV)->is_primary)
#define IS_LLA_DEV_SYNC_SECONDARY(DEV)	((DEV)->sync_dev && (!((DEV)->is_primary)))

extern int lla_init_module(void);

extern void lla_exit_module(void);

extern int lla_init_clk(
	struct lla_hw_clk *hwclk,
	const char *name,
	phys_addr_t reg_addr,
	u32 rate,
	u32 counts_up,
	struct lla_hw_clk_ops *ops,
	void *context);

extern int lla_register_clk(struct lla_hw_clk *hwclk);

extern void lla_unregister_clk(struct lla_hw_clk *hwclk);

extern u32 lla_clock_read(void);

extern int lla_init_dev(
	struct lla_dev *adev,
	struct device *dev,
	const char *name,
	enum lla_dev_type type,
	struct lla_ops *ops,
	void *context);

extern int lla_sync_to_dev(
	struct lla_dev *adev,
	enum lla_dev_type primary);

extern int lla_register_dev(struct lla_dev *adev);

extern void lla_unregister_dev(struct lla_dev *adev);

extern struct lla_shared *lla_allocate_shared_buffers(
		struct lla_dev *adev,
		u32 channel_len,
		u32 channels_per_frame,
		u32 frames_per_buffer,
		u32 buffer_cnt,
		u32 ring_cnt,
		struct lla_ptr *audio_buf);

void lla_free_shared_buffers(struct lla_dev *adev);

extern void lla_update_sample_rate(
	struct lla_dev *adev,
	u32 sample_rate);

extern int lla_map_to_user(
	struct vm_area_struct *vma,
	struct lla_ptr *ptr,
	unsigned long size,
	unsigned long *offset,
	bool uncached);

#endif

#endif
