/*
 * Copyright (c) 2016-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Standard amplifier control for all platforms.
 *
 */

#ifndef AMPCTL_H
#define AMPCTL_H

#if defined(SONOS_ARCH_ATTR_SOC_IS_A113)
#include "ak4432.h"
#endif

#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX6)
#include "ak4458.h"
#include "ak461x.h"
#include "cs4265.h"
#endif

#if defined(SONOS_ARCH_ATTR_SOC_IS_MT8521P)
#include "ak4458.h"
#include "pcm1690.h"
#endif

#define AMPCTL_MAJOR_NUMBER	(214)
#define AMPCTL_DEVICE_NAME	"ampctl"
#define AMPCTL_DEVICE_NODE	"/dev/"AMPCTL_DEVICE_NAME

#define AMPCTL_VERSION		(1)
#define AMPCTL_VERSION_2	(2)

enum amp_type {
	AMP_AK4490,
	AMP_CS431XX,
	AMP_TAS5720,
	AMP_TAS58XX,
	AMP_MA2304,
	AMP_SSM3582,
	AMP_RT9120S
};

enum {
	_AMPCTL_GET_VERSION = 0,
	_AMPCTL_SET_POWER,
	_AMPCTL_SET_MUTE,
	_AMPCTL_SET_HIPOWER,
	_AMPCTL_GET_RAIL,
	_AMPCTL_GET_CAPABILITY,
	_AMPCTL_SET_CONFIG,
	_AMPCTL_SPK_DETECT_GOTO_READY,
	_AMPCTL_SPK_DETECT_START_CAL,
	_AMPCTL_SPK_DETECT_START_MEAS,
	_AMPCTL_SPK_DETECT_GET_STATUS,
	_AMPCTL_SPK_DETECT_READ_MEAS_L,
	_AMPCTL_SPK_DETECT_READ_MEAS_R,
	_AMPCTL_RESET_AND_INIT,
	_AMPCTL_CONTROL_BOOST,
	_AMPCTL_GET_AMP_TYPE,
	_AMPCTL_MAX
};

#define AMPCTL_GET_VERSION	_IOR(AMPCTL_MAJOR_NUMBER, _AMPCTL_GET_VERSION, uint32_t)
#define AMPCTL_GET_CAPABILITY	_IOR(AMPCTL_MAJOR_NUMBER, _AMPCTL_GET_CAPABILITY, uint32_t)
#define AMPCTL_SET_POWER	_IOW(AMPCTL_MAJOR_NUMBER, _AMPCTL_SET_POWER, uint8_t)
#define AMPCTL_SET_MUTE		_IOW(AMPCTL_MAJOR_NUMBER, _AMPCTL_SET_MUTE, uint8_t)
#define AMPCTL_SET_HIPOWER	_IOW(AMPCTL_MAJOR_NUMBER, _AMPCTL_SET_HIPOWER, uint8_t)
#define AMPCTL_GET_RAIL		_IOR(AMPCTL_MAJOR_NUMBER, _AMPCTL_GET_RAIL, int)
#define AMPCTL_SET_CONFIG	_IOW(AMPCTL_MAJOR_NUMBER, _AMPCTL_SET_CONFIG, void *)
#define AMPCTL_SPK_DETECT_GOTO_READY   _IO(AMPCTL_MAJOR_NUMBER, _AMPCTL_SPK_DETECT_GOTO_READY)
#define AMPCTL_SPK_DETECT_START_CAL    _IOW(AMPCTL_MAJOR_NUMBER, _AMPCTL_SPK_DETECT_START_CAL, int)
#define AMPCTL_SPK_DETECT_START_MEAS   _IOW(AMPCTL_MAJOR_NUMBER, _AMPCTL_SPK_DETECT_START_MEAS, int)
#define AMPCTL_SPK_DETECT_GET_STATUS   _IOR(AMPCTL_MAJOR_NUMBER, _AMPCTL_SPK_DETECT_GET_STATUS, int *)
#define AMPCTL_SPK_DETECT_READ_MEAS_L  _IOR(AMPCTL_MAJOR_NUMBER, _AMPCTL_SPK_DETECT_READ_MEAS_L, int *)
#define AMPCTL_SPK_DETECT_READ_MEAS_R  _IOR(AMPCTL_MAJOR_NUMBER, _AMPCTL_SPK_DETECT_READ_MEAS_R, int *)
#define AMPCTL_RESET_AND_INIT	_IOW(AMPCTL_MAJOR_NUMBER, _AMPCTL_RESET_AND_INIT, int)
#define AMPCTL_CONTROL_BOOST	_IOW(AMPCTL_MAJOR_NUMBER, _AMPCTL_CONTROL_BOOST, void *)
#define AMPCTL_GET_AMP_TYPE	_IOR(AMPCTL_MAJOR_NUMBER, _AMPCTL_GET_AMP_TYPE, int)

struct ampctl_boost_control {
	int mode;
	int duty_cycle;
};

#if defined(SONOS_ARCH_ATTR_SOC_IS_IMX6)
#define AMPCTL_HW_INITS { \
	NULL, \
	ak4458_i2c_init, \
	ak461x_i2c_InitDriver, \
	cs4265_i2c_init, \
	NULL \
} ;
#define AMPCTL_HW_EXITS { \
	NULL, \
	ak4458_i2c_exit, \
	ak461x_i2c_ExitDriver, \
	cs4265_i2c_exit, \
	NULL \
} ;
#define AMPCTL_MORE_WORK_FUNCS { \
	NULL, \
	ak4458_dac_smute_i2c, \
	ak4458_dac_reset_i2c, \
	ak461x_mute_i2c, \
	ak461x_dac_reset_i2c, \
	cs4265_init_device, \
	NULL \
} ;
#endif

#if defined(SONOS_ARCH_ATTR_SOC_IS_MT8521P)
#define AMPCTL_HW_INITS { \
	NULL, \
	ak4458_i2c_init, \
	pcm1690_i2c_init, \
	NULL \
} ;
#define AMPCTL_HW_EXITS { \
	NULL, \
	ak4458_i2c_exit, \
	pcm1690_i2c_exit, \
	NULL \
} ;
#define AMPCTL_MORE_WORK_FUNCS { \
	NULL, \
	ak4458_dac_smute_i2c, \
	ak4458_dac_reset_i2c, \
	pcm1690_dac_smute_i2c, \
	pcm1690_dac_reset_i2c, \
	NULL \
} ;
#endif

#if defined(SONOS_ARCH_ATTR_SOC_IS_A113)
#define AMPCTL_HW_INITS { \
	NULL, \
	ak4432_i2c_init, \
	NULL \
} ;
#define AMPCTL_HW_EXITS { \
	NULL, \
	ak4432_i2c_exit, \
	NULL \
} ;
#define AMPCTL_MORE_WORK_FUNCS { \
	NULL, \
	ak4432_dac_smute_i2c, \
	ak4432_dac_reset_i2c, \
	NULL \
} ;
#endif

#ifdef __KERNEL__
#include <linux/gpio.h>

#define AMP_ENABLED	0
#define AMP_SW_RESET	1
#define AMP_HW_RESET	2

struct amp_ops {
	void (*enable)(struct amp_ops *, int);
	int (*is_enabled)(struct amp_ops *);
	void (*write_reg)(struct amp_ops *, int, int);
	void (*enable_dither)(struct amp_ops *, int);
	int (*get_faults)(struct amp_ops *);
	void (*handle_faults)(struct amp_ops *);
	void (*clear_faults)(struct amp_ops *);
	int (*set_config)(struct amp_ops *, void *);
	void (*reset_and_init)(struct amp_ops *, int);
	enum amp_type (*get_amp_type)(void);

	int (*spk_detect_goto_ready)(struct amp_ops *);
	int (*spk_detect_start_cal)(struct amp_ops *, void *);
	int (*spk_detect_start_meas)(struct amp_ops *, void *);
	int (*spk_detect_get_status)(struct amp_ops *, void *);
	int (*spk_detect_read_meas_l)(struct amp_ops *, void *);
	int (*spk_detect_read_meas_r)(struct amp_ops *, void *);

	struct amp_ops *next;
};
void ampctl_register_callbacks(struct amp_ops *ops);

enum ampctl_event {
	AMPCTL_REQ_POWER_ON,
	AMPCTL_REQ_POWER_OFF,
	AMPCTL_EVENT_POWER_ON,
	AMPCTL_EVENT_POWER_OFF,

	AMPCTL_REQ_MUTE_ON,
	AMPCTL_REQ_MUTE_OFF,
	AMPCTL_EVENT_MUTE_ON,
	AMPCTL_EVENT_MUTE_OFF,

	AMPCTL_REQ_HIPOWER_ON,
	AMPCTL_REQ_HIPOWER_OFF,
	AMPCTL_EVENT_HIPOWER_ON,
	AMPCTL_EVENT_HIPOWER_OFF,

	AMPCTL_EVENT_DAC_RESET_ON,
	AMPCTL_EVENT_DAC_RESET_OFF,

	AMPCTL_EVENT_I2C_DONE,
	AMPCTL_EVENT_ERROR,
	AMPCTL_REQ_AMP_FAULT,

	AMPCTL_EVENT_MAX
};

enum ampctl_signal {
	AMPCTL_SIG_DACRESET,
	AMPCTL_SIG_POWER,
	AMPCTL_SIG_MUTE,
	AMPCTL_SIG_HIPOWER,
	AMPCTL_SIG_BOUNDARY
};

enum ampctl_init_states {
	AMPCTL_INIT_NONE,
	AMPCTL_INIT_ERROR,
	AMPCTL_INIT_DONE
};

struct ampctl_signal_data {
	int config;
	int actual;
	unsigned int on_time;
	unsigned int off_time;
	bool supports_off;
	struct timer_list timer;
	enum ampctl_event on_event;
	enum ampctl_event off_event;
	enum ampctl_event on_req;
	enum ampctl_event off_req;
	struct gpio pin;
	int active_low;
	int (*more_work)(int);

	unsigned int more_work_delay;
};

extern void ampctl_shutdown(void);

#endif

#endif
