/*
 * Copyright (c) 2016-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * SONOS LED Driver
 * Using i.MX6 PWM
 *
 */
#include <linux/gpio.h>
#include <linux/module.h>
#include <linux/of_platform.h>
#include <linux/pwm.h>
#include <linux/reboot.h>
#include <linux/io.h>
#include <asm/io.h>

#include <linux/semaphore.h>
#include "button_event.h"
#include "sonos_led.h"
#include "blackbox.h"
#include "ledctl.h"

static struct pwm_device *leds[4];

struct pwm_scale {
	int pwm_led_scale;
	int pwm_led_period;
};

static struct pwm_scale scales = { 0 };

#define WHITE_LED		leds[0]
#define RED_LED			leds[1]
#define GREEN_LED		leds[2]
#define AMBER_LED		leds[3]

#define IMX_PWM_GPIO_PIN_REGS  \
    {0x020e0384, 0x000010b0},  \
    {0x020e0388, 0x000010b0},  \
    {0x020e0390, 0x000010b0},  \
    {0x020e053c, 0x000010b0},  \
    {0x020e003c, 0x00000005},  \
    {0x020e0040, 0x00000005},  \
    {0x020e0048, 0x00000005},  \
    {0x020e01f4, 0x00000005}

int imx_pwm_led_update(struct led_step *step)
{
	bb_log_dbg(BB_MOD_LEDCTL, "Got step %02x%02x%02x hold time: %u", step->r, step->g, step->b, step->hold_time);
	if (ledctl_is_white(step)) {
		pwm_config(RED_LED, 0, scales.pwm_led_period);
		pwm_config(GREEN_LED, 0, scales.pwm_led_period);
		pwm_config(AMBER_LED, 0, scales.pwm_led_period);
		pwm_config(WHITE_LED, (step->r * scales.pwm_led_scale), scales.pwm_led_period);
	} else {
		pwm_config(WHITE_LED, 0, scales.pwm_led_period);
		pwm_config(RED_LED, (step->r * scales.pwm_led_scale), scales.pwm_led_period);
		pwm_config(GREEN_LED, (step->g * scales.pwm_led_scale), scales.pwm_led_period);
		pwm_config(AMBER_LED, (step->b * scales.pwm_led_scale), scales.pwm_led_period);
	}
	return step->hold_time;
}

int imx_pwm_led_scale(struct led_step *step)
{
	if (step->hold_time > 150) {
		uint16_t mod = step->hold_time % 750;
		if (mod) {
			step->hold_time = step->hold_time + 750 - mod;
		}
	}
	if (ledctl_is_white(step)) {
		return 0;
	} else {
		if (step->r > step->g) {
			if (step->g > 0x20) {
				step->b = 0xff;
				step->r = 0;
				step->g = 0;
			} else {
				step->r = 0xff;
				step->g = 0;
				step->b = 0;
			}
		} else if (step->g > step->r) {
			step->g = 0xff;
			step->r = 0;
			step->b = 0;
		} else {
			step->r = 0;
			step->g = 0;
			step->b = 0;
			return -1;
		}
	}
	return 0;
}

#if defined(IMX_PWM_GPIO_PIN_REGS)
struct imx_pwm_reboot_reg_addr_data {
	u32  addr;
	u32  data;
};
struct imx_pwm_reboot_reg_addr_data imx_pwm_reboot_regs[] = {IMX_PWM_GPIO_PIN_REGS};
struct notifier_block imx_pwm_led_reboot_notifier;

int imx_pwm_led_reboot_callback(struct notifier_block *nb, unsigned long x, void *y)
{
	int i;
	int num = sizeof(imx_pwm_reboot_regs) / sizeof(struct imx_pwm_reboot_reg_addr_data);
	for (i = 0; i < num; i++) {
		struct imx_pwm_reboot_reg_addr_data *reg = &imx_pwm_reboot_regs[i];
		u32 __iomem *p = (u32 *)ioremap(reg->addr, 4);
		if (p == NULL) {
			break;
		}
		writel(reg->data, p);
	}
	return NOTIFY_DONE;
}
#endif

int imx_pwm_led_init(struct device_node *np)
{
	if (!np) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Could not find node led-pwm");
		return -ENODEV;
	}

	if (of_property_read_u32(np, "scale", &scales.pwm_led_scale)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Unable to get scale");
		return -EINVAL;
	}
	if (of_property_read_u32(np, "period", &scales.pwm_led_period)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Unable to get period");
		return -EINVAL;
	}

#if defined(IMX_PWM_GPIO_PIN_REGS)
	imx_pwm_led_reboot_notifier.notifier_call = imx_pwm_led_reboot_callback;
	imx_pwm_led_reboot_notifier.priority = 0;
	register_reboot_notifier(&imx_pwm_led_reboot_notifier);
#endif

	return 0;
}

static struct ledctl_hw_ops ops = {
	.update		= imx_pwm_led_update,
	.scale		= imx_pwm_led_scale,
};

static int imx_pwm_led_probe(struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	int ret = 0;

	if (imx_pwm_led_init(node)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_INFO, "imx PWM control driver init failed");
		return -EINVAL;
	}

	bb_log(BB_MOD_LEDCTL, BB_LVL_INFO, "Starting Sonos PWM Control driver");

	WHITE_LED = of_pwm_get(node, "white");
	if (IS_ERR(WHITE_LED)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "of_pwm_get(white) returned %p", WHITE_LED);
		ret = -1;
		goto exit;
	} else {
		pwm_config(WHITE_LED, 1000000000, 2000000000);
		pwm_enable(WHITE_LED);
	}
	RED_LED = of_pwm_get(node, "red");
	if (IS_ERR(RED_LED)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "of_pwm_get(red) returned %p", RED_LED);
		ret = -1;
		goto exit;
	} else {
		pwm_config(RED_LED, 0, 1000);
		pwm_enable(RED_LED);
	}
	GREEN_LED = of_pwm_get(node, "green");
	if (IS_ERR(GREEN_LED)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "of_pwm_get(green) returned %p", GREEN_LED);
		ret = -1;
		goto exit;
	} else {
		pwm_config(GREEN_LED, 0, 1000);
		pwm_enable(GREEN_LED);
	}
	AMBER_LED = of_pwm_get(node, "amber");
	if (IS_ERR(AMBER_LED)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "of_pwm_get(amber) returned %p", AMBER_LED);
		ret = -1;
		goto exit;
	} else {
		pwm_config(AMBER_LED, 0, 1000);
		pwm_enable(AMBER_LED);
	}

	ledctl_hw_register(&ops);

exit:
	return ret;
}

static int imx_pwm_led_remove(struct platform_device *pdev)
{
#if defined(IMX_PWM_GPIO_PIN_REGS)
	unregister_reboot_notifier(&imx_pwm_led_reboot_notifier);
#endif
	ledctl_hw_unregister();
	pwm_put(WHITE_LED);
	pwm_put(RED_LED);
	pwm_put(GREEN_LED);
	pwm_put(AMBER_LED);
	return 0;
}

const static struct of_device_id imx_pwm_match[] = {
	{ .compatible = "sonos,pwm-led", },
	{}
};
MODULE_DEVICE_TABLE(of, imx_pwm_match);

static struct platform_driver imx_pwm_driver = {
	.probe = imx_pwm_led_probe,
	.remove = imx_pwm_led_remove,
	.driver = {
		.name = "Sonos PWM LED Driver",
		.owner = THIS_MODULE,
		.of_match_table = imx_pwm_match,
	},
};

module_platform_driver(imx_pwm_driver);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for HW using imx_pwm to control the LED");
MODULE_LICENSE("GPL");
