/*
 * Copyright (c) 2016-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * LED control driver for iCE40 FPGA on Elrey
 */

#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include "blackbox.h"
#include "ledctl.h"
#include "ice40_fpga.h"

#define ICE40_REG_RED_BLUE_INT_A         0x00
#define ICE40_REG_GREEN_WHITE_INT_A      0x01
#define ICE40_REG_RED_BLUE_INT_B         0x02
#define ICE40_REG_GREEN_WHITE_INT_B      0x03
#define ICE40_REG_RED_FRACT_A            0x04
#define ICE40_REG_BLUE_FRACT_A           0x05
#define ICE40_REG_GREEN_FRACT_A          0x06
#define ICE40_REG_WHITE_FRACT_A          0x07
#define ICE40_REG_RED_FRACT_B            0x08
#define ICE40_REG_BLUE_FRACT_B           0x09
#define ICE40_REG_GREEN_FRACT_B          0x0A
#define ICE40_REG_WHITE_FRACT_B          0x0B
#define ICE40_REG_ON_TIME_A              0x0C
#define ICE40_REG_ON_TIME_B              0x0D
#define ICE40_REG_RAMP_TIME_A_B          0x0E
#define ICE40_REG_TRAN_TIME_A_B          0x0F
#define ICE40_REG_RGBW_LED_CONTROL       0x40

#define ICE40_MSK_RGBW_OUTPUT_EN         BIT(7)
#define ICE40_MSK_RGBW_CURRENT_EN        BIT(6)
#define ICE40_MSK_RGBW_BLINK_EN          BIT(5)
#define ICE40_MSK_RGBW_A_ON_OFF          BIT(4)
#define ICE40_MSK_RGBW_B_ON_OFF          BIT(3)
#define ICE40_MSK_INT_BLUE_WHITE         (BIT(3)|BIT(2)|BIT(1)|BIT(0))
#define ICE40_MSK_INT_RED_GREEN          (BIT(7)|BIT(6)|BIT(5)|BIT(4))

#define ICE40_LED_RAMP_TIME_STEP_US      2732
#define ICE40_LED_ON_TIME_STEP_MS        11
#define ICE40_LED_MIN_ON_TIME            1
#define ICE40_LED_MIN_RAMP_TIME          0x10
#define ICE40_LED_MAX_RAMP_TIME          0xff

struct ice40_leds_state {
	bool slot_a_used;
};

struct ice40_leds_state *ice40_global_state = NULL;

static inline struct ice40_leds_state *ice40_leds_get_state(void)
{
	return ice40_global_state;
}

#if !defined(SONOS_ARCH_ELREY)
int ice40_fpga_write_reg(uint8_t reg, uint8_t val)
{
    return 0;
}
#endif

void ice40_leds_calculate_slew(uint8_t bright, uint8_t ramp, uint8_t *slew_int, uint8_t *slew_frac)
{
	uint32_t scaled_slope = 0;

	scaled_slope = ((bright * 10000) / ramp);
	*slew_int = bright / ramp;
	*slew_frac = (uint8_t)(((scaled_slope - (*slew_int * 10000)) << 8) / 10000 );
}

int ice40_leds_update(struct led_step *step)
{
	uint8_t red_slew_int = 0;
	uint8_t green_slew_int = 0;
	uint8_t blue_slew_int = 0;
	uint8_t white_slew_int = 0;
	uint8_t red_slew_frac = 0;
	uint8_t green_slew_frac = 0;
	uint8_t blue_slew_frac = 0;
	uint8_t white_slew_frac = 0;
	uint16_t ramp_time = 0;
	uint8_t on_time = 0;
	uint32_t time_spent_in_step = 0;
	struct ice40_leds_state *state = ice40_leds_get_state();

	if (IS_ERR(state)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "The iCE40 LED state structure is not intialized!");
		return PTR_ERR(state);
	}


	bb_log_dbg(BB_MOD_LEDCTL, "Got step %02x%02x%02x hold time: %u fade time: %u", step->r, step->g, step->b, step->hold_time, step->fade);

	if (step->fade) {
		ramp_time = ((step->fade * 1000) / ICE40_LED_RAMP_TIME_STEP_US);
	}

	on_time = ICE40_LED_MIN_ON_TIME;

	if (ramp_time < ICE40_LED_MIN_RAMP_TIME) {
		ramp_time = ICE40_LED_MIN_RAMP_TIME;
	} else if (ramp_time > ICE40_LED_MAX_RAMP_TIME) {
		ramp_time = ICE40_LED_MAX_RAMP_TIME;
	}

	time_spent_in_step = (uint32_t)(step->fade + step->hold_time);
	if (!step->fade) {
		time_spent_in_step += ((ramp_time * ICE40_LED_RAMP_TIME_STEP_US) / 1000);
	}
	if (!step->hold_time) {
		time_spent_in_step += (on_time * ICE40_LED_ON_TIME_STEP_MS);
	}


	if (ledctl_is_white(step)) {
		ice40_leds_calculate_slew(step->r, ramp_time, &white_slew_int, &white_slew_frac);
	} else {
		ice40_leds_calculate_slew(step->r, ramp_time, &red_slew_int, &red_slew_frac);
		ice40_leds_calculate_slew(step->g, ramp_time, &green_slew_int, &green_slew_frac);
		ice40_leds_calculate_slew(step->b, ramp_time, &blue_slew_int, &blue_slew_frac);
	}

	ramp_time--;

	if (!state->slot_a_used) {
		ice40_fpga_write_reg(ICE40_REG_RED_BLUE_INT_A,
			(((red_slew_int << 4 ) & ICE40_MSK_INT_RED_GREEN) | ((blue_slew_int) & ICE40_MSK_INT_BLUE_WHITE)));

		ice40_fpga_write_reg(ICE40_REG_GREEN_WHITE_INT_A,
			(((green_slew_int << 4 ) & ICE40_MSK_INT_RED_GREEN) | ((white_slew_int) & ICE40_MSK_INT_BLUE_WHITE)));

		ice40_fpga_write_reg(ICE40_REG_RED_FRACT_A, red_slew_frac);
		ice40_fpga_write_reg(ICE40_REG_GREEN_FRACT_A, green_slew_frac);
		ice40_fpga_write_reg(ICE40_REG_BLUE_FRACT_A, blue_slew_frac);
		ice40_fpga_write_reg(ICE40_REG_WHITE_FRACT_A, white_slew_frac);
		ice40_fpga_write_reg(ICE40_REG_ON_TIME_A, on_time);
		state->slot_a_used = true;
	} else {
		ice40_fpga_write_reg(ICE40_REG_RED_BLUE_INT_B,
			(((red_slew_int << 4 ) & ICE40_MSK_INT_RED_GREEN) | ((blue_slew_int) & ICE40_MSK_INT_BLUE_WHITE)));

		ice40_fpga_write_reg(ICE40_REG_GREEN_WHITE_INT_B,
			(((green_slew_int << 4 ) & ICE40_MSK_INT_RED_GREEN) | ((white_slew_int) & ICE40_MSK_INT_BLUE_WHITE)));

		ice40_fpga_write_reg(ICE40_REG_RED_FRACT_B, red_slew_frac);
		ice40_fpga_write_reg(ICE40_REG_GREEN_FRACT_B, green_slew_frac);
		ice40_fpga_write_reg(ICE40_REG_BLUE_FRACT_B, blue_slew_frac);
		ice40_fpga_write_reg(ICE40_REG_WHITE_FRACT_B, white_slew_frac);
		ice40_fpga_write_reg(ICE40_REG_ON_TIME_B, on_time);
		state->slot_a_used = false;
	}

	ice40_fpga_write_reg(ICE40_REG_RAMP_TIME_A_B, ramp_time);

	if (state->slot_a_used) {
		ice40_fpga_write_reg(ICE40_REG_RGBW_LED_CONTROL, (ICE40_MSK_RGBW_OUTPUT_EN | ICE40_MSK_RGBW_CURRENT_EN | ICE40_MSK_RGBW_A_ON_OFF));
	} else {
		ice40_fpga_write_reg(ICE40_REG_RGBW_LED_CONTROL, (ICE40_MSK_RGBW_OUTPUT_EN | ICE40_MSK_RGBW_CURRENT_EN | ICE40_MSK_RGBW_B_ON_OFF));
	}

	return time_spent_in_step;
}

int ice40_leds_scale(struct led_step *step)
{
	if (!ledctl_is_white(step) && step->r > step->g && step->g) {
		step->r = 0xff;
		step->g >>= 2;
	}

	return 0;
}

static struct ledctl_hw_ops ops = {
	.update		= ice40_leds_update,
	.scale		= ice40_leds_scale,
};

static int __init ice40_leds_init(void)
{
	struct ice40_leds_state *state = kzalloc(sizeof(struct ice40_leds_state), GFP_KERNEL);
	if (IS_ERR(state)) {
		bb_log(BB_MOD_LEDCTL, BB_LVL_ERR, "Failed to allocate memory for the state structure!");
		return -ENOMEM;
	}
	ice40_global_state = state;

	ice40_fpga_write_reg(ICE40_REG_GREEN_WHITE_INT_A, 0x01);
	ice40_fpga_write_reg(ICE40_REG_GREEN_WHITE_INT_B, 0x00);
	ice40_fpga_write_reg(ICE40_REG_WHITE_FRACT_A, 0x24);
	ice40_fpga_write_reg(ICE40_REG_WHITE_FRACT_B, 0x00);
	ice40_fpga_write_reg(ICE40_REG_ON_TIME_A, 0x3c);
	ice40_fpga_write_reg(ICE40_REG_ON_TIME_B, 0x3c);
	ice40_fpga_write_reg(ICE40_REG_RAMP_TIME_A_B, 0xdd);
	ice40_fpga_write_reg(ICE40_REG_TRAN_TIME_A_B, 0x01);
	ice40_fpga_write_reg(ICE40_REG_RGBW_LED_CONTROL, (ICE40_MSK_RGBW_OUTPUT_EN | ICE40_MSK_RGBW_CURRENT_EN | ICE40_MSK_RGBW_BLINK_EN));

	state->slot_a_used = false;

	ledctl_hw_register(&ops);

	bb_log(BB_MOD_LEDCTL, BB_LVL_INFO, "Initialized Sonos iCE40 LED Control driver");

	return 0;
}
module_init(ice40_leds_init);

static void __exit ice40_leds_exit(void)
{
	ledctl_hw_unregister();
	ice40_fpga_write_reg(ICE40_REG_RGBW_LED_CONTROL, 0);
	kzfree(ice40_global_state);
	bb_log(BB_MOD_LEDCTL, BB_LVL_INFO, "Exited Sonos iCE40 LED Control driver");
}
module_exit(ice40_leds_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for HW using ice40_fpga to control the LED");
MODULE_LICENSE("GPL");
