/*
 * Copyright (c) 2017, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The purpose of this driver is to load the ice40X FPGA bitstream binary via fpga-manager
 * and provide necessary exported functions to talk to the ice40 FPGA over i2c.
 */

#include <asm/uaccess.h>
#include <linux/err.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/proc_fs.h>
#include <linux/gpio/consumer.h>
#include <linux/i2c.h>
#include <linux/fpga/fpga-mgr.h>
#include <linux/pinctrl/consumer.h>
#include "blackbox.h"
#include "mdp.h"

#define ICE40_FIRMWARE_IMAGE_ROOTFS       "ice40_rev_20_00.bin"
#define ICE40_FIRMWARE_IMAGE_IN_JFFS      "fpga.bin"

#define ICE40_MSG_PRINT(LEVEL, FORMAT, ARG...) bb_log(BB_MOD_FPGA, LEVEL, FORMAT, ## ARG)
#define ICE40_MSG_INFO(FORMAT, ARG...) ICE40_MSG_PRINT(BB_LVL_INFO, FORMAT, ## ARG)
#define ICE40_MSG_ERROR(FORMAT, ARG...) ICE40_MSG_PRINT(BB_LVL_ERR, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)

#define ICE40_I2C_WRITE(REG, VAL)    i2c_smbus_write_byte_data(ice40_fpga_i2c_client, REG, VAL)

static struct gpio_desc *ice40_soft_reset_pin;
static struct gpio_desc *spi_mosi_pin;
static struct gpio_desc *spi_miso_pin;

struct i2c_client *ice40_fpga_i2c_client;

uint8_t ice40_fpga_reg_state[19] = {
	0x00,
	0x0f,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x00,
	0x30,
	0x30,
	0x08,
	0x30,
	0x00,
	0x00,
	0x40,
};

int ice40_fpga_release_reset(void)
{
	gpiod_set_value(ice40_soft_reset_pin, 0);
	return 0;
}
EXPORT_SYMBOL(ice40_fpga_release_reset);

int ice40_fpga_read_reg(uint8_t reg)
{
	if (reg > 0xf) {
		reg = ((reg >> 6) | 0x10) - 1;
	}
	return ice40_fpga_reg_state[reg];
}
EXPORT_SYMBOL(ice40_fpga_read_reg);

int ice40_fpga_write_reg(uint8_t reg, uint8_t val)
{
	int ret;
	ret = ICE40_I2C_WRITE(reg, val);
	if (ret < 0) {
		ICE40_MSG_ERROR("i2c write to iCE40 FPGA failed with error(%d)!", ret);
	} else {
		if (reg > 0xf) {
			reg = ((reg >> 6) | 0x10) - 1;
		}
		ice40_fpga_reg_state[reg] = val;
	}
	return ret;
}
EXPORT_SYMBOL(ice40_fpga_write_reg);

static ssize_t ice40_fpga_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	char buf[200];
	char *reg;
	char *val;
	unsigned long v,r;

	if (count > 199)
		return -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
		*strchr(buf, '=') = '\0';
		reg = buf;
		val = buf + strlen(reg) + 1;
		if (kstrtoul(val, 0, &v)) {
			ICE40_MSG_ERROR("Could not parse register value %s.", val);
			return -EIO;
		}
		if (kstrtoul(reg, 0, &r)) {
			ICE40_MSG_ERROR("Could not parse register number %s.", reg);
		} else {
			ice40_fpga_write_reg(r, (char)v);
		}
	}
	return count;
}

static int ice40_fpga_proc_show(struct seq_file *m, void *v)
{
	u32 i;

	seq_printf(m, "FPGA rev %#02x\n", i2c_smbus_read_byte_data(ice40_fpga_i2c_client, 0x00));
	seq_printf(m, "NOTE: These values are based on the known state of the FPGA. They may not be accurate if the FPGA altered registers without us knowing.\n");
	for (i = 0; i < sizeof(ice40_fpga_reg_state); i++) {
		seq_printf(m, "%02x: %02x\n", (i > 0xf ? ((i + 1) << 6) & 0xff : i), ice40_fpga_reg_state[i]);
	}

	return 0;
}

static int ice40_fpga_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ice40_fpga_proc_show, PDE_DATA(inode));
}

static const struct file_operations ice40_fpga_proc_ops = {
	.owner = THIS_MODULE,
	.open = ice40_fpga_proc_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.write = ice40_fpga_proc_write,
	.release = single_release,
};

static int ice40_fpga_proc_init(void)
{
	struct proc_dir_entry *file = NULL;
	file = proc_create("driver/ice40_fpga", 0666, NULL, &ice40_fpga_proc_ops);
	if (!file) {
		ICE40_MSG_ERROR("Error creating proc file.");
		return -ENOMEM;
	}
	return 0;
}

static void ice40_fpga_proc_exit(void)
{
	remove_proc_entry("driver/ice40_fpga", NULL);
}

static int ice40_driver_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	ice40_fpga_i2c_client = client;

	ice40_fpga_proc_init();
	ICE40_MSG_INFO("iCE40 slave i2c device detected and driver probed");
	return 0;
}

static int ice40_driver_remove(struct i2c_client *client)
{
	ice40_fpga_proc_exit();
	ICE40_MSG_INFO("iCE40 slave i2c device driver removed");
	return 0;
}


static const struct i2c_device_id ice40_i2c_id[] = {
	{ "ice40-fpga", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, ice40_i2c_id);

static const struct of_device_id ice40_driver_ids[] = {
	{ .compatible = "Sonos,ice40-fpga-i2c"},
	{  }
};
MODULE_DEVICE_TABLE(of, ice40_driver_ids);

static struct i2c_driver ice40_i2c_driver_struct = {
	.driver = {
		.name		= "ice40-fpga",
		.owner		= THIS_MODULE,
		.of_match_table	= ice40_driver_ids,
	},
	.id_table	= ice40_i2c_id,
	.probe		= ice40_driver_probe,
	.remove		= ice40_driver_remove,
};


static int __init ice40_driver_init(void)
{
	int ret = 0;
	struct fpga_manager *manager;
	struct device_node *node;
	struct pinctrl *spi_pins;

	node = of_find_node_by_name(NULL, "ice40x");
	if (IS_ERR(node)) {
		ICE40_MSG_ERROR("Failed to find device node!");
		return PTR_ERR(node);
	}

	manager = of_fpga_mgr_get(node);
	if (IS_ERR(manager)) {
		ICE40_MSG_ERROR("Failed to attach to fpga-manager framework!");
		return PTR_ERR(manager);
	}

	ret = i2c_add_driver(&ice40_i2c_driver_struct);
	if (ret) {
		ICE40_MSG_ERROR("Failed to add the i2c driver!");
		return ret;
	}

	spi_pins = devm_pinctrl_get_select_default(&ice40_fpga_i2c_client->dev);
	if (IS_ERR(spi_pins)) {
		ICE40_MSG_ERROR("Failed to configure the SPI pinmux!");
		i2c_del_driver(&ice40_i2c_driver_struct);
		return ret;
	}

	if (!fpga_mgr_firmware_load(manager, 0, ICE40_FIRMWARE_IMAGE_IN_JFFS)) {
		ICE40_MSG_INFO("** Loaded Custom FPGA binary(%s) from JFFS! **", ICE40_FIRMWARE_IMAGE_IN_JFFS);
	} else {
		ret = fpga_mgr_firmware_load(manager, 0, ICE40_FIRMWARE_IMAGE_ROOTFS);
	}

	fpga_mgr_put(manager);
	if (ret) {
		ICE40_MSG_ERROR("Failed to load firmware image on to ice40!");
		i2c_del_driver(&ice40_i2c_driver_struct);
		return ret;
	} else {
		ICE40_MSG_INFO("Loaded Sonos iCE40 FPGA driver: FPGA programmed successfully!");
	}

	ice40_soft_reset_pin = devm_gpiod_get(&ice40_fpga_i2c_client->dev, "soft-reset", GPIOD_OUT_HIGH);
	if (IS_ERR(ice40_soft_reset_pin)) {
		ICE40_MSG_ERROR("Failed to get the 'soft-reset' gpio pin!");
		return -ENODEV;
	}

	spi_mosi_pin = devm_gpiod_get(&ice40_fpga_i2c_client->dev, "spi-mosi", GPIOD_IN);
	if (IS_ERR(spi_mosi_pin)) {
		ICE40_MSG_ERROR("Failed to get the 'spi-mosi' gpio pin!");
		i2c_del_driver(&ice40_i2c_driver_struct);
		return -ENODEV;
	}

	spi_miso_pin = devm_gpiod_get(&ice40_fpga_i2c_client->dev, "spi-miso", GPIOD_IN);
	if (IS_ERR(spi_miso_pin)) {
		ICE40_MSG_ERROR("Failed to get the 'spi-miso' gpio pin!");
		i2c_del_driver(&ice40_i2c_driver_struct);
		return -ENODEV;
	}

	return ret;
}


static void __exit ice40_driver_exit(void)
{
	i2c_del_driver(&ice40_i2c_driver_struct);
	ICE40_MSG_INFO("Unloaded Sonos iCE40 FPGA driver");
}

module_init(ice40_driver_init);
module_exit(ice40_driver_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Sonos iCE40 FPGA driver");
MODULE_LICENSE("GPL");
MODULE_FIRMWARE(ICE40_FIRMWARE_IMAGE_ROOTFS);
