/*
 * Copyright (c) 2019-2021, Sonos, Inc.  All rights reserved.
 *
 * SPDX-License-Identifier: GPL-2.0
 *
 * Events generated by hardware use this driver to send
 * notifications to interested parties using Netlink.
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/skbuff.h>
#include <linux/workqueue.h>
#include <linux/timer.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,1,0)
#include <linux/export.h>
#endif
#include <net/genetlink.h> 
#include <linux/circ_buf.h>
#include <linux/time.h>
#include <linux/types.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/uaccess.h>
#include <linux/slab.h>
#include <linux/debugfs.h>
#include <linux/rwsem.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,1,0)
#include "blackbox.h"
#endif
#include "hwevent_queue_api.h"
#include "hwevent_queue_msg_api.h"
#include "hwevent_queue_private.h"
#include "sonos_device.h"

static int devno = -1;
struct cdev hwevtq_chr_dev;


struct workqueue_struct *hwevtq_work;
static spinlock_t hwevtq_producer_lock;
static spinlock_t hwevtq_consumer_lock;
struct circ_buf hwevtq_deferred_events_cb;
struct hwevtq_defer_event hwevtq_de[HWEVTQ_NUM_DEFERRED_EVENTS];

static struct dentry *debugfs_dir;

#define HWEVTQ_REG_MAGIC        0xC0FFEE
struct list_head hwevtq_registration_list = LIST_HEAD_INIT(hwevtq_registration_list);
DECLARE_RWSEM(hwevtq_registration_lock);

#ifdef DEBUG_HWEVTQ
struct timer_list hwevtq_timer;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,4,0)
static int wake_on_monaco_buttons = 0;
#endif

enum HWEVTQ_EventInfo hwevtq_last_info[HWEVTQSOURCE_NUM_SOURCES];

char *hwevtq_get_source_string(enum HWEVTQ_EventSource src);
char *hwevtq_get_event_string(enum HWEVTQ_EventInfo info);

char *hwevtq_get_source_string(enum HWEVTQ_EventSource src)
{
    switch (src) {
        case HWEVTQSOURCE_NO_SOURCE:           return "NO_SOURCE";
        case HWEVTQSOURCE_AMP:                 return "AMP";
        case HWEVTQSOURCE_LEDS:                return "LEDS";
        case HWEVTQSOURCE_IR:                  return "IR";
	case HWEVTQSOURCE_HEADPHONE:	       return "HEADPHONE";
        case HWEVTQSOURCE_SUBWOOFER:           return "SUBWOOFER";
        case HWEVTQSOURCE_ORIENTATION:         return "ORIENTATION";
        case HWEVTQSOURCE_BUTTON_PLAYPAUSE:    return "BUTTON_PLAYPAUSE";
        case HWEVTQSOURCE_BUTTON_VOL_UP:       return "BUTTON_VOL_UP";
        case HWEVTQSOURCE_BUTTON_VOL_DN:       return "BUTTON_VOL_DN";
        case HWEVTQSOURCE_BUTTON_JOIN:         return "BUTTON_JOIN";
        case HWEVTQSOURCE_BUTTON_MODE:         return "BUTTON_MODE";
	case HWEVTQSOURCE_BUTTON_POWER:        return "BUTTON_POWER";
        case HWEVTQSOURCE_BUTTON_MICMUTE:      return "BUTTON_MICMUTE";
        case HWEVTQSOURCE_CAPZONEA:            return "CAPZONEA";
        case HWEVTQSOURCE_CAPZONEB:            return "CAPZONEB";
        case HWEVTQSOURCE_CAPZONEC:            return "CAPZONEC";
        case HWEVTQSOURCE_CAPZONEM:            return "CAPZONEM";
        case HWEVTQSOURCE_CAPZONECAT:          return "CAPZONECAT";
        case HWEVTQSOURCE_LINEIN:              return "LINEIN";
        case HWEVTQSOURCE_DAC:                 return "DAC";
        case HWEVTQSOURCE_ADC:                 return "ADC";
        case HWEVTQSOURCE_CPU:                 return "CPU";
        case HWEVTQSOURCE_SOC:                 return "SOC";
        case HWEVTQSOURCE_SONGLE:              return "SONGLE";
        case HWEVTQSOURCE_POWER:               return "POWER";
        case HWEVTQSOURCE_AVTRIGGER:           return "AVTRIGGER";
        case HWEVTQSOURCE_LIGHT_SENSOR:        return "LIGHT_SENSOR";
        case HWEVTQSOURCE_LINEOUT:             return "LINEOUT";
        case HWEVTQSOURCE_SPDIF_OUT:           return "SPDIF_OUT";
	case HWEVTQSOURCE_HDMI:		       return "HDMI";
	case HWEVTQSOURCE_NFC:		       return "NFC";
        case HWEVTQSOURCE_MOTION_DETECTOR:     return "MOTION_DETECTOR";
        case HWEVTQSOURCE_BLE:                 return "BLE";
        case HWEVTQSOURCE_WIFI:                return "WIFI";
        case HWEVTQSOURCE_BATTERY:             return "BATTERY";
        case HWEVTQSOURCE_CHARGER:             return "CHARGER";
        case HWEVTQSOURCE_RTC:                 return "RTC";
        case HWEVTQSOURCE_FUEL_GAUGE:          return "FUEL_GAUGE";
        case HWEVTQSOURCE_NUM_SOURCES:         return "*";
    }
    return "Source unknown";
}
EXPORT_SYMBOL(hwevtq_get_source_string);

char *hwevtq_get_event_string(enum HWEVTQ_EventInfo info)
{
    switch (info) {
        case HWEVTQINFO_NO_EVENT:               return "NO_EVENT";
        case HWEVTQINFO_PRESSED:                return "PRESSED";
        case HWEVTQINFO_RELEASED:               return "RELEASED";
        case HWEVTQINFO_REPEATED:               return "REPEATED";
        case HWEVTQINFO_CONNECTED:              return "CONNECTED";
        case HWEVTQINFO_DISCONNECTED:           return "DISCONNECTED";
        case HWEVTQINFO_SWIPED:                 return "SWIPED";
        case HWEVTQINFO_ORIENT_HORIZONTAL:      return "HORIZONTAL";
        case HWEVTQINFO_ORIENT_TABLE:           return "TABLE";
        case HWEVTQINFO_ORIENT_VERTICAL:        return "VERTICAL";
        case HWEVTQINFO_ORIENT_VERTICAL_LEFT:   return "VERTICAL_LEFT";
        case HWEVTQINFO_ORIENT_VERTICAL_RIGHT:  return "VERTICAL_RIGHT";
        case HWEVTQINFO_ORIENT_WALL_ABOVE:      return "WALL_ABOVE";
        case HWEVTQINFO_ORIENT_WALL_BELOW:      return "WALL_BELOW";
	case HWEVTQINFO_ORIENT_HORIZONTAL_WALL:	return "HORIZONTAL_WALL";
	case HWEVTQINFO_ORIENT_VERTICAL_WALL:	return "VERTICAL_WALL";
	case HWEVTQINFO_ORIENT_VERTICAL_LEFT_WALL:	return "VERTICAL_LEFT_WALL";
	case HWEVTQINFO_ORIENT_VERTICAL_RIGHT_WALL:	return "VERTICAL_RIGHT_WALL";
	case HWEVTQINFO_QUEUE_OVERRUN:		return "QUEUE OVERRUN";
	case HWEVTQINFO_TIMEOUT:		return "TIMEOUT";
        case HWEVTQINFO_CLIP:                   return "CLIP";
        case HWEVTQINFO_HW_ERROR:               return "HW_ERROR";
        case HWEVTQINFO_HW_OK:                  return "HW_OK";
        case HWEVTQINFO_TEMP_WARNING:           return "TEMP_WARNING";
        case HWEVTQINFO_TEMP_FAULT:             return "TEMP_FAULT";
        case HWEVTQINFO_TEMP_OK:                return "TEMP_OK";
        case HWEVTQINFO_AMP_HI_RAIL:            return "AMP_HI_RAIL";
        case HWEVTQINFO_AMP_LO_RAIL:            return "AMP_LO_RAIL";
        case HWEVTQINFO_AMP_OFF:                return "AMP_OFF";
        case HWEVTQINFO_UPDATE:                 return "UPDATE";
        case HWEVTQINFO_MOTION_DETECTED:        return "MOTION_DETECTED";
        case HWEVTQINFO_MOTION_SETTLED:         return "MOTION_SETTLED";
	case HWEVTQINFO_HW_FAULT:		return "HW_FAULT";
        case HWEVTQINFO_WAKEUP:                 return "WAKEUP";
        case HWEVTQINFO_ON_BATTERY:             return "ON_BATTERY";
        case HWEVTQINFO_ON_CHARGER:             return "ON_CHARGER";
        case HWEVTQINFO_LOW_BATTERY:            return "LOW_BATTERY";
        case HWEVTQINFO_CRIT_BATTERY:           return "CRIT_BATTERY";
        case HWEVTQINFO_ALARM:                  return "ALARM";
	case HWEVTQINFO_INVALID_USB_CHARGER:	return "INVALID_CHARGER";
        case HWEVTQINFO_NUM_EVENTS:             return "*";
    }
    return "Event unknown";
}
EXPORT_SYMBOL(hwevtq_get_event_string);

static inline int hwevtq_match_src(struct hwevtq_reg *hweqr, enum HWEVTQ_EventSource src)
{
        if ((hweqr->source == HWEVTQSOURCE_NUM_SOURCES) || (hweqr->source == src)) {
                return 1;
        }
        return 0;
}

static inline int hwevtq_match_info(struct hwevtq_reg *hweqr, enum HWEVTQ_EventInfo info)
{
        if ((hweqr->info == HWEVTQINFO_NUM_EVENTS) || (hweqr->info == info)) {
                return 1;
        }
        return 0;
}

static inline int hwevtq_match_callback(struct hwevtq_reg *hweqr, hwevtq_cb cbfunc)
{
        return (hweqr->cbfunc == cbfunc);
}

static inline int hwevtq_match_param(struct hwevtq_reg *hweqr, void *param)
{
        return (hweqr->param == param);
}

struct hwevtq_reg *
hwevtq_register_event(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info, hwevtq_cb cb, void *param, char *name)
{
        struct hwevtq_reg *hweqr;
        int found = 0;

        down_read(&hwevtq_registration_lock);
        list_for_each_entry(hweqr, &hwevtq_registration_list, list)
        {
                if (hwevtq_match_src(hweqr, source) &&
                        hwevtq_match_info(hweqr, info) &&
                        hwevtq_match_callback(hweqr, cb) &&
                        hwevtq_match_param(hweqr, param)) {
                        found = 1;
                        break;
                }
        }
        up_read(&hwevtq_registration_lock);
        if (found) {
                goto event_register_exit;
        }

        hweqr = kmalloc(sizeof(*hweqr), GFP_KERNEL);
        if (hweqr == NULL) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,1,0)
                bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: unable to alloc EQR", __func__);
#endif
		return NULL;
        }
        hweqr->param = param;
        hweqr->cbfunc = cb;
        hweqr->source = source;
        hweqr->info = info;
        hweqr->magic = HWEVTQ_REG_MAGIC;
        strncpy(hweqr->name, name, HWEVTQ_REG_NAME_LEN);
        hweqr->name[HWEVTQ_REG_NAME_LEN-1] = '\0';
        down_write(&hwevtq_registration_lock);
        list_add(&hweqr->list, &hwevtq_registration_list);
        up_write(&hwevtq_registration_lock);
event_register_exit:
        return hweqr;
}
EXPORT_SYMBOL(hwevtq_register_event);

struct hwevtq_reg *
hwevtq_register_source(enum HWEVTQ_EventSource source, hwevtq_cb cb, void *param, char *name)
{
        return hwevtq_register_event(source, HWEVTQINFO_NUM_EVENTS, cb, param, name);
}
EXPORT_SYMBOL(hwevtq_register_source);

struct hwevtq_reg *
hwevtq_register_info(enum HWEVTQ_EventInfo info, hwevtq_cb cb, void *param, char *name)
{
	return hwevtq_register_event(HWEVTQSOURCE_NUM_SOURCES, info, cb, param, name);
}
EXPORT_SYMBOL(hwevtq_register_info);

struct hwevtq_reg *
hwevtq_register_all(hwevtq_cb cb, void *param, char *name)
{
        return hwevtq_register_event(HWEVTQSOURCE_NUM_SOURCES, HWEVTQINFO_NUM_EVENTS,
                                cb, param, name);
}
EXPORT_SYMBOL(hwevtq_register_all);

static void hwevtq_init_events(void)
{
        enum HWEVTQ_EventSource source;
        for (source = HWEVTQSOURCE_NO_SOURCE; source < HWEVTQSOURCE_NUM_SOURCES; source++) {
                hwevtq_last_info[source] = HWEVTQINFO_NO_EVENT;
        }
}

enum HWEVTQ_EventInfo hwevtq_get_event(enum HWEVTQ_EventSource source)
{
        if (source >= HWEVTQSOURCE_NUM_SOURCES) {
                return HWEVTQINFO_NO_EVENT;
        }
        return hwevtq_last_info[source];
}
EXPORT_SYMBOL(hwevtq_get_event);

int hwevtq_unregister_event(struct hwevtq_reg **peqr)
{
        int error = 0;
        struct hwevtq_reg *hweqr = *peqr;
        if (hweqr != NULL) {
                if (hweqr->magic == HWEVTQ_REG_MAGIC) {
                        list_del(&hweqr->list);
                        hweqr->magic = 0;
                        kfree(hweqr);
                        *peqr = NULL;
                } else {
                        error = -1;
                }
        }
        return error;
}
EXPORT_SYMBOL(hwevtq_unregister_event);

void hwevtq_process_regs(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
        struct hwevtq_reg *hweqr;

        down_read(&hwevtq_registration_lock);
        list_for_each_entry(hweqr, &hwevtq_registration_list, list)
        {
                if (hwevtq_match_src(hweqr, source) && hwevtq_match_info(hweqr, info)) {
                        hweqr->cbfunc(hweqr->param, source, info);
                        hweqr->calls++;
                }
        }
        up_read(&hwevtq_registration_lock);
}


int hwevtq_get_mcgrp_from_source(enum HWEVTQ_EventSource src, enum HWEVTQ_EventInfo info)
{
	int group = -1;

	static uint16_t mcast_grp_by_src[HWEVTQSOURCE_NUM_SOURCES + 1] = {
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_LED,
		HWEVTQ_MCAST_GRP_BUTTON,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_SENSOR,
		HWEVTQ_MCAST_GRP_BUTTON,
		HWEVTQ_MCAST_GRP_BUTTON,
		HWEVTQ_MCAST_GRP_BUTTON,
		HWEVTQ_MCAST_GRP_BUTTON,
		HWEVTQ_MCAST_GRP_BUTTON,
		HWEVTQ_MCAST_GRP_BUTTON_POWER,
		HWEVTQ_MCAST_GRP_BUTTON,
		HWEVTQ_MCAST_GRP_CAPZONE,
		HWEVTQ_MCAST_GRP_CAPZONE,
		HWEVTQ_MCAST_GRP_CAPZONE,
		HWEVTQ_MCAST_GRP_CAPZONE,
		HWEVTQ_MCAST_GRP_CAPZONE,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_TEMP,
		HWEVTQ_MCAST_GRP_TEMP,
		HWEVTQ_MCAST_GRP_HT,
		HWEVTQ_MCAST_GRP_TEMP,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_SENSOR,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_AUDIO,
		HWEVTQ_MCAST_GRP_BUTTON,
		HWEVTQ_MCAST_GRP_SENSOR,
		HWEVTQ_MCAST_GRP_WAKEUP,
		HWEVTQ_MCAST_GRP_WAKEUP,
		HWEVTQ_MCAST_GRP_BATTERY,
		HWEVTQ_MCAST_GRP_WAKEUP,
		HWEVTQ_MCAST_GRP_WAKEUP,
		HWEVTQ_MCAST_GRP_BATTERY,
		HWEVTQ_MCAST_GRP_AUDIO
	};

	if ((src == HWEVTQSOURCE_NO_SOURCE) || (src == HWEVTQSOURCE_NUM_SOURCES))  {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "Invalid Source for event.  None sent.");
#endif
	}

	if (src < HWEVTQSOURCE_NUM_SOURCES) {
		group = mcast_grp_by_src[src];
	}

	if ((src == HWEVTQSOURCE_AMP) &&
		((info == HWEVTQINFO_TEMP_FAULT) ||
		 (info == HWEVTQINFO_TEMP_WARNING) ||
		 (info == HWEVTQINFO_TEMP_OK)))  {
			group = HWEVTQ_MCAST_GRP_TEMP;
	}
	if ((info == HWEVTQINFO_WAKEUP) &&
		((src == HWEVTQSOURCE_BATTERY) ||
		 (src == HWEVTQSOURCE_BUTTON_POWER))) {
			group = HWEVTQ_MCAST_GRP_WAKEUP;
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,4,0)
	if (wake_on_monaco_buttons) {
		if ((info == HWEVTQINFO_WAKEUP) &&
			((src == HWEVTQSOURCE_BUTTON_VOL_UP) ||
			(src == HWEVTQSOURCE_BUTTON_VOL_DN) ||
			(src == HWEVTQSOURCE_BUTTON_PLAYPAUSE) ||
			(src == HWEVTQSOURCE_BUTTON_MICMUTE))) {
			group = HWEVTQ_MCAST_GRP_WAKEUP;
		}
        }
#endif
	return group;
}

static int hwevtq_dbgfs_show(struct seq_file *m, void *p);

static int hwevtq_dbgfs_open(struct inode *inodep, struct file *filp)
{
        return single_open(filp, hwevtq_dbgfs_show, inodep->i_private);
}

static const struct file_operations hwevtq_dbgfs_operations = {
        .owner  = THIS_MODULE,
        .open   = hwevtq_dbgfs_open,
        .write  = NULL,
        .read   = seq_read,
        .llseek = seq_lseek,
        .release = seq_release,
};


void hwevtq_init_debugfs(void)
{
        struct dentry *dbg_file;

        debugfs_dir = debugfs_create_dir("hwevtq", NULL);
        if (IS_ERR_OR_NULL(debugfs_dir)) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
                bb_log(BB_MOD_EVENT, BB_LVL_ERR,
                                "failed to create debugfs dir (%ld)", PTR_ERR(debugfs_dir));
#endif
		debugfs_dir = NULL;
		return;
        }

        dbg_file = debugfs_create_file("hwevtq_registration", S_IRUGO,
                                debugfs_dir, NULL, &hwevtq_dbgfs_operations);
        if (dbg_file == NULL) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
                bb_log(BB_MOD_EVENT, BB_LVL_ERR,
                                        "failed to create debugfs file hwevtq_registration.");
#endif
	}
}

static int hwevtq_dbgfs_show(struct seq_file *m, void *p)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
        struct hwevtq_reg *hweqr;

        down_read(&hwevtq_registration_lock);
        if (list_empty(&hwevtq_registration_list)) {
                seq_printf(m, "\nNo Event Callback registrations exist\n\n");
        } else {
                seq_printf(m, "\nHardware Event Queue Callback Registrations\n\n");
                seq_printf(m, "%15s%22s%9s %9s %s\n", "Source", "Event", "Param", "Callback", "Name");
                list_for_each_entry(hweqr, &hwevtq_registration_list, list)
                {
                        seq_printf(m, "%15s", hwevtq_get_source_string(hweqr->source));
                        seq_printf(m, "%22s", hwevtq_get_event_string(hweqr->info));
                        seq_printf(m, "%9p", hweqr->param);
                        seq_printf(m, " %9p", hweqr->cbfunc);
                        seq_printf(m, " %s\n", hweqr->name);
                }
                seq_printf(m, "\n");
        }
        up_read(&hwevtq_registration_lock);
#endif
        return 0;
}

static int hwevtq_open(struct inode *inode, struct file *filp)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
	struct hwevtctl *evtqctl = container_of(inode->i_cdev, struct hwevtctl, c_dev);

	filp->private_data = evtqctl;
        bb_log(BB_MOD_EVENT, BB_LVL_INFO,"%s:", __func__);
	bb_log_dbg(BB_MOD_EVENT, "ptr=%p", evtqctl);
#endif
	return 0;
}

static int hwevtq_release(struct inode *inode, struct file *filp)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
	bb_log(BB_MOD_EVENT, BB_LVL_INFO, "%s:", __func__);
	bb_log_dbg(BB_MOD_EVENT, "");
#endif
	return 0;
}

const struct file_operations hwevtq_fops = {
	.open = hwevtq_open,
	.release = hwevtq_release,
};

int hwevtq_send_event(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
        int		ret = 0;
        struct timeval	timestamp;
	struct sk_buff	*skb;
	void		*msg_head;
        u64		time_secs=0;
        u64		time_usecs=0;
	int		flags = GFP_KERNEL;
	int		group = hwevtq_get_mcgrp_from_source(source, info);
        ktime_t		now;

	if ( group < 0) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR,
			"Multicast Group not defined. Event not sent.");
#endif
		return 0;
	}
	hwevtq_last_info[source] = info;

	hwevtq_process_regs(source, info);
	skb = genlmsg_new(NLMSG_DEFAULT_SIZE, flags);
        if (!skb) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR,
			"Error allocating event message buffer. Event not sent.");
#endif
                return -ENOMEM;
        }
        msg_head = genlmsg_put(skb, 0, 0, &hwevtq_genl_family, flags, HWEVTQ_CMD_EVENT);
        if (!msg_head) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "Message header not correct.  Event not sent.");
#endif
		goto msg_free;
        }

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
        now = ktime_get_boottime();
#else
	now = ktime_get();
#endif
	timestamp = ktime_to_timeval(now);
        time_secs = timestamp.tv_sec;
        time_usecs = timestamp.tv_usec;

#if LINUX_VERSION_CODE > KERNEL_VERSION(4,7,0)
	if (nla_put_u64_64bit(skb, HWEVTQ_ATTR_TIMESTAMP_SECS, time_secs, 0) ||
                nla_put_u64_64bit(skb, HWEVTQ_ATTR_TIMESTAMP_USECS, time_usecs, 0) ||
                nla_put_u16(skb, HWEVTQ_ATTR_EVTSRC, source) ||
                nla_put_u16(skb, HWEVTQ_ATTR_EVTINFO, info ) ||
                nla_put_u32(skb, HWEVTQ_ATTR_PRIVATE, group)) {
                goto msg_free;
        }
#else
        if (nla_put_u64(skb, HWEVTQ_ATTR_TIMESTAMP_SECS, time_secs) ||
		nla_put_u64(skb, HWEVTQ_ATTR_TIMESTAMP_USECS, time_usecs) ||
		nla_put_u16(skb, HWEVTQ_ATTR_EVTSRC, source) ||
                nla_put_u16(skb, HWEVTQ_ATTR_EVTINFO, info ) ||
		nla_put_u32(skb, HWEVTQ_ATTR_PRIVATE, group)) {
		goto msg_free;
	}
#endif
	genlmsg_end(skb, msg_head);
#if LINUX_VERSION_CODE > KERNEL_VERSION(4,4,0)
	ret = genlmsg_multicast(&hwevtq_genl_family, skb, 0, group, flags);
#else
	ret = genlmsg_multicast(skb, 0, hwevtq_mcgrps[group].id, flags);
#endif
	if (ret != 0) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_INFO,
			"Multicast event not sent. ret = %d, %s, %s, group = %d",
			ret, hwevtq_get_event_string(info),
			hwevtq_get_source_string(source), group);
#else
		printk("Multicast event not sent.  ret = %d, %s, %s, group = %d\n",
				ret, hwevtq_get_event_string(info),
				hwevtq_get_source_string(source), group);
#endif
		return 0;
	}
        if (info != HWEVTQINFO_UPDATE) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_INFO, "HWEVTQ: Sent event %s from %s.",
                        hwevtq_get_event_string(info),
                        hwevtq_get_source_string(source));
#else
		printk("HWEVTQ: Sent event %s from %s\n", hwevtq_get_event_string(info),
				hwevtq_get_source_string(source));
#endif
        } else {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
                bb_log_dbg(BB_MOD_EVENT, "HWEVTQ: Sent event %s from %s.",
                        hwevtq_get_event_string(info),
                        hwevtq_get_source_string(source));
#else
		printk("HWEVTQ: Sent event %s from %s\n", hwevtq_get_event_string(info),
                                hwevtq_get_source_string(source));
#endif
	}
	return 0;
msg_free:
	nlmsg_free(skb);
	return 0;
}
EXPORT_SYMBOL(hwevtq_send_event);

void hwevtq_worker(struct work_struct *d)
{
        struct hwevtq_defer_event *hwevtqwork = container_of(d, struct hwevtq_defer_event, worker);
        struct hwevtq_defer_event event_work;
	unsigned long head, tail;
	int event_found = 1;

        spin_lock(&hwevtq_consumer_lock);
	head = hwevtq_deferred_events_cb.head;
	tail = ACCESS_ONCE(hwevtq_deferred_events_cb.tail);
	if (CIRC_CNT(head, tail, HWEVTQ_NUM_DEFERRED_EVENTS) >= 1) {
		smp_read_barrier_depends();
		event_work = *hwevtqwork;
		smp_mb();
		hwevtq_deferred_events_cb.tail = (tail + 1) & (HWEVTQ_NUM_DEFERRED_EVENTS - 1);
	} else {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: no deferred event queued.", __func__);
#else
		printk("HWEVTQ: No deferred event queued.\n");
#endif
		event_found = 0;
	}
	spin_unlock(&hwevtq_consumer_lock);

	if (event_found) {
		hwevtq_send_event(event_work.source, event_work.info);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_DEBUG,
				"hwevtq_work: deferred work eventinfo = %s, eventsource = %s",
				hwevtq_get_event_string(event_work.info),
				hwevtq_get_source_string(event_work.source));
#endif
	}
        return;
}

int hwevtq_send_event_defer(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
        int ret = 0;
	int success;
        unsigned long   lock_flags;
	unsigned long head, tail;

        spin_lock_irqsave(&hwevtq_producer_lock, lock_flags);

	head = hwevtq_deferred_events_cb.head;
	tail = ACCESS_ONCE(hwevtq_deferred_events_cb.tail);
	if (CIRC_SPACE(head, tail, HWEVTQ_NUM_DEFERRED_EVENTS) >= 1) {
		struct hwevtq_defer_event *de = &hwevtq_de[head];
		 INIT_WORK(&de->worker, hwevtq_worker);
		 de->source = source;
		 de->info = info;
		 smp_wmb();
		 hwevtq_deferred_events_cb.head = (head + 1) & (HWEVTQ_NUM_DEFERRED_EVENTS -1);
		 success = queue_work(hwevtq_work, &de->worker);
		 if (!success)  {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
			bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: queue_work not successful", __func__);
#else
			printk("HWEVTQ: Queue_work not successful for deferred event.\n");
#endif
			ret = -EIO;
		}
	} else {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: no space in circ buf for event", __func__);
#else
		printk("HWEVTQ: No space in circ buf for deferred event\n");
#endif
		ret = -EIO;
	}

        if (info != HWEVTQINFO_UPDATE) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_DEBUG, "hwevtq_defer: will send event %s from %s.",
                        hwevtq_get_event_string(info),
                        hwevtq_get_source_string(source));
#endif
        } else {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
                bb_log_dbg(BB_MOD_EVENT, "hwevtq_defer: will send event %s from %s.",
                        hwevtq_get_event_string(info),
                        hwevtq_get_source_string(source));
#else
		printk("Multicast event deferred. event = %s, %s\n",
                                hwevtq_get_event_string(info),
                                hwevtq_get_source_string(source));
#endif
        }

	spin_unlock_irqrestore(&hwevtq_producer_lock, lock_flags);
        return ret;
}
EXPORT_SYMBOL(hwevtq_send_event_defer);

#ifdef DEBUG_HWEVTQ
static void greet_group(unsigned int group)
{
	void		*hdr;
	int		result;
	int		flags = GFP_KERNEL;
	char		msg[HWEVTQ_TEST_MSG_MAX_SIZE];
	struct sk_buff	*skb = genlmsg_new(NLMSG_DEFAULT_SIZE, flags);

	if (!skb) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR,
			"Multicast message allocation failed");
#endif
		return;
	}

	hdr = genlmsg_put(skb, 0, 0, &hwevtq_genl_family, flags, HWEVTQ_CMD_EVENT);
	if (!hdr) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR,
			"Error loading header for netlink multicast message.");
#endif
			goto msg_fail;
	}

	snprintf(msg, HWEVTQ_TEST_MSG_MAX_SIZE, "Hello group %s\n",hwevtq_mcgrp_names[group]);

	result = nla_put_string(skb, HWEVTQ_ATTR_MSG, msg);
	if (result) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR,
			"Error adding message string to multicast.");
#endif
			goto msg_fail;
	}

	genlmsg_end(skb, hdr);
	result = genlmsg_multicast(&hwevtq_genl_family, skb, 0, group, flags);
	if (result != 0) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "Multicast from kernel failed.");
#endif
	}
	return;

msg_fail:
	nlmsg_free(skb);
	return;
}

static void hwevtq_mctest_callback(unsigned long data)
{

	greet_group(HWEVTQ_MCAST_GRP0);
	greet_group(HWEVTQ_MCAST_GRP1);
	greet_group(HWEVTQ_MCAST_GRP2);

	mod_timer(&hwevtq_timer, jiffies + msecs_to_jiffies(HWEVTQ_TEST_HELLO_INTERVAL));

	return;
}
#endif

int hwevtq_rx_msg(struct sk_buff *skb, struct genl_info *info)
{
	if (!info->attrs[HWEVTQ_ATTR_MSG]) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
                printk("Empty message from %d...\n", info->snd_portid);
#else
		printk("Empty message from %d...\n", info->snd_pid);
#endif
		printk("%p\n", info->attrs[HWEVTQ_ATTR_MSG]);
                return -EINVAL;
        }
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
	printk("%u says %s\n", info->snd_portid,
		(char *)nla_data(info->attrs[HWEVTQ_ATTR_MSG]));
#else
	printk("%u says %s\n", info->snd_pid,
                (char *)nla_data(info->attrs[HWEVTQ_ATTR_MSG]));
#endif
        return 0;
}

static int __init hwevtq_init(void)
{
        int             err = 0;
        struct device   *class_dev = NULL;
#if LINUX_VERSION_CODE < KERNEL_VERSION(4,4,0)
	int	i;
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,4,0)
	struct device_node *np;
#endif

	devno = MKDEV(HWEVTQ_DEVICE_MAJOR, 0);

	hwevtq_work = create_singlethread_workqueue("hwevtq");
	if (hwevtq_work == NULL) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "Deferred hwevtq could not be created.");
#else
		printk("HWEVTQ: Deferred hwevtq could not be created.\n");
#endif
		return -EIO;
	}
	hwevtq_deferred_events_cb.head = 0;
	hwevtq_deferred_events_cb.tail = 0;

        hwevtq_init_events();

#ifdef CONFIG_DEVTMPFS
        err = alloc_chrdev_region(&devno, 0, 1, HWEVTQ_DEVICE_NAME);
#else
	err = register_chrdev_region(devno, 1, HWEVTQ_DEVICE_NAME);
#endif
        if (err < 0) {
		printk("HWEVTQ: Error - failed to allocate driver major number. (%d)\n", err);
		destroy_workqueue(hwevtq_work);
                return(err);
        } else {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
		bb_log(BB_MOD_EVENT, BB_LVL_INFO, "Alloc'd driver major number %d", MAJOR(devno));
#else
		printk("HWEVTQ: Alloc'd driver major number %d\n", HWEVTQ_DEVICE_MAJOR);
#endif
        }

        cdev_init(&hwevtq_chr_dev, &hwevtq_fops);
        hwevtq_chr_dev.owner = THIS_MODULE;

        err = cdev_add(&hwevtq_chr_dev, devno, 1);
        if (err) {
		printk("HWEVTQ: Error - failed to setup char device node %d:0 (%d)\n",
				HWEVTQ_DEVICE_MAJOR, err);
                unregister_chrdev_region(devno, 1);
		destroy_workqueue(hwevtq_work);
                return(err);
        }

        class_dev = sonos_device_create(NULL, devno, NULL, HWEVTQ_DEVICE_NAME);
        if (IS_ERR(class_dev)) {
		printk("HWEVTQ: Error - failed to create sonos device\n");
                cdev_del(&hwevtq_chr_dev);
                unregister_chrdev_region(devno, 1);
		destroy_workqueue(hwevtq_work);
                return PTR_ERR(class_dev);
	}

#if LINUX_VERSION_CODE > KERNEL_VERSION(4,4,0)
        err = genl_register_family(&hwevtq_genl_family);
#else
	err = genl_register_family(&hwevtq_genl_family);
	err |= genl_register_ops(&hwevtq_genl_family, &hwevtq_ops);
	for ( i= 0; i < HWEVTQ_MCAST_GRP_MAX; i++ ) {
		err |= genl_register_mc_group(&hwevtq_genl_family, &hwevtq_mcgrps[i]);
	}
#endif
        if (err) {
		printk("HWEVTQ: Error - failure to register family message\n");
		sonos_device_destroy(devno);
		cdev_del(&hwevtq_chr_dev);
		unregister_chrdev_region(devno, 1);
                destroy_workqueue(hwevtq_work);
                return -EINVAL;
        }

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,4,0)
	np = of_find_node_by_name(NULL, "wake-events");
	if (np) {
		if (of_property_read_bool(np, "monaco-buttons")) {
			bb_log(BB_MOD_SENSORS, BB_LVL_INFO, "monaco button wake events");
			wake_on_monaco_buttons = 1;
		}
	}
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
	hwevtq_init_debugfs();
#endif

#ifdef DEBUG_HWEVTQ
        setup_timer(&hwevtq_timer, hwevtq_mctest_callback, 0);
        mod_timer(&hwevtq_timer, jiffies + msecs_to_jiffies(HWEVTQ_TEST_HELLO_INTERVAL));
#endif
	spin_lock_init(&hwevtq_producer_lock);
	spin_lock_init(&hwevtq_consumer_lock);

	printk("Registered hwevent_queue driver\n");
        return err;
}
module_init(hwevtq_init);

static void __exit hwevtq_exit(void)
{
#ifdef DEBUG_HWEVTQ
	del_timer(&hwevtq_timer);
#endif
	debugfs_remove_recursive(debugfs_dir);
	genl_unregister_family(&hwevtq_genl_family);
        sonos_device_destroy(devno);
        cdev_del(&hwevtq_chr_dev);
        unregister_chrdev_region(devno, 1);
	destroy_workqueue(hwevtq_work);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0)
	bb_log(BB_MOD_EVENT, BB_LVL_INFO, "%s: unregistering platform driver", __func__);
#else
	printk("HWEVTQ: Info - %s, unregistering platform driver\n", __func__);
#endif
	return;
}
module_exit(hwevtq_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for Hardware Event Notifications");
MODULE_LICENSE("GPL");
