/*
 * Copyright (c) 2018-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * SONOS HDMI control module - Main file.
 * HDMI features without a standard Linux interface
 * are controlled here.
 *
 * This includes:
 *	+5v (pin 18) source control
 */

#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/module.h>
#include <linux/uaccess.h>

#include "sonos_device.h"
#ifndef SONOS_EXTERNAL_BUILD
#include "blackbox.h"
#else
#include "external_build_stubs.h"
#endif

#include "hdmictl.h"
#include "hdmictl_priv.h"

static int hdmictl_open(struct inode *inode, struct file *filp)
{
	struct hdmictl *hdmictl = container_of(inode->i_cdev, struct hdmictl, cdev);
	filp->private_data = hdmictl;
	bb_log_dbg(BB_MOD_HDMICTL, "ptr=%p", hdmictl);
	return 0;
}

static int hdmictl_release(struct inode *inode, struct file *filp)
{
	bb_log_dbg(BB_MOD_HDMICTL, "");
	return 0;
}

static long hdmictl_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct hdmictl *hdmictl = filp->private_data;
	long ret = 0;
	void __user *parg = (void __user *)arg;

	bb_log_dbg_dev(hdmictl->dev, BB_MOD_HDMICTL, "ptr=%p cmd=%u(%d) arg=%ld", hdmictl, cmd, _IOC_NR(cmd), arg);

	switch (cmd) {
	case HDMICTL_GET_VERSION:
		{
			__u32 ver = HDMICTL_VERSION;
			if (copy_to_user(parg, &ver, sizeof(__u32))) {
				ret = -EACCES;
			}
		}
		break;

	case HDMICTL_GET_5V_STATE:
		{
			__u8 val = gpio_get_5v_line(hdmictl);
			if (copy_to_user(parg, &val, sizeof(__u8))) {
				ret = -EACCES;
			}
		}
		break;

	case HDMICTL_SET_5V_STATE:
		{
			__u8 val;
			if (copy_from_user(&val, parg, sizeof(__u8)) == 0) {
				ret = gpio_set_5v_line(hdmictl, !!val);
			} else {
				ret = -EACCES;
			}
		}
		break;

	case HDMICTL_GET_FAULT_STATE:
		{
			__u8 val = gpio_get_fault_state(hdmictl);
			if (copy_to_user(parg, &val, sizeof(__u8))) {
				ret = -EACCES;
			}
		}
		break;

	default:
		ret = -EPERM;
		bb_log(BB_MOD_HDMICTL, BB_LVL_WARNING, "unrecognized ioctl %u (dir=%d type=%d nr=%d size=%d)",
			cmd, _IOC_DIR(cmd), _IOC_TYPE(cmd), _IOC_NR(cmd), _IOC_SIZE(cmd));
		break;
	}

	return ret;
}

const struct file_operations hdmictl_fops = {
	.open		= hdmictl_open,
	.unlocked_ioctl	= hdmictl_ioctl,
	.release	= hdmictl_release,
};

static int hdmictl_probe(struct platform_device *pdev)
{
	int err;
	struct hdmictl *hdmictl;
	struct device *class_dev = NULL;

	hdmictl = devm_kzalloc(&(pdev->dev), sizeof(struct hdmictl), GFP_KERNEL);
	if (!hdmictl) {
		bb_log_dev(&(pdev->dev), BB_MOD_HDMICTL, BB_LVL_ERR, "mem allocation failed");
		return -ENOMEM;
	}
	platform_set_drvdata(pdev, hdmictl);
	hdmictl->dev = &(pdev->dev);

	bb_log_dbg(BB_MOD_HDMICTL, "ptr=%p", hdmictl);

	hdmictl->devno = MKDEV(HDMICTL_MAJOR_NUMBER, 0);
#ifdef CONFIG_DEVTMPFS
	err = alloc_chrdev_region(&hdmictl->devno, 0, 1, HDMICTL_DEVICE_NAME);
#else
	err = register_chrdev_region(hdmictl->devno, 1, HDMICTL_DEVICE_NAME);
#endif
	if (err) {
		bb_log(BB_MOD_HDMICTL, BB_LVL_ERR, "failed to register character device region (%d)", err);
		return err;
	}

	cdev_init(&hdmictl->cdev, &hdmictl_fops);
	hdmictl->cdev.owner = THIS_MODULE;

	err = cdev_add(&hdmictl->cdev, hdmictl->devno, 1);
	if (err) {
		bb_log(BB_MOD_HDMICTL, BB_LVL_ERR, "failed add hdmictl chrdev %d:0 (%d)", hdmictl->devno, err);
		unregister_chrdev_region(hdmictl->devno, 1);
		return err;
	}

	class_dev = sonos_device_create(NULL, hdmictl->devno, NULL, HDMICTL_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		bb_log(BB_MOD_HDMICTL, BB_LVL_ERR, "failed creating hdmictl class (%ld)", PTR_ERR(class_dev));
		cdev_del(&hdmictl->cdev);
		unregister_chrdev_region(hdmictl->devno, 1);
		return PTR_ERR(class_dev);
	}

	hdmictl->debugfs_dir = debugfs_create_dir("hdmictl", NULL);
	if (IS_ERR_OR_NULL(hdmictl->debugfs_dir)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "failed to create debugfs dir (%ld)\n", PTR_ERR(hdmictl->debugfs_dir));
		hdmictl->debugfs_dir = NULL;
		return PTR_ERR(hdmictl->debugfs_dir);
	}

        if (gpio_init(hdmictl)) {
                bb_log(BB_MOD_HDMICTL, BB_LVL_ERR, "gpio_init failed ");
        }


	err = adc_init(hdmictl);
	if (err) {
                bb_log(BB_MOD_HDMICTL, BB_LVL_ERR, "adc_init failed");
	}

	bb_log(BB_MOD_HDMICTL, BB_LVL_INFO, "probe complete");

	return err;
}

static int hdmictl_remove(struct platform_device *pdev)
{
	struct hdmictl *hdmictl = platform_get_drvdata(pdev);

	adc_exit(hdmictl);
	gpio_exit(hdmictl);
	debugfs_remove_recursive(hdmictl->debugfs_dir);
	sonos_device_destroy(hdmictl->devno);
	cdev_del(&hdmictl->cdev);
	unregister_chrdev_region(hdmictl->devno, 1);
	return 0;
}

static int hdmictl_suspend(struct platform_device *pdev, pm_message_t state)
{
	return 0;
}

static int hdmictl_resume(struct platform_device *pdev)
{
	return 0;
}

static const struct of_device_id hdmictl_ids[] = {
	{ .compatible = "sonos,hdmi-ctrl", },
	{}
};

static struct platform_driver hdmictl_drv = {
	.probe = hdmictl_probe,
	.remove = hdmictl_remove,
	.suspend = hdmictl_suspend,
	.resume = hdmictl_resume,
	.driver = {
		.name = HDMICTL_DEVICE_NAME,
		.owner = THIS_MODULE,
		.of_match_table = hdmictl_ids,
	},
};

static int __init hdmictl_init(void)
{
	return platform_driver_register(&hdmictl_drv);
}
module_init(hdmictl_init);

static void __exit hdmictl_exit(void)
{
	platform_driver_unregister(&hdmictl_drv);
	bb_log(BB_MOD_HDMICTL, BB_LVL_INFO, "exit complete");
}
module_exit(hdmictl_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for special Sonos HDMI subsystems");
MODULE_LICENSE("GPL");
