/*
 * Copyright (c) 2018-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * SONOS HDMI control module - GPIO access
 */

#include <linux/of.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <linux/uaccess.h>
#include <linux/kernel.h>
#include <linux/kthread.h>
#include <linux/delay.h>

#ifndef SONOS_EXTERNAL_BUILD
#include "blackbox.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#else
#include "external_build_stubs.h"
#endif

#include "hdmictl.h"
#include "hdmictl_priv.h"

#define HDMI_FAULT_POLL_PERIOD_MSEC	(500)

#define HDMI_HPD_NAME	"HDMI HPD"
#define HDMI_FAULT_NAME	"HDMI Fault"

int gpio_set_5v_line(struct hdmictl *hdmictl, int val)
{
	bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_INFO, "5v=%d", !!val);
	gpio_set_value(hdmictl->gpio.power_ctrl_gpio, val);
	return 0;
}

int gpio_get_5v_line(struct hdmictl *hdmictl)
{
	int val;
	val = gpio_get_value(hdmictl->gpio.power_ctrl_gpio);
	bb_log_dbg_dev(hdmictl->dev, BB_MOD_HDMICTL, "val=%d", val);
	return val;
}

int gpio_set_songle(struct hdmictl *hdmictl, int val)
{
	int ret = 0;

	if (gpio_is_valid(hdmictl->gpio.arc_hpd_ctrl_gpio)) {
		spin_lock_irq(&hdmictl->gpio.hpd_lock);
		if (hdmictl->gpio.hpd_last_state == 0) {
			gpio_set_value(hdmictl->gpio.arc_hpd_ctrl_gpio, val);
		} else {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_WARNING, "hpd active, ignoring songle=%d", val);
			ret = -EBUSY;
		}

		spin_unlock_irq(&hdmictl->gpio.hpd_lock);
	}
	return ret;
}

int gpio_get_fault_state(struct hdmictl *hdmictl) {
	int gpio_val = gpio_get_value(hdmictl->gpio.fault_gpio);
	int simulated_fault_state = READ_ONCE(hdmictl->gpio.sim_fault_state);
	if (gpio_val >= 0) {
		return ((!gpio_val) | simulated_fault_state);
	} else {
		return gpio_val;
	}
}

static int debugfs_gpio_5v_write(void *data, u64 val)
{
	struct hdmictl *hdmictl = (struct hdmictl *)data;
	gpio_set_5v_line(hdmictl, val);
	return 0;
}

static int debugfs_gpio_5v_show(void *data, u64 *val)
{
	struct hdmictl *hdmictl = (struct hdmictl *)data;
	*val = gpio_get_5v_line(hdmictl);
	return 0;

}

DEFINE_SIMPLE_ATTRIBUTE(debugfs_gpio_5v_ops, debugfs_gpio_5v_show, debugfs_gpio_5v_write, "%llu\n");

static int debugfs_hdmi_fault_show(void *data, u64 *val)
{
	struct hdmictl *hdmictl = (struct hdmictl *)data;
	*val = gpio_get_fault_state(hdmictl);
	return 0;
}

static int debugfs_hdmi_fault_write(void *data, u64 val)
{
	struct hdmictl *hdmictl = (struct hdmictl *)data;
	int new_sim_fault_state = !!val;


	int prev_sim_fault_state = READ_ONCE(hdmictl->gpio.sim_fault_state);
	if (prev_sim_fault_state != new_sim_fault_state) {
		if (new_sim_fault_state) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_INFO, "user set simulated load fault!");
		} else {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_INFO, "user cleared simulated load fault!");
		}
	}

	WRITE_ONCE(hdmictl->gpio.sim_fault_state, new_sim_fault_state);
	return 0;
}

DEFINE_SIMPLE_ATTRIBUTE(debugfs_hdmi_fault_ops, debugfs_hdmi_fault_show, debugfs_hdmi_fault_write, "%llu\n");

static struct hdmictl *g_hdmictl = NULL;

void hdmictl_set_virtual_hpd(bool vhpd)
{
	struct hdmictl *hdmictl = g_hdmictl;

	if (gpio_is_valid(hdmictl->gpio.mipi_hpd_ctrl_gpio)) {
		int hpd_state = gpio_get_value(hdmictl->gpio.hpd_gpio);

		if (hpd_state == 1) {
			gpio_set_value(hdmictl->gpio.mipi_hpd_ctrl_gpio, !!vhpd);
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_INFO, "virtual hpd=%d", !!vhpd);
		} else {
			if (hpd_state == 0) {
				bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_DEBUG, "virtual hpd=%d while hpd=%d", !!vhpd, hpd_state);
			} else {
				bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "virtual hpd gpio read error (%d)", hpd_state);
			}
		}
	}
}
EXPORT_SYMBOL(hdmictl_set_virtual_hpd);

static irqreturn_t hpd_gpio_isr(int irq, void *data)
{
	int hpd_state;
	int ret;
	struct hdmictl *hdmictl = (struct hdmictl*)data;
	(void)irq;

	spin_lock(&hdmictl->gpio.hpd_lock);

	hpd_state = gpio_get_value(hdmictl->gpio.hpd_gpio);
	if (hpd_state >= 0) {
		if (hdmictl->gpio.hpd_last_state != hpd_state) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_INFO, "hpd=%d", hpd_state);

			if (gpio_is_valid(hdmictl->gpio.arc_hpd_ctrl_gpio)) {
				gpio_set_value(hdmictl->gpio.arc_hpd_ctrl_gpio, hpd_state);
			}
			if (gpio_is_valid(hdmictl->gpio.mipi_hpd_ctrl_gpio)) {
				gpio_set_value(hdmictl->gpio.mipi_hpd_ctrl_gpio, hpd_state);
			}
			ret = irq_set_irq_type(
				hdmictl->gpio.hpd_gpio_irq_num,
				hpd_state ? IRQF_TRIGGER_LOW : IRQF_TRIGGER_HIGH);
			if (ret) {
				bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "hpd irq type set error (%d)", ret);
			}
		}

		hdmictl->gpio.hpd_last_state = hpd_state;
	} else {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "hpd gpio read error (%d)", hpd_state);
	}

	spin_unlock(&hdmictl->gpio.hpd_lock);

	return IRQ_HANDLED;
}


#ifndef SONOS_EXTERNAL_BUILD
extern
#endif
enum HWEVTQ_EventInfo sensors_hdmi_fault_state;

static int hdmi_fault_poll_thread_fn(void *data)
{
	struct hdmictl *hdmictl = (struct hdmictl *)data;
        int fault_last_state = 0;
	while (!kthread_should_stop()) {
		int fault_state = gpio_get_fault_state(hdmictl);
		if (fault_state >= 0) {
			if (fault_last_state != fault_state) {
				if (fault_state) {
					event_queue_send_event(HWEVTQSOURCE_HDMI, HWEVTQINFO_HW_FAULT);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
					hwevtq_send_event(HWEVTQSOURCE_HDMI, HWEVTQINFO_HW_FAULT);
#endif
					bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "load fault!");
					WRITE_ONCE(sensors_hdmi_fault_state, HWEVTQINFO_HW_FAULT);
				} else {
					bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_INFO, "load fault cleared");
					WRITE_ONCE(sensors_hdmi_fault_state, HWEVTQINFO_NO_EVENT);
				}
			}

			fault_last_state = fault_state;
		} else {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "fault state read error (%d)", fault_state);
		}

		msleep(HDMI_FAULT_POLL_PERIOD_MSEC);
	}
	return 0;
}

int gpio_init(struct hdmictl *hdmictl)
{
	struct dentry *debug_file;

	hdmictl->gpio.power_ctrl_gpio = of_get_named_gpio(hdmictl->dev->of_node, "power-ctrl-gpio", 0);
	if (!gpio_is_valid(hdmictl->gpio.power_ctrl_gpio)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not find power-ctrl-gpio (%d)", hdmictl->gpio.power_ctrl_gpio);
		return hdmictl->gpio.power_ctrl_gpio;
	}
	bb_log_dbg_dev(hdmictl->dev, BB_MOD_HDMICTL, "power-ctrl-gpio=%d", hdmictl->gpio.power_ctrl_gpio);

	if (devm_gpio_request_one(hdmictl->dev, hdmictl->gpio.power_ctrl_gpio, GPIOF_OUT_INIT_LOW, "HDMI Power ctrl")) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not get power-ctrl-gpio");
		return -EFAULT;
	}

	hdmictl->gpio.arc_hpd_ctrl_gpio = of_get_named_gpio(hdmictl->dev->of_node, "arc-hpd-ctrl-gpio", 0);
	if (gpio_is_valid(hdmictl->gpio.arc_hpd_ctrl_gpio)) {
		bb_log_dbg_dev(hdmictl->dev, BB_MOD_HDMICTL, "arc_hpd_ctrl_gpio=%d", hdmictl->gpio.arc_hpd_ctrl_gpio);

		if (devm_gpio_request_one(hdmictl->dev, hdmictl->gpio.arc_hpd_ctrl_gpio, GPIOF_OUT_INIT_LOW, "ARC HPD ctrl")) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not get arc-hpd-ctrl-gpio");
			return -EFAULT;
		}
	} else {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_DEBUG, "could not find arc-hpd-ctrl-gpio (%d); disabled", hdmictl->gpio.arc_hpd_ctrl_gpio);
	}

	hdmictl->gpio.mipi_hpd_ctrl_gpio = of_get_named_gpio(hdmictl->dev->of_node, "mipi-hpd-ctrl-gpio", 0);
	if (gpio_is_valid(hdmictl->gpio.mipi_hpd_ctrl_gpio)) {
		bb_log_dbg_dev(hdmictl->dev, BB_MOD_HDMICTL, "mipi_hpd_ctrl_gpio=%d", hdmictl->gpio.mipi_hpd_ctrl_gpio);

		if (devm_gpio_request_one(hdmictl->dev, hdmictl->gpio.mipi_hpd_ctrl_gpio, GPIOF_OUT_INIT_LOW, "MIPI HPD ctrl")) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not get mipi-hpd-ctrl-gpio");
			return -EFAULT;
		}
	} else {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_DEBUG, "could not find mipi-hpd-ctrl-gpio (%d); disabled", hdmictl->gpio.mipi_hpd_ctrl_gpio);
	}

	spin_lock_init(&hdmictl->gpio.hpd_lock);

	hdmictl->gpio.hpd_gpio = of_get_named_gpio(hdmictl->dev->of_node, "hpd-gpio", 0);
	if (gpio_is_valid(hdmictl->gpio.hpd_gpio)) {
		int ret;

		bb_log_dbg_dev(hdmictl->dev, BB_MOD_HDMICTL, "hpd_gpio=%d", hdmictl->gpio.hpd_gpio);

		if (devm_gpio_request_one(hdmictl->dev, hdmictl->gpio.hpd_gpio, GPIOF_IN, HDMI_HPD_NAME)) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not get hpd-gpio");
			return -EFAULT;
		}

		hdmictl->gpio.hpd_gpio_irq_num = gpio_to_irq(hdmictl->gpio.hpd_gpio);
		ret = devm_request_irq(hdmictl->dev, hdmictl->gpio.hpd_gpio_irq_num,
					hpd_gpio_isr,
					IRQF_TRIGGER_HIGH,
					HDMI_HPD_NAME, hdmictl);
		if (ret) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "hpd irq request failed (err=%d)", ret);
			return -EFAULT;
		}
	} else {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_DEBUG, 
			   "could not find hpd-gpio (%d); disabled", hdmictl->gpio.hpd_gpio);
	}

	hdmictl->gpio.fault_gpio = of_get_named_gpio(hdmictl->dev->of_node, "fault-gpio", 0);
	if (gpio_is_valid(hdmictl->gpio.fault_gpio)) {
		bb_log_dbg_dev(hdmictl->dev, BB_MOD_HDMICTL, "fault_gpio=%d", hdmictl->gpio.fault_gpio);

		if (devm_gpio_request_one(hdmictl->dev, hdmictl->gpio.fault_gpio, GPIOF_IN, HDMI_FAULT_NAME)) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "fault gpio not detected, power fault polling disabled");
		} else {
			hdmictl->gpio.fault_poll_thread = kthread_run(hdmi_fault_poll_thread_fn, hdmictl, "hdmi_fault_poll");
			if (IS_ERR(hdmictl->gpio.fault_poll_thread)) {
				bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, 
					   "could not start hdmi fault polling thread (%ld)", 
					   PTR_ERR(hdmictl->gpio.fault_poll_thread));
			}


		}
	} else {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_INFO, "fault-gpio not configured in kernel");
	}

	debug_file = debugfs_create_file("gpio_fault", S_IRUGO|S_IWUGO, hdmictl->debugfs_dir, 
					 hdmictl, &debugfs_hdmi_fault_ops);
	if (IS_ERR_OR_NULL(debug_file)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, 
			  "failed to create debugfs fault file (%ld)\n", PTR_ERR(debug_file));
	}


	debug_file = debugfs_create_file("gpio_5v", S_IRUGO|S_IWUGO, hdmictl->debugfs_dir, hdmictl, &debugfs_gpio_5v_ops);
	if (IS_ERR_OR_NULL(debug_file)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "failed to create debugfs 5v file (%ld)\n", PTR_ERR(debug_file));
	}
	g_hdmictl = hdmictl;

	return 0;
}

void gpio_exit(struct hdmictl *hdmictl)
{
	if (hdmictl->gpio.fault_poll_thread) {
		kthread_stop(hdmictl->gpio.fault_poll_thread);
	}
	disable_irq(hdmictl->gpio.hpd_gpio_irq_num);
	g_hdmictl = NULL;
}

