/*
 * Copyright (c) 2018-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * SONOS HDMI control module - ADC access for songle detection
 */

#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/delay.h>


#ifndef SONOS_EXTERNAL_BUILD
#include "blackbox.h"
#include "event_queue_api.h"
#else
#include "external_build_stubs.h"
#endif

#include "hdmictl_priv.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif

static u32 poll_adc_period_msec = 100;
static u32 poll_adc_debounce_count = 10;

#ifndef SONOS_EXTERNAL_BUILD
extern
#endif
enum HWEVTQ_EventInfo sensors_songle_state;

static u32 adc_get_value(struct hdmictl *hdmictl)
{
	int value = 0;
	int ignored = 0;
	int ret = 0;

	ret = hdmictl->adc.iio_dev->info->read_raw(hdmictl->adc.iio_dev,
						hdmictl->adc.channel,
						&value,
						&ignored,
						IIO_CHAN_INFO_PROCESSED);
	if (ret != IIO_VAL_INT) {
		value = hdmictl->adc.value;

		if (ret == -ETIMEDOUT) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_WARNING, "adc read_raw timeout");
		} else if (ret < 0) {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "adc read_raw error (%d)", ret);
		} else {
			bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "adc unsupported value type (%d)", ret);
		}
	}
	return (u32)value;
}

static int adc_poll_thread_fn(void *data)
{
	struct hdmictl *hdmictl = (struct hdmictl *)data;
	int counts = 0;

	WRITE_ONCE(sensors_songle_state, HWEVTQINFO_DISCONNECTED);
	hdmictl->adc.songle = 0;

	while (!kthread_should_stop()) {
		hdmictl->adc.value = adc_get_value(hdmictl);

		if ((hdmictl->adc.value > hdmictl->adc.thresh_low) && (hdmictl->adc.value < hdmictl->adc.thresh_high)) {
			if (!hdmictl->adc.songle) {
				counts++;
				if (counts >= poll_adc_debounce_count) {
					if (gpio_set_songle(hdmictl, 1) == 0) {
						bb_log(BB_MOD_HDMICTL, BB_LVL_INFO, "songle connected, value = %d", hdmictl->adc.value);
						event_queue_send_event(HWEVTQSOURCE_SONGLE, HWEVTQINFO_CONNECTED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
						hwevtq_send_event(HWEVTQSOURCE_SONGLE, HWEVTQINFO_CONNECTED);
#endif
						WRITE_ONCE(sensors_songle_state, HWEVTQINFO_CONNECTED);
						hdmictl->adc.songle = 1;
					}
					counts = 0;
				}
			}
		} else {
			if (hdmictl->adc.songle) {
				gpio_set_songle(hdmictl, 0);
				bb_log(BB_MOD_HDMICTL, BB_LVL_INFO, "songle disconnected, value = %d", hdmictl->adc.value);
				event_queue_send_event(HWEVTQSOURCE_SONGLE, HWEVTQINFO_DISCONNECTED);
				gpio_set_songle(hdmictl, 0);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(HWEVTQSOURCE_SONGLE, HWEVTQINFO_DISCONNECTED);
#endif
				gpio_set_songle(hdmictl, 0);
				WRITE_ONCE(sensors_songle_state, HWEVTQINFO_DISCONNECTED);
				hdmictl->adc.songle = 0;
			}
		}

		msleep(poll_adc_period_msec);
	}
	return 0;
}

void adc_exit(struct hdmictl *hdmictl)
{

	if (hdmictl->adc.poll_thread) {
		kthread_stop(hdmictl->adc.poll_thread);
	}

	if (hdmictl->adc.iio_dev) {
		iio_device_put(hdmictl->adc.iio_dev);
		hdmictl->adc.iio_dev = NULL;
	}
}

int adc_init(struct hdmictl *hdmictl)
{
	int adc_dev_num;
	int channel_idx;
	struct device *dev;
	struct dentry *debug_file;
	int ret;

	ret = of_property_read_u32(hdmictl->dev->of_node, "songle-adc-device", &adc_dev_num);
	if (ret) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not find dtb value songle-adc-device");
		return 1;
	}
	ret = of_property_read_u32(hdmictl->dev->of_node, "songle-adc-channel", &channel_idx);
	if (ret) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not find dtb value songle-adc-channel");
		return 1;
	}
	ret = of_property_read_u32(hdmictl->dev->of_node, "songle-adc-high", &(hdmictl->adc.thresh_high));
	if (ret) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not find dtb value songle-adc-high");
		return 1;
	}
	ret = of_property_read_u32(hdmictl->dev->of_node, "songle-adc-low", &(hdmictl->adc.thresh_low));
	if (ret) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not find dtb value songle-adc-low");
		return 1;
	}

	dev = subsys_find_device_by_id(&iio_bus_type, adc_dev_num, NULL);
	if (!dev) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "adc %d not found", adc_dev_num);
		goto failed_init;
	}

	hdmictl->adc.iio_dev = iio_device_get(dev_to_iio_dev(dev));
	if (!hdmictl->adc.iio_dev) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not get iio_dev");
		goto failed_init;
	}

	bb_log_dbg_dev(hdmictl->dev, BB_MOD_HDMICTL, "found adc %s (%s)", dev_name(dev), hdmictl->adc.iio_dev->name);

	if (channel_idx >= hdmictl->adc.iio_dev->num_channels) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "channel %d outside of range %d", channel_idx, hdmictl->adc.iio_dev->num_channels);
		goto failed_init;
	}
	if (hdmictl->adc.iio_dev->channels[channel_idx].type != IIO_VOLTAGE) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "channel %d not voltage type (%d)", channel_idx, hdmictl->adc.iio_dev->channels[channel_idx].type);
		goto failed_init;
	}
	hdmictl->adc.channel = &(hdmictl->adc.iio_dev->channels[channel_idx]);

	hdmictl->adc.poll_thread = kthread_run(adc_poll_thread_fn, hdmictl, "songle_poll");
	if (IS_ERR(hdmictl->adc.poll_thread)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "could not start polling thread (%ld)", PTR_ERR(hdmictl->adc.poll_thread));
		goto failed_init;
	}

	debug_file = debugfs_create_u32("adc_value", S_IRUGO, hdmictl->debugfs_dir, &(hdmictl->adc.value));
	if (IS_ERR_OR_NULL(debug_file)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "failed to create debugfs value file (%ld)\n", PTR_ERR(debug_file));
		goto failed_init;
	}

	debug_file = debugfs_create_u32("adc_songle", S_IRUGO, hdmictl->debugfs_dir, &(hdmictl->adc.songle));
	if (IS_ERR_OR_NULL(debug_file)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "failed to create debugfs songle file (%ld)\n", PTR_ERR(debug_file));
		goto failed_init;
	}

	debug_file = debugfs_create_u32("adc_thresh_high", S_IRUGO|S_IWUGO, hdmictl->debugfs_dir, &(hdmictl->adc.thresh_high));
	if (IS_ERR_OR_NULL(debug_file)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "failed to create debugfs thresh_high file (%ld)\n", PTR_ERR(debug_file));
		goto failed_init;
	}

	debug_file = debugfs_create_u32("adc_thresh_low", S_IRUGO|S_IWUGO, hdmictl->debugfs_dir, &(hdmictl->adc.thresh_low));
	if (IS_ERR_OR_NULL(debug_file)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "failed to create debugfs thresh_low file (%ld)\n", PTR_ERR(debug_file));
		goto failed_init;
	}

	debug_file = debugfs_create_u32("adc_period", S_IRUGO|S_IWUGO, hdmictl->debugfs_dir, &poll_adc_period_msec);
	if (IS_ERR_OR_NULL(debug_file)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "failed to create debugfs period file (%ld)\n", PTR_ERR(debug_file));
		goto failed_init;
	}

	debug_file = debugfs_create_u32("adc_debounce", S_IRUGO|S_IWUGO, hdmictl->debugfs_dir, &poll_adc_debounce_count);
	if (IS_ERR_OR_NULL(debug_file)) {
		bb_log_dev(hdmictl->dev, BB_MOD_HDMICTL, BB_LVL_ERR, "failed to create debugfs debounce file (%ld)\n", PTR_ERR(debug_file));
		goto failed_init;
	}

	return 0;
failed_init:
	adc_exit(hdmictl);
	return 1;
}

