/*
 * Copyright (c) 2016-2017, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_platform.h>
#include <linux/platform_device.h>
#include <linux/of_gpio.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/cdev.h>

#define DEVICE_NAME "fuse_ctrl"

static int fuse_ctrl_gpio = -1;
static int device_open = 0;

static int fuse_ctrl_open(struct inode *inode, struct file *file)
{
	if (device_open) {
		return -EBUSY;
	}

	device_open++;
	try_module_get(THIS_MODULE);
	if (gpio_is_valid(fuse_ctrl_gpio)) {
		gpio_set_value(fuse_ctrl_gpio, 0);
	}
	return 0;
}

static int fuse_ctrl_release(struct inode *inode, struct file *file)
{
	device_open--;
	module_put(THIS_MODULE);
	if (gpio_is_valid(fuse_ctrl_gpio)) {
		gpio_set_value(fuse_ctrl_gpio, 0);
	}
	return 0;
}

static ssize_t fuse_ctrl_read(struct file *filp, char *buffer,
		size_t length, loff_t * offset)
{
	int ret;

	if (*offset != 0) {
		return -EFAULT;
	}

	if (length != 1) {
		return -EFAULT;
	}

	if (gpio_is_valid(fuse_ctrl_gpio)) {
		char ch;
		ret = gpio_get_value(fuse_ctrl_gpio);
		ch = !!ret;
		if (__put_user(ch, buffer)) {
			return -EFAULT;
		}
	} else {
		return -EFAULT;
	}
	return 1;
}

static ssize_t fuse_ctrl_write(struct file *filp,
		const char *buff, size_t len, loff_t * off)
{
	char c;

	if (*off != 0) {
		return 0;
	}

	if (len != 1) {
		return 0;
	}

	if (__get_user(c, buff)) {
		return -EFAULT;
	}

	if (gpio_is_valid(fuse_ctrl_gpio)) {
		gpio_set_value(fuse_ctrl_gpio, c);
	} else {
		return -EFAULT;
	}
	return 1;
}

const static struct file_operations fuse_ctrl_fops = {
	.owner = THIS_MODULE,
	.read = fuse_ctrl_read,
	.write = fuse_ctrl_write,
	.open = fuse_ctrl_open,
	.release = fuse_ctrl_release
};

static struct miscdevice fuse_ctrl_dev = {
        MISC_DYNAMIC_MINOR,
        DEVICE_NAME,
        &fuse_ctrl_fops
};

static int fuse_ctrl_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	int ret;

	fuse_ctrl_gpio = of_get_named_gpio(np, "fuse-ctrl-gpio", 0);
	if (gpio_is_valid(fuse_ctrl_gpio)) {
		ret = devm_gpio_request_one(&pdev->dev, fuse_ctrl_gpio,
					    GPIOF_OUT_INIT_LOW, "Fuse control");
		if (ret) {
			dev_err(&pdev->dev, "unable to get fuse ctrl gpio\n");
			return ret;
		}
	}
	ret = misc_register(&fuse_ctrl_dev);
	if ( ret != 0 ) {
		goto out;
	}
	return 0;
out:
	if (gpio_is_valid(fuse_ctrl_gpio)) {
		devm_gpio_free(&pdev->dev, fuse_ctrl_gpio);
	}
	return ret;
}

static int fuse_ctrl_remove(struct platform_device *pdev)
{
	if (gpio_is_valid(fuse_ctrl_gpio)) {
		gpio_set_value(fuse_ctrl_gpio, 0);
		devm_gpio_free(&pdev->dev, fuse_ctrl_gpio);
	}

	misc_deregister(&fuse_ctrl_dev);
	return 0;
}

static const struct of_device_id fuse_ctrl_ids[] = {
        { .compatible = "Sonos,fuse_ctrl", },
        { },
};
MODULE_DEVICE_TABLE(of, fuse_ctrl_ids);

static struct platform_driver fuse_ctrl_driver = {
	.probe = fuse_ctrl_probe,
	.remove = fuse_ctrl_remove,
	.driver = {
		.name = "fuse_ctrl",
		.owner  = THIS_MODULE,
		.of_match_table = fuse_ctrl_ids,
	},
};
module_platform_driver(fuse_ctrl_driver);

MODULE_LICENSE("GPL v2");
MODULE_AUTHOR ("Sonos, Inc.");
MODULE_DESCRIPTION ("Fuse Control");
