/*
 * Copyright (c) 2016-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/kallsyms.h>
#include <asm/uaccess.h>
#include "blackbox.h"
#include "event_queue.h"

#define EVENT_REG_PROC_FILENAME "driver/event_queue_registrations"

#define case_enum_ret_str(e) \
	case e: \
	if (strchr(#e, '_') == NULL) return #e; \
	return (strchr(#e, '_')+1)

char * event_queue_get_source_string(enum HWEVTQ_EventSource source)
{
	switch(source) {
		case_enum_ret_str(HWEVTQSOURCE_NO_SOURCE);
		case_enum_ret_str(HWEVTQSOURCE_AMP);
		case_enum_ret_str(HWEVTQSOURCE_LEDS);
		case_enum_ret_str(HWEVTQSOURCE_IR);
		case_enum_ret_str(HWEVTQSOURCE_HEADPHONE);
		case_enum_ret_str(HWEVTQSOURCE_SUBWOOFER);
		case_enum_ret_str(HWEVTQSOURCE_ORIENTATION);
		case_enum_ret_str(HWEVTQSOURCE_BUTTON_PLAYPAUSE);
		case_enum_ret_str(HWEVTQSOURCE_BUTTON_VOL_UP);
		case_enum_ret_str(HWEVTQSOURCE_BUTTON_VOL_DN);
		case_enum_ret_str(HWEVTQSOURCE_BUTTON_JOIN);
		case_enum_ret_str(HWEVTQSOURCE_BUTTON_MODE);
		case_enum_ret_str(HWEVTQSOURCE_BUTTON_POWER);
		case_enum_ret_str(HWEVTQSOURCE_BUTTON_MICMUTE);
		case_enum_ret_str(HWEVTQSOURCE_CAPZONEA);
		case_enum_ret_str(HWEVTQSOURCE_CAPZONEB);
		case_enum_ret_str(HWEVTQSOURCE_CAPZONEC);
		case_enum_ret_str(HWEVTQSOURCE_CAPZONEM);
		case_enum_ret_str(HWEVTQSOURCE_CAPZONECAT);
		case_enum_ret_str(HWEVTQSOURCE_LINEIN);
		case_enum_ret_str(HWEVTQSOURCE_DAC);
		case_enum_ret_str(HWEVTQSOURCE_ADC);
		case_enum_ret_str(HWEVTQSOURCE_CPU);
		case_enum_ret_str(HWEVTQSOURCE_SOC);
		case_enum_ret_str(HWEVTQSOURCE_SONGLE);
		case_enum_ret_str(HWEVTQSOURCE_POWER);
		case_enum_ret_str(HWEVTQSOURCE_AVTRIGGER);
		case_enum_ret_str(HWEVTQSOURCE_LIGHT_SENSOR);
		case_enum_ret_str(HWEVTQSOURCE_LINEOUT);
		case_enum_ret_str(HWEVTQSOURCE_SPDIF_OUT);
                case_enum_ret_str(HWEVTQSOURCE_HDMI);
                case_enum_ret_str(HWEVTQSOURCE_NFC);
		case_enum_ret_str(HWEVTQSOURCE_MOTION_DETECTOR);
		case_enum_ret_str(HWEVTQSOURCE_BLE);
		case_enum_ret_str(HWEVTQSOURCE_WIFI);
		case_enum_ret_str(HWEVTQSOURCE_BATTERY);
                case_enum_ret_str(HWEVTQSOURCE_CHARGER);
                case_enum_ret_str(HWEVTQSOURCE_RTC);
                case_enum_ret_str(HWEVTQSOURCE_FUEL_GAUGE);
		case HWEVTQSOURCE_NUM_SOURCES:
			return "*";
	}
	return "source-unknown";
}
EXPORT_SYMBOL(event_queue_get_source_string);

char * event_queue_get_event_string(enum HWEVTQ_EventInfo event)
{
	switch (event) {
		case_enum_ret_str(HWEVTQINFO_NO_EVENT);
		case_enum_ret_str(HWEVTQINFO_PRESSED);
		case_enum_ret_str(HWEVTQINFO_RELEASED);
		case_enum_ret_str(HWEVTQINFO_REPEATED);
		case_enum_ret_str(HWEVTQINFO_CONNECTED);
		case_enum_ret_str(HWEVTQINFO_DISCONNECTED);
		case_enum_ret_str(HWEVTQINFO_SWIPED);
		case_enum_ret_str(HWEVTQINFO_ORIENT_HORIZONTAL);
		case_enum_ret_str(HWEVTQINFO_ORIENT_TABLE);
		case_enum_ret_str(HWEVTQINFO_ORIENT_VERTICAL);
		case_enum_ret_str(HWEVTQINFO_ORIENT_VERTICAL_LEFT);
		case_enum_ret_str(HWEVTQINFO_ORIENT_VERTICAL_RIGHT);
		case_enum_ret_str(HWEVTQINFO_ORIENT_WALL_ABOVE);
		case_enum_ret_str(HWEVTQINFO_ORIENT_WALL_BELOW);
		case_enum_ret_str(HWEVTQINFO_ORIENT_HORIZONTAL_WALL);
		case_enum_ret_str(HWEVTQINFO_ORIENT_VERTICAL_WALL);
		case_enum_ret_str(HWEVTQINFO_ORIENT_VERTICAL_LEFT_WALL);
		case_enum_ret_str(HWEVTQINFO_ORIENT_VERTICAL_RIGHT_WALL);
		case_enum_ret_str(HWEVTQINFO_QUEUE_OVERRUN);
		case_enum_ret_str(HWEVTQINFO_TIMEOUT);
		case_enum_ret_str(HWEVTQINFO_CLIP);
		case_enum_ret_str(HWEVTQINFO_HW_ERROR);
		case_enum_ret_str(HWEVTQINFO_HW_OK);
		case_enum_ret_str(HWEVTQINFO_TEMP_WARNING);
		case_enum_ret_str(HWEVTQINFO_TEMP_FAULT);
		case_enum_ret_str(HWEVTQINFO_TEMP_OK);
		case_enum_ret_str(HWEVTQINFO_AMP_HI_RAIL);
		case_enum_ret_str(HWEVTQINFO_AMP_LO_RAIL);
		case_enum_ret_str(HWEVTQINFO_AMP_OFF);
		case_enum_ret_str(HWEVTQINFO_UPDATE);
		case_enum_ret_str(HWEVTQINFO_MOTION_DETECTED);
		case_enum_ret_str(HWEVTQINFO_MOTION_SETTLED);
		case_enum_ret_str(HWEVTQINFO_HW_FAULT);
		case_enum_ret_str(HWEVTQINFO_WAKEUP);
		case_enum_ret_str(HWEVTQINFO_ON_BATTERY);
		case_enum_ret_str(HWEVTQINFO_ON_CHARGER);
		case_enum_ret_str(HWEVTQINFO_LOW_BATTERY);
		case_enum_ret_str(HWEVTQINFO_CRIT_BATTERY);
		case_enum_ret_str(HWEVTQINFO_ALARM);
		case_enum_ret_str(HWEVTQINFO_INVALID_USB_CHARGER);
		case HWEVTQINFO_NUM_EVENTS:
			return "*";
	}
	return "event-unknown";
}
EXPORT_SYMBOL(event_queue_get_event_string);

static int eqr_show(struct seq_file *m, void *v)
{
	struct event_reg *eqr;

	down_read(&event_queue_reg_lock);
	if (list_empty(&event_queue_reg_list)) {
		seq_printf(m, "\nno Event Callback registrations exist\n\n");
	} else {
		seq_printf(m, "\nEvent Callback registrations\n\n");
		seq_printf(m, "%15s%22s%9s %9s %s\n", "Source", "Event", "Param", "Callback", "Name");
		list_for_each_entry(eqr, &event_queue_reg_list, list)
		{
			seq_printf(m, "%15s", event_queue_get_source_string(eqr->source));
			seq_printf(m, "%22s", event_queue_get_event_string(eqr->info));
			seq_printf(m, "%9p", eqr->param);
			seq_printf(m, " %9p", eqr->cbfunc);
			seq_printf(m, " %s\n", eqr->name);
		}
		seq_printf(m, "\n");
	}
	up_read(&event_queue_reg_lock);
	return 0;
}

static int eqr_open(struct inode *inode, struct file *file)
{
	return single_open(file, eqr_show, PDE_DATA(inode));
}

static struct file_operations eqr_proc_operations = {
	.owner	= THIS_MODULE,
	.open	= eqr_open,
	.write	= NULL,
	.read	= seq_read,
	.llseek	= seq_lseek,
	.release = single_release,
};

int event_queue_init_proc(void)
{
	struct proc_dir_entry *entry;
	entry = proc_create_data(EVENT_REG_PROC_FILENAME, 0666, NULL, &eqr_proc_operations, NULL);
	if (entry == NULL) {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "proc init failed\n");
		return(-EIO);
	}
	return 0;
}

void event_queue_remove_proc(void)
{
	remove_proc_entry(EVENT_REG_PROC_FILENAME, NULL);
}
