/*
 * Copyright (c) 2016-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/hardirq.h>
#include <linux/rwsem.h>
#include "blackbox.h"
#include "event_queue.h"

#define EVENT_REG_MAGIC		0xdef1ade4

struct list_head event_queue_reg_list = LIST_HEAD_INIT(event_queue_reg_list);
DECLARE_RWSEM(event_queue_reg_lock);

static inline int eqr_match_source(struct event_reg *eqr, enum HWEVTQ_EventSource source)
{
	if ((eqr->source == HWEVTQSOURCE_NUM_SOURCES) || (eqr->source == source)) {
		return 1;
	}
	return 0;
}

static inline int eqr_match_info(struct event_reg *eqr, enum HWEVTQ_EventInfo info)
{
	if ((eqr->info == HWEVTQINFO_NUM_EVENTS) || (eqr->info == info)) {
		return 1;
	}
	return 0;
}

static inline int eqr_match_callback(struct event_reg *eqr, event_cbfunc cbfunc)
{
	return (eqr->cbfunc == cbfunc);
}

static inline int eqr_match_param(struct event_reg *eqr, void *param)
{
	return (eqr->param == param);
}

struct event_reg *
event_register_event(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info, event_cbfunc cb, void *param, char *name)
{
	struct event_reg *eqr;
	int found = 0;

	down_read(&event_queue_reg_lock);
	list_for_each_entry(eqr, &event_queue_reg_list, list)
	{
		if (eqr_match_source(eqr, source) &&
			eqr_match_info(eqr, info) &&
			eqr_match_callback(eqr, cb) &&
			eqr_match_param(eqr, param)) {
			found = 1;
			break;
		}
	}
	up_read(&event_queue_reg_lock);
	if (found) {
		goto event_register_exit;
	}

	eqr = kmalloc(sizeof(*eqr), GFP_KERNEL);
	if (eqr == NULL) {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: unable to alloc EQR", __func__);
		return NULL;
	}
	eqr->param = param;
	eqr->cbfunc = cb;
	eqr->source = source;
	eqr->info = info;
	eqr->magic = EVENT_REG_MAGIC;
	strncpy(eqr->name, name, EVENT_REG_NAME_LEN);
	eqr->name[EVENT_REG_NAME_LEN-1] = '\0';
	down_write(&event_queue_reg_lock);
	list_add(&eqr->list, &event_queue_reg_list);
	up_write(&event_queue_reg_lock);

event_register_exit:
	return eqr;
}
EXPORT_SYMBOL(event_register_event);

struct event_reg *
event_register_source(enum HWEVTQ_EventSource source, event_cbfunc cb, void *param, char *name)
{
	return event_register_event(source, HWEVTQINFO_NUM_EVENTS, cb, param, name);
}
EXPORT_SYMBOL(event_register_source);

struct event_reg *
event_register_info(enum HWEVTQ_EventInfo info, event_cbfunc cb, void *param, char *name)
{
	return event_register_event(HWEVTQSOURCE_NUM_SOURCES, info, cb, param, name);
}
EXPORT_SYMBOL(event_register_info);

struct event_reg *
event_register_all(event_cbfunc cb, void *param, char *name)
{
	return event_register_event(HWEVTQSOURCE_NUM_SOURCES, HWEVTQINFO_NUM_EVENTS, cb, param, name);
}
EXPORT_SYMBOL(event_register_all);

int event_unregister_event(struct event_reg **peqr)
{
	int error = 0;
	struct event_reg *eqr = *peqr;
	if (eqr != NULL) {
		if (eqr->magic == EVENT_REG_MAGIC) {
			list_del(&eqr->list);
			eqr->magic = 0;
			kfree(eqr);
			*peqr = NULL;
		} else {
			error = -1;
		}
	}
	return error;
}
EXPORT_SYMBOL(event_unregister_event);

void event_queue_process_regs(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
	struct event_reg *eqr;

	if (in_atomic()) {
		panic("event generated from illegal context, source %s event %s\n",
			  event_queue_get_source_string(source), event_queue_get_event_string(info));
	}

	down_read(&event_queue_reg_lock);
	list_for_each_entry(eqr, &event_queue_reg_list, list)
	{
		if (eqr_match_source(eqr, source) && eqr_match_info(eqr, info)) {
			eqr->cbfunc(eqr->param, source, info);
			eqr->calls++;
		}
	}
	up_read(&event_queue_reg_lock);
}
