/*
 * Copyright (c) 2016-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */

#include <generated/autoconf.h>
#include <linux/kernel.h>
#include <linux/export.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/barrier.h>
#include <linux/spinlock.h>
#include <linux/circ_buf.h>
#include <linux/watchdog.h>
#include <linux/semaphore.h>
#include <linux/proc_fs.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/hardirq.h>
#include "sonos_device.h"
#include "blackbox.h"
#include "button_event.h"
#include "event_queue.h"
#include "event_queue_hal.h"

static int devno = -1;

struct event_queue_deferred_event {
	struct work_struct worker;
	enum HWEVTQ_EventSource source;
	enum HWEVTQ_EventInfo info;
};

#define EVENT_QUEUE_NUM_DEFERRED_EVENTS  32
struct event_queue_deferred_event event_queue_deferred_events[EVENT_QUEUE_NUM_DEFERRED_EVENTS];
struct circ_buf event_queue_deferred_events_cb;
struct workqueue_struct	*event_queue_deferred_queue;
static spinlock_t event_queue_producer_lock;
static spinlock_t event_queue_consumer_lock;

static struct button_event_queue *event_queue_default = NULL;
static wait_queue_head_t event_queue_wait_queue;
static spinlock_t event_queue_lock;

struct file_operations event_queue_fops;

static int device_init_success = 0;
static int device_readers = 0;
static int device_writers = 0;

struct button_event_queue ** event_queue_get_default_queue(void)
{
	return &event_queue_default;
}
EXPORT_SYMBOL(event_queue_get_default_queue);

unsigned int event_queue_circular_buffer_size(void)
{
	return EVENT_QUEUE_NUM_DEFERRED_EVENTS;
}
EXPORT_SYMBOL(event_queue_circular_buffer_size);

static void event_queue_worker(struct work_struct *d)
{
	struct event_queue_deferred_event *decb = container_of(d, struct event_queue_deferred_event, worker);
	struct event_queue_deferred_event de;
	unsigned long head, tail;
	int event_present = 1;

	spin_lock(&event_queue_consumer_lock);
	head = event_queue_deferred_events_cb.head;
	tail = ACCESS_ONCE(event_queue_deferred_events_cb.tail);
	if (CIRC_CNT(head, tail, EVENT_QUEUE_NUM_DEFERRED_EVENTS) >= 1) {
		smp_read_barrier_depends();
		de = *decb;
		smp_mb();
		event_queue_deferred_events_cb.tail = (tail + 1) & (EVENT_QUEUE_NUM_DEFERRED_EVENTS - 1);
	} else {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: event missing", __func__);
		event_present = 0;
	}
	spin_unlock(&event_queue_consumer_lock);

	if (event_present) {
		event_queue_process_regs(de.source, de.info);
	}
}

enum HWEVTQ_EventInfo event_queue_last_info[HWEVTQSOURCE_NUM_SOURCES];

static void event_queue_init_events(void)
{
	enum HWEVTQ_EventSource source;
	for (source = HWEVTQSOURCE_NO_SOURCE; source < HWEVTQSOURCE_NUM_SOURCES; source++) {
		event_queue_last_info[source] = HWEVTQINFO_NO_EVENT;
	}
}

enum HWEVTQ_EventInfo event_queue_get_event(enum HWEVTQ_EventSource source)
{
	if (source >= HWEVTQSOURCE_NUM_SOURCES) {
		return HWEVTQINFO_NO_EVENT;
	}
	return event_queue_last_info[source];
}
EXPORT_SYMBOL(event_queue_get_event);

int event_queue_send_event(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
	event_queue_last_info[source] = info;
	event_queue_process_regs(source, info);
	return button_event_send(event_queue_default, source, info);
}
EXPORT_SYMBOL(event_queue_send_event);

int event_queue_send_event_defer(enum HWEVTQ_EventSource source, enum HWEVTQ_EventInfo info)
{
	int error, success;
	unsigned long flags, head, tail;

	error = button_event_send(event_queue_default, source, info);
	if (error) {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: event couldn't be sent, error %d", __func__, error);
		return error;
	}

	spin_lock_irqsave(&event_queue_producer_lock, flags);
	head = event_queue_deferred_events_cb.head;
	tail = ACCESS_ONCE(event_queue_deferred_events_cb.tail);
	if (CIRC_SPACE(head, tail, EVENT_QUEUE_NUM_DEFERRED_EVENTS) >= 1) {
		struct event_queue_deferred_event *de = &event_queue_deferred_events[head];
		INIT_WORK(&de->worker, event_queue_worker);
		de->source = source;
		de->info = info;
		smp_wmb();
		event_queue_deferred_events_cb.head = (head + 1) & (EVENT_QUEUE_NUM_DEFERRED_EVENTS - 1);
		success = queue_work(event_queue_deferred_queue, &de->worker);
		if (!success) {
			bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: error queuing work", __func__);
			error = -EIO;
		}
	} else {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "%s: no space for event", __func__);
		error = -EIO;
	}
	spin_unlock_irqrestore(&event_queue_producer_lock, flags);
	return error;
}
EXPORT_SYMBOL(event_queue_send_event_defer);

struct cdev event_queue_chr_dev;

static int __init event_queue_init(void)
{
	int error;
	struct device *class_dev = NULL;

	devno = MKDEV(EVENT_QUEUE_MAJOR_NUMBER, 0);

	event_queue_deferred_queue = create_singlethread_workqueue("deferred events");
	if (event_queue_deferred_queue == NULL) {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "deferred event queue could not be created");
		return -EIO;
	}
	event_queue_init_events();

#ifdef CONFIG_DEVTMPFS
	error = alloc_chrdev_region(&devno, 0, 1, EVENT_QUEUE_DEVICE_NAME);
#else
	error = register_chrdev_region(devno, 1, EVENT_QUEUE_DEVICE_NAME);
#endif
	if (error) {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "Error %d registering device %s", error, EVENT_QUEUE_DEVICE_NAME);
		goto event_queue_fail1;
	}
	cdev_init(&event_queue_chr_dev, &event_queue_fops);
	event_queue_chr_dev.owner = THIS_MODULE;

	error = cdev_add(&event_queue_chr_dev, devno, 1);
	if (error) {
		printk("Couldn't add character device (%d).", error);
		goto event_queue_fail2;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, EVENT_QUEUE_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "Error creating event queue sonos device.\n");
		cdev_del(&event_queue_chr_dev);
		error = PTR_ERR(class_dev);
		goto event_queue_fail2;
	}

	spin_lock_init(&event_queue_lock);
	spin_lock_init(&event_queue_producer_lock);
	spin_lock_init(&event_queue_consumer_lock);
	init_waitqueue_head(&event_queue_wait_queue);
	event_queue_default = button_event_queue_create(100, &event_queue_wait_queue, &event_queue_lock);
	if (event_queue_default == NULL) {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "default event queue couldn't be created");
		goto event_queue_fail3;
	}
	error = event_queue_init_proc();
	if (error) {
		bb_log(BB_MOD_EVENT, BB_LVL_ERR, "event queue proc init failed, error %d", error);
		goto event_queue_fail4;
	}
	event_queue_deferred_events_cb.head = 0;
	event_queue_deferred_events_cb.tail = 0;
	bb_log(BB_MOD_EVENT, BB_LVL_INFO, "Event queue registered at %d, version %d", MAJOR(devno), EVENT_QUEUE_VERSION);
	device_init_success = 1;
	return 0;

event_queue_fail4:
	button_event_queue_free(event_queue_default);
	event_queue_default = NULL;

event_queue_fail3:
	sonos_device_destroy(devno);
	cdev_del(&event_queue_chr_dev);

event_queue_fail2:
	unregister_chrdev_region(devno, 1);

event_queue_fail1:
	destroy_workqueue(event_queue_deferred_queue);

	return error;
}
module_init(event_queue_init);

static void __exit event_queue_exit(void)
{
	event_queue_remove_proc();
	button_event_queue_free(event_queue_default);
	event_queue_default = NULL;
	sonos_device_destroy(devno);
	cdev_del(&event_queue_chr_dev);
	unregister_chrdev_region(devno, 1);
	destroy_workqueue(event_queue_deferred_queue);
}
module_exit(event_queue_exit);

static int event_queue_device_open(struct inode *inode, struct file *file)
{

	if (file->f_mode & FMODE_READ) {
		if (!device_readers++) {
			button_event_queue_flush(event_queue_default);
		}
	}

	if (file->f_mode & FMODE_WRITE) {
		++device_writers;
	}

	return (device_init_success ? 0 : -ENODEV);
};

static int event_queue_device_release(struct inode *inode, struct file *file)
{
	if (file->f_mode & FMODE_READ) {
		--device_readers;
	}

	if (file->f_mode & FMODE_WRITE) {
		--device_writers;
	}

	return 0;
}

static ssize_t event_queue_device_read(struct file *filp, char *data, size_t len, loff_t *ppos)
{
	int ret;

	if (len < sizeof(struct button_evt)) {
		return -EINVAL;
	}

	if (filp->f_flags & O_NONBLOCK) {
		if (button_event_queue_empty(event_queue_default)) {
			return -EWOULDBLOCK;
		}
	}

	ret = button_event_receive(event_queue_default, data);
	if (ret < 0) {
		return ret;
	}
	return sizeof(struct button_evt);
}

static ssize_t event_queue_device_write(struct file *file, const char *data, size_t len, loff_t *ppos)
{
	return 0;
}

static unsigned int event_queue_device_poll(struct file *fp, struct poll_table_struct *pt)
{
	unsigned int mask=0;

	poll_wait(fp,&event_queue_wait_queue, pt);
	if (!button_event_queue_empty(event_queue_default)) {
		mask = POLLIN | POLLRDNORM;
	}

	return mask;
}

static long event_queue_device_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = EVENT_QUEUE_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case 12:
	case 28:
	case 58:
	case 61:
	case 65:
	case 66:
	case 68:
		break;
	default:
		printk("event_queue: Unrecognized IOCTL command %u\n", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}

struct file_operations event_queue_fops =
{
	.unlocked_ioctl = event_queue_device_ioctl,
	.open		= event_queue_device_open,
	.release	= event_queue_device_release,
	.read		= event_queue_device_read,
	.write		= event_queue_device_write,
	.poll		= event_queue_device_poll,
};

MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("Sonos, Inc.");
MODULE_DESCRIPTION ("Event Queue Module");
