/* Copyright (c) 2017-2018 Sonos. Inc.
 * All rights reserved.
 * I2C EEPROM driver
 *@File i2c-eeprom.c, impletent the i2c eeprom read/write
 *
 */

#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/slab.h>
#include <linux/mutex.h>
#include <linux/delay.h>
#include <linux/sysfs.h>
#include <linux/of_gpio.h>
#include <linux/version.h>

struct eeprom_data_t {
    unsigned int page_size;
    unsigned int addr_len;
    unsigned int chip_size;
    struct bin_attribute bin;
    struct mutex buf_mutex;
    struct gpio wp;
    u8 buffer[];
};

static ssize_t i2c_eeprom_bin_read(struct file *filp, struct kobject *kobj,
		struct bin_attribute *attr, char *buf, loff_t off, size_t count)
{
    int ret = 0;
    unsigned int bytes = off;
    unsigned int read_size = 0;
    struct eeprom_data_t *eeprom = 0;
    struct i2c_client* client = 0;

    printk(KERN_INFO "%s: cout: %zu, offset: %lld", __func__, count, off);

    client = kobj_to_i2c_client(kobj);
    eeprom = i2c_get_clientdata(client);
    if ((off + count) > eeprom->bin.size)
    {
        printk(KERN_ERR "exceed the eeprom size.");
        return -1;
    }

    mutex_lock(&eeprom->buf_mutex);

    if (eeprom->addr_len == 1)
    {
        eeprom->buffer[0] = (unsigned char)(bytes & 0x00ff);
    }
    else if (eeprom->addr_len == 2)
    {
        eeprom->buffer[0] = bytes >> 8;
        eeprom->buffer[1] = (unsigned char)(bytes & 0x00ff);
    }
    i2c_master_send(client, &eeprom->buffer[0], eeprom->addr_len);

    while (bytes < (off + count))
    {
        read_size = (off + count - bytes) > eeprom->page_size ? eeprom->page_size : (off + count - bytes);
        ret = i2c_master_recv(client, &eeprom->buffer[0], read_size);
        if (ret < 0)
        {
            mutex_unlock(&eeprom->buf_mutex);
            printk(KERN_ERR "i2c bus read failed.");
            return ret;
        }
        memcpy(buf + bytes - off, &eeprom->buffer[0], read_size);
        bytes += read_size;
    }
    mutex_unlock(&eeprom->buf_mutex);
    printk(KERN_INFO "%s: read %lld bytes.", __func__, bytes - off);
    return count;
}

static ssize_t i2c_eeprom_bin_write(struct file *filp, struct kobject *kobj,
		struct bin_attribute *attr, char *buf, loff_t off, size_t count)
{
    int ret = 0;
    unsigned int bytes = off;
    unsigned int write_size = 0;
    struct eeprom_data_t *eeprom;
    struct i2c_client* client = 0;

    printk(KERN_INFO "%s: count: %zu, offset: %lld", __func__, count, off);

    client = kobj_to_i2c_client(kobj);
    eeprom = i2c_get_clientdata(client);
    if ((off + count) > eeprom->bin.size)
    {
        printk(KERN_ERR "exceed the eeprom size.");
        return -1;
    }

    mutex_lock(&eeprom->buf_mutex);
    if (gpio_is_valid(eeprom->wp.gpio))
    {
        gpio_set_value(eeprom->wp.gpio, 0);
    }
    while (bytes < (off + count))
    {
        write_size = (off + count - bytes) > eeprom->page_size ? eeprom->page_size : (off + count - bytes);
        if (eeprom->addr_len == 1)
        {
            eeprom->buffer[0] = (unsigned char)(bytes & 0x00ff);
        }
        else if (eeprom->addr_len == 2)
        {
            eeprom->buffer[0] = bytes >> 8;
            eeprom->buffer[1] = (unsigned char)(bytes & 0x00ff);
        }
        memcpy(&eeprom->buffer[eeprom->addr_len], (buf + bytes - off), write_size);
        ret = i2c_master_send(client, &eeprom->buffer[0], write_size + eeprom->addr_len);
        if (ret < 0)
        {
            mutex_unlock(&eeprom->buf_mutex);
            printk(KERN_ERR "i2c bus write failed.");
            if (gpio_is_valid(eeprom->wp.gpio))
            {
                gpio_set_value(eeprom->wp.gpio, 1);
            }
            return ret;
        }
        bytes += write_size;
        msleep(1);
    }
    if (gpio_is_valid(eeprom->wp.gpio))
    {
        gpio_set_value(eeprom->wp.gpio, 1);
    }
    mutex_unlock(&eeprom->buf_mutex);
    printk(KERN_INFO "%s: write %d bytes", __func__, bytes);
    return count;
}

static int i2c_eeprom_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
    int ret;
    unsigned int chip_size = 0;
    unsigned int page_size = 0;
    unsigned int addr_len = 0;
    struct gpio wp;
    struct device_node *node;
    struct eeprom_data_t *eeprom;
    node = client->dev.of_node;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,4,24)
    client->flags |= I2C_CLIENT_SLAVE;
#endif
    of_property_read_u32(node, "chip-size", &chip_size);
    of_property_read_u32(node, "page-size", &page_size);
    of_property_read_u32(node, "addr-len", &addr_len);
    wp.gpio = of_get_named_gpio(node, "write-protect", 0);

    printk(KERN_INFO "chip size: 0x%x, page size: 0x%02x, addr len: %d.", chip_size, page_size, addr_len);
    if (!(chip_size && page_size && addr_len))
    {
        printk(KERN_ERR "unsufficient eeprom info, needed chip-size, page-size and addr-len in dts.");
        return -EINVAL;
    }
    if (addr_len > 2)
    {
        addr_len = 2;
    }

    printk(KERN_INFO "i2c eeprom probe %s, addr 0x%02x.", client->name, client->addr);

    eeprom = devm_kzalloc(&client->dev, sizeof(struct eeprom_data_t) + page_size + addr_len, GFP_KERNEL);
    if (!eeprom)
    {
        return -ENOMEM;
    }
    eeprom->page_size = page_size;
    eeprom->addr_len = addr_len;
    eeprom->chip_size = chip_size;
    eeprom->wp = wp;

    mutex_init(&eeprom->buf_mutex);
    i2c_set_clientdata(client, eeprom);

    ret = i2c_master_send(client, &eeprom->buffer[0], eeprom->addr_len);
    if ( ret < 0 ) {
        printk(KERN_INFO "%s: i2c_eeprom not present.\n", __func__);
        return -ENXIO;
    }

    sysfs_bin_attr_init(&eeprom->bin);
    eeprom->bin.attr.name = "eeprom";
    eeprom->bin.attr.mode = S_IRUSR | S_IWUSR;
    eeprom->bin.read = i2c_eeprom_bin_read;
    eeprom->bin.write = i2c_eeprom_bin_write;
    eeprom->bin.size = chip_size;

    ret = sysfs_create_bin_file(&client->dev.kobj, &eeprom->bin);
    if (ret)
    {
        return ret;
    }

    ret = sysfs_create_link(NULL, &client->dev.kobj, "eeprom");
    if (ret)
    {
        return ret;
    }

    return 0;
};

static int i2c_eeprom_remove(struct i2c_client *client)
{
    struct eeprom_data_t *eeprom = i2c_get_clientdata(client);

    mutex_destroy(&eeprom->buf_mutex);
    sysfs_remove_link(NULL, "eeprom");
    sysfs_remove_bin_file(&client->dev.kobj, &eeprom->bin);

    return 0;
}

static const struct i2c_device_id i2c_eeprom_id[] = {
    { "m24c64", 0 },
    { "m24512", 0 },
    { }
};
MODULE_DEVICE_TABLE(i2c, i2c_eeprom_id);

static struct i2c_driver i2c_eeprom_driver = {
    .driver = {
	    .name = "i2c-eeprom",
    },
    .probe = i2c_eeprom_probe,
    .remove = i2c_eeprom_remove,
    .id_table = i2c_eeprom_id,
};
module_i2c_driver(i2c_eeprom_driver);

MODULE_AUTHOR("Sonos. Inc.");
MODULE_DESCRIPTION("I2C EEPROM Driver");
MODULE_LICENSE("GPL v2");
