/*
 * Copyright (c) 2016-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * BLACKBOX: Kernel trace functionality for Sonos kernel modules
 */

#include <linux/hardirq.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/spinlock.h>
#include <linux/time.h>
#include <linux/uaccess.h>
#include <linux/vmalloc.h>
#include <linux/sched.h>

#include "blackbox.h"

struct bb_entry {
	uint32_t	mask;
	uint16_t	length;
	uint16_t	flags;
	struct timespec	time;
	char		str[0];
};

static void		*bb_buf = NULL;
static void		*bb_writep;
static void		*bb_readp;

static uint8_t		bb_print_rel_time = 0;
static uint8_t		bb_has_wrapped = 0;
static uint32_t		bb_mask = BB_MASK_DEFAULT;
module_param(bb_mask, uint, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
MODULE_PARM_DESC(bb_mask, "Blackbox mask");
static uint32_t		bb_printk_mask = BB_PRINTK_DEFAULT;
module_param(bb_printk_mask, uint, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
MODULE_PARM_DESC(bb_printk_mask, "Blackbox printk mask");
static unsigned int	bb_buf_size = 0x20000;
module_param(bb_buf_size, uint, S_IRUSR | S_IRGRP | S_IROTH);
MODULE_PARM_DESC(bb_buf_size, "Blackbox ringbuffer size");

static DEFINE_RWLOCK(bb_rwlock);

#define BB_CASE_STR(type, val) \
	case BB_##type##_##val: \
		return #val;

const char *bb_mod_to_str(unsigned mod)
{
	switch (BB_MASK_MOD(mod)) {
	BB_CASE_STR(MOD, BLACKBOX)
	BB_CASE_STR(MOD, LLA)
	BB_CASE_STR(MOD, LEDCTL)
	BB_CASE_STR(MOD, AMPCTL)
	BB_CASE_STR(MOD, EVENT)
	BB_CASE_STR(MOD, FPGA)
	BB_CASE_STR(MOD, SENSORS)
	BB_CASE_STR(MOD, SDD)
	BB_CASE_STR(MOD, SWD)
	BB_CASE_STR(MOD, CHK)
	BB_CASE_STR(MOD, APPLE)
	BB_CASE_STR(MOD, VBLOCK)
	BB_CASE_STR(MOD, CAAM)
	BB_CASE_STR(MOD, PSMON)
	BB_CASE_STR(MOD, M4)
	BB_CASE_STR(MOD, MICCTL)
	BB_CASE_STR(MOD, EEPROM)
	BB_CASE_STR(MOD, IR)
	BB_CASE_STR(MOD, CCGSWD)
	BB_CASE_STR(MOD, MCU)
	BB_CASE_STR(MOD, HDMICTL)
	BB_CASE_STR(MOD, USBPD)
	BB_CASE_STR(MOD, QCADDFA)
	BB_CASE_STR(MOD, PROCFS)
	BB_CASE_STR(MOD, BLUTOOTH)
	BB_CASE_STR(MOD, USR2)
	BB_CASE_STR(MOD, USR3)
	BB_CASE_STR(MOD, USR4)
	}
	return "UNKNOWN";
}

const char *bb_lvl_to_str(unsigned lvl)
{
	switch (BB_MASK_LVL(lvl)) {
	BB_CASE_STR(LVL, ERR)
	BB_CASE_STR(LVL, WARNING)
	BB_CASE_STR(LVL, INFO)
	BB_CASE_STR(LVL, DEBUG)
	}
	return "UNKNOWN";
}

static void bb_advance_readp(int str_len)
{
	struct bb_entry *entry;
	int adv_len = ALIGN((sizeof(struct bb_entry) + str_len) - (bb_readp - bb_writep), sizeof(int));
	while (adv_len > 0) {
		entry = bb_readp;
		if (!(entry->mask)) {
			bb_readp += adv_len;
			break;
		}
		bb_readp += entry->length;
		adv_len -= entry->length;
	}
}

inline static void bb_get_time_entry(struct timespec* time)
{
	u64 nsec = local_clock();
	time->tv_nsec = do_div(nsec, 1000000000);
	time->tv_sec = nsec;
}

void bb_log_dev(struct device *dev, unsigned mod, unsigned lvl, const char *fmt, ...)
{
	char scratch[BB_ENTRY_MAX] = { 0 };
	struct bb_entry *entry;
	va_list args;
	size_t buf_size, fmt_size;
	unsigned increment;
	ssize_t dev_size = 0, va_size = 0;
	unsigned long flags;
	uint8_t context = BB_CONT_NONE;
	if (in_nmi()) {
		panic("Logging not supported from NMI context.\n");
		return;
	}
	if (dev) {
		dev_size = snprintf(scratch, BB_ENTRY_MAX, "%.24s@%.24s: ", dev_driver_string(dev), dev_name(dev));
		if (unlikely(dev_size < 0)) {
			printk(KERN_ERR "BLACKBOX ASSERT: snprintf returned %zd while parsing dev name!\n", dev_size);
			return;
		}
	}
	va_start(args, fmt);
	va_size = vsnprintf(scratch + dev_size, BB_ENTRY_MAX - dev_size, fmt, args);
	va_end(args);
	if (unlikely(va_size < 0)) {
		printk(KERN_ERR "BLACKBOX ASSERT: vsnprintf returned %zd while parsing format string!\n", va_size);
		return;
	}
	fmt_size = min(va_size + dev_size + 1, (ssize_t)BB_ENTRY_MAX);
	if ((bb_printk_mask & mod) && (bb_printk_mask & lvl)) {
		printk("{%c%c} <%s, %s> %s %s\n", ((in_irq()) ? 'H' : '-'),
						  ((in_softirq()) ? 'S' : '-'),
						  bb_mod_to_str(mod), bb_lvl_to_str(lvl),
						  ((lvl & BB_LVL_ERR) ? "(ASSERT)" : ""),
						  scratch);
	}
	write_lock_irqsave(&bb_rwlock, flags);
	buf_size = (bb_buf + bb_buf_size) - bb_writep;
	if (buf_size < (sizeof(struct bb_entry) + fmt_size)) {
		memset(bb_writep, 0x00, buf_size);
		bb_writep = bb_buf;
		bb_readp = bb_writep;
		bb_has_wrapped = 1;
	}
	if (bb_has_wrapped) {
		bb_advance_readp(fmt_size);
	}
	entry = (struct bb_entry *)bb_writep;
	entry->mask = (mod | lvl);
	bb_get_time_entry(&(entry->time));
	if (in_irq()) {
		context |= BB_CONT_HIRQ;
	}
	if (in_softirq()) {
		context |= BB_CONT_SIRQ;
	}
	entry->flags = context;
	increment = ALIGN(sizeof(struct bb_entry) + fmt_size, sizeof(int));
	entry->length = increment;
	strncpy(entry->str, scratch, increment - sizeof(struct bb_entry));
	bb_writep += increment;
	write_unlock_irqrestore(&bb_rwlock, flags);
}
EXPORT_SYMBOL(bb_log_dev);

static int bb_proc_show(struct seq_file *m, void *v)
{
	struct bb_entry *entry;
	void *readp;
	unsigned increment = 0, bytes = 0;
	unsigned long flags;
	struct timespec last_time = { 0, 0 };
	read_lock_irqsave(&bb_rwlock, flags);
	readp = bb_readp;
	do {
		entry = (struct bb_entry *)readp;
		if ((bb_mask & BB_MASK_MOD(entry->mask)) && (bb_mask & BB_MASK_LVL(entry->mask))) {
			if (bb_print_rel_time) {
				struct timespec temp = timespec_sub(entry->time, last_time);
				last_time = entry->time;
				seq_printf(m, "[%3ld.%06ld] ", temp.tv_sec, (temp.tv_nsec / NSEC_PER_USEC));
			} else {
				seq_printf(m, "[%3ld.%06ld] ", entry->time.tv_sec, (entry->time.tv_nsec / NSEC_PER_USEC));
			}
			seq_printf(m, "{%c%c} <%s, %s> %s\n", ((entry->flags & BB_CONT_HIRQ) ? 'H' : '-'),
							      ((entry->flags & BB_CONT_SIRQ) ? 'S' : '-'),
							      bb_mod_to_str(entry->mask), bb_lvl_to_str(entry->mask),
							      entry->str);
		}
		if (entry->mask) {
			increment = entry->length;
		} else {
			increment = 4;
		}
		if ((readp + increment) >= (bb_buf + bb_buf_size)) {
			readp = bb_buf;
		} else if (increment) {
			bytes += increment;
			if (unlikely(bytes >= bb_buf_size)) {
				printk(KERN_ERR "BLACKBOX ASSERT: Read exceeds size of buffer. Something terrible has happened! Aborting to avoid deadlock.\n");
				break;
			}
			readp += increment;
		} else {
			printk(KERN_ERR "BLACKBOX ASSERT: Encountered an increment of zero. Likely buffer corruption. Bailing out. Current entry mask: %#010x\n", entry->mask);
			break;
		}
	} while (readp != bb_writep);
	read_unlock_irqrestore(&bb_rwlock, flags);
	return 0;
}

static ssize_t bb_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[BB_ENTRY_MAX];
	char *str;
	uint32_t lvl;
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}
	str = buf;

	if (!strncmp(str, "ERR ", 4)) {
		str += 4;
		lvl = BB_LVL_ERR;
	} else {
		lvl = BB_LVL_INFO;
	}
	bb_log(BB_MOD_PROCFS, lvl, "%s", str);
	return count;
}

static int bb_config_show(struct seq_file *m, void *v)
{
	uint64_t mask = 0x00000001;
	seq_printf(m, "BLACKBOX CONFIG: mask=%#010x printk_mask=%#010x time=%s\n", bb_mask, bb_printk_mask,
		   (bb_print_rel_time ? "relative" : "absolute"));
	seq_printf(m, "BLACKBOX COMMANDS: clear, defaults\n");
	seq_printf(m, "BLACKBOX MODULES:\n");
	for (; mask <= 0x8000000; mask <<= 2) {
		seq_printf(m, "%#010llx: %-10s\t", mask, bb_mod_to_str(mask));
		seq_printf(m, "%#010llx: %-10s\n", mask << 1, bb_mod_to_str(mask << 1));
	}
	seq_printf(m, "BLACKBOX LEVELS:\n");
	for (; mask <= 0x80000000; mask <<= 2) {
		seq_printf(m, "%#010llx: %-10s\t", mask, bb_lvl_to_str(mask));
		seq_printf(m, "%#010llx: %-10s\n", mask << 1, bb_lvl_to_str(mask << 1));
	}
	return 0;
}

static ssize_t bb_config_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[64];
	char *cmd, *str, *val;
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}
	str = buf;
	while (str) {
		cmd = strsep(&str, "=");
		val = strsep(&str, " ");
		switch (cmd[0]) {
		case 'p':
		{
			uint32_t mask;
			if (!val) {
				bb_log(BB_MOD_BLACKBOX, BB_LVL_WARNING, "Configuration option %s requires an argument.", cmd);
				return count;
			}
			if (kstrtou32(val, 16, &mask)) {
				bb_log(BB_MOD_BLACKBOX, BB_LVL_WARNING, "Error parsing printk mask %s.", val);
				break;
			}
			bb_printk_mask = mask;
			bb_log(BB_MOD_BLACKBOX, BB_LVL_INFO, "Printk mask changed to %#010x.", bb_printk_mask);
			break;
		}
		case 'm':
		{
			uint32_t mask;
			if (!val) {
				bb_log(BB_MOD_BLACKBOX, BB_LVL_WARNING, "Configuration option %s requires an argument.", cmd);
				return count;
			}
			if (kstrtou32(val, 16, &mask)) {
				bb_log(BB_MOD_BLACKBOX, BB_LVL_WARNING, "Error parsing mask %s.", val);
				break;
			}
			bb_mask = mask;
			bb_log(BB_MOD_BLACKBOX, BB_LVL_INFO, "Blackbox mask changed to %#010x.", bb_mask);
			break;
		}
		case 't':
		{
			if (!val) {
				bb_log(BB_MOD_BLACKBOX, BB_LVL_WARNING, "Configuration option %s requires an argument.", cmd);
				return count;
			}
			if (!strncmp(val, "rel", 3)) {
				bb_print_rel_time = 1;
				bb_log(BB_MOD_BLACKBOX, BB_LVL_INFO, "Now using relative time.");
			} else if (!strncmp(val, "abs", 3)) {
				bb_print_rel_time = 0;
				bb_log(BB_MOD_BLACKBOX, BB_LVL_INFO, "Now using absolute time.");
			} else {
				bb_log(BB_MOD_BLACKBOX, BB_LVL_WARNING, "Unrecognized argument to time: %s.", val);
			}
			break;
		}
		case 'd':
		{
			bb_mask = BB_MASK_DEFAULT;
			bb_printk_mask = BB_PRINTK_DEFAULT;
			bb_log(BB_MOD_BLACKBOX, BB_LVL_INFO, "Reset masks to default values.");
			break;
		}
		case 'c':
		{
			unsigned long flags;
			write_lock_irqsave(&bb_rwlock, flags);
			bb_writep = bb_buf;
			bb_readp = bb_buf;
			bb_has_wrapped = 0;
			memset(bb_buf, 0x00, bb_buf_size);
			write_unlock_irqrestore(&bb_rwlock, flags);
			bb_log(BB_MOD_BLACKBOX, BB_LVL_INFO, "Log cleared.");
			break;
		}
		default:
			bb_log(BB_MOD_BLACKBOX, BB_LVL_WARNING, "Unrecognized command %s.", cmd);
			break;
		}
	}
	return count;
}

static int bb_time_show(struct seq_file *m, void *v)
{
	struct timeval timeofday;
	u64 ts_nsec;
	uint32_t rem_nsec;

	do_gettimeofday(&timeofday);

	ts_nsec = local_clock();
	rem_nsec = do_div(ts_nsec, 1000000000);

	seq_printf(m, "Sonos Kernel Time Information\n");
	seq_printf(m, "Jiffies: %llu \nTime of day: %lu.%06lu \nPrintk time: %u.%06u\n",
				get_jiffies_64(), timeofday.tv_sec,
				timeofday.tv_usec, (uint32_t)ts_nsec, rem_nsec/1000);
	return 0;
}

static int bb_proc_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, bb_proc_show, PDE_DATA(inode));
}

static int bb_config_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, bb_config_show, PDE_DATA(inode));
}

static int bb_time_open(struct inode *inode, struct file *filp)
{
	return single_open(filp, bb_time_show, PDE_DATA(inode));
}

const static struct file_operations bb_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= bb_proc_open,
	.write		= bb_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

const static struct file_operations bb_config_fops = {
	.owner		= THIS_MODULE,
	.open		= bb_config_open,
	.write		= bb_config_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};

const static struct file_operations bb_time_fops = {
	.owner		= THIS_MODULE,
	.open		= bb_time_open,
	.write		= NULL,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};

static struct proc_dir_entry *bb_parent_dir;

static int __init bb_init(void)
{
	printk(KERN_INFO "Sonos Blackbox kernel trace module, buffer size %d\n", bb_buf_size);
	if (bb_buf_size % sizeof(int)) {
		printk(KERN_WARNING "WARNING: Requested buffer size %d, not word-aligned. Rounding to %d.\n", bb_buf_size, ALIGN(bb_buf_size, sizeof(int)));
		bb_buf_size = ALIGN(bb_buf_size, sizeof(int));
	}
	bb_buf = vzalloc(bb_buf_size);
	if (!bb_buf) {
		printk(KERN_ERR "Unable to allocate blackbox ringbuffer.\n");
		return ENOMEM;
	}
	bb_writep = bb_buf;
	bb_readp = bb_buf;
	bb_parent_dir = proc_mkdir("blackbox", 0);
	if (!bb_parent_dir) {
		printk(KERN_ERR "BLACKBOX ASSERT: Unable to create /proc/blackbox/.\n");
		goto out_err;
	}
	if (!proc_create("log", 0666, bb_parent_dir, &bb_proc_fops)) {
		printk(KERN_ERR "BLACKBOX ASSERT: Unable to create /proc/blackbox/log.\n");
		goto out_proc_err;
	}
	if (!proc_create("config", 0666, bb_parent_dir, &bb_config_fops)) {
		printk(KERN_ERR "BLACKBOX ASSERT: Unable to create /proc/blackbox/config.\n");
		remove_proc_entry("log", bb_parent_dir);
		goto out_proc_err;
	}
	if (!proc_create("timeinfo", 0444, NULL, &bb_time_fops)) {
		printk(KERN_ERR "BLACKBOX ASSERT: Unable to create /proc/timeinfo.\n");
	}
	bb_log(BB_MOD_BLACKBOX, BB_LVL_INFO, "Sonos Blackbox kernel trace module, buffer size %d", bb_buf_size);
	return 0;
out_proc_err:
	remove_proc_entry("blackbox", NULL);
out_err:
	vfree(bb_buf);
	return -1;
}
module_init(bb_init);

static void __exit bb_exit(void)
{
	remove_proc_entry("log", bb_parent_dir);
	remove_proc_entry("config", bb_parent_dir);
	remove_proc_entry("blackbox", NULL);
	remove_proc_entry("timeinfo", NULL);
	vfree(bb_buf);
}
module_exit(bb_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Sonos Blackbox kernel logging module");
MODULE_LICENSE("GPL");
