/*
 * Copyright (c) 2010-2020 Sonos Inc.
 * All rights reserved.
 */
#include "audioctl.h"
#include "audioctl_i2c.h"
#include "hwevent_queue_user_api.h"
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/workqueue.h>
#include <asm/uaccess.h>
#include "mdp.h"
#include "button_event.h"
#include "hwevent_queue_api.h"

#define MAX_THERM_INSTS   AUDIOCTL_MAX_AVAILABLE_TEMPERATURE_SENSORS
#define THERM_POLL_TIME   msecs_to_jiffies(5000)

#define THERM_TYPE_S5851A 0
#define THERM_TYPE_ADC    1

#define MAX_RANGE	120
#define MIN_RANGE	-40

#define SEN_ERROR	-212
#define SEN_RANGE	-211

typedef struct _THERM_INST
{
   char *name;
   int   inst;
   int   type;
   int   warnTemp;
   int   faultTemp;
   enum  HWEVTQ_EventSource evtSource;
   int   warnStatus;
   int   faultStatus;
   struct i2c_board_info *info;
} THERM_INST;


static struct i2c_board_info FenwayThermalSensorI2cInfo = {
	I2C_BOARD_INFO("Thermal Sensor", THERMAL_SENSOR_I2C_ADDRESS),
	.irq		= -1,
};

static struct i2c_board_info AmoebaThermalSensorI2cInfo = {
	I2C_BOARD_INFO("Thermal Sensor", THERMAL_SENSOR_I2C_ADDRESS),
	.irq		= -1,
};

static struct i2c_board_info AnvilCpuThermalSensorI2cInfo = {
	I2C_BOARD_INFO("CPU Thermal Sensor", CPU_THERMAL_SENSOR_I2C_ADDRESS),
	.irq		= -1,
};

static struct i2c_board_info AnvilEthThermalSensorI2cInfo = {
	I2C_BOARD_INFO("Eth Thermal Sensor", ETH_THERMAL_SENSOR_I2C_ADDRESS),
	.irq		= -1,
};

static struct i2c_board_info AnvilWifiThermalSensorI2cInfo = {
	I2C_BOARD_INFO("WiFi Thermal Sensor", WIFI_THERMAL_SENSOR_I2C_ADDRESS),
	.irq		= -1,
};

static struct i2c_board_info AnvilCoreThermalSensorI2cInfo = {
	I2C_BOARD_INFO("Core Thermal Sensor", CORE_THERMAL_SENSOR_I2C_ADDRESS),
	.irq		= -1,
};

static struct i2c_board_info AnvilThermalSensorI2cInfo_4a = {
	I2C_BOARD_INFO("therm-4a", 0x4a),
	.irq		= -1,
};
static struct i2c_board_info AnvilThermalSensorI2cInfo_4c = {
	I2C_BOARD_INFO("therm-4c", 0x4c),
	.irq		= -1,
};

THERM_INST FenwayThermalSensors[] = {
   {.name = "CPU",
    .inst = AUDIOCTL_CPU_TEMPERATURE_SENSOR,
    .warnTemp = FENWAY_CPU_WARN_TEMPERATURE,
    .faultTemp = FENWAY_CPU_FAULT_TEMPERATURE,
    .evtSource = HWEVTQSOURCE_CPU,
    .warnStatus = SYSTAT_CPU_WARN_TEMPERATURE_STATUS,
    .faultStatus = SYSTAT_CPU_FAULT_TEMPERATURE_STATUS,
    .info = &FenwayThermalSensorI2cInfo},
   {}};

THERM_INST AnvilThermalSensors[] = {
   {.name = "CPU",
    .inst = AUDIOCTL_CPU_TEMPERATURE_SENSOR,
    .warnTemp = ANVIL_CPU_WARN_TEMPERATURE,
    .faultTemp = ANVIL_CPU_FAULT_TEMPERATURE,
    .evtSource = HWEVTQSOURCE_CPU,
    .warnStatus = SYSTAT_CPU_WARN_TEMPERATURE_STATUS,
    .faultStatus = SYSTAT_CPU_FAULT_TEMPERATURE_STATUS,
    .info = &AnvilCpuThermalSensorI2cInfo},
   {}};

THERM_INST AmoebaThermalSensors[] = {
   {.name = "CPU",
    .inst = AUDIOCTL_CPU_TEMPERATURE_SENSOR,
    .warnTemp = AMOEBA_CPU_WARN_TEMPERATURE,
    .faultTemp = AMOEBA_CPU_FAULT_TEMPERATURE,
    .evtSource = HWEVTQSOURCE_CPU,
    .warnStatus = SYSTAT_CPU_WARN_TEMPERATURE_STATUS,
    .faultStatus = SYSTAT_CPU_FAULT_TEMPERATURE_STATUS,
    .info = &AmoebaThermalSensorI2cInfo},
   {}};

typedef struct _THERM_DATA {
   struct i2c_client *pI2cClient;
   char *pInstName;
   int   instId;
   int   debugTemp;
   int   type;
   int   initFailed;
   int   warnTemp;
   int   faultTemp;
   enum HWEVTQ_EventSource evtSource;
   int   warnStatus;
   int   faultStatus;
   unsigned long   status;
   int   minTemp;
   int   maxTemp;
   unsigned int   readCount;
   unsigned int   writeCount;
   unsigned int   inWarnCount;
   unsigned int   enterWarnCount;
   unsigned int   backoffWarnCount;
   unsigned int   inFaultCount;
   unsigned int   enterFaultCount;
} THERM_DATA;

static THERM_DATA ThermData[MAX_THERM_INSTS];
static struct button_event_queue *button_event_queue;
static struct delayed_work thermal_work;

static int
thermalsensor_proc_read( char *page, char **start, off_t off, int count, int*eof, void *data)
{
   return thermalSensor_PrintStatus(page);
}

static int
thermalsensor_proc_write(struct file *file, const char __user * buffer,
                         unsigned long count, void *data)
{
	int result = 0;
   long longtemp, i2caddr;
   char buf[40];
   char *peq;
   const char warn_str[] = "warn-temp-";
   const char fault_str[] = "fault-temp-";
   const char temp_str[] = "temp-";

   if (count >= sizeof(buf))
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	}
   else {
      buf[count] = 0;
      peq = strchr(buf, '=');
      if (peq != NULL) {
         *peq = 0;
         if (strncmp(buf, temp_str, strlen(temp_str)) == 0) {
            strict_strtol(peq-2, 16, &i2caddr);
            strict_strtol(peq+1, 10, &longtemp);
            thermalsensor_SetDebugTemp(i2caddr, longtemp);
         }
		 else if (strncmp(buf, fault_str, strlen(fault_str)) == 0) {
            strict_strtol(peq-2, 16, &i2caddr);
            strict_strtol(peq+1, 10, &longtemp);
            thermalsensor_SetFaultTemp(i2caddr, longtemp);
		 }
		 else if (strncmp(buf, warn_str, strlen(warn_str)) == 0) {
            strict_strtol(peq-2, 16, &i2caddr);
            strict_strtol(peq+1, 10, &longtemp);
            thermalsensor_SetWarnTemp(i2caddr, longtemp);
		 }
	  }
      result = count;
   }

	return result;
}

int thermalsensor_proc_init( void )
{
	 struct proc_dir_entry *proc;

    proc = create_proc_read_entry("driver/temp-sensor", 0, 0, thermalsensor_proc_read, 0);
    if( !proc ) {
        return( -EIO );
    }
    proc->write_proc = (write_proc_t *) thermalsensor_proc_write;

    return( 0 );
}

static inline THERM_DATA * thermalsensor_GetData(int id)
{
   THERM_DATA *p = NULL;

   if (id < AUDIOCTL_MAX_AVAILABLE_TEMPERATURE_SENSORS ) {
	   p = &ThermData[id];
   }

   return(p);
}

int thermalsensor_InitDevice(THERM_DATA *pThermData)
{
   int error=0;
   char reg;
   char value;
   int  attempts;
   unsigned int errorCount=0;
   unsigned int eioCount=0;

   if (pThermData->type == THERM_TYPE_ADC) {
      return 0;
   }

   attempts = 0;
   error = -1;
   while (error) {
      reg = 1;
      value = 0;
      pThermData->writeCount++;
      error = i2c_WriteReg(pThermData->pI2cClient, reg, &value, 1);
      if (!error) {
         printk("%s Thermal sensor init success, %d attempts, %d errors, %d EIOs\n",
                pThermData->pInstName, attempts, errorCount, eioCount);
         break;
      }

      errorCount++;
      if (error == -5)
         eioCount++;

      if (attempts++ >= 3) {
	      printk("%s Thermal sensor init aborted, %d attempts, %d errors, %d EIOs\n",
                pThermData->pInstName, attempts, errorCount, eioCount);
         pThermData->initFailed = 1;
         break;
      }
      mdelay(1);
   }

   return error;
}

static inline THERM_INST *MySensors(void)
{
   if (IS_ANVIL) {
      return AnvilThermalSensors;
   } else if (IS_AMOEBA) {
      return AmoebaThermalSensors;
   } else {
      return FenwayThermalSensors;
   }
}


static void thermalsensor_Poll(struct work_struct *work)
{
   thermalsensor_GetStatus();
   schedule_delayed_work(&thermal_work, THERM_POLL_TIME);
}

void thermalsensor_InitDriver(struct i2c_adapter *pI2cAdapter)
{
   THERM_DATA *pThermData;
   THERM_INST *pThermalSensor;

   pThermalSensor = MySensors();

   while (pThermalSensor->info != NULL) {

      pThermData = thermalsensor_GetData(pThermalSensor->inst);
      if (!pThermData) {
          pThermalSensor++;
          continue;
      }

      memset(pThermData, 0, sizeof(*pThermData));
      pThermData->pInstName = "unused";

      pThermData->pI2cClient = i2c_new_device(pI2cAdapter, pThermalSensor->info);
      if (pThermData->pI2cClient == NULL) {
         printk("  Error creating %s thermal sensor I2C device\n", pThermalSensor->name);
      }

      pThermData->pInstName = pThermalSensor->name;
      pThermData->instId = pThermalSensor->inst;
      pThermData->type = pThermalSensor->type;
      pThermData->faultTemp = pThermalSensor->faultTemp;
      pThermData->faultStatus = pThermalSensor->faultStatus;
      pThermData->warnTemp = pThermalSensor->warnTemp;
      pThermData->warnStatus = pThermalSensor->warnStatus;
      pThermData->evtSource = pThermalSensor->evtSource;
      pThermData->initFailed = 0;
      pThermData->debugTemp = 0;
      pThermData->readCount = 0;
      pThermData->writeCount = 0;
      pThermData->minTemp = 127;
      pThermData->maxTemp = -128;

      thermalsensor_InitDevice(pThermData);

      pThermalSensor++;
   }

   thermalsensor_proc_init();
}

void thermalsensor_InitEventQueue(struct button_event_queue *beq)
{
   button_event_queue = beq;
   INIT_DELAYED_WORK(&thermal_work, thermalsensor_Poll);
   schedule_delayed_work(&thermal_work, THERM_POLL_TIME);
}

void thermalsensor_ExitDriver()
{
   THERM_DATA *pThermData;
   THERM_INST *pThermalSensor;

   pThermalSensor = MySensors();

   while (pThermalSensor->info != NULL) {
      pThermData = thermalsensor_GetData(pThermalSensor->inst);
      if (pThermData && pThermData->pI2cClient != NULL) {
         i2c_unregister_device(pThermData->pI2cClient);
         pThermData->pI2cClient = NULL;
      }
      pThermalSensor++;
   }

   button_event_queue = NULL;

}

static int ReadTempSensor(THERM_DATA *pThermData, int *pRawTemp)
{
   int   error;
   short rawTemp;
   static int msgs;

   pThermData->readCount++;
   error = i2c_ReadReg(pThermData->pI2cClient, 0, (char *)&rawTemp, 2);
   if (error) {
      if (!pThermData->initFailed) {
         if(msgs++<10)printk("Error reading %s thermal sensor\n", pThermData->pInstName);
         return -EIO;
      }
   }
   else {
      *pRawTemp = rawTemp / 16;
   }

   return error;
}

int thermalSensor_GetTemperature(int inst, int *pCelsiusTemp)
{
   THERM_DATA *pThermData = thermalsensor_GetData(inst);
   int   error;
   int   rawTemp;
   int   celsiusTemp;

   if (!pThermData)
	return -EIO;

   if (pThermData->debugTemp == SEN_ERROR)
      return -EIO;

   error = ReadTempSensor(pThermData, &rawTemp);
   if (error) {
      return error;
   }

   if (pThermData->debugTemp != 0)
      rawTemp = (pThermData->debugTemp&0xff) * 16;

   celsiusTemp = rawTemp / 16;
   if (celsiusTemp & 0x80)
      celsiusTemp |= 0xffffff00;

   if (celsiusTemp > MAX_RANGE)
      return -EINVAL;
   if (celsiusTemp < MIN_RANGE)
      return -EINVAL;

   if (celsiusTemp > pThermData->maxTemp)
      pThermData->maxTemp = celsiusTemp;
   if (celsiusTemp < pThermData->minTemp)
      pThermData->minTemp = celsiusTemp;

   *pCelsiusTemp = celsiusTemp;

   return 0;
}

int thermalSensor_GetAdcTemperature(int inst, int *pRawTemp)
{
   THERM_DATA *pThermData = thermalsensor_GetData(inst);
   int   error;
   short rawTemp;

   if (!pThermData)
      return -EIO;

   pThermData->readCount++;
   error = i2c_ReadReg(pThermData->pI2cClient, 0, (char *)&rawTemp, 2);
   if (error) {
      printk("Error reading %s thermal sensor\n", pThermData->pInstName);
      return -EIO;
   }

   if (rawTemp > MAX_RANGE)
      return -EINVAL;
   if (rawTemp < MIN_RANGE)
      return -EINVAL;

   *pRawTemp = rawTemp;

   return error;
}

int thermalSensor_PrintStatus(char *buf)
{
   THERM_DATA *pThermData;
   THERM_INST *pThermalSensor;
   int temp=-67;
   int error;
   int i=0;
   char *name, namestr[40];;

   pThermalSensor = MySensors();

   while (pThermalSensor->info != NULL) {
      pThermData = thermalsensor_GetData(pThermalSensor->inst);

      name = pThermalSensor->name;
      if (name == NULL) {
         sprintf(namestr, "I2C-addr-%02x", pThermalSensor->info->addr);
         name = namestr;
      }

      if (!pThermData || pThermData->pI2cClient == NULL) {
         i += sprintf(buf + i, "%04x: %s Temperature sensor not present\n",
                      pThermalSensor->info->addr, name);
      }
      else if (pThermData->initFailed) {
         i += sprintf(buf + i, "%04x: %s Temperature sensor init failed\n",
                      pThermalSensor->info->addr, name);
      }
      else {
         if (pThermalSensor->type == THERM_TYPE_ADC) {
            error = thermalSensor_GetAdcTemperature(pThermalSensor->inst, &temp);
            if (error) {
               i += sprintf(buf + i, "%04x: %s Temperature sensor: ERROR %d\n",
                            pThermalSensor->info->addr, name, error);
            }
            else {
               i += sprintf(buf + i, "%04x: 0x%04x raw: %s Temperature sensor ",
                            pThermalSensor->info->addr, temp, pThermalSensor->name);
               if (pThermData->debugTemp != 0) {
                  i += sprintf(buf + i, "(debug %d)", pThermData->debugTemp);
               }
               i += sprintf(buf + i, "\n");
            }
         }
         else {
            error = thermalSensor_GetTemperature(pThermalSensor->inst, &temp);
            if (error == -EIO) {
               i += sprintf(buf + i, "%04x: ERROR: %s Temperature sensor (", pThermalSensor->info->addr, name);
            }
            else if (error == -EINVAL) {
               i += sprintf(buf + i, "%04x: RANGE: %s Temperature sensor (", pThermalSensor->info->addr, name);
            }
            else {
               i += sprintf(buf + i, "%04x: %5d Celsius: %s Temperature sensor (",
                            pThermalSensor->info->addr, temp, pThermalSensor->name);
            }
            if (pThermData->debugTemp != 0)
               i += sprintf(buf + i, "debug %d, ", pThermData->debugTemp);
            i += sprintf(buf + i, "fault %d, warn %d)\n",
                         pThermData->faultTemp, pThermData->warnTemp);

	    i += sprintf(buf + i, "%9u warning polls\n", pThermData->inWarnCount);
	    i += sprintf(buf + i, "%9u warning enters\n", pThermData->enterWarnCount);
	    i += sprintf(buf + i, "%9u fault->warning backoffs\n", pThermData->backoffWarnCount);
	    i += sprintf(buf + i, "%9u fault polls\n", pThermData->inFaultCount);
	    i += sprintf(buf + i, "%9u fault enters\n", pThermData->enterFaultCount);
	    i += sprintf(buf + i, "%9d min temperature\n", pThermData->minTemp);
	    i += sprintf(buf + i, "%9d max temperature\n", pThermData->maxTemp);
	    i += sprintf(buf + i, "\n");

         }
      }
      pThermalSensor++;
   }

   return i;
}

THERM_DATA * thermalsensor_GetDataI2c(short i2cAddr)
{
   THERM_INST *pThermalSensor;

   pThermalSensor = MySensors();
   while (pThermalSensor->info != NULL) {
      if (i2cAddr == pThermalSensor->info->addr) {
         return thermalsensor_GetData(pThermalSensor->inst);
      }
      pThermalSensor++;
   }
   printk("Temperature sensor with I2C addr %X not found\n", i2cAddr);
   return NULL;
}

void thermalsensor_SetDebugTemp(short i2cAddr, long longtemp)
{
   THERM_DATA *pThermData = thermalsensor_GetDataI2c(i2cAddr);

   if (pThermData != NULL) {
	   pThermData->debugTemp = longtemp;
   }
}

void thermalsensor_SetFaultTemp(short i2cAddr, long longtemp)
{
   THERM_DATA *pThermData = thermalsensor_GetDataI2c(i2cAddr);

   if (pThermData != NULL) {
	   if (longtemp == 0) {
		   int temp;
		   if (IS_ANVIL) {
			   temp = ANVIL_CPU_FAULT_TEMPERATURE;
		   } else if (IS_AMOEBA) {
			   temp = AMOEBA_CPU_FAULT_TEMPERATURE;
		   } else {
			   temp = FENWAY_CPU_FAULT_TEMPERATURE;
		   }
		   pThermData->faultTemp = temp;
	   }
	   else {
		   pThermData->faultTemp = longtemp;
	   }
   }
}

void thermalsensor_SetWarnTemp(short i2cAddr, long longtemp)
{
   THERM_DATA *pThermData = thermalsensor_GetDataI2c(i2cAddr);

   if (pThermData != NULL) {
	   if (longtemp == 0) {
		   int temp;
		   if (IS_ANVIL) {
			   temp = ANVIL_CPU_WARN_TEMPERATURE;
		   } else if (IS_AMOEBA) {
			   temp = AMOEBA_CPU_WARN_TEMPERATURE;
		   } else {
			   temp = FENWAY_CPU_WARN_TEMPERATURE;
		   }
		   pThermData->warnTemp = temp;
	   }
	   else {
		   pThermData->warnTemp = longtemp;
	   }
   }
}


void thermalsensor_logStatus(THERM_DATA *pThermData, unsigned long newStatus)
{
	enum HWEVTQ_EventSource hwevtq_source=(enum HWEVTQ_EventSource)pThermData->evtSource;

	if (newStatus == pThermData->warnStatus) {
		if (pThermData->status == pThermData->warnStatus) {
			pThermData->inWarnCount++;
		}
		else if (pThermData->status == pThermData->faultStatus) {
			pThermData->backoffWarnCount++;
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_WARNING);
			printk("%s temperature warning (back off from fault)\n", pThermData->pInstName);
			hwevtq_send_event(hwevtq_source, HWEVTQINFO_TEMP_WARNING);
		}
		else {
			pThermData->enterWarnCount++;
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_WARNING);
			printk("%s temperature warning detected\n", pThermData->pInstName);
			hwevtq_send_event(hwevtq_source, HWEVTQINFO_TEMP_WARNING);
		}
	}
	else if (newStatus == pThermData->faultStatus) {
		if (pThermData->status == pThermData->faultStatus) {
			pThermData->inFaultCount++;
		}
		else {
			pThermData->enterFaultCount++;
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_FAULT);
			printk("%s temperature fault detected\n", pThermData->pInstName);
			hwevtq_send_event(hwevtq_source, HWEVTQINFO_TEMP_FAULT);
		}
	}
	if (newStatus == 0) {
		if (pThermData->status == pThermData->warnStatus) {
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_OK);
			printk("%s temperature warning cleared\n", pThermData->pInstName);
			hwevtq_send_event(hwevtq_source, HWEVTQINFO_TEMP_OK);
		}
		else if (pThermData->status == pThermData->faultStatus) {
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_OK);
			printk("%s temperature fault cleared\n", pThermData->pInstName);
			hwevtq_send_event(hwevtq_source, HWEVTQINFO_TEMP_OK);
		}
	}

	pThermData->status = newStatus;
}

long thermalsensor_GetStatus(void)
{
   THERM_DATA *pThermData;
   THERM_INST *pThermalSensor;
   int error;
   int temp;
   int faultCount=0;
   long status=0;

   pThermalSensor = MySensors();

   while (pThermalSensor->info != NULL) {
      pThermData = thermalsensor_GetData(pThermalSensor->inst);
      if (!pThermData) {
         pThermalSensor++;
         continue;
      }
      error = thermalSensor_GetTemperature(pThermalSensor->inst, &temp);
      if (!error) {
         if ((pThermData->faultTemp != 0) && (temp >= pThermData->faultTemp)) {
            printk("thermalsensor: inst %d, temp %d, fault-temp %d\n",
                   pThermalSensor->inst, temp, pThermData->faultTemp);
            faultCount++;
            status |= pThermData->faultStatus;
         }
         else if ((pThermData->warnTemp != 0) && (temp >= pThermData->warnTemp)) {
            status |= pThermData->warnStatus;
         }
	 thermalsensor_logStatus(pThermData, status);
      }
      pThermalSensor++;
   }

   if (faultCount > 0) {
      audioctl_mute(1);
   }

   return status;
}
