/*
 * Copyright (c) 2010 Sonos Inc.
 * All rights reserved.
 */

#include <asm/io.h>
#include <sysdev/fsl_soc.h>

#define CONFIG_MPC8315
#include <asm/immap_83xx.h>
#include "audioctl.h"
#include "mdp.h"

#define BUS_CLOCK_FREQ  (125 * 1000 * 1000)
#define VCXO_PWM_FREQ   (20 * 1000)
#define VCXO_TIMER_FREQ (VCXO_PWM_FREQ / 2)

#define VCXO_CALC_TMV        ((BUS_CLOCK_FREQ / VCXO_TIMER_FREQ) / 2)

#define MAX_DUTY_CYCLE         1024
#define DUTY_CYCLE_50_PERCENT (MAX_DUTY_CYCLE / 2)

static DEFINE_SPINLOCK(Pwm_Lock);
gtm83xx_t *Gtm83xx=NULL;

unsigned long PwmCurrentDutyCycle=2007;

static void pwm_VcxoCalcTimer2(unsigned long reqDutyCycle)
{
   unsigned long dutyCycle = reqDutyCycle;
   unsigned long skew;
   unsigned long flags;
   unsigned short current, cnr1, cnr2;

   if (dutyCycle > MAX_DUTY_CYCLE) {
      dutyCycle = MAX_DUTY_CYCLE;
   }

   if (dutyCycle < MAX_DUTY_CYCLE) {
      skew = (dutyCycle * VCXO_CALC_TMV) / MAX_DUTY_CYCLE;
   }
   else {
      skew = VCXO_CALC_TMV - 2;
   }
   
	spin_lock_irqsave(&Pwm_Lock, flags);

   current = Gtm83xx->cnr1 + skew;
   if (current < VCXO_CALC_TMV) {
      Gtm83xx->cnr2 = current;
   }
   else {
      Gtm83xx->cnr2 = current - VCXO_CALC_TMV;
   }
   
   cnr1 = Gtm83xx->cnr1;
   cnr2 = Gtm83xx->cnr2;

	spin_unlock_irqrestore(&Pwm_Lock, flags);
}


void pwm_UpdateVcxoDutyCycle(unsigned long reqDutyCycle)
{
   unsigned long flags;


   Gtm83xx->cfr1 = 0;
   Gtm83xx->cfr1 = 0x33;

   Gtm83xx->mdr1 = 0x000A;  
   Gtm83xx->mdr2 = 0x000A;

   Gtm83xx->rfr1 = VCXO_CALC_TMV;
   Gtm83xx->rfr2 = VCXO_CALC_TMV;
   Gtm83xx->psr1 = 0;
   Gtm83xx->psr2 = 0;

   Gtm83xx->cnr1 = 0;
   pwm_VcxoCalcTimer2(reqDutyCycle);

	spin_lock_irqsave(&Pwm_Lock, flags);
   Gtm83xx->cfr1 = 0x11;
	spin_unlock_irqrestore(&Pwm_Lock, flags);

   PwmCurrentDutyCycle = reqDutyCycle;
}

int pwm_InitVcxo(void)
{
   immap_t *immr = (immap_t *)get_immrbase();

   if (Gtm83xx == NULL) {
      Gtm83xx = (gtm83xx_t *)ioremap((unsigned long)(&immr->gtm[0]), 0x100);
   }
   if (Gtm83xx == NULL) {
      printk("Error mapping VCXO GTM registers\n");
      return -1;
   }

   pwm_UpdateVcxoDutyCycle(DUTY_CYCLE_50_PERCENT);
   printk("pwm_InitVcxo: done %08x\n", VCXO_CALC_TMV);
   return 0;
}

void pwm_InitAnvilP3MCLK(void)
{
   immap_t *immr = (immap_t *)get_immrbase();
   if (Gtm83xx == NULL) {
      Gtm83xx = (gtm83xx_t *)ioremap((unsigned long)(&immr->gtm[0]), 0x100);
   }
   Gtm83xx->cfr1=0x00;
   Gtm83xx->cfr1=0x33;
   Gtm83xx->mdr1=0x000A;
   Gtm83xx->rfr1=14;
   Gtm83xx->psr1=0;
   Gtm83xx->cfr1=0x31;
}

int pwm_PrintStatus(char *buf)
{
   int bytes = 0;
   bytes += sprintf(buf+bytes, "PWM: duty-cycle %ld, rfr1 %x, rfr2 %x, cnr1 %x, cnr2 %x\n",
                    PwmCurrentDutyCycle, 
                    Gtm83xx->rfr1, Gtm83xx->rfr1, 
                    Gtm83xx->cnr1, Gtm83xx->cnr2);

   return bytes;
}

