// #define TDM_MAJOR 253 - defined in tdm_ioctl.h
#define TDM_NAME "dspdev"
/*
 * drivers/misc/mpc8xxxtdm.c
 *
 * Copyright (C) 2007-2008 Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * TDM driver for MPC8xxx RDB board.
 * This driver can interface with SLIC device to run VOIP kind of
 * applications.
 *
 * Created by P. V. Suresh <pala@freescale.com>
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write  to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#undef TDMRX

#include <generated/autoconf.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/dma-mapping.h>
#include <linux/of_platform.h>
#include <linux/spinlock.h>
#include <linux/uaccess.h>
#include <linux/math64.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/jiffies.h>
#include <linux/sonos_procfs.h>

#include <sysdev/fsl_soc.h>

#include "tdm_hardware.h"
#include "tdm_shared.h"
#include "mpc8xxxtdm_new.h"
#include "tdm_platform.h"

#include "tdm_ioctl.h"

#include "mdp.h"

#if defined(CONFIG_SONOS_LIMELIGHT)
// Recieve is active for Limelight
#define LLRX
#endif

extern struct manufacturing_data_page sys_mdp;

#define DRV_DESC "Freescale MPC8xxx TDM Driver"
#define DRV_NAME "fsl_8xxx_tdm"

#define PROCFS_DIR         "driver/tdm"
#define PROCFS_STATS_FILE  "stats"
#define PROCFS_REGS_FILE   "regs"
#define PROCFS_RXRING_FILE "rxring"
#define PROCFS_TXRING_FILE "txring"
#define PROCFS_RXSHARED_FILE "rxshared"
#define PROCFS_TXSHARED_FILE "txshared"

struct tdm_priv *mpc8xxx_tdm_priv;

#undef DEBUG_TDM_ERROR_TEST
#ifdef DEBUG_TDM_ERROR_TEST
#define PROCFS_ERROR_FILE "test_error"
struct proc_dir_entry *tdm_error;
#endif

#ifdef CONFIG_SONOS_FENWAY
extern void amoeba_dac_xsmt(int active);
#endif

static struct device_audio_params *dparms;

static int init_devparms(void)
{
#ifdef CONFIG_SONOS_LIMELIGHT
		dparms = &devparms_limelight;
		printk("tdm init Limelight.\n");
#endif
#ifdef CONFIG_SONOS_FENWAY
		if (IS_FENWAY) {
			dparms = &devparms_fenway;
			printk("tdm init Fenway.\n");
		}
		if (IS_ANVIL) {
			dparms = &devparms_anvil;
			printk("tdm init Anvil.\n");
		}
		if (IS_AMOEBA) {
			dparms = &devparms_amoeba;
			printk("tdm init Amoeba.\n");
		}
#endif
		return 1;
}

void tdm_debug_dump_tcd(struct tcd *tcd)
{
	int i;
	for (i = 0; i < WORDS_PER_TCD; i++) {
		printk("tcd[%d] = 0x%08x\n", i, tcd->tcd[i]);
	}

	printk("tcd.saddr           = 0x%08x\n", tcd->tcd[0]);
	printk("tcd.smod            = 0x%02x\n", (tcd->tcd[1] >> 27) & 0x1f);
	printk("tcd.ssize           = 0x%01x\n", (tcd->tcd[1] >> 25) & 0x3);
	printk("tcd.dmod            = 0x%02x\n", (tcd->tcd[1] >> 20) & 0x1f);
	printk("tcd.dsize           = 0x%01x\n", (tcd->tcd[1] >> 16) & 0x3);
	printk("tcd.soff            = 0x%04x\n", tcd->tcd[1] & 0xffff);
	printk("tcd.nbytes          = 0x%08x\n", tcd->tcd[2]);
	printk("tcd.slast           = 0x%08x\n", tcd->tcd[3]);
	printk("tcd.daddr           = 0x%08x\n", tcd->tcd[4]);
	printk("tcd.citer           = 0x%04x\n", (tcd->tcd[5] >> 16) & 0xffff);
	printk("tcd.doff            = 0x%04x\n", tcd->tcd[5] & 0xffff);
	printk("tcd.dlast_sga       = 0x%08x\n", tcd->tcd[6]);
	printk("tcd.biter           = 0x%04x\n", (tcd->tcd[7] >> 16) & 0xffff);
	printk("tcd.control/statuss = 0x%04x\n", tcd->tcd[7] & 0xffff);
}

static int tdm_ring_procfs_read(struct seq_file *m, void *v)
{
	struct tdm_shared *shared = (struct tdm_shared *)m->private;
	int desc, word;

	for (desc = 0; desc < shared->nds; desc++) {
		seq_printf(m, "%02d:", desc);
		for (word = 0; word < 8; word++) {
			seq_printf(m, " %08x", shared->descrs_k[desc].pd[word]);
		}
		seq_printf(m, "\n");
	}

	return 0;
}

static int tdm_shared_procfs_read(struct seq_file *m, void *v)
{
	struct tdm_shared *shared = (struct tdm_shared *)m->private;

	seq_printf(m, "nds:               %u\n", shared->nds);
	seq_printf(m, "dh_active:         %u\n", shared->dh_active);
	seq_printf(m, "dh_consumer:       %u\n", shared->dh_consumer);
	seq_printf(m, "buffer_physlen:    %u\n", shared->buffer_physlen);
	seq_printf(m, "buffer_ll:         %u\n", shared->buffer_ll);

	seq_printf(m, "flags:             0x%x ( %s%s)\n", shared->flags,
		   (shared->flags & TDM_SHARED_FLAG_RUNNING) ? "RUNNING " : "",
		   (shared->flags & TDM_SHARED_FLAG_OVERRUN) ? "OVERRUN " : "");

	seq_printf(m, "framelen:          %u\n", shared->framelen);
	seq_printf(m, "stridelen:         %u\n", shared->stridelen);
	seq_printf(m, "clocks_per_buffer: %u\n", shared->clocks_per_buffer);
	seq_printf(m, "polltime:          %llu\n", shared->polltime);
	seq_printf(m, "latency:           %u\n", shared->latency);
	seq_printf(m, "underflows:        %u\n", shared->underflows);
	seq_printf(m, "buffer_ll_min:     %u\n", shared->buffer_ll_min);
	seq_printf(m, "buffer_ll_max:     %u\n", shared->buffer_ll_max);
	seq_printf(m, "buffer_ll_default: %u\n", shared->buffer_ll_default);
	seq_printf(m, "u_cursor:          %u\n", shared->u_cursor);
	seq_printf(m, "u_dno:             %u\n", shared->u_dno);
	seq_printf(m, "u_ftime:           %llu\n", shared->u_ftime);

	seq_printf(m, "error:             0x%x ( %s)\n", shared->error,
		   (shared->error & TDM_ERROR_STOPPED) ? "STOPPED " : "");

	return 0;
}

static int tdm_regs_procfs_read(struct seq_file *m, void *v)
{
	struct tdm_priv *priv=mpc8xxx_tdm_priv;
	int num, num2;

	seq_printf(m, "tdmgir:    %08x\n", priv->tdm_regs->gir);
	seq_printf(m, "tdmrir:    %08x\n", priv->tdm_regs->rir);
	seq_printf(m, "tdmtir:    %08x\n", priv->tdm_regs->tir);
	seq_printf(m, "tdmrfp:    %08x\n", priv->tdm_regs->rfp);
	seq_printf(m, "tdmtfp:    %08x\n", priv->tdm_regs->tfp);
	seq_printf(m, "tdmrfp:    %08x\n", priv->tdm_regs->rfp);

	for (num = 0; num < 4; num++)
	{
		seq_printf(m, "tdmrcen%d:  %08x\n", num, priv->tdm_regs->rcen[num]);
		seq_printf(m, "tdmtcen%d:  %08x\n", num, priv->tdm_regs->tcen[num]);
		seq_printf(m, "tdmtcma%d:  %08x\n", num, priv->tdm_regs->tcma[num]);
	}

	seq_printf(m, "tdmrcr:    %08x\n", priv->tdm_regs->rcr);
	seq_printf(m, "tdmtcr:    %08x\n", priv->tdm_regs->tcr);
	seq_printf(m, "tdmrier:   %08x\n", priv->tdm_regs->rier);
	seq_printf(m, "tdmtier:   %08x\n", priv->tdm_regs->tier);
	seq_printf(m, "tdmrer:    %08x\n", priv->tdm_regs->rer);
	seq_printf(m, "tdmter:    %08x\n", priv->tdm_regs->ter);
	seq_printf(m, "tdmrsr:    %08x\n", priv->tdm_regs->rsr);
	seq_printf(m, "tdmtsr:    %08x\n", priv->tdm_regs->tsr);

	seq_printf(m, "dmacr:     %08x\n", priv->dmac_regs->dmacr);
	seq_printf(m, "dmaes:     %08x\n", priv->dmac_regs->dmaes);
#if defined(CONFIG_SONOS_LIMELIGHT)
	seq_printf(m, "dmaerqh:   %08x\n", priv->dmac_regs->dmaerqh);
#endif
	seq_printf(m, "dmaerql:   %08x\n", priv->dmac_regs->dmaerql);
#if defined(CONFIG_SONOS_LIMELIGHT)
	seq_printf(m, "dmaeeih:   %08x\n", priv->dmac_regs->dmaeeih);
#endif
	seq_printf(m, "dmaeeil:   %08x\n", priv->dmac_regs->dmaeeil);
	seq_printf(m, "dmaserq:   %02x\n", priv->dmac_regs->dmaserq);
	seq_printf(m, "dmacerq:   %02x\n", priv->dmac_regs->dmacerq);
	seq_printf(m, "dmaseei:   %02x\n", priv->dmac_regs->dmaseei);
	seq_printf(m, "dmaceei:   %02x\n", priv->dmac_regs->dmaceei);
	seq_printf(m, "dmacint:   %02x\n", priv->dmac_regs->dmacint);
	seq_printf(m, "dmacerr:   %02x\n", priv->dmac_regs->dmacerr);
	seq_printf(m, "dmassrt:   %02x\n", priv->dmac_regs->dmassrt);
	seq_printf(m, "dmacdne:   %02x\n", priv->dmac_regs->dmacdne);
#if defined(CONFIG_SONOS_LIMELIGHT)
	seq_printf(m, "dmainth:   %08x\n", priv->dmac_regs->dmainth);
#endif
	seq_printf(m, "dmaintl:   %08x\n", priv->dmac_regs->dmaintl);
#if defined(CONFIG_SONOS_LIMELIGHT)
	seq_printf(m, "dmaerrh:   %08x\n", priv->dmac_regs->dmaerrh);
#endif
	seq_printf(m, "dmaerrl:   %08x\n", priv->dmac_regs->dmaerrl);
#if defined(CONFIG_SONOS_LIMELIGHT)
	seq_printf(m, "dmahrsh:   %08x\n", priv->dmac_regs->dmahrsh);
#endif
	seq_printf(m, "dmahrsl:   %08x\n", priv->dmac_regs->dmahrsl);
	seq_printf(m, "dmagpor:   %08x\n", priv->dmac_regs->dmagpor);

	for (num = 0; num < 2; num++)
	{
		seq_printf(m, "dchpri%d:   %02x\n", num, priv->dmac_regs->dchpri[num]);

		seq_printf(m, "tcd%d:", num);
		for (num2 = 0; num2 < WORDS_PER_TCD; num2++)
		{
			if ((num2 % 4) == 0)
			{
				seq_printf(m, "\n          ");
			}

			seq_printf(m, " 0x%08x", priv->dmac_regs->tcd[num].tcd[num2]);
		}
		seq_printf(m, "\n");
	}

	return 0;
}

static void tdm_stats_calc_rate(struct tdm_priv *priv)
{
	unsigned long j = jiffies;

	// Only calculate after 1 sec has elapsed
	if (time_after_eq(j, priv->stats.last_sample + HZ)) {
		// If the rate hasn't been updated in 2 sec assume the rate is 0 because the DMAC has stopped
		if (time_after_eq(j, priv->stats.last_sample + (2*HZ))) {
			priv->stats.tx_bps = 0;
			priv->stats.rx_bps = 0;
		}
		else {
			priv->stats.tx_bps = priv->stats.tx_bytes - priv->stats.last_tx_bytes;
			priv->stats.rx_bps = priv->stats.rx_bytes - priv->stats.last_rx_bytes;
		}

		priv->stats.last_tx_bytes = priv->stats.tx_bytes;
		priv->stats.last_rx_bytes = priv->stats.rx_bytes;
		priv->stats.last_sample = j;
	}
}

static void tdm_stats_reset(struct tdm_priv *priv)
{
	// Zero out all statistics
	memset(&(priv->stats), 0, sizeof(priv->stats));

	// Set initial timestamp
	priv->stats.last_sample = jiffies;
}

static int tdm_stats_procfs_read(struct seq_file *m, void *v)
{
	struct tdm_priv *priv=mpc8xxx_tdm_priv;

	// Double check the rate in case the DMAC done isr stopped getting called
	tdm_stats_calc_rate(priv);

	seq_printf(m, "Tx Bytes:          %12llu\n", priv->stats.tx_bytes);
	seq_printf(m, "Tx Bytes/Sec:      %12d\n", priv->stats.tx_bps);
	seq_printf(m, "Tx FIFO Underruns: %12d\n", priv->stats.tx_fifo_under_err);
	seq_printf(m, "Tx Sync Errors:    %12d\n", priv->stats.tx_sync_err);
#ifdef LLRX
	seq_printf(m, "Rx Bytes:          %12llu\n", priv->stats.rx_bytes);
	seq_printf(m, "Rx Bytes/Sec:      %12d\n", priv->stats.rx_bps);
	seq_printf(m, "Rx FIFO Overruns:  %12d\n", priv->stats.rx_fifo_over_err);
	seq_printf(m, "Rx Sync Errors:    %12d\n", priv->stats.rx_sync_err);
	seq_printf(m, "Rx Status:         %12s\n", (priv->rxshared->flags & TDM_SHARED_FLAG_RUNNING) ? "running" : "stopped");
#endif // LLRX
	seq_printf(m, "DMAC Errors:       %12d\n", priv->stats.dmac_err);

	return 0;
}

static ssize_t tdm_stats_procfs_write(struct file *file, const char __user *buffer,
			 size_t count, loff_t *data)
{
	// Reset statistics if the file is written to
	tdm_stats_reset(mpc8xxx_tdm_priv);
	return count;
}

#ifdef DEBUG_TDM_ERROR_TEST
ssize_t tdm_error_procfs_write(struct file *file, const char __user *buffer,
			unsigned long count, loff_t *data)
{
	struct tdm_priv *priv=mpc8xxx_tdm_priv;
	char opt;

	if (copy_from_user(&opt, buffer, 1))
		return -EFAULT;

	switch (opt)
	{
		case 'T':
			printk(KERN_INFO "Injecting TDM error\n");
			// Disable all TDM DMA
			out_be32(&(priv->dmac_regs->dmaerql), 0);
			break;
		case 'D':
			printk(KERN_INFO "Injecting TDM DMAC error\n");
			// Write garbage into the DMAC TCDs
			memset(&(priv->dmac_regs->tcd), 0xcd, (2 * WORDS_PER_TCD));
			break;
		case '0':
			tdm_debug_dump_tcd(&(priv->dmac_regs->tcd[0]));
			break;
		default:
			break;
	}

	return count;
}

static int tdm_error_procfs_read(struct seq_file *m, void *v)
{
	return -EPERM;
}
#endif // DEBUG_TDM_ERROR_TEST


SONOS_PROCFS_INIT(tdm_regs_procfs_read, NULL, tdm_regs_procfs_fops);
SONOS_PROCFS_INIT(tdm_ring_procfs_read, NULL, tdm_ring_procfs_fops);
SONOS_PROCFS_INIT(tdm_shared_procfs_read, NULL, tdm_shared_procfs_fops);
SONOS_PROCFS_INIT(tdm_stats_procfs_read, tdm_stats_procfs_write, tdm_stats_procfs_fops);
#ifdef DEBUG_TDM_ERROR_TEST
SONOS_PROCFS_INIT(tdm_error_procfs_read, tdm_error_procfs_write, tdm_error_procfs_fops);
#endif

static void tdm_procfs_setup(struct tdm_priv *priv)
{
	tdm_stats_reset(priv);

	// Setup procfs dir
	priv->ftdm_dir = proc_mkdir(PROCFS_DIR, NULL);
	if (!priv->ftdm_dir) {
		printk(KERN_ERR "failed to create procfs dir %s/n", PROCFS_DIR);
		return;
	}

	// Setup procfs stats file
	priv->fstats = proc_create_data(PROCFS_STATS_FILE, (SONOS_PROCFS_PERM_READ | SONOS_PROCFS_PERM_WRITE), priv->ftdm_dir, &tdm_stats_procfs_fops, NULL);
	if (!priv->fstats) {
		printk(KERN_ERR "failed to create procfs file %s\n", PROCFS_STATS_FILE);
		return;
	}

	// Setup procfs regs file
	priv->fregs = proc_create_data(PROCFS_REGS_FILE, SONOS_PROCFS_PERM_READ, priv->ftdm_dir, &tdm_regs_procfs_fops, NULL);
	if (!priv->fregs) {
		printk(KERN_ERR "failed to create procfs file %s\n", PROCFS_REGS_FILE);
		return;
	}

#ifdef LLRX
	// Setup procfs rxring file
	priv->frxring = proc_create_data(PROCFS_RXRING_FILE, SONOS_PROCFS_PERM_READ, priv->ftdm_dir, &tdm_ring_procfs_fops, priv->rxshared);
	if (!priv->frxring) {
		printk(KERN_ERR "failed to create procfs file %s\n", PROCFS_RXRING_FILE);
		return;
	}

	// Setup procfs rxshared file
	priv->frxshared = proc_create_data(PROCFS_RXSHARED_FILE, SONOS_PROCFS_PERM_READ, priv->ftdm_dir, &tdm_shared_procfs_fops, priv->rxshared);
	if (!priv->frxshared) {
		printk(KERN_ERR "failed to create procfs file %s\n", PROCFS_RXSHARED_FILE);
		return;
	}
#endif // LLRX

	// Setup procfs txring file
	priv->ftxring = proc_create_data(PROCFS_TXRING_FILE, SONOS_PROCFS_PERM_READ, priv->ftdm_dir, &tdm_ring_procfs_fops, priv->txshared);
	if (!priv->ftxring) {
		printk(KERN_ERR "failed to create procfs file %s\n", PROCFS_TXRING_FILE);
		return;
	}

	// Setup procfs txshared file
	priv->ftxshared = proc_create_data(PROCFS_TXSHARED_FILE, SONOS_PROCFS_PERM_READ, priv->ftdm_dir, &tdm_shared_procfs_fops, priv->txshared);
	if (!priv->ftxshared) {
		printk(KERN_ERR "failed to create procfs file %s\n", PROCFS_TXSHARED_FILE);
		return;
	}

#ifdef DEBUG_TDM_ERROR_TEST
	// Setup procfs error injector
	tdm_error = proc_create_data(PROCFS_ERROR_FILE, (SONOS_PROCFS_PERM_READ | SONOS_PROCFS_PERM_WRITE), priv->ftdm_dir, &tdm_error_procfs_fops, NULL);
	if (!tdm_error) {
		printk(KERN_ERR "failed to create procfs file %s\n", PROCFS_ERROR_FILE);
		return;
	}
#endif // DEBUG_TDM_ERROR_TEST

	return;
}

#undef DESCR_DEBUG
inline int paddr_to_descrno(struct tdm_shared *s, dma_addr_t p)
{
	int i;
#ifdef DESCR_DEBUG
	if (p<s->descrs_p) { printk("paddr_to_descrno: %08x is before the base pointer\n",p);}
#endif
	p -= s->descrs_p;
#ifdef DESCR_DEBUG
	if (p % sizeof(struct tdm_descr)) { printk("paddr_to_descrno: offset %u is not properly aligned\n",p);}
#endif
	i = (p/sizeof(struct tdm_descr));
#ifdef DESCR_DEBUG
	if (i >= s->nds) { printk("paddr_to_descrno: %d is not a valid descriptor number\n",i);}
#endif
	return i;
}

#ifdef LLRX
static inline int get_last_completed_rx_descrno(struct tdm_priv *priv)
{
	u32 x;
	int a;
	// Ethan: Get the address of the next TCD and convert it to descriptor number
	x=in_be32(&priv->dmac_regs->tcd[TDMRX_DMA_CH].tcd[6]);
	a=paddr_to_descrno(priv->rxshared,x);
	// Ethan: Since a = current + 1 then last = a - 2
	a += priv->rxshared->nds;
	a -= 2;
	a = a % priv->rxshared->nds;
	return a;
}
#endif

static inline int get_last_completed_tx_descrno(struct tdm_priv *priv)
{
	u32 x;
	int a;
	// Ethan: Get the address of the next TCD and convert it to descriptor number
	x=in_be32(&priv->dmac_regs->tcd[TDMTX_DMA_CH].tcd[6]);
	a=paddr_to_descrno(priv->txshared,x);
	// Ethan: Since a = current + 1 then last = a - 2
	a += priv->txshared->nds;
	a -= 2;
	a = a % priv->txshared->nds;
	return a;
}

#ifdef LLRX
void service_rx_ring(struct tdm_priv *priv)
{
	int a,b;
	u32 overflow_idx;

	// If the producer gets here there is an overrun
	overflow_idx = priv->rxshared->dh_consumer + priv->rxshared->nds - 1;
	overflow_idx %= priv->rxshared->nds;

	a=get_last_completed_rx_descrno(priv);
	// The producer, dh_active, is the lastest received buffer
	while (priv->rxshared->dh_active!=a) {
		// Ethan: Go to the next unserviced descriptor
		b = (priv->rxshared->dh_active) + 1;
		b = b % priv->rxshared->nds;
		// The number of clocks is CITER in the descriptor
		// note: this assumes that frame length and stride length are the same
		priv->rxshared->descrs_k[b].buf_clocks = DMA_TCD5_CITER_GET(priv->rxshared->descrs_k[b].pd[5]);
		// Reset the CITER/BITER in the descriptor back to default
		// This is done here to make sure a single long buffer is reset to default
		// even if the program fails.
		priv->rxshared->descrs_k[b].pd[5] = DMA_TCD5_DOFF(priv->rxshared->stridelen) |
			DMA_TCD5_CITER_DISABLE_LINK(priv->rxshared->buffer_ll_default/priv->rxshared->stridelen);
		priv->rxshared->descrs_k[b].pd[7] = DMA_TCD7_E_SG | DMA_TCD7_INT_MAJ |
			DMA_TCD7_BITER_DISABLE_LINK(priv->rxshared->buffer_ll_default/priv->rxshared->stridelen);
		// The dsc is the dsc from the previous buffer + the number of clocks in the previous buffer
		priv->rxshared->descrs_k[b].dsc = priv->rxshared->descrs_k[priv->rxshared->dh_active].dsc +
			priv->rxshared->descrs_k[priv->rxshared->dh_active].buf_clocks;

		if (b == overflow_idx) {
		    // The overflow is cleared in a startup sequence.
		    priv->rxshared->flags |= TDM_SHARED_FLAG_OVERRUN;
		}

		priv->stats.rx_bytes += priv->rxshared->descrs_k[b].buf_ll;
		priv->rxshared->dh_active = b;
	}
}
#endif

void service_tx_ring(struct tdm_priv *priv)
{
	int a,b,x;
	a=get_last_completed_tx_descrno(priv);
	if (priv->txshared->dh_active==a) return;
	x=0;
	// Ethan: Loop until the active descriptor is the last completed descriptor
	while (priv->txshared->dh_active!=a) {
		// Ethan: Go to the next unserviced descriptor
		b = (priv->txshared->dh_active) + 1;
		b = b % priv->txshared->nds;
		// Ethan: If the buffer is already silence then userspace didn't commit audio data in time.
		// This is considered an underflow however it may have been intentional (not playing).
		if (priv->fwriter && (priv->txshared->descrs_k[b].buf==0)) priv->txshared->underflows++;
		// reset the buffer to the zero buffer so if this doesn't get filled
		// from the app by the next time it comes around, we'll play silence
		priv->txshared->descrs_k[b].buf = 0;

		// Following lines valid only for fixed size buffers
		// Ethan: Set the starting frame for the descriptor
		priv->txshared->descrs_k[b].dsc = priv->txshared->descrs_k[priv->txshared->dh_active].dsc + priv->txshared->clocks_per_buffer;
		// Ethan: Set the descriptor number for the descriptor
		priv->txshared->descrs_k[b].dno = priv->txshared->descrs_k[priv->txshared->dh_active].dno + 1;
		// Ethan: Set the source address for the descriptor (silence buffer)
		priv->txshared->descrs_k[b].pd[0] = (u32)(priv->txshared->buffers_p + priv->txshared->descrs_k[b].buf);

		priv->stats.tx_bytes += priv->txshared->descrs_k[b].buf_ll;
		priv->txshared->dh_active = b;
		x++;
	}
	// Ethan: Record the last descriptor number serviced
	priv->txsdno=priv->txshared->descrs_k[a].dno;
}

static inline void _tcd_init(struct tdm_priv *priv,struct tdm_shared *shared,int rx);

#ifdef LLRX
static void rx_tcd_init(struct tdm_priv *priv)
{
	_tcd_init(priv,priv->rxshared,1);
}
#endif

static void tx_tcd_init(struct tdm_priv *priv)
{
	_tcd_init(priv,priv->txshared,0);
}

/* Initialize the Transmit Control Discriptor parameters*/
static inline void _tcd_init(struct tdm_priv *priv,struct tdm_shared *shared,int rx)
{
	int i;
	u32 iter;

	/*
	   iter = number of frames * number_of_channels/frame
	   * numberof_bytes/channel * Ntx/8bytes
	 */

	for (i = 0; i < shared->nds; i++) {
		iter = shared->descrs_k[i].buf_ll / shared->stridelen;
		/*
		 * Ethan:
		 * Rx - Set source address to the Rx data register
		 * Tx - Set source address to the corresponding descriptor buffer address (currently silence)
		 */
		if (rx) {
			shared->descrs_k[i].pd[0] = TDM_RDR_OFFSET + priv->ptdm_base;
		}
		else {
			shared->descrs_k[i].pd[0] = (u32)(shared->buffers_p + shared->descrs_k[i].buf);
		}

		/*
		 * Ethan:
		 * Rx - ssize=dsize=64bit, soff=smod=dmod=0
		 * Tx - ssize=dsize=64bit, soff=stride length, smod=dmod=0
		 */
		if (rx) {
			shared->descrs_k[i].pd[1] = DMA_TCD1_SSIZE(SSIZE_64BITS) | DMA_TCD1_DSIZE(SSIZE_64BITS);
		}
		else {
			shared->descrs_k[i].pd[1] = DMA_TCD1_SSIZE(SSIZE_64BITS) | DMA_TCD1_DSIZE(SSIZE_64BITS) | DMA_TCD1_SOFF(shared->stridelen);
		}

		/* number of bytes for minor loop, wide fifo 8bytes for dma */
		shared->descrs_k[i].pd[2] = shared->stridelen;

		/* slast = 0 */
		shared->descrs_k[i].pd[3] = 0;

		/*
		 * Ethan:
		 * Rx - Set destination address to the corresponding descriptor buffer address
		 * Tx - Set destination address to the Tx data register
		 */
		if (rx) {
			shared->descrs_k[i].pd[4] = (u32)(shared->buffers_p + shared->descrs_k[i].buf);
		}
		else {
			shared->descrs_k[i].pd[4] = TDM_TDR_OFFSET + priv->ptdm_base;
		}

		/*
		   ch linking=0, doff=inc dadr by 8,
		   citer = no of transfers for frame
		 */
		if (rx) {
			shared->descrs_k[i].pd[5] = DMA_TCD5_DOFF(shared->stridelen) | DMA_TCD5_CITER_DISABLE_LINK(iter);
		}
		else {
			shared->descrs_k[i].pd[5] = DMA_TCD5_CITER_DISABLE_LINK(iter);
		}

		/* enable scatter-gather, interrupt on end of frame, */
		shared->descrs_k[i].pd[7] = DMA_TCD7_BITER_DISABLE_LINK(iter) | DMA_TCD7_E_SG | DMA_TCD7_INT_MAJ;

		// Ethan: Link this buffer descriptor to the next in the chain, if this is the last descriptor then set the next to the first (making a ring)
		if (i==(shared->nds - 1)) {
			shared->descrs_k[i].pd[6]=((u32)(shared->descrs_p));
		}
		else {
			shared->descrs_k[i].pd[6]=((u32)((shared->descrs_p)+((i+1)*sizeof(struct tdm_descr))));
		}
	}
}

static irqreturn_t dmac_err_isr(int irq, void *p)
{
	u32 err;
	u8 ch;
	struct tdm_priv *priv = p;

	err = in_be32(&priv->dmac_regs->dmaes);
	if (!(err & DMAES_VLD))
		return IRQ_NONE;

	ch = DMAES_ERRCHN(err);
	dev_err(priv->device, "TDM DMAC error on channel %d\n", ch);

	if (err & DMAES_CPE)
		dev_err(priv->device, "TDM DMAC::Channel priority error\n");
	if (err & DMAES_GPE)
		dev_err(priv->device, "TDM DMAC::Group priority error\n");
	if (err & DMAES_SAE)
		dev_err(priv->device, "TDM DMAC::Source address error\n");
	if (err & DMAES_SOE)
		dev_err(priv->device, "TDM DMAC::Source offset error\n");
	if (err & DMAES_DAE)
		dev_err(priv->device, "TDM DMAC::Destination address error\n");
	if (err & DMAES_DOE)
		dev_err(priv->device, "TDM DMAC::Destination offset error\n");
	if (err & DMAES_NCE)
		dev_err(priv->device, "TDM DMAC::Nbytes/citer error\n");
	if (err & DMAES_SGE)
		dev_err(priv->device, "TDM DMAC::Scatter/gather error\n");
	if (err & DMAES_DBE)
		dev_err(priv->device, "TDM DMAC::Destination bus error \n");
	if (err & DMAES_SBE)
		dev_err(priv->device, "TDM DMAC::Source bus error \n");

#ifdef LLRX
	if ((ch == TDMRX_DMA_CH ) || (ch == TDMTX_DMA_CH)) {
#else // LLRX
	if (ch == TDMTX_DMA_CH) {
#endif // LLRX
		tdm_debug_dump_tcd(&(priv->dmac_regs->tcd[ch]));
	}

	priv->stats.dmac_err++;

	/* Clear the error */
	out_8(&priv->dmac_regs->dmacerr, ch);
	return IRQ_HANDLED;
}

static irqreturn_t tdm_err_isr(int irq, void *p)
{
	int ret = IRQ_NONE;
	u32 status, mask, val;
	struct tdm_priv *priv = p;

	/* transmit errors */
	status = in_be32(&priv->tdm_regs->ter);
	mask = in_be32(&priv->tdm_regs->tier);
	val = status & mask;
	out_be32(&priv->tdm_regs->ter, val);

	/* Transmit under Run error */
	if (val & TER_TUE) {
		dev_err(priv->device, "TDM::Transmit Under Run error \n");
		priv->stats.tx_fifo_under_err++;
	}

	/* Transmit Sync Error */
	if (val & TER_TSEE) {
		dev_err(priv->device, "TDM::Transmit Sync error \n");
		priv->stats.tx_sync_err++;
	}

	if (val)
		ret = IRQ_HANDLED;

	/* receive errors */
	status = in_be32(&priv->tdm_regs->rer);
	mask = in_be32(&priv->tdm_regs->rier);
	val = status & mask;
	out_be32(&priv->tdm_regs->rer, val);

	/* Receiver Over run error */
	if (val & RER_ROE) {
		dev_err(priv->device, "TDM::Receive Over Run error \n");
		priv->stats.rx_fifo_over_err++;
	}

	/* Receive Sync Error  */
	if (val & RER_RSEE) {
		dev_err(priv->device, "TDM::Receive Sync error \n");
		priv->stats.rx_sync_err++;
	}

	if (val)
		ret = IRQ_HANDLED;

	return ret;
}

static irqreturn_t dmac_done_isr(int irq, void *p)
{
	u32 ch, err;
	int ret = IRQ_NONE;
	struct tdm_priv *priv = p;

	spin_lock(&priv->tdmlock);

	err = in_be32(&priv->dmac_regs->dmaes);

#if defined(CONFIG_SONOS_LIMELIGHT)
	if (err & DMAES_VLD) return dmac_err_isr(irq, p);
#endif

	ch = in_be32(&priv->dmac_regs->dmaintl);

	/* clear interrupt */
#ifdef LLRX
	if (ch & DMAC_RX_INT) {
		unsigned long flags;
		out_8(&priv->dmac_regs->dmacint, TDMRX_DMA_CH);
		ret = IRQ_HANDLED;
		local_irq_save(flags);
		service_rx_ring(priv);
		local_irq_restore(flags);
		wake_up_interruptible(&mpc8xxx_tdm_priv->rd_wakeup_event);
	}
#endif
	if (ch & DMAC_TX_INT) {
		unsigned long flags;
		out_8(&priv->dmac_regs->dmacint, TDMTX_DMA_CH);
		ret = IRQ_HANDLED;
		local_irq_save(flags);
		service_tx_ring(priv);
		local_irq_restore(flags);
		wake_up_interruptible(&mpc8xxx_tdm_priv->wr_wakeup_event);
	}

	tdm_stats_calc_rate(priv);

	spin_unlock(&priv->tdmlock);
	return ret;
}

/* TDM register programming */
static int mpc8xxx_tdm_reg_init(struct tdm_priv *priv)
{
	int i;
	__be32 __iomem *psccr;
#ifdef CONFIG_SONOS_FENWAY
	__be32 __iomem *psicr;
#endif
	phys_addr_t base = get_immrbase();

#if defined(CONFIG_SONOS_LIMELIGHT)
	// Ethan: Configures pin mux for input from FPGA?
	psccr = ioremap(base, 0xF0000);
	*(psccr + P1014_PMUXCR2) = (*(psccr + P1014_PMUXCR2) & P1014_PMUX_TDMCLKMASK) | P1014_PMUX_TDMCLKEN;
	*(psccr + P1014_PMUXCR1) = (*(psccr + P1014_PMUXCR1) & P1014_PMUX_TDMENMASK) | P1014_PMUX_TDMEN;
	*(psccr + P1014_DEVDISR1) &= ~0x20000000;
	iounmap(psccr);
#endif
#ifdef CONFIG_SONOS_FENWAY
	psccr = ioremap(base + SCCR_OFFSET, 4);
	if (!psccr)
		return -1;
	/* CSB:TDM clk =1 */
	clrsetbits_be32(psccr, SCCR_TDM_MASK, TDM_CM_01);
	iounmap(psccr);

	psicr = ioremap(base + SICRL_OFFSET, 4);
	if (!psicr)
		return -1;
	/* enable TDM in SICR */
	clrbits32(psicr, SICRL_TDM_MASK);
	iounmap(psicr);
#endif

	/* channel/group round robin */
	out_be32(&priv->dmac_regs->dmacr, DMACR_ERGA | DMACR_ERCA);
	/* Enable error Interrupts for TDM Rx &Tx */
	out_8(&priv->dmac_regs->dmaseei, TDMTX_DMA_CH);
	out_be32(&priv->dmac_regs->dmagpor, DMAGPOR_SNOOP);

	tx_tcd_init(priv);
#ifdef LLRX
	rx_tcd_init(priv);
#endif

	out_be32(&priv->tdm_regs->gir, 0);

	out_be32(&priv->clk_regs->tx, 480); // For Anvil, irrelevant for others
					    // Ethan: What?
	out_be32(&priv->clk_regs->rx, 1);

	/*
	   Rx Water mark 0,  FIFO enable,  Wide fifo, DMA enable for RX,
	   Receive Sync out, syncwidth = ch width, Rx clk out,zero sync,
	   falling edge , data order
	 */

#ifdef LLRX
	out_be32(&priv->tdm_regs->rir, dparms->rir);
#endif
	out_be32(&priv->tdm_regs->tir, dparms->tir);

	/* no of channels ,Channel size-coding */
#ifdef LLRX
	out_be32(&priv->tdm_regs->rfp,
		 dparms->rfp);
#endif
	out_be32(&priv->tdm_regs->tfp,
		 dparms->tfp);
#ifdef LLRX
	out_be32(&priv->tdm_regs->rer, RER_RSEE|RER_ROE);
#endif
	// Ethan: Setup tx events and interrupts
	/*out_be32(&priv->tdm_regs->rier, RIER_RSEEE|RIER_ROEE);*/
	out_be32(&priv->tdm_regs->ter, TER_TSEE|TER_TUE);
	out_be32(&priv->tdm_regs->tier, TIER_TSEEE|TIER_TUEE);

	/* clear all receive and transmit chs */
	for (i = 0; i < 4; i++) {
		out_be32(&priv->tdm_regs->tcma[i], 0);
		out_be32(&priv->tdm_regs->tcen[i], 0);
#ifdef LLRX
		out_be32(&priv->tdm_regs->rcen[i], 0);
#endif
	}

	return 0;
}

static int mpc8xxx_tdm_enable(struct tdm_priv *priv)
{
	int i;
	unsigned long timeout;

	for (i = 0; i < 4; i++) {
#ifdef LLRX
		out_be32(&priv->tdm_regs->rcen[i], dparms->rcen[i]);
#endif
		out_be32(&priv->tdm_regs->tcen[i], dparms->tcen[i]);
		out_be32(&priv->tdm_regs->tcma[i], dparms->tcma[i]);
	}

	/* Clear the DONE bit */
#ifdef LLRX
	out_8(&priv->dmac_regs->dmacdne, TDMRX_DMA_CH);
#endif
	out_8(&priv->dmac_regs->dmacdne, TDMTX_DMA_CH);

	/* Load the Tx  transfer control descriptors */
	for (i = 0; i < 8; i++)
		out_be32(&priv->dmac_regs->tcd[TDMTX_DMA_CH].tcd[i],
			 priv->txshared->descrs_k[0].pd[i]);

#ifdef LLRX
	/* Load the Rx  transfer control descriptors */
	for (i = 0; i < 8; i++)
		out_be32(&priv->dmac_regs->tcd[TDMRX_DMA_CH].tcd[i],
			 priv->rxshared->descrs_k[0].pd[i]);
#endif

	/* enable the dma request */
	out_8(&priv->dmac_regs->dmaserq, TDMTX_DMA_CH);

	/* Enable Receiver, transmitter */
	timeout = jiffies + TDM_ENABLE_TIMEOUT;
	out_be32(&priv->tdm_regs->tcr, TCR_TEN);
	while (!(in_be32(&priv->tdm_regs->tsr) & TSR_TENS)) {
		if (time_after(jiffies, timeout)) {
			dev_err(priv->device, "timeout to enable TDM Tx\n");
			return -ETIMEDOUT;
		}
		cpu_relax();
	}

#ifdef CONFIG_SONOS_FENWAY
	if (IS_AMOEBA) {
		mdelay(10);
		amoeba_dac_xsmt(0); /* unmute the DAC */
	}
#endif
	return 0;
}

#ifdef LLRX
void _mpc8xxx_tdm_handle_rxstop(void);
void mpc8xxx_tdm_handle_rxstop(void)
{
	unsigned long flags;
	struct tdm_priv *priv=mpc8xxx_tdm_priv;
	spin_lock_irqsave(&priv->tdmlock,flags);
	_mpc8xxx_tdm_handle_rxstop();
	spin_unlock_irqrestore(&priv->tdmlock,flags);
}

void _mpc8xxx_tdm_handle_rxstop(void)
{
	struct tdm_priv *priv=mpc8xxx_tdm_priv;
	if (!(priv->rxshared->flags & TDM_SHARED_FLAG_RUNNING)) return; //Do nothing if already stopped
	out_be32(&priv->tdm_regs->rcr, 0); // Stop the receiver
	priv->rxshared->flags &= ~TDM_SHARED_FLAG_RUNNING;
	priv->rxshared->error = TDM_ERROR_STOPPED;
	// Ethan: Wake up any threads polling on the Rx window
	wake_up_interruptible(&mpc8xxx_tdm_priv->rd_wakeup_event);

}

void mpc8xxx_tdm_handle_rxstart(void)
{
	unsigned long flags;
	struct tdm_priv *priv=mpc8xxx_tdm_priv;
	int i,a;
	u32 x;

	spin_lock_irqsave(&priv->tdmlock,flags);
	if (priv->rxshared->flags & TDM_SHARED_FLAG_RUNNING) _mpc8xxx_tdm_handle_rxstop(); //If starting while running, restart

	out_be32(&priv->tdm_regs->rcr, 0); // Stop the receiver
	i=0;
	while ((in_be32(&priv->tdm_regs->rsr)) & RSR_RENS) {
		if (i>1000000) {
			dev_err(priv->device,"mpc8xxx_tdm_handle_rxstart: receiver did not disable, giving up\n");
			spin_unlock_irqrestore(&priv->tdmlock,flags);
			return; //XXX if we got here we really can't proceed with the rest of this - apparently the clock isn't running now
		}
		i++;
		cpu_relax();
	}

	out_8(&priv->dmac_regs->dmacerq, TDMRX_DMA_CH); //disable DMA requests

	// wait for the current descriptor ACTIVE bit to be clear
	while ((in_be32(&priv->dmac_regs->tcd[TDMRX_DMA_CH].tcd[7])&DMA_TCD7_ACTIVE)) {
		cpu_relax();
	}
	out_8(&priv->dmac_regs->dmacdne, TDMRX_DMA_CH); //clear the DONE bit

	x=in_be32(&priv->dmac_regs->tcd[TDMRX_DMA_CH].tcd[6]);
	a=paddr_to_descrno(priv->rxshared,x);
	// When DMA restarts, it will be one descriptor ahead of the one
	// during which the stop took place
	for (i = 0; i < 8; i++)
		out_be32(&priv->dmac_regs->tcd[TDMRX_DMA_CH].tcd[i],
			 priv->rxshared->descrs_k[a].pd[i]);
	service_rx_ring(priv);

	out_8(&priv->dmac_regs->dmaserq, TDMRX_DMA_CH); // enable DMA requests for receiver
	out_be32(&priv->tdm_regs->rcr, RCR_REN); // Start the receiver

	priv->rxshared->flags |= TDM_SHARED_FLAG_RUNNING;
	spin_unlock_irqrestore(&priv->tdmlock,flags);
}
#endif

static void mpc8xxx_tdm_stop(struct tdm_priv *priv)
{
	/* stop the Tx & Rx */
	out_be32(&priv->tdm_regs->tcr, 0);
	out_be32(&priv->tdm_regs->rcr, 0);

	/* Clear DMA error Enable Request DMAEEIH/L */
	out_8(&priv->dmac_regs->dmaceei, TDMTX_DMA_CH);
	out_8(&priv->dmac_regs->dmaceei, TDMRX_DMA_CH);
	out_8(&priv->dmac_regs->dmacint, TDMRX_DMA_CH);
	out_8(&priv->dmac_regs->dmacint, TDMTX_DMA_CH);

	/* disable the dma request */
	out_8(&priv->dmac_regs->dmacerq, TDMRX_DMA_CH);
	out_8(&priv->dmac_regs->dmacerq, TDMTX_DMA_CH);
}

static int init_tdm(struct tdm_priv *priv)
{
	int i;
	int buf_size;
	dma_addr_t physaddr = 0;
	int ret = 0;


#ifdef LLRX
	// Ethan: Allocate shared Rx structure
	// XXX BT this one doesn't really need dma_alloc_coherent but this is
	// XXX convenient and we do want it to be page-aligned
	// XXX Ethan: ...because it's mapped not DMA-ed (could have just used alloc_pages)
	priv->rxshared = dma_alloc_coherent(NULL, sizeof(struct tdm_shared), &physaddr, GFP_KERNEL);
	if (!priv->rxshared) {
		ret = -ENOMEM;
		goto err_alloc_rxshared;
	}
	printk("Allocated %d bytes for rxshared @ %p (%08x)\n",sizeof(struct tdm_shared),priv->rxshared,physaddr);

	// Ethan: Initialize shared Rx structure
	priv->rxshared->this_p=physaddr;
	priv->rxshared->nds=dparms->rx_descrs;
	priv->rxshared->buffer_physlen=dparms->rx_buflen_max;
	priv->rxshared->buffer_ll_default=dparms->rx_buflen_default;
	priv->rxshared->buffer_ll_min=dparms->rx_buflen_min;
	priv->rxshared->buffer_ll_max=dparms->rx_buflen_max;
	priv->rxshared->buffer_ll=dparms->rx_buflen_default;
	priv->rxshared->flags=0;
	priv->rxshared->error=TDM_ERROR_NONE;
	priv->rxshared->framelen=dparms->rx_framelen;
	priv->rxshared->stridelen=dparms->rx_stridelen;
	priv->rxshared->clocks_per_buffer=((priv->rxshared->buffer_ll_default)/(priv->rxshared->framelen));

	// Ethan: Allocate Rx audio buffers
	buf_size = priv->rxshared->buffer_physlen * priv->rxshared->nds;
	priv->rxshared->buffers_k = dma_alloc_coherent(NULL, buf_size, &(priv->rxshared->buffers_p), GFP_KERNEL);
	if (!priv->rxshared->buffers_k) {
		ret = -ENOMEM;
		goto err_alloc_rxbuffers;
	}
	printk("Allocated %d bytes for rxbuffers @ %p (%08x)\n",buf_size,priv->rxshared->buffers_k,priv->rxshared->buffers_p);

	// Ethan: Allocate Rx DMAC descriptors
	buf_size = sizeof(struct tdm_descr) * priv->rxshared->nds;
	priv->rxshared->descrs_k = dma_alloc_coherent(NULL, buf_size, &(priv->rxshared->descrs_p), GFP_KERNEL);
	if (!priv->rxshared->descrs_k) {
		ret = -ENOMEM;
		goto err_alloc_rxdescrs;
	}
	printk("Allocated %d bytes for rxdescrs @ %p (%08x)\n",buf_size,priv->rxshared->descrs_k,priv->rxshared->descrs_p);

	// Ethan: Initialize Rx DMAC descriptors (for pd[] init see _tcd_init)
	for (i=0;i<priv->rxshared->nds;i++) {
		priv->rxshared->descrs_k[i].self=i;
		priv->rxshared->descrs_k[i].dno=i;
		priv->rxshared->descrs_k[i].dsc=0;
		priv->rxshared->descrs_k[i].buf_ll=priv->rxshared->buffer_ll_default;
		priv->rxshared->descrs_k[i].buf=(i*priv->rxshared->buffer_physlen);
	}
	priv->rxshared->dh_active=priv->rxshared->nds - 1;
	// Jason: dh_consumer must be equal to dh_active at startup,
	// otherwise we won't block on select until there's a lock
	priv->rxshared->dh_consumer=priv->rxshared->dh_active;
	priv->rxshared->flags=0;
#endif //LLRX

	// Ethan: Allocate shared Tx structure
	priv->txshared = dma_alloc_coherent(NULL, sizeof(struct tdm_shared), &physaddr, GFP_KERNEL);
	if (!priv->txshared) {
		ret = -ENOMEM;
		goto err_alloc_txshared;
	}
	printk("Allocated %d bytes for txshared @ %p (%08x)\n",sizeof(struct tdm_shared),priv->txshared,physaddr);

	// Ethan: Initialize shared Tx structure
	priv->txshared->this_p=physaddr;
	priv->txshared->nds=dparms->tx_descrs;
	priv->txshared->buffer_physlen=dparms->tx_buflen_max;
	priv->txshared->buffer_ll=dparms->tx_buflen_default;
	priv->txshared->buffer_ll_min=dparms->tx_buflen_min;
	priv->txshared->buffer_ll_max=dparms->tx_buflen_max;
	priv->txshared->u_cursor=0;
	priv->txshared->u_dno=0;
	priv->txshared->u_ftime=0LL;
	priv->txshared->flags=0;
	priv->txshared->error=TDM_ERROR_NONE;
	priv->txshared->underflows=0;
	priv->txshared->framelen=dparms->tx_framelen;
	priv->txshared->stridelen=dparms->tx_stridelen;
	priv->txshared->clocks_per_buffer=((priv->txshared->buffer_ll)/(priv->txshared->framelen));
	priv->txshared->polltime=0xffffffffffffffffULL;
	priv->txshared->latency=dparms->tx_descrs;

	// Ethan: Allocate Tx audio buffers
	buf_size = priv->txshared->buffer_physlen * (priv->txshared->nds + 1);
	priv->txshared->buffers_k = dma_alloc_coherent(NULL, buf_size, &(priv->txshared->buffers_p), GFP_KERNEL);
	if (!priv->txshared->buffers_k) {
		ret = -ENOMEM;
		goto err_alloc_txbuffers;
	}
	printk("Allocated %d bytes for txbuffers @ %p (%08x)\n",buf_size,priv->txshared->buffers_k,priv->txshared->buffers_p);

	// Ethan: Allocate Tx DMAC descriptors
	buf_size = sizeof(struct tdm_descr) * priv->txshared->nds;
	priv->txshared->descrs_k = dma_alloc_coherent(NULL, buf_size, &(priv->txshared->descrs_p), GFP_KERNEL);
	if (!priv->txshared->descrs_k) {
		ret = -ENOMEM;
		goto err_alloc_txdescrs;
	}
	printk("Allocated %d bytes for txdescrs @ %p (%08x)\n",buf_size,priv->txshared->descrs_k,priv->txshared->descrs_p);

	// Ethan: Initialize Tx DMAC descriptors (for pd[] init see _tcd_init)
	for (i=0;i<priv->txshared->nds;i++) {
		priv->txshared->descrs_k[i].self=i;
		priv->txshared->descrs_k[i].dno=i;
		priv->txshared->descrs_k[i].dsc=(i*priv->txshared->clocks_per_buffer);
		priv->txshared->descrs_k[i].buf_ll=priv->txshared->buffer_ll;
		priv->txshared->descrs_k[i].buf_clocks = priv->txshared->clocks_per_buffer;
		priv->txshared->descrs_k[i].buf=0; //start of the zero buffer
	}
	priv->txsdno=priv->txshared->descrs_k[priv->txshared->nds-1].dno;

	// Ethan: Setup zero template and set all buffers to silence
	memcpy(priv->txshared->zeropat,dparms->tx_zerotemplate,dparms->tx_framelen);
	for (i=0;i<dparms->tx_buflen_max;i+=dparms->tx_framelen) {
		memcpy(((unsigned char *)priv->txshared->buffers_k)+i,dparms->tx_zerotemplate,dparms->tx_framelen);
	}
	priv->txshared->dh_active=priv->txshared->nds - 1;
	priv->txshared->flags=0;

	return 0;

err_alloc_txdescrs:
err_alloc_txbuffers:
err_alloc_txshared:
#ifdef LLRX
err_alloc_rxdescrs:
err_alloc_rxbuffers:
err_alloc_rxshared:
#endif
	printk("A DMA page allocation error occurred\n");

	return ret;
}

static int tdm_open( struct inode *, struct file *);
static int tdm_release( struct inode *, struct file *);
static int mpc8xxx_tdm_ioctl(struct inode *, struct file *, unsigned int, unsigned long);
static unsigned int mpc8xxx_tdm_poll(struct file *, struct poll_table_struct *);
static int mpc8xxx_tdm_mmap(struct file *,struct vm_area_struct *);

struct file_operations tdm_fops =
{
	.open		= tdm_open,
	.release	= tdm_release,
	.ioctl		= mpc8xxx_tdm_ioctl,
	.poll		= mpc8xxx_tdm_poll,
	.mmap		= mpc8xxx_tdm_mmap
};

static int
tdm_open( struct inode *inode, struct file *file)
{
	return 0;
}

static int
tdm_release( struct inode *inode, struct file *file )
{
	struct tdm_priv *priv=mpc8xxx_tdm_priv;
	if (file==priv->freader) priv->freader=0;
	if (file==priv->fwriter) priv->fwriter=0;
	return 0;
}

static int mpc8xxx_tdm_mmap(struct file *file,struct vm_area_struct *vma)
{
	struct tdm_priv *priv=mpc8xxx_tdm_priv;
	u32 x,o,p;
	int r=0,w=0;

	// Ethan: Determine if this is a reader or writer map
	if (file==priv->freader) r=1;
	if (file==priv->fwriter) w=1;

	if ((r==0)&&(w==0)) return -EIO;
	if ((vma->vm_pgoff!=0)&&(vma->vm_pgoff!=2)) {
		printk("vm_pgoff is an unexpected value\n");
		return -EFAULT;
	}

	// Ethan: If the page offset is 2 jump to the other map operations
	if (vma->vm_pgoff==2) goto do_mem;

	/*
	 * Ethan:
	 * Map up the DMAC registers
	 * Uncached because they are HW registers
	 * Userspace should mmap these as read-only
	 */
	x=0x2000;
	p=(priv->pdmac_base)>>PAGE_SHIFT;
	if (remap_pfn_range(vma,vma->vm_start,p,x,pgprot_noncached(vma->vm_page_prot))) goto err;
	return 0;

	// Ethan: All maps beyond this point are presumed to be read-write
do_mem:
	// Ethan: Map up the Rx shared structure (offset 0 pages)
	o=0;
	x=4096; // XXX assuming struct tdm_shared is no larger than a page
#ifdef LLRX
	if (r) {
		p=priv->rxshared->this_p>>PAGE_SHIFT;
		priv->rxshared->this_u=(struct tdm_shared *)(vma->vm_start + o);
		if (remap_pfn_range(vma,vma->vm_start+o,p,x,pgprot_cached(vma->vm_page_prot))) goto err;
	}
#endif
	// Ethan: Map up the Tx shared structure
	o+=x;
	x=4096; // Ethan: 1 page
	if (w) {
		p=priv->txshared->this_p>>PAGE_SHIFT;
		priv->txshared->this_u=(struct tdm_shared *)(vma->vm_start + o);
		if (remap_pfn_range(vma,vma->vm_start+o,p,x,pgprot_cached(vma->vm_page_prot))) goto err;
	}

	// Ethan: Map up the Rx DMA descriptors
	o+=x;
#ifdef LLRX
	x=sizeof(struct tdm_descr) * priv->rxshared->nds;
	x+=4095;
	x&=(~4095); // Ethan: Round up to the nearest page size
	if (r) {
		p=priv->rxshared->descrs_p>>PAGE_SHIFT;
		priv->rxshared->descrs_u=(struct tdm_descr *)(vma->vm_start + o);
		if (remap_pfn_range(vma,vma->vm_start+o,p,x,pgprot_cached(vma->vm_page_prot))) goto err;
	}

	// Ethan: Map up the Rx audio buffers
	o+=x;
	x=priv->rxshared->buffer_physlen * (priv->rxshared->nds + 1);
	x+=4095;
	x&=(~4095); // Ethan: Round up to the nearest page size
	if (r) {
		p=priv->rxshared->buffers_p>>PAGE_SHIFT;
		priv->rxshared->buffers_u=(void *)(vma->vm_start + o);
		if (remap_pfn_range(vma,vma->vm_start+o,p,x,pgprot_cached(vma->vm_page_prot))) goto err;
	}

	// Ethan: Map up the Tx DMA descriptors
	o+=x;
#endif
	x=sizeof(struct tdm_descr) * priv->txshared->nds;
	x+=4095;
	x&=(~4095); // Ethan: Round up to the nearest page size
	if (w) {
		p=priv->txshared->descrs_p>>PAGE_SHIFT;
		priv->txshared->descrs_u=(struct tdm_descr *)(vma->vm_start + o);
		if (remap_pfn_range(vma,vma->vm_start+o,p,x,pgprot_cached(vma->vm_page_prot))) goto err;
	}

	// Ethan: Map up the Tx audio buffers
	o+=x;
	x=priv->txshared->buffer_physlen * (priv->txshared->nds + 1);
	x+=4095;
	x&=(~4095); // Ethan: Round up to the nearest page size
	if (w) {
		p=priv->txshared->buffers_p>>PAGE_SHIFT;
		priv->txshared->buffers_u=(void *)(vma->vm_start + o);
		if (remap_pfn_range(vma,vma->vm_start+o,p,x,pgprot_cached(vma->vm_page_prot))) goto err;
	}
	return 0;
err:
	// XXX BT this is woefully inadequate, there should be a whole
	// XXX unwinding in here
	printk("In tdm_mmap, remap_pfn_range failed\n");
	return -EFAULT;

}

#ifdef LLRX
extern void fpga_EnableSpdif(void);
extern void fpga_CheckSpdif(int);
extern unsigned long fpga_GetSpdifMinPulse(void);
#endif

#define OFNODE ofdev->dev.of_node

static int fsl_8xxx_tdm_probe(struct of_device *ofdev,
			      const struct of_device_id *match)
{
	int ret = 0;
	struct tdm_priv *priv;
	struct resource res;

	/*
	 * The size of tdm_descr must be a multiple of 32 so that when
	 * the DMAC to reads the pd field it's aligned properly.
	 *
	 * If this check fails the build will error out with something like:
	 *	error: negative width in bit-field '<anonymous>'
	 */
	BUILD_BUG_ON(sizeof(struct tdm_descr) % 32);

	printk("fsl_8xxx_tdm_probe\n");
	priv = kmalloc(sizeof(struct tdm_priv), GFP_KERNEL);
	if (!priv) {
		ret = -ENOMEM;
		printk("kmalloc failed\n");
		goto err_alloc;
	}

	mpc8xxx_tdm_priv = priv;
	dev_set_drvdata(&ofdev->dev, priv);
	priv->device = &ofdev->dev;
	priv->txsdno = 0;
	priv->freader = 0;
	priv->fwriter = 0;

	ret = of_address_to_resource(OFNODE, 0, &res);
	if (ret) {
		printk("of_address_to_resource failed\n");
		ret = -EINVAL;
		goto err_resource;
	}
	priv->ptdm_base = res.start;

	priv->tdm_regs = of_iomap(OFNODE, 0);
	if (!priv->tdm_regs) {
		printk("of_iomap 0 failed\n");
		ret = -ENOMEM;
		goto err_tdmregs;
	}

	ret = of_address_to_resource(OFNODE, 1, &res);
	if (ret) {
		printk("of_address_to_resource failed\n");
		ret = -EINVAL;
		goto err_dmacreg;
	}
	priv->pdmac_base = res.start;
	priv->dmac_regs = of_iomap(OFNODE, 1);
	if (!priv->dmac_regs) {
		printk("of_iomap 1 failed\n");
		ret = -ENOMEM;
		goto err_dmacreg;
	}
	printk("DMAC regs @ k%p\n",priv->dmac_regs);

	/* tdmrd tmdtd at immrbar+0x16100 */
	priv->data_regs =
	    (struct tdm_data *)(TDM_DATAREG_OFFSET + (u8 *)priv->tdm_regs);
	/* TDMCLK_DIV_VAL_RX/TX at TDMBASE+0x180 */
	priv->clk_regs =
	    (struct tdm_clock *)(TDM_CLKREG_OFFSET + (u8 *)priv->tdm_regs);

	/* irqs mapping for tdm err, dmac err, dmac done */
	priv->tdm_err_intr = irq_of_parse_and_map(OFNODE, dparms->tdm_irq);
	if (priv->tdm_err_intr == NO_IRQ) {
		printk("can't figure out the TDM error irq\n");
		ret = -EINVAL;
		goto err_tdmerr_irqmap;
	}

	priv->dmac_err_intr = irq_of_parse_and_map(OFNODE, dparms->dmac_err_irq);
	if (priv->dmac_err_intr == NO_IRQ) {
		printk("can't figure out the DMAC error irq\n");
		ret = -EINVAL;
		goto err_dmacerr_irqmap;
	}

	priv->dmac_done_intr = irq_of_parse_and_map(OFNODE, dparms->dmac_irq);
	if (priv->dmac_done_intr == NO_IRQ) {
		printk("can't figure out the DMAC done irq\n");
		ret = -EINVAL;
		goto err_dmacdone_irqmap;
	}

	ret =
	    request_irq(priv->tdm_err_intr, tdm_err_isr, IRQF_SHARED, "tdm_err_isr",
			priv);
	if (ret)
		goto err_tdmerrisr;

	ret =
	    request_irq(priv->dmac_err_intr, dmac_err_isr, IRQF_SHARED, "dmac_err_isr",
			priv);
	if (ret)
		goto err_dmacerrisr;

	ret =
	    request_irq(priv->dmac_done_intr, dmac_done_isr, 0, "dmac_done_isr",
			priv);
	if (ret)
		goto err_dmacdoneisr;

	/* Wait q initilization */
	init_waitqueue_head(&mpc8xxx_tdm_priv->wr_wakeup_event);
#ifdef LLRX
	init_waitqueue_head(&mpc8xxx_tdm_priv->rd_wakeup_event);
#endif

	printk("calling init_tdm\n");
	ret = init_tdm(priv);
	if (ret)
		goto err_tdminit;

	ret = mpc8xxx_tdm_reg_init(priv);
	if (ret)
		goto err_tdminit;

	// Setup procfs now that the tdm shared pointers are valid
	tdm_procfs_setup(priv);

	ret = mpc8xxx_tdm_enable(priv);
	if (ret)
		goto err_tdminit;

	/* Set tx running */
	priv->txshared->flags |= TDM_SHARED_FLAG_RUNNING;

	spin_lock_init(&priv->tdmlock);
	spin_lock(&priv->tdmlock);
	spin_unlock(&priv->tdmlock);
	if (register_chrdev(TDM_MAJOR, TDM_NAME, &tdm_fops)!=0) {
		printk("tdm: register_chrdev failed\n");
	}
#ifdef LLRX
	fpga_EnableSpdif();
	fpga_CheckSpdif(1);
#endif

	return 0;

err_tdminit:
	free_irq(priv->dmac_done_intr, priv);
err_dmacdoneisr:
	free_irq(priv->dmac_err_intr, priv);
err_dmacerrisr:
	free_irq(priv->tdm_err_intr, priv);
err_tdmerrisr:
	irq_dispose_mapping(priv->dmac_done_intr);
err_dmacdone_irqmap:
	irq_dispose_mapping(priv->dmac_err_intr);
err_dmacerr_irqmap:
	irq_dispose_mapping(priv->tdm_err_intr);
err_tdmerr_irqmap:
	iounmap(priv->dmac_regs);
err_dmacreg:
	iounmap(priv->tdm_regs);
err_tdmregs:
err_resource:
	kfree(priv);
err_alloc:
	return ret;
}

static int __devexit fsl_8xxx_tdm_remove(struct of_device *ofdev)
{
	// Ethan: TODO - This code isn't adequate to remove the driver
	struct tdm_priv *priv = dev_get_drvdata(&ofdev->dev);

	mpc8xxx_tdm_stop(priv);

	/* free the irqs and dispose their mapping */
	free_irq(priv->dmac_err_intr, priv);
	free_irq(priv->tdm_err_intr, priv);
	free_irq(priv->dmac_done_intr, priv);
	irq_dispose_mapping(priv->dmac_err_intr);
	irq_dispose_mapping(priv->tdm_err_intr);
	irq_dispose_mapping(priv->dmac_done_intr);
	iounmap(priv->tdm_regs);
	iounmap(priv->dmac_regs);
	dev_set_drvdata(&ofdev->dev, NULL);
	kfree(priv);
	return 0;
}

static int mpc8xxx_tdm_ioctl(struct inode *inode, struct file *file, unsigned int ioctl_num, unsigned long ioctl_param)
{
	int ret = 0;
	int nbytes;
	char par[16];
	int nr = _IOC_NR(ioctl_num);
	struct tdm_priv *priv = mpc8xxx_tdm_priv;

	if ( _IOC_DIR(ioctl_num)&_IOC_WRITE ) {
		nbytes = _IOC_SIZE(ioctl_num);
		if (nbytes > sizeof(par) ) nbytes = sizeof(par);
		if (copy_from_user((unsigned char *)&par, (unsigned char *)ioctl_param, nbytes)) return (-EFAULT);
	}

	switch (nr) {
		case NR_TDM_SETMODE:
			{
#ifdef LLRX
			if (priv->freader && (ioctl_param == TDM_SETMODE_INPUT)) {
				ret= -EIO;
				printk("ASSERT (%s): LLA TDM RX already in use\n", __FUNCTION__);
				break;
			}
#else
			if (ioctl_param == TDM_SETMODE_INPUT) {
				ret= -EIO;
				printk("ASSERT (%s): LLA TDM RX not supported\n", __FUNCTION__);
				break;
			}
#endif
			if (priv->fwriter && (ioctl_param == TDM_SETMODE_OUTPUT)) {
				ret= -EIO;
				printk("ASSERT (%s): LLA TDM TX already in use\n", __FUNCTION__);
				break;
			}
			if (ioctl_param == TDM_SETMODE_INPUT) {
				priv->freader=file;
			}
			else if (ioctl_param == TDM_SETMODE_OUTPUT) {
				priv->fwriter=file;
			}
			else {
				ret = -EPERM;
			}
			break;
			}
		case NR_TDM_GETRATE:
			{
			if (file == priv->fwriter) {
				/* All TX is 44.1k */
				*(unsigned long*)par = 44100;
			}
#ifdef LLRX
			else if (file == priv->freader) {
				/* RX sample rate is read from FPGA */
				*(unsigned long*)par = fpga_GetSpdifMinPulse();
			}
#endif
			else {
				ret = -ENOENT;
			}
			break;
			}
		default:
			return -EPERM;
	}
	if (ret<0) return ret;

	if ( _IOC_DIR(ioctl_num)&_IOC_READ ) {
		nbytes = sizeof(par);
		if (nbytes > _IOC_SIZE(ioctl_num)) nbytes = _IOC_SIZE(ioctl_num);
		if (copy_to_user( (char *)ioctl_param, par, nbytes)) ret = -EFAULT;
	}
	return (ret);
}

static unsigned int
mpc8xxx_tdm_poll( struct file *fp, struct poll_table_struct *pt)
{
	unsigned int mask=0;
	struct tdm_priv *priv = mpc8xxx_tdm_priv;

	// Ethan: Wait until a Tx/Rx window update
	if (pt&&(pt->key&(POLLOUT|POLLWRNORM))) poll_wait(fp,&priv->wr_wakeup_event,pt);
#ifdef LLRX
	if (pt&&(pt->key&(POLLIN|POLLRDNORM))) poll_wait(fp,&priv->rd_wakeup_event,pt);
#endif

	spin_lock_irq(&priv->tdmlock);
#ifdef LLRX
	/*
	 * rxshared->dh_consumer is RX_CON in the library and maintained
	 * there
	 */
	// start of most recently received buffer is at least polltime
	if (priv->rxshared->dh_consumer != priv->rxshared->dh_active) {
		mask |= POLLIN|POLLRDNORM;
	}
#endif
	// Ethan: If the polltime is now within the Tx window
	if ((((signed int)(priv->txsdno - (priv->txshared->nds - priv->txshared->latency))) - ((signed int)priv->txshared->polltime)) >= 0) {
		mask |= POLLOUT|POLLWRNORM;
	}
	spin_unlock_irq(&priv->tdmlock);
	return mask;
}

static struct of_device_id fsl_8xxx_tdm_match[] = {
	{
#if defined(CONFIG_SONOS_LIMELIGHT)
		.compatible = "fsl,mpc8xxx-tdm",
#endif
#ifdef CONFIG_SONOS_FENWAY
		.compatible = "fsl,mpc8315-tdm",
#endif
	},
	{},
};

MODULE_DEVICE_TABLE(of, fsl_8xxx_tdm_match);

static struct of_platform_driver fsl_8xxx_tdm_driver = {
	.driver	= {
		.owner = THIS_MODULE,
		.name = DRV_NAME,
		.of_match_table = fsl_8xxx_tdm_match,
	},
	.probe = fsl_8xxx_tdm_probe,
	.remove = fsl_8xxx_tdm_remove,
};

int fsl_8xxx_tdm_init(void)
{
	pr_info(DRV_NAME ": " DRV_DESC "\n");
	init_devparms();
	return of_register_platform_driver(&fsl_8xxx_tdm_driver);
}

MODULE_LICENSE("GPL");
MODULE_AUTHOR("P.V.Suresh, Freescale Semiconductor");
MODULE_DESCRIPTION("TDM Driver For MPC8xxx");
MODULE_VERSION("1.0");
